/*
 *	Network Queueing System (NQS)
 *  This version of NQS is Copyright (C) 1992  John Roman
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
*  PROJECT:     Network Queueing System
*  AUTHOR:      John Roman
*
*  Modification history:
*
*       Version Who     When            Description
*       -------+-------+---------------+-------------------------
*       V01.10  JRR                     Initial version.
*       V01.20  JRR     16-Jan-1992	Added support for RS6000.
*       V01.3   JRR     12-Feb-1992	Fixed static declaration of newmach.
*	V01.4	JRR	17-Jun-1992	Added header.
*	V01.5	JRR	01-Dec-1992	Modified for C prototypes.
*/
/*++ macnam.c - Network Queueing System
 *
 * $Source: /usr2/jrroma/nqs/nqs-3.30/lib/RCS/macnam.c,v $
 *
 * DESCRIPTION:
 *
 *	This module contains the two functions:  getmacnam()  and
 *	endmacnam() which return the principal name of the machine
 *	for the specified machine-id, and discard the machine-name
 *	cache respectively.
 *
 *
 *	Author:
 *	-------
 *	Brent A. Kingsbury, Sterling Software Incorporated.
 *	August 12, 1985.
 *
 *
 * STANDARDS VIOLATIONS:
 *   None.
 *
 * REVISION HISTORY: ($Revision: 1.5 $ $Date: 1992/12/22 15:46:11 $ $State: Exp $)
 * $Log: macnam.c,v $
 * Revision 1.5  1992/12/22  15:46:11  jrroma
 * Version 3.30
 *
 * Revision 1.4  92/06/18  13:24:08  jrroma
 * Added gnu header
 * 
 * Revision 1.3  92/02/12  12:39:20  jrroma
 * Fixed static declaration of newmach,
 * ,
 * 
 * Revision 1.2  92/01/16  16:46:48  jrroma
 * Added support for RS6000
 * 
 * Revision 1.1  92/01/16  16:08:53  jrroma
 * Initial revision
 * 
 *
 */

#include "nqs.h"
#include <string.h>
#include <malloc.h>

#ifndef __CEXTRACT__
#if __STDC__

static struct m_cache *newmach ( char *name );

#else /* __STDC__ */

static struct m_cache *newmach (/* char *name */);

#endif /* __STDC__ */
#endif /* __CEXTRACT__ */
/*
 *	Configurable parameters.
 */
#define	MAX_M_CACHESIZE	200	/* We will remember this many machine names */


/*
 *	Data structure definitions local to this module.
 */
struct m_cache {
	struct m_cache *prev;	/* Previous machine name cache entry */
	struct m_cache *next;	/* Next machine name cache entry */
	Mid_t mid;		/* Machine-id */
	char *name;		/* Machine principal name for mid */
};


/*
 *	Variables which are global to this module but not public.
 */
static struct m_cache *m_set = (struct m_cache *) 0;
					/* Machine-id/name cache */
static int m_count = 0;			/* # of machine-id/name cache entries */


/*** getmacnam
 *
 *
 *	char *getmacnam():
 *
 *	Return the principal name of the machine whose machine-id
 *	is: mid.
 */
char *getmacnam (mid)
Mid_t mid;				/* Machine-id */
{
	struct m_cache *newmach();	/* Allocate a new machine cache entry */
	register struct m_cache *scan;	/* Current machine cache entry */
	register struct m_cache *prev;	/* Previous machine cache entry */
	register char *name;		/* Machine-name */

	prev = (struct m_cache *) 0;
	scan = m_set;			/* Scan machine cache */
	while (scan != (struct m_cache *) 0 && scan->mid != mid) {
		prev = scan;
		scan = scan->next;
	}
	if (scan == (struct m_cache *) 0) {
		/*
		 *  The machine-id/name was not in the cache.
		 */
		name = fmtmidname (mid);	/* Format name */
		if (m_count < MAX_M_CACHESIZE) scan = newmach (name);
		while (scan == (struct m_cache *) 0 &&
		       prev != (struct m_cache *) 0) {
			/*
			 *  Discard the least recently used mapping and
			 *  try to add the new mapping to the head of
			 *  the mapping set.
			 */
			free (prev->name);	/* Dispose of LRU name part */
			scan = prev;
			prev = prev->prev;	/* Backup one entry */
			free ((char *) scan);	/* Dispose of LRU entry */
			m_count--;		/* One less entry */
			if (prev != (struct m_cache *) 0) {	/* Disconnect */
				prev->next = (struct m_cache *) 0;
			}
			else {			/* No more entries left */
				m_set = (struct m_cache *) 0;
			}
			scan = newmach (name);	/* Try to allocate new entry */
		}
		if (scan == (struct m_cache *) 0) {
			/*
			 *  Insufficient memory existed to add the mapping
			 *  cache.  m_set points to nothing.
			 */
			return (name);
		}
		/*
		 *  Add the new mapping to the head of the mapping cache.
		 */
		if (m_set != (struct m_cache *) 0) m_set->prev = scan;
		scan->prev = (struct m_cache *) 0;
		scan->next = m_set;
		m_set = scan;
		scan->mid = mid;		/* Save machine-id */
		strcpy (scan->name, name);	/* Save machine-name */
	}
	else {
		/*
		 *  The machine-id/name pair has been found in the cache.
		 *  Move the entry to the front of the cache to keep track
		 *  of the least-recently used order of the cache entries.
		 */
		if (scan != m_set) {	/* If not already at the front... */
			if (prev != (struct m_cache *) 0) {
				prev->next = scan->next;
			}
			if (scan->next != (struct m_cache *) 0) {
				scan->next->prev = prev;
			}
			scan->prev = (struct m_cache *) 0;
			scan->next = m_set;
			m_set = scan;
		}
	}
	return (scan->name);	/* Return ptr to machine-name */
}


/*** endmacnam
 *
 *
 *	void endmacnam():
 *	Clear the machine-id/name cache.
 */
void endmacnam ()
{
	register struct m_cache *walk;
	register struct m_cache *next;

	walk = m_set;
	while (walk != (struct m_cache *) 0) {
		next = walk->next;
		free (walk->name);
		free ((char *) walk);
		walk = next;
	}
	m_count = 0;			/* Zero machine-id/name cache entries */
	m_set = (struct m_cache *) 0;
}


/*** newmach
 *
 *
 *	struct m_cache *newmach():
 *	Returns:
 *		A pointer to a new machine-id/name cache entry if
 *		adequate heap space exists; otherwise a null ptr
 *		is returned.
 */
static struct m_cache *newmach (name)
char *name;
{
	register struct m_cache *new;

	if ((new = (struct m_cache *)
		   malloc (sizeof (struct m_cache))) != (struct m_cache *) 0) {
		/*
		 *  We successfully got a new cache entry.
		 *  Now try to allocate sufficient name space.
		 */
		if ((new->name = malloc (strlen (name) + 1)) == (char *) 0) {
			/*
			 *  Insufficient heap space for name.
			 */
			free ((char *) new);
			new = (struct m_cache *) 0;
		}
		else m_count++;	/* One more entry */
	}
	return (new);
}
