// Program     : x10-amh
// Version     : 1.01
// Description : X10 Model CP-290 control program (available at Radio Shack)
//               (computer control for X10 modules)
//               This application allows you to control lights and appliances
//               in your home either directly, or through a crontab
// Author      : Copyright (C) 1995 Aaron Hightower (aaron@paradigmsim.com)
//               1217 Shirley Way
//               Bedford, TX  76022 (817)267-6001
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <strings.h>
#include <fcntl.h>
#include <termios.h>
#include <unistd.h>
#include <time.h>
#include <ctype.h>

#include "x10-amh.h"

#define X10_ON  	2
#define X10_OFF 	3
#define X10_DIM 	5
#define VERSION 	"v1.01"

/*
  Write n bytes from file descriptor, and don't return until done
*/

static int
write_n ( int fd, u_char *buf, int len )
{
  int num, rval;

  for ( num=0; num<len; num+=rval ) {

    if ( ( rval = write( fd, &buf[num], len-num ) ) == -1 ) break;
  }

  return ( num>0 ) ? num : ( -1 );
}

/*
  Read n bytes from file descriptor, and don't return until done
*/

static int
read_n ( int fd, u_char *buf, int len )
{
  int num, rval;

  for ( num=0; num<len; num+=rval ) {

    if ( ( rval = read( fd, &buf[num], len-num ) ) < 0 ) rval = 0;
  }

  return ( num>0 ) ? num : ( -1 );
}

/*
  Constructors
*/

x10::x10( int _fd )
{
  this->fd = _fd;

  config_serial( );
}

x10::~x10( )
{
  close( this->fd );
}

/*
  Wait for ACK from CP-290 - return 0 if wack is ok
*/

int
x10::wreport( )
{
  u_char result[7];

  read_n( this->fd, result, 7 );
  {
    int ok=1;

    for( int i=0; i<6; i++ ) if( result[i] != 0xff ) ok=0;

    if( !ok ) {

      printf( "Leading sync from command upload is corrupt\n" );
      return 1;
    }
    if( !result[6] )
      printf( "*** Warning: the CP-290's memory has been reset" );

    read_n( this->fd, result, 5 );

    if( this->verbose ) {

      printf( "Received report from CP-290 that something has changed\n" );
    }
  }
  return 0;
}

int
x10::wack( )
{
  u_char result[7];

  read_n( this->fd, result, 7 );
  {
    int ok=1;

    for( int i=0; i<6; i++ ) if( result[i] != 0xff ) ok=0;

    if( !ok ) {

      printf("ACK from CP-290 was not received properly.  Fatal error\n" );
      return 1;
    }
    if( !result[6] )
      printf( "*** Warning: the CP-290's memory has been reset" );
  }
  return 0;
}

/*
  Dump out each byte received from the CP-290 - used for debugging
*/

void
x10::dump( )
{
  u_char c;

  while( 1 ) {

    read_n( this->fd, &c, 1 );
    printf( "%d\n", c );
  }
}

/*
  Send 16 bytes of 0xff's then send a message of len bytes to the CP-290
*/

void
x10::msg( u_char *msg, int len )
{
  static u_char sync[16]={ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
                           0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

  if( write_n( this->fd, sync, 16 ) < 16  ) printf( "Error sending sync\n"    );
  if( write_n( this->fd, msg, len ) < len ) printf( "Error sending message\n" );

  tcdrain( this->fd );
}

/*
  Set the base housecode for the unit
*/

void
x10::send_housecode( )
{
  static char hc[2]={ 0, 0 }, answer[2];

  hc[1]=this->hc;

  printf("Resetting your base housecode will erase any memory in your "
         "X10 control unit.\n\n"
         "Are you sure you want to set the base housecode on your "
         "control unit to %c? ", this->hc_char );

  scanf( "%1s", answer );

  if( tolower( answer[0] )=='y' ) msg( hc, 2                  );
  else                            printf( "Operation aborted!\n" );
}

/*
  Return sum of len bytes
*/

u_char
x10::checksum( u_char *bytes, int len )
{
  u_char sum = 0;

  for( int i=0; i<len; i++ ) sum+=bytes[i];

  return sum;
}

/*
  convert string of numbers in the form of "1,2,3,4" to a bitmap appropriate
  for the X10 control module
*/

static u_short
strtobm( char *string )
{
#define ITOBM(i)                (((i)<=16&&(i)>0)?(1<<(16-(i))):(0))
  u_short rval=0, cur=0;

  if( string ) {

    for( int i=0; string[i]; i++ ) {

      if( isdigit( string[i] ) ) cur = cur*10 + string[i] - '0';
      else {

        rval |= ITOBM( cur );
        cur = 0;
      }
    }

    rval |= ITOBM( cur );
  }

  return rval;
}

/*
  Send a direct message to the CP-290
*/

void
x10::direct( u_char command, char *str )
{
#define X10_MKBM_1_8(a)         (((a)&0xff00)>>8)
#define X10_MKBM_9_16(a)        (((a)&0xff))

  static u_char dc[6];
  u_short       bm;

  if( bm = strtobm( str ) ) {

    if( this->verbose ) {

      char *brightness[] = {"Full brightness", "94% brightness",
           "88% brightness", "80% brightness", "three fourths brightness",
           "68% brightness", "62% brightness", "56% brightness", "half bright",
           "43% brightness", "37% brightness", "31% brightness",
           "25% brightness", "18% brightness", "12% brightness",
           "06% brightness", "Totally dark (you might as well turn it off!)" };

      switch( command & 15 ) {

        case X10_DIM:
          printf( "Dimming to %s: ", brightness[( command&0xf0 ) >> 4] );
          break;

        case X10_ON : 
          printf( "Turning on: " );
          break;

        case X10_OFF:
          printf( "Turning off: " );
          break;
      }

      for( int i=0;i<16;i++ ) {
        if( bm & ( ITOBM( i ) ) ) printf( "%d,", i );
      }
      printf( "\b \n" );
    }

    {
      const int cmd = command&15;

      if( cmd != X10_DIM && cmd != X10_ON && cmd != X10_OFF ) {
        printf( "Attempt to perform unknown command!\n" );
        return;
      }
    }

    dc[0] = 1; /* Direct command */
    dc[1] = command;
    dc[2] = this->hc;
    dc[3] = X10_MKBM_9_16( bm );
    dc[4] = X10_MKBM_1_8( bm );
    dc[5] = checksum( dc+1, 4 );

    msg( dc, 6 );
  }
  wack( );
  wreport( );
}

/*
  Configure the serial port that the CP-290 is attached to
*/

void
x10::config_serial( )
{
  static struct termios tios;

  if( ! this->fd ) {

    printf("Must set_fd() before config_serial()");
    return;
  }

  tcgetattr( this->fd, &tios );

  tios.c_iflag &= ~( BRKINT | IGNPAR | PARMRK | INPCK |
                     ISTRIP | INLCR  | IGNCR  | ICRNL | IXON );
  tios.c_iflag |=    IGNBRK | IXOFF;

  tios.c_lflag &= ~( ECHO   | ECHOE  | ECHOK  | ECHONL |
                     ICANON | ISIG   | NOFLSH | TOSTOP );

  tios.c_cflag &= ~(  CSIZE | CSTOPB | HUPCL  | PARENB );
  tios.c_cflag |=    CLOCAL | CREAD  | CS8 ;

  cfsetospeed( &tios, B600 );
  cfsetispeed( &tios, B600 );

  tcsetattr( this->fd, TCSADRAIN, &tios );
}

/*
  Explain a little bit of how to use this app
*/

static void
usage( char *name )
{
  printf( "x10 "VERSION" control program "
         "by Aaron Hightower (aaron@paradigmsim.com)\n"
         "\nUsage:\n"
         "%s: [-v] [-c housecode] [-n list] [-f list] [-d dimlevel,list]\n\n"
         "-v : verbose , -t : self-test, -q : query CP-290's day and time\n"
         "-s : set CP-290's time according to CPU's day and time\n"
         "-c [a-p] : use alternate house code (default a)\n"
         "-n list : turn oN devices in list\n"
         "-f list : turn oFf devices in list\n"
         "-d dimlevel,list : dim devices in list to dimlevel\n"
         "list is a comma separated list of devices, each ranging from 1 to 16"
         "\ndimlevel is an integer from 0 to 15 (0 brightest)\n"
         "\nExamples:\n"
         "\tTurn on devices a1 a2 a3\n"
         "\t%s -n 1,2,3\n\n"
         "\tTurn off devices b1 b2 b3\n"
         "\t%s -c b -f 4,6\n\n"
         "\tdim lights 2 and 3 to 5th brightest setting\n"
         "\t%s -d 5,2,3\n\n"
          , name, name, name, name, name );
  exit( 0 );
}

/*
  Issue the selftest command to the CP-290 and report what it says
*/

void
x10::selftest( )
{
  u_char stest=7, result[7];

  printf( "Performing self-test (This will take about 10 seconds)\n" );
  msg( &stest, 1 );
  read_n( this->fd, result, 7 );

  {
    int i, ok = 1;

    for( i=0; i<6; i++ ) if( result[i] != 0xff ) ok = 0;
    if( result[6]!=0 ) ok = 0;

    if( ok ) printf( "Everything is okay!\n" );
    else     printf( "Sorry, something is wrong\n" );
  }
}

/*
  Set the time of the CP-290 based on the time of the CPU
*/

void
x10::settime( )
{
  char  *days[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
  u_char dow[7] = { 64, 1, 2, 4, 8, 16, 32 };
  u_char set[5] = { 2, 30, 7, 1, 0 };
  struct tm *tp;
  time_t secs;

  secs = time( NULL );
  tp = localtime( &secs );

  set[2] = tp->tm_hour;
  set[1] = tp->tm_min;
  set[3] = dow[tp->tm_wday];

  printf( "Setting time to %s %d:%02d\n", days[tp->tm_wday], set[2], set[1] );
  set[4] = checksum( set+1, 3 );
  msg( set, 5 );
}

/*
  Find out what the time of the CP-290 is currently set to
*/

void
x10::gettime( )
{
  u_char ask=4, result[12], *day;

  msg( &ask,    1 );
  read_n( this->fd, result, 12 );

  switch( result[9] ) {

    case 64: day = "Sun";break;
    case 32: day = "Sat";break;
    case 16: day = "Fri";break;
    case  8: day = "Thu";break;
    case  4: day = "Wed";break;
    case  2: day = "Tue";break;
    case  1: day = "Mon";break;
    default: day = "???";break;
  }

  printf( "The CP-290's internal clock is set to: %s, %d:%02d\n",
    day, result[8], result[7] );
}

/*
  List out the events that have been set in the CP-290 memory
*/

void
x10::getevents( )
{
  int    i, newline=0;
  u_char result[8],
         ask=5;

  msg( &ask, 1   );
  read_n( this->fd, result, 7 );

  for( i=0; i<6; i++ ) if( result[i] != 255 ) {

    printf( "Did not receive sync from CP-290\n" );
    return;
  }
  if( result[6] == 0 ) {

    printf( "Warning!  CP-290 has been reset.  All memory has been lost\n" );
  }


  for( i=0; i<128; i++ ) {

    read_n( this->fd, result, 1 );

    if( result[0] != 255 ) {

      read_n( this->fd, result+1, 7 );

      {
	int mode, daymap, hour, minute, devbm, housecode, level, function;

	mode      = result[0] & 0x0f;
	daymap    = result[1] & 0x7f;
	hour      = result[2] & 0x1f;
	minute    = result[3] & 0x3f;
	devbm     = ( ( ( int )result[4] ) << 8 ) + ( ( int ) result[5] );
	housecode = result[6] >> 4;
	level     = result[7] >> 4;
	function  = result[7] & 0x0f;

	if( newline ) printf( "\n" );

	printf( "mode=%d, daymap=0x%x, time=%d:%02d, devs=0x%x, "
	        "hcode=%d, level=%d, function=%d\n",
	         mode, daymap, hour, minute, devbm,
	         housecode, level, function );

        newline = 0;
      }
    }
    else {
      static int numdots = 0;

      printf( "." );

      if( ++numdots > 10 ) {

        printf( "\n" );
        numdots = 0;
        newline = 0;
      }
      else newline = 1;
    }
  }

  printf( "\n" );
}

/*
  Turn on verbose mode printing
*/

void
x10::verbose_on( )
{
  this->verbose = 1;
}

/*
  Change which file descriptor to use for direct commands, etc
*/

void
x10::fd( int _fd )
{
  this->fd = _fd;
}

/*
  Change which housecode to use for direct commands, etc
*/

void
x10::housecode( int c )
{
  static char *housecodes = "meckogainfdlphbj";

  c = tolower( c );

  if( c<'a' || c>'p' ) c = 'a';

  this->hc_char = c;
  this->hc = 
    ( ( strlen( housecodes ) - strlen( index( housecodes, c ) ) ) <<4 );
}

/*
  Main
*/

int
main( int argc, char * const argv[] )
{
  if( argc <= 1 ) {usage( argv[0] );exit( 0 );}

  char *portname = ( char * )getenv( "X10_PORTNAME" );

  if( !portname ) portname = "/dev/x10";

  int fd;

  if( ( fd = open( portname, O_RDWR|O_NOCTTY, 0 ) ) == -1 ) {

    printf( "Could not access serial port %s\nPlease set environment variable "
           "X10_PORTNAME to proper serial device\n"
           "or else make the appropriate soft link as superuser.\n", portname );
    exit( -1 );
  }

  x10 *x = new x10( fd );

  x->housecode('a');

  int c;

  while( ( c = getopt( argc, argv, "z:d:evhc:n:f:qst" ) ) != -1 ) {

    switch( c ) {

      case 'z':
        x->housecode( optarg?( *optarg ):'a' );
        x->send_housecode( );
        break;

      case 'v':
        x->verbose_on( );
        break;

      case 'c':
        x->housecode( optarg ? ( *optarg ) : 'a' );
        break;

      case 'd':
        if( optarg ) { /* First number is the dimlevel (0..15) */
	  int dimlevel;
	  char *dim;

          dim = strdup( optarg );
          sscanf( optarg, "%d,%s", &dimlevel, dim );
          x->direct( X10_DIM + ( ( dimlevel & 0xf ) << 4 ), dim );
        }
        break;

      case 'e':
        x->getevents( );
        break;

      case 'n':
        x->direct( X10_ON, optarg );
        break;

      case 'f':
        x->direct( X10_OFF, optarg );
        break;

      case 't':
        x->selftest( );
        break;

      case 's':
        x->settime( );
        x->gettime( );
        break;

      case 'q':
        x->gettime( );
        break;

      case 'h':
      case '?':
        usage( argv[0] );
        break;
    }
  }

  exit( 0 );

  return 0;
}
