/*
 * NAME
 *   GetSelectedArticlesInGroup.c
 * USAGE
 *   Usage: GetSelectedArticlesInGroup Group
 * DESCRIPTION
 *   Retrieve the selected articles from the news server. 
 * INPUT
 *   The article numbers of the selected articles are read from
 *   standard input. One article number per line.
 * OUTPUT
 *   Every article is stored in a separate file in the directory
 *   $SKIMDIR/Articles. The file name is GroupName.ArticleNumber.
 * COPYRIGHT
 *   Skim - Off-line news reading package optimized for slow lines.
 *   Copyright (C) 1995  Rene W.J. Pijlman
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 * 
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * VERSION
 *   Skim version 0.6.
 */

#include <unistd.h>
#include <string.h>

#include "VarBuf.h"
#include "Skim.h"

FILE_ID("/home/rene/sys/CVS_MasterSourceRepository/skim/GetSelectedArticlesInGroup.c,v 1.10 1995/08/15 19:33:35 rpijlman Exp");


int main( int argc, char * argv[] )
{
    VarBuf ArticleNumber = VBCreate();
    VarBuf ArticleFileName = VBCreate();
    VarBuf IndexFileName = VBCreate();
    FILE * Index = NULL;
    FILE * NewsServer = NULL;
    FILE * ArticleFile;

    if ( argc != 2 )
    {
        fprintf( stderr, "%s\n",
                 "Usage: GetSelectedArticlesInGroup Group" );
        exit( EXIT_FAILURE );
    }

    VBPrintf( IndexFileName, "%s/Indexes/Articles/%s", SkimDir(), argv[1] );

    while ( VBReadLine( ArticleNumber, stdin, WITHOUT_NEWLINE ) )
    {
	if ( NewsServer == NULL )
	{
	    NewsServer = NNTPStreamOpen();

	    /* Make group current. */
	    if ( fprintf( NewsServer, "group %s\r\n", argv[1] ) == EOF ||
		 fflush( NewsServer ) == EOF )
	    {
		fprintf( stderr, "Error writing to NewsServer.\n" );
		exit( EXIT_FAILURE );
	    }

	    CheckStatusResponse( NewsServer, argv[1], "2" );
	}

	if ( VBSize( ArticleNumber ) == 0 ||
	     atoi( VBAsString(ArticleNumber) ) < 1 )
	{
	    fprintf( stderr, "Invalid article number: '%s'\n", 
	             VBAsString(ArticleNumber) );
	    exit( EXIT_FAILURE );
	}

	VBPrintf( ArticleFileName, "%s/Articles/%s.%V",
	          SkimDir(), argv[1], ArticleNumber );

	/* Request an article. */
	if ( fprintf( NewsServer, "article %s\r\n",
	              VBAsString(ArticleNumber) ) == EOF ||
	     fflush( NewsServer) == EOF )
        {
	    fprintf( stderr, "Error writing to NewsServer.\n" );
	    exit( EXIT_FAILURE );
        }

	/* Check status response. */
	CheckStatusResponse( NewsServer, argv[1], "2" );

	if ( ( ArticleFile =
	       fopen( VBAsString(ArticleFileName), "w" ) ) == NULL )
	{
	    fprintf( stderr, "Cannot open article file %s\n", 
	             VBAsString(ArticleFileName) );
	    exit( EXIT_FAILURE );
	}

	/* Write the article in a file. */
	TextResponseToFile( NewsServer, ArticleFile );

	if ( fclose(ArticleFile) == EOF )
	{
	    fprintf(stderr, "Error closing %s\n", VBAsString(ArticleFileName));
	    exit( EXIT_FAILURE );
	}

        if ( Index == NULL )
        {
	    Index = fopen( VBAsString(IndexFileName), "a" );
	    if ( Index == NULL )
	    {
		fprintf( stderr, "Cannot open index file %s\n", 
			 VBAsString(IndexFileName) );
		exit( EXIT_FAILURE );
	    }
	}

	AddArticleToIndex( ArticleFileName, ArticleNumber, Index );

	VBReset( ArticleFileName );
	VBReset( ArticleNumber );
    }

    if ( Index != NULL && fclose(Index) == EOF )
    {
	fprintf(stderr, "Error closing %s\n", VBAsString(IndexFileName));
	exit( EXIT_FAILURE );
    }

    NNTPStreamClose( NewsServer );

    VBDestroy(ArticleNumber);
    VBDestroy(ArticleFileName);
    VBDestroy(IndexFileName);

    return EXIT_SUCCESS;
}
