#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/un.h>

#include "runpipe.h"
#include "config.h"




/* Inspect a filename, make sure it's my pipe */
int ismypipe(const char *filename, uid_t myuid)
{
  struct stat pipestats;

  if (stat(filename, &pipestats) || !S_ISFIFO(pipestats.st_mode) ||
      (myuid && (myuid != pipestats.st_uid))) {
    return FALSE;
  }
  return TRUE;
}

/* Inspect a filename, make sure it's my socket */
int ismysocket(const char *filename, uid_t myuid)
{
  struct stat sockstats;

  if (stat(filename, &sockstats) || !S_ISSOCK(sockstats.st_mode) ||
      (myuid && (myuid != sockstats.st_uid))) {
    return FALSE;
  }
  return TRUE;
}


int ismydir(const char *filename, uid_t myuid)
{
  struct stat dirstats;

  if (stat(filename, &dirstats) || !S_ISDIR(dirstats.st_mode) ||
      (myuid && (myuid != dirstats.st_uid))) {
    return FALSE;
  }
  return TRUE;
}


/* Generate the path to the socket. If called by the daemon, and the socket
   permissions are not respected, then also create the subdirectory to
   hide the socket.							      */
void gensockpath(char *socketpath, int calledby)
{
  char pathname[MAXPATHLEN+1];
#ifdef USERMODE
  char *homeptr;
#endif
#ifdef NOSOCKETPERMS
  char *strpname;
#endif  

#ifdef USERMODE	      /* Manufacture path name of socket */

  if (strlen(USERPATH) > MAXPATHLEN) {
    fprintf(stderr, "Error: Configured user path name is too long.\n");
    exit(BADUSAGE);
  }
  strcpy(pathname, USERPATH);
  if (pathname[0] != '/') {   /* Relative path, get $HOME */
    if (((homeptr = getenv(HOMEENV)) == NULL) || homeptr[0] != '/') {
      fprintf(stderr, "Error: unable to resolve your $%s for default socket path.\n", HOMEENV);
      fprintf(stderr, "Please supply a complete path name for the socket in the configuration\n");
      fprintf(stderr, "and rebuild the package.\n");
      exit(BADUSAGE);
    }
    if (strlen(homeptr) > MAXPATHLEN - 1) {
      fprintf(stderr, "Error: $%s is too long: %s\n", HOMEENV, homeptr);
      exit(BADUSAGE);
    }
    strcpy(pathname, homeptr);
    strcat(pathname, "/");
  } else {
    pathname[0] = 0;
  }

  if (strlen(pathname) + strlen(USERPATH) + 1 > MAXPATHLEN) {
    fprintf(stderr, "Error: socket path %s%s is too long\n", pathname, USERPATH);
    exit(BADUSAGE);
  }

  strcat(pathname, USERPATH);

/* If the socket permissions aren't respected, hide the socket in a
   subdirectory								  */
#ifdef NOSOCKETPERMS
  if (calledby == DAEMON && !ismydir(pathname, getuid()) && mkdir(pathname, 0700)) {
    fprintf(stderr, "Error creating directory to hide the socket.\n");
    exit(BADUSAGE);
  }
  if ((strpname = strrchr(USERPATH, '/')) == NULL) {
    if (strlen(pathname) + strlen(USERPATH) + 1 > MAXPATHLEN) {
      fprintf(stderr, "Error: socket path %s/%s is too long\n", pathname, USERPATH);
      exit(BADUSAGE);
    }
    strcat(pathname, "/");
    strcat(pathname, USERPATH);
  } else {
    strpname++;
    if (strlen(pathname) + strlen(strpname) + 1 > MAXPATHLEN) {
      fprintf(stderr, "Error: socket path %s/%s is too long\n", pathname, USERPATH);
      exit(BADUSAGE);
    }
    strcat(pathname, "/");
    strcat(pathname, strpname);
  }  
#endif
  
#else
  if (strlen(SYSTEMPATH) > MAXPATHLEN) {
    fprintf(stderr, "Error: socket path %s is too long\n", pathname);
    exit(BADUSAGE);
  }

  strcat(pathname, SYSTEMPATH);
  if (pathname[0] != '/') {
    fprintf(stderr, "Error: the configured System path must be a complete path name.\n");
    exit(BADUSAGE);
  }

/* If the socket permissions aren't respected, hide the socket in a
   subdirectory								  */
#ifdef NOSOCKETPERMS
  if (calledby == DAEMON && !ismydir(pathname, getuid()) && mkdir(pathname, 0770)) {
    fprintf(stderr, "Error creating directory to hide the socket.\n");
    exit(BADUSAGE);
  }
  if ((strpname = strrchr(SYSTEMPATH, '/')) == NULL) {
    if (strlen(pathname) + strlen(SYSTEMPATH) + 1 > MAXPATHLEN) {
      fprintf(stderr, "Error: socket path %s/%s is too long\n", pathname, SYSTEMPATH);
      exit(BADUSAGE);
    }
    strcat(pathname, "/");
    strcat(pathname, SYSTEMPATH);
  } else {
    strpname++;
    if (strlen(pathname) + strlen(strpname) + 1 > MAXPATHLEN) {
      fprintf(stderr, "Error: socket path %s/%s is too long\n", pathname, SYSTEMPATH);
      exit(BADUSAGE);
    }
    strcat(pathname, "/");
    strcat(pathname, strpname);
  }  
#endif
  
#endif



  if (realpath(pathname, socketpath) == NULL && errno != ENOENT) {
    fprintf(stderr, "Error: failed to expand socket path at %s\n", socketpath);
    exit(BADUSAGE);
  }

  if (strlen(socketpath) > UNIX_PATH_MAX) {
    fprintf(stderr, "Error. Cannot bind to a path name more than %d characters long.\n", UNIX_PATH_MAX);
    fprintf(stderr, "Tried to open path %s\n", socketpath);
    exit(BADUSAGE);
  }
}
