/*  backupsd.c -- Simple Daemon to catch power failure signals from a
 *                Back-UPS (from APC).
 * 
 *  Parts of the code are from Miquel van Smoorenburg's powerd.c
 *  Other parts are original from Christian Holtje <docwhat@uiuc.edu>
 *  I believe that it is okay to say that this is Public Domain, just
 *  give credit, where credit is due.
 *
 *  Disclaimer:  We make NO claims to this software, and take no
 *               resposibility for it's use/misuse.
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>

/* This is the file needed by SysVInit */
#define PWRSTAT         "/etc/powerstatus"
#define PWRFAIL		"/etc/powerfail"

void powerfail(int fail);

/* Main program. */
int main(int argc, char **argv)
{
  int fd;
  int killpwr_bit = TIOCM_RTS;
  int flags;
  int status, oldstat = -1;
  int count = 0;

  if (argc < 2) {
        fprintf(stderr, "Usage: %s <device> [killpower]\n", argv[0]);
        exit(1);
  }

  /* Open the the device */
  if ((fd = open(argv[1], O_RDWR | O_NDELAY)) < 0) {
        fprintf(stderr, "%s: %s: %s\n", argv[0], argv[1], sys_errlist[errno]);
        exit(1);
  }

  if ( argc >= 3  && (strcmp(argv[2], "killpower")==0) )
      {
	  /* Let's kill the power! */
	  fprintf(stderr, "%s: Attempting to kill the power!\n",argv[0] );
	  ioctl(fd, TIOCMBIS, &killpwr_bit); 
	  /* Hmmm..... If you have a power outtage, you won't make it! */
	  exit(0);
      }
  else
      /* Since we don't want to kill the power, clear the RTS. (killpwr_bit) */
      ioctl(fd, TIOCMBIC, &killpwr_bit); 

#ifndef DEBUG
/* Become a daemon. */
  switch(fork()) {
  case 0: /* I am the child. */
                setsid();
                break;
  case -1: /* Failed to become daemon. */
                fprintf(stderr, "%s: can't fork.\n", argv[0]);
                exit(1);
  default: /* I am the parent. */
                exit(0);
  }
#endif

  /* Now sample the DCD line. */
  while(1) {
      ioctl(fd, TIOCMGET, &flags);
      status = (flags & TIOCM_CD); 
      /* Did DCD jumps to high? Then the power has failed. */
      if (oldstat == 0 && status != 0) {
	  count++;
	  if (count > 3)
	  {
		make_file(PWRFAIL);
	 	powerfail(0);
	  }
	  else { sleep(1); continue; }
      }
      /* Did DCD go down again? Then the power is back. */
      if (oldstat > 0 && status == 0) {
	  count++;
	  if (count > 3)
	  {
		remove_file(PWRFAIL);
	  	powerfail(1);
	  }
	  else { sleep(1); continue; }
      }
      /* Reset count, remember status and sleep 2 seconds. */
      count = 0;
      oldstat = status;
      sleep(2);
  }
  /* Error! (shouldn't happen) */
  return(1);
}


/* Tell init the power has either gone or is back. */
void powerfail(ok)
int ok;
{
  int fd;

  /* Create an info file needed by init to shutdown/cancel shutdown */
  unlink(PWRSTAT);
  if ((fd = open(PWRSTAT, O_CREAT|O_WRONLY, 0644)) >= 0) {
        if (ok)
                write(fd, "OK\n", 3);
        else
                write(fd, "FAIL\n", 5);
        close(fd);
  }
  kill(1, SIGPWR);
}

int
make_file(char *s)
{
	int fd;
	if ( (fd = open(s,O_CREAT|O_WRONLY, 0644)) >= 0)
		close(fd);
	return fd;
}
int
remove_file(char *s)
{
	unlink(s);
}
