/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module:  tools.c                                                      */
/* Author:  Matthias Hanisch & Peter Rupp                                */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/

#include <strings.h>
#include <stdlib.h>
#include <unistd.h>
#include <dirent.h>
#include <ctype.h>
#include <sys/stat.h>
#include <ntape.h>


/*************************************************************************/
/* Name:     backup_database                                             */
/* Purpose:  makes security copies from the database files               */
/*************************************************************************/
int backup_database(int typ)
{
    FILE *fp1,*fp2;
    char destname[MAXLEN];
    int c;

    print_footer("Making security copy ...");
    if (typ == BACKUP_TAPE) /* Tape */
    {
	strcpy(destname,tape_db);
	strcat(destname,"~");
	if ((fp1 = fopen(tape_db,"r")) == NULL)
	{
	    print_footer("An error has occurred. No security copy made!");
	    return(-1);
	}
	if ((fp2 = fopen(destname,"w")) == NULL)
	{
	    print_footer("An error has occurred. No security copy made!");
	    fclose(fp1);
	    return(-1);
	}
    }
    else if (typ == BACKUP_ARCHIVE) /* Archive */
    {
	strcpy(destname,archive_db);
	strcat(destname,"~");
	if ((fp1 = fopen(archive_db, "r")) == NULL)
	{
	    print_footer("An error has occurred. No security copy made!");
	    return (-1);
	}
	if ((fp2 = fopen(destname, "w")) == NULL)
	{
	    print_footer("An error has occurred. No security copy made!");
	    fclose(fp1);
	    return (-1);
	}
	strcat(destname,"~");
    }
    else if (typ == BACKUP_ARCHIVE_RVS)	/* Archive back */
    {
	strcpy(destname, archive_db);
	strcat(destname, "~");
	if ((fp1 = fopen(destname, "r")) == NULL)
	{
	    print_footer("A bad error has occurred. Database is damaged!");
	    return (-1);
	}
	if ((fp2 = fopen(archive_db, "w")) == NULL)
	{
	    print_footer("A bad error has occurred. Database is damaged!");
	    fclose(fp1);
	    return (-1);
	}
    }
    else
	return (-1);
    while ((c = getc(fp1)) != EOF)
	putc(c, fp2);
    fclose(fp1);
    fclose(fp2);
    print_footer("");
    return (1);
}      


/*************************************************************************/
/* Name:     read_header                                                 */
/* Purpose:  Reads the ntape id from the tape header                     */
/*************************************************************************/
int read_header()
{
    FILE *tp;
    char cmd[MAXLEN];

    print_footer("Rewind tape ...");
    if (offline())
    {
	print_footer("Tape is offline. No active tape set!");
	return(0);
    }
    if (do_tape_command("rewind",0) < 0)
    {
	print_footer("Error on rewinding tape!");
	return(-1);
    }
    print_footer("Reading tape header ...");
    if ((tp = fopen(tapedevice,"r")) == NULL)
    {
	print_footer("Error on accessing tape!");
	return(-1);
    }
    if (fgets(cmd, MAXLEN-1, tp) == NULL)
    {
	print_footer("Sorry! No valid tape!");
	fclose(tp);
	return(0);
    }
    if (strcmp(cmd, NTAPEHEADER))
    {
	print_footer("Sorry! No valid tape!");
	fclose(tp);
	return(0);
    }
    if (fgets(cmd, MAXLEN-1, tp) == NULL)
    {
	print_footer("Sorry! No valid tape!");
	fclose(tp);
	return(0);
    }
    fclose(tp);
    if (atoi(cmd) <= 0)
    {
	print_footer("Sorry! No valid tape!");
	return(0);
    }
    return(atoi(cmd));
}


/*************************************************************************/
/* Name:     get_pwd                                                     */
/* Purpose:  Get current path                                            */
/*************************************************************************/
void get_pwd(char *path)
{
    getcwd(path,MAXLEN - 1);

#ifdef DEBUG
    fprintf(stderr,"get_pwd: %s\n",path);
#endif

    if (path == NULL)
	print_footer("Error on finding path!");
}


/*************************************************************************/
/* Name:     get_dir_size                                                */
/* Purpose:  Calculates the file sizes recursively in directory dir      */
/*************************************************************************/
int get_dir_size(char *dir)
{
    DIR *dirp;
    struct dirent *entry;
    struct stat pstat;
    char newdir[MAXLEN];
    int total_size = 0;

#ifdef DEBUG
    fprintf(stderr,"Entering directory %s ...\n",dir);
#endif
    if ((dirp = opendir(dir)) == NULL)
    {
#ifdef DEBUG
	fprintf(stderr,"Opendir returns NULL...\n");
#endif
	return(0);
    }
    
    while ((entry = readdir(dirp)) != NULL)
    {
	if ((!strcmp(entry->d_name,".")) || (!strcmp(entry->d_name,"..")))
	    continue;
	strcpy(newdir, dir);
	strcat(newdir, "/");
	strcat(newdir, entry->d_name);

#ifdef DEBUG
	fprintf(stderr,"Calculating file size of %s...\n",newdir);
#endif

	if (lstat(newdir,&pstat) < 0)
	    continue;
	total_size += pstat.st_size;
	if (S_ISLNK(pstat.st_mode))
	    continue;
	if (S_ISDIR(pstat.st_mode))
	    total_size += get_dir_size(newdir);
    }
    closedir(dirp);
#ifdef DEBUG
    fprintf(stderr,"Leaving directory %s...\n",dir);
#endif
    return(total_size);
}


/*************************************************************************/
/* Name:     change_info                                                 */
/* Purpose:  applies the info changes in database                        */
/*************************************************************************/
int change_info(int change_what,char *name,long tapeno,long position,
		char *platf,char *categ,char *desc)
{
    char line_name[MAXLEN],line_tape[MAXLEN],line_pos[MAXLEN],line[MAXLEN];
    FILE *fp1,*fp2;
    int i,addinfo = 0; 
    
    /* Prepare name for database */
    for (i = strlen(name) - 1; name[i] == '\t' || name[i] == ' '; i--)
	name[i] = '\0';
    for (i = 0; i < strlen(name); i++)
	name[i] = toupper(name[i]);

    if (backup_database(BACKUP_ARCHIVE) < 0)
	return(-1);
  
    sprintf(line,"%s~",archive_db);
    if ((fp1 = fopen(line,"r")) == NULL)
	goto error1;
    if ((fp2 = fopen(archive_db,"w")) == NULL)
	goto error2;

    if (!check_database_version(ARCHIVE_DATABASE,fp1,fp2))
	goto error;
    
    while (fgets(line_name, MAXLEN-1, fp1) != NULL)
    {
	line_name[strlen(line_name)-1] = '\0';

	if (fgets(line_tape, MAXLEN-1,fp1) == NULL)
	    goto error;

	if (fgets(line, MAXLEN-1,fp1) == NULL)     /* size */
	    goto error;
	
	if (fgets(line_pos, MAXLEN-1,fp1) == NULL)
	    goto error;
	      
	if ((atoi(line_tape) == tapeno) && (atoi(line_pos) == position) &&
	    (change_what == CHANGE_NAME_DESC_PF_CAT))
	    fprintf(fp2,"%s\n",name);
	else
	    fprintf(fp2,"%s\n",line_name);

	fprintf(fp2,"%s",line_tape);
	fprintf(fp2,"%s",line);                    /* size */
	fprintf(fp2,"%s",line_pos);

	if (fgets(line, MAXLEN-1, fp1) == NULL)    /* addinfo */
	    goto error;
	if (change_what == CHANGE_ADDINFO_COMMENT)
	    addinfo = atoi(line) | DELETED;
	else if (change_what == CHANGE_ADDINFO_SAVED)
	    addinfo = atoi(line) | CONTENTS_SAVED;
	else if (change_what == CHANGE_ADDINFO_NOTSAVED)
	    addinfo = atoi(line) & ~CONTENTS_SAVED;
	if ((atoi(line_tape) == tapeno) && (atoi(line_pos) == position) &&
	    (change_what > CHANGE_ADDINFO))
	    fprintf(fp2,"%d\n",addinfo);
	else
	    fprintf(fp2,"%s",line);

	if (fgets(line, MAXLEN-1, fp1) == NULL)    /* pf */
	    goto error;
	if ((atoi(line_tape) == tapeno) && (atoi(line_pos) == position) &&
	    (change_what == CHANGE_NAME_DESC_PF_CAT))
	    fprintf(fp2,"%s\n",platf);
	else
	    fprintf(fp2,"%s",line);

	if (fgets(line, MAXLEN-1, fp1) == NULL)    /* category */
	    goto error;
        if ((atoi(line_tape) == tapeno) && (atoi(line_pos) == position) &&
            (change_what == CHANGE_NAME_DESC_PF_CAT))
            fprintf(fp2,"%s\n",categ);
        else
	    fprintf(fp2,"%s",line);

	if (fgets(line, MAXLEN-1, fp1) == NULL)    /* date */
	    goto error;
	fprintf(fp2,"%s",line);
	if (fgets(line, MAXLEN-1, fp1) == NULL)    /* desc */
	    goto error;
	if ((atoi(line_tape) == tapeno) && (atoi(line_pos) == position) &&
	    (change_what == CHANGE_NAME_DESC_PF_CAT))
	    fprintf(fp2,"%s\n",desc);
	else
	    fprintf(fp2,"%s",line);
    }
    fclose(fp1);
    fclose(fp2);
    return(1);

error: fclose(fp2);
    MessageBox("Error 1. Look in the man pages for more info!",1,
	       OKCANCEL,0);
    
error2: fclose(fp1);
    
error1: print_footer("An error has occurred! Nothing changed!");
    backup_database(BACKUP_ARCHIVE_RVS);
    return(-1);
}


/*************************************************************************/
/* Name:     check_database_version                                      */
/* Purpose:  checks if you have the right archive database for the       */
/*           right ntape                                                 */
/*************************************************************************/
int check_database_version(int which_database,FILE *fp1,FILE *fp2)
{
    int i;
    char line[MAXLEN];
    
    if (fgets(line, MAXLEN-1, fp1) != NULL)
    {
	line[strlen(line) - 1] = '\0';
#ifdef DEBUG
	fprintf(stderr,"check_database_version: %s\n",line);
#endif
	for (i = 0; i < strlen(line); i++)
	    if (line[i] == 'V')
		break;
	if (i < strlen(line))
	    if (((which_database == ARCHIVE_DATABASE) &&
		(!strncmp(&line[i+1],ARCHIVE_VERSION,strlen(ARCHIVE_VERSION)))) ||
		((which_database == TAPE_DATABASE) &&
		(!strncmp(&line[i+1],TAPE_VERSION,strlen(TAPE_VERSION)))))
	    {
		if (fp2 != NULL)
		    fprintf(fp2,"%s\n",line);
		return(1);
	    }
    }
    print_footer("Error in database! You may have an obsolete version...");
    return(0);
}


/*************************************************************************/
/* Name:     get_archive_entry                                           */
/* Purpose:  returns one entry of the archive database in a struct       */
/*************************************************************************/
int get_archive_entry(fp,pentry)
    FILE *fp;
    file_entry *pentry;
{
    char line[MAXLEN];
    
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* name */
	return(0);
    line[strlen(line) - 1] = '\0';
    strncpy(pentry->name, line, ARCHIVENAMELENGTH);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* tape id */
	goto serious_error;
    line[strlen(line) - 1] = '\0';
    pentry->tapeno = atol(line);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* size */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    pentry->size = atol(line);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* position */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    pentry->position = atol(line);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* additional info */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    pentry->add_info = atoi(line);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* platform */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    strncpy(pentry->pf,line,PFLENGTH);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* category */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    strncpy(pentry->category,line,CATEGORYLENGTH);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* date */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    strncpy(pentry->date,line,DATELENGTH);
    if (fgets(line, MAXLEN - 1, fp) == NULL)        /* description */
        goto serious_error;
    line[strlen(line) - 1] = '\0';
    strncpy(pentry->description,line,DESCRIPTIONLENGTH);
    
    return(1);
    
serious_error:
    print_footer("Serious error in database! Exiting...");
    nta_exit(-1);
    return(0);
}

    
/*************************************************************************/
/* Name:     strmatch                                                    */
/* Purpose:  Returns 1 if needle is a substring of haystack. It          */
/*           supports the wildcards '*' and '?' in needle                */
/*************************************************************************/
int strmatch(haystack, needle)
    char *haystack,*needle;
{
    int i,j;
    
    if (!strlen(haystack))
	return(1);          /* per definition */
    
    for (i = j = 0; i < strlen(needle) && j < strlen(haystack); j++)
    {
	switch(needle[i])
	{
	case '*':
	    if (needle[i+1] == '\0')
		return(1);
	    if (strmatch(&haystack[j],&needle[i+1]))
		return(1);
	    break;
	case '?':
	    i++;
	    break;
	default:
	    if (haystack[j] == needle[i])
		i++;
	    else
		i = 0;
	}
    }
    
    if (i == strlen(needle))
	return(1);
    else
	return(0);
}
    
/*************************************************************************/
/* Copyright (C) 1994,1995 Matthias Hanisch & Peter Rupp, Wuerzburg      */
/*************************************************************************/
