/*************************************************************************/
/* ntape - a tape archiver                                               */
/* Module: mt.c                                                          */
/* Authors: Peter Rupp, Matthias Hanisch                                 */
/*          A lot of things stolen from mt.c written by the University   */
/*          of California modified by Kai Makisara and Udo Christ for    */
/*          the Linux SCSI driver (Copyright notices included below).    */
/*************************************************************************/
/*                                                                       */
/* This program is free software; you can redistribute it and/or modify  */
/* it under the terms of the GNU General Public License as published by  */
/* the Free Software Foundation; either version 2 of the License, or     */
/* (at your option) any later version.                                   */
/*                                                                       */
/* This program is distributed in the hope that it will be useful,       */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/* GNU General Public License for more details.                          */
/*                                                                       */
/* You should have received a copy of the GNU General Public License     */
/* along with this program; if not, write to the Free Software           */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             */
/*                                                                       */
/*************************************************************************/
/*
 * Copyright (c) 1980 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * mt -- magnetic tape operation program
 *
 * Modified for the Linux SCSI tape driver by Kai Makisara
 *
 * Updated by Udo Christ 11.1994
 */

/* Some new tape densities stolen from mt-st 0.2 by Kai Makisara */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mtio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <ntape.h>

#define STATUS_WIN_HEIGHT 17
#define STATUS_WIN_WIDTH  62

#define DENS(A)         (((A) & MT_ST_DENSITY_MASK) >> MT_ST_DENSITY_SHIFT)
#define BLKS(A)         (((A) & MT_ST_BLKSIZE_MASK) >> MT_ST_BLKSIZE_SHIFT)
#define SFTERR(A)       (((A) & MT_ST_SOFTERR_MASK) >> MT_ST_SOFTERR_SHIFT)
struct commands 
{
    const char *c_name;
    int c_code;
    int c_ronly;
} 
com[] = 
{   
    { "rewind",	MTREW,	 1 },
    { "eod",	MTEOM,	 1 },
    { "seek",	MTSEEK,  1 },
    { "tell",	MTTELL,  1 },
    { "erase",	MTERASE, 0 },
    { "status", MTNOP,   1 },
    { 0 }
};

static struct mtpos mt_pos;
static struct mtop mt_com;
static struct mtget mt_status;
static struct mt_tape_info tapes[] = MT_TAPE_INFO;


/*************************************************************************/
/* Name:     do_tape_command                                             */
/* Purpose:  sends various commands to magnetic tape                     */
/*************************************************************************/
long int do_tape_command(char *cp,long int how_often)
{
    int mtfd;
    struct commands *comp;

    for (comp = com; comp->c_name != NULL; comp++)
	if (strncmp(cp, comp->c_name, strlen(cp)) == 0)	
	    break;
    if (comp->c_name == NULL)
    {
	print_footer("Unknown tape command. This is a bug in the sources!");
	return(-1);
    }
    
#ifdef DEBUG
    fprintf(stderr,"do_tape_command : %s\n",comp->c_name);
#endif

    if ((mtfd = open(tapedevice, comp->c_ronly ? O_RDONLY : O_RDWR )) < 0)
    {
	return(-1);
    }
    if (comp->c_code == MTTELL)
    {
	if (ioctl(mtfd,MTIOCPOS, (char *)&mt_pos) < 0)
	{
	    return (-1);
	}
	close(mtfd);
	return (mt_pos.mt_blkno);
    }	
    else if (comp->c_code != MTNOP)
    {
	mt_com.mt_op = comp->c_code;
	mt_com.mt_count = how_often;
	if (ioctl(mtfd,MTIOCTOP, (char *)&mt_com) <0)
	{
	    return (-1);
	}
	close(mtfd);
	return(0);
    }
    else
    {
	if (ioctl(mtfd, MTIOCGET, (char *)&mt_status) < 0)
	{
	    return(-1);
	}
	close(mtfd);
	return(mt_status.mt_gstat);
    }
}


/*************************************************************************/
/* Name:     offline                                                     */
/* Purpose:  returns 1 if tape is offline                                */
/*************************************************************************/
int offline()
{
#ifdef NO_OFFLINE
    return(0);
#else
    int ret;
    
    if (GMT_ONLINE(do_tape_command("status",0)))
	ret = 0;
    else
	ret = 1;

#ifdef DEBUG
    fprintf(stderr,"Tape offline: %s\n",
	    ret ? "Yes" : "No");
#endif /* DEBUG */    

    return(ret);
#endif
}


/*************************************************************************/
/* Name:     wr_prot                                                     */
/* Purpose:  returns 1 if tape is write protected                        */
/*************************************************************************/
int wr_prot()
{
#ifdef NO_WRITE_PROTECTED
    return(0);
#else
    int ret;
    
    if (GMT_WR_PROT(do_tape_command("status",0)))
	ret = 1;
    else
	ret = 0;

#ifdef DEBUG
    fprintf(stderr,"Tape write protected: %s\n",
	    ret ? "Yes" : "No");
#endif /* DEBUG */
    
    return(ret);
#endif
}

/*************************************************************************/
/* Name:     status_window                                               */
/* Purpose:  displays a window with various tape status information      */
/*************************************************************************/
void status_window()
{
    WINDOW *status_win;
    struct mt_tape_info *mt;
    int i = 2, fertig = 0;
    long absblocks;
    char line[30];

    /* Get the tape status struct. We'll find all in mt_status */
    do_tape_command("status",0);
    absblocks = do_tape_command("tell",0);
    
    status_win = newwin(STATUS_WIN_HEIGHT,STATUS_WIN_WIDTH,
			(LINES - STATUS_WIN_HEIGHT) / 2,
			(COLS - STATUS_WIN_WIDTH) / 2);
    leaveok(status_win, TRUE);
    scrollok(status_win, TRUE);
    keypad(status_win,TRUE);
    set_color_pair(status_win,TAPESTATUSWIN);
    winclr(status_win);
    
    for (mt = tapes; mt->t_type; mt++)
	if (mt->t_type == mt_status.mt_type)
	    break;
    mvwaddstr(status_win,i,2,"Drive type       :");
    mvwaddstr(status_win,i++,22,mt->t_name); i++;
    
    mvwaddstr(status_win,i,2,"Generic status   :");
    sprintf(line,"%lx",mt_status.mt_gstat);
    mvwaddstr(status_win,i++,22,line);
    mvwaddstr(status_win,i,2,"Online           :");
    mvwaddstr(status_win,i,22,GMT_ONLINE(mt_status.mt_gstat) ? "Yes":"No");
    mvwaddstr(status_win,i,30,"Write protected  :");
    mvwaddstr(status_win,i++,50,GMT_WR_PROT(mt_status.mt_gstat) ? "Yes":"No");
    i++;

    /* Cartridge information by SCSI Sequential-Access density code specs.
       Docus from Kai */
    mvwaddstr(status_win,i,2,"Cartridge info   :");
    switch (DENS(mt_status.mt_dsreg))
    {
    case 0x1:
	mvwaddstr(status_win,i,22,"800 bpi, 9 Tracks, NRZI");
	break;
    case 0x2:
	mvwaddstr(status_win,i,22,"1600 bpi, 9 Tracks, PE");
	break;
    case 0x3:
	mvwaddstr(status_win,i,22,"6250 bpi, 9 Tracks, GCR");
	break;
    case 0x4:
	mvwaddstr(status_win,i,22,"8000 bpi, 4/9 Tracks, GCR");
	break;
    case 0x5:
	mvwaddstr(status_win,i,22,"8000 bpi, 4/9 Tracks, QIC-45/60, GCR");
	break;
    case 0x6:
	mvwaddstr(status_win,i,22,"3200 bpi, 9 Tracks, PE");
	break;
    case 0x7:
	mvwaddstr(status_win,i,22,"6400 bpi, 4 Tracks, IMFM");
	break;
    case 0x8:
	mvwaddstr(status_win,i,22,"8000 bpi, 4 Tracks, GCR");
	break;
    case 0x9:
	mvwaddstr(status_win,i,22,"37871 bpi, 18 Tracks, GCR");
	break;
    case 0xa:
	mvwaddstr(status_win,i,22,"6667 bpi, 22 Tracks, MFM");
	break;
    case 0xb:
	mvwaddstr(status_win,i,22,"1600 bpi, 4 Tracks, PE");
	break;
    case 0xc:
	mvwaddstr(status_win,i,22,"12690 bpi, 24 Tracks, GCR");
	break;
    case 0xd:
	mvwaddstr(status_win,i,22,"25380 bpi, 24 Tracks, GCR");
	break;
    case 0xe:
	mvwaddstr(status_win,i,22,"Reserved for ECMA");
	break;
    case 0xf:
	mvwaddstr(status_win,i,22,"10000 bpi, 15 Tracks, QIC-120, GCR");
	break;
    case 0x10:
	mvwaddstr(status_win,i,22,"10000 bpi, 18 Tracks, QIC-150/250, GCR");
	break;
    case 0x11:
	mvwaddstr(status_win,i,22,"16000 bpi, 26 Tracks, QIC-320/525, GCR");
	break;
    case 0x12:
	mvwaddstr(status_win,i,22,"51667 bpi, 30 Tracks, QIC-1350, RLL");
	break;
    case 0x13:
	mvwaddstr(status_win,i,22,"61000 bpi, 1 Track, DDS");
	break;
    case 0x14:
	mvwaddstr(status_win,i,22,"43245 bpi, ? Tracks, EXB-8200, RLL");
	break;
    case 0x15:
	mvwaddstr(status_win,i,22,"45434 bpi, ? Tracks, EXB-8500, RLL");
	break;
    case 0x16:
	mvwaddstr(status_win,i,22,"10000 bpi, ? Tracks, MFM");
	break;
    case 0x17:
	mvwaddstr(status_win,i,22,"42500 bpi, ? Tracks, MFM");
	break;
    case 0x24:
	mvwaddstr(status_win,i,22,"DDS-2");
	break;
    default:
	mvwaddstr(status_win,i,22,"Unknown bpi");
	break;
    }
    i += 2;
    
    mvwaddstr(status_win,i,2,"Block number     :");
    sprintf(line,"Absolute : %ld  Relative: %d",
	    absblocks,mt_status.mt_blkno);
    mvwaddstr(status_win,i++,22,line);
    
    mvwaddstr(status_win,i,2,"Block size       :");
    sprintf(line,"%ld",BLKS(mt_status.mt_dsreg));
    mvwaddstr(status_win,i++,22,line);

    mvwaddstr(status_win,i,2,"File number      :");
    sprintf(line,"%d",mt_status.mt_fileno);
    mvwaddstr(status_win,i++,22,line);
    
    mvwaddstr(status_win,i,2,"Residual count   :");
    sprintf(line,"%ld",mt_status.mt_resid);
    mvwaddstr(status_win,i++,22,line);

    mvwaddstr(status_win,i,2,"Recovered errors :");
    sprintf(line,"%ld",SFTERR(mt_status.mt_erreg));
    mvwaddstr(status_win,i++,22,line);

    box(status_win,ACS_VLINE,ACS_HLINE);
    mvwaddstr(status_win,0,STATUS_WIN_WIDTH / 2 - 10,
	      " Tape Status Report ");

    while (!fertig)
    {
	set_color_pair(status_win,TAPESTATUSBTN);
	mvwaddstr(status_win,STATUS_WIN_HEIGHT - 2,
		  STATUS_WIN_WIDTH / 2 - 4," Cancel ");
	touchwin(status_win);
	wrefresh(status_win);
	i = getch();
	if (i == CR_KEY)
	    fertig = 1;
    }
    delwin(status_win);
}

/*************************************************************************/
/* Copyright (C) 1994,1995 Peter Rupp & Matthias Hanisch, Wuerzburg      */
/*************************************************************************/
