// Filename:   ropchatd.C
// Contents:   the private chat daemon
// Author:     Gregory Shaw
// Created:    4/17/95

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _ROPCHATD_C_
#define _ROPCHATD_C_

#include "bbshdr.h"
#include "lang/bbsstring.h"

#undef DEBUG

// This is C (c++), so let's do prototypes

int check_admin_connect(void);
int shuffle_data(void);
int check_allocated(void);
int client_master_connect(void);
void delete_client(RoomInfo    *, ClientInfo *);
void getout(int);
void init_data(void);
void init_masters(void);

// declared global so that the 'getout' function can close the socket
bbsipc master_client;           // master client port
bbsipc master_info_admin;       // master admin information port
                                // room tracking information
RoomInfo   roomlist[MAX_CHAT_ROOMS];
errlog error_logger;            // error logging connection
                                // list of available ports
char   portlist[MAX_PRIVATE_CLIENTS];
                                // do we have any clients issued, but waiting?
int        clients_wo_connected=0;
int        clients=0;           // do we have any clients?
int    room_number;             // room number (unique)
long  connections;
long  messages_sent;
long  messages_received;
time_t boot;


// Function:   check_admin_connect
// Purpose:    check for an administration connection and handle appropriately
// Input:  none
// Output: checks for connect.  if successful, admin data is sent to the client.
// Author: Gregory Shaw
// Created:    4/9/95

int check_admin_connect(void)
{
	char msg[255];
	char *timestr;              // boot time

                                // connection?
	if (master_info_admin.do_connect(0) == 0)
	{
		// build information message
		// send to client
		timestr = ctime(&boot);
		messages_sent++;
		messages_received++;
		master_info_admin.send("The rocat BBS System: ropchatd\n");
		sprintf(msg,"Up since: %s\n",timestr);
		master_info_admin.send(msg);
		master_info_admin.send("Messages:\n");
		sprintf(msg,"    Received: %ld\n",messages_received);
		master_info_admin.send(msg);
		sprintf(msg,"    Sent: %ld\n",messages_sent);
		master_info_admin.send(msg);
		sprintf(msg,"\nConnections: %ld\n",connections);
		master_info_admin.send(msg);
                                // disconnect the bugger
		master_info_admin.close_sock(1);
	}
	return(0);
}


// Function:   check_allocated
// Purpose:    check the lists for clients with allocated ports that never connected
// Input:  none
// Output: checks for connect.  If successful, data structures are updated.  If not, it checks
//     against a timeout to see if port has expired.
// Author: Gregory Shaw
// Created:    4/9/95

int check_allocated(void)
{
	ClientInfo *tmp,*tmp2;      // new record
	int    x;

	// check for non-connected (but allocated) clients
	// delete client if it hasn't responded in 10 seconds
	if (clients_wo_connected > 0)
	{                           // run through the list trying to connect
		// attempt to connect
		#ifdef DEBUG
		printf("Checking allocated clients\n");
		fflush(stdout);
		#endif
		for (x=0; x< MAX_CHAT_ROOMS; x++)
		{
			if (roomlist[x].list != NULL)
			{
				tmp = roomlist[x].list;
				while (tmp != NULL)
				{
					if (!tmp->online)
						if (tmp->port->do_connect(0) == 0)
					{
						#ifdef DEBUG
						printf("Client connect successful.\n");
						fflush(stdout);
						#endif
						tmp->online = 1;
						clients_wo_connected--;
					}
					else
					{
						#ifdef DEBUG
						printf("Client connect failed.\n");
						fflush(stdout);
						#endif
						if (time(NULL) - tmp->connected > PORT_EXPIRATION_TIME)
						{       // give up
							#ifdef DEBUG
							printf("Gave up on client %s\n",tmp->login_name);
							fflush(stdout);
							#endif
							delete_client(&roomlist[x],tmp);
							clients_wo_connected--;
						}
					}
					tmp2 = tmp;
					tmp = tmp->next;
				}
			}
		}
	}
	return(0);
}


// Function:   client_master_connect
// Purpose:    check the master client port for connections and connect, as necessary.
// Input:  none
// Output: the socket is connected, and data structures are configured for the client
// Author: Gregory Shaw
// Created:    4/9/95

int client_master_connect(void)
{
	ClientInfo *tmp,*newrec;    // new record
	int    x,y,done;
	char       msg[255];
                                // login name
	char   login[MAX_LOGIN_LENGTH];
                                // person to chat with
	char   target[MAX_LOGIN_LENGTH];
	char       tmpstr[255];     // string to output to file

                                // connection?
	if (master_client.do_connect(0) == 0)
	{
		#ifdef DEBUG
		printf("got connect\n");
		fflush(stdout);
		#endif
		clients++;              // more clients
		// receive target person and other info
		if (master_client.receive(msg) < 0)
		{
			master_client.close_sock(1);
			return(1);
		}
		messages_received++;
		connections++;
		// create new port
		if (sscanf(msg,"%s %s",login,target) != 2)
		{
			sprintf(tmpstr,"ropchatd: bad message received on client master %s",msg);
			error_logger.ap_log(tmpstr);
                                // disconnect
			master_client.close_sock(1);
			return(0);
		}
		#ifdef DEBUG
		printf("information: %s\n",msg);
		fflush(stdout);
		#endif
		// connect to new port, if possible
		clients_wo_connected++; // not connected, but waiting
		x = 0;
		done = 0;
		while (x < MAX_CHAT_ROOMS && !done)
		{
                                // used?
			if (roomlist[x].room != -1)
			{                   // let's see if this is our target
				tmp = roomlist[x].list;
				while (tmp != NULL)
				{
					if (!strcmp(tmp->login_name,target))
						done++; // yup.
					tmp = tmp->next;
				}
			}
			if (!done)          // don't increment if we found it.
				x++;
		}
		if (!done)              // someone waiting?
		{                       // nope.  allocate new room
			#ifdef DEBUG
			printf("no room found.  creating...\n");
			fflush(stdout);
			#endif
			x = 0;
			while (roomlist[x].room != -1 && x < MAX_CHAT_ROOMS)
				x++;
			if (x == MAX_CHAT_ROOMS)
			{                   // out of rooms
				error_logger.ap_log("ropchatd: out of chat rooms!");
				error_logger.ap_log("ropchatd: please recompile with more rooms");
                                // disconnect
				master_client.close_sock(1);
				return(0);
			}
			// init room data
			strcpy(roomlist[x].target,target);
                                // unique room
			roomlist[x].room = room_number++;
			roomlist[x].movement = 1;
			roomlist[x].users = 0;
			roomlist[x].avg_participants = 1.0;
			roomlist[x].avg_lurkers = 1.0;
			roomlist[x].output_message = NULL;
			roomlist[x].list = NULL;
		}
		// allocate new client record
		#ifdef DEBUG
		printf("creating new client record...\n");
		fflush(stdout);
		#endif
		roomlist[x].users++;
                                // list not empty
		if (roomlist[x].list != NULL)
		{                       // go to end of list
			tmp = roomlist[x].list;
			while (tmp->next != NULL)
				tmp = tmp->next;
		}
		else
			tmp = roomlist[x].list;
		newrec = (ClientInfo *)malloc(sizeof(ClientInfo));
		if (newrec == NULL)
		{
			error_logger.ap_log("ropchatd: out of memory!");
                                // disconnect
			master_client.close_sock(1);
			return(0);
		}
		if (roomlist[x].list == NULL)
	                                // start
			roomlist[x].list = newrec;
		else
			tmp->next = newrec; // add to end
		// add port object
		newrec->port = new bbsipc;
		newrec->mess_sent = 0;
		time(&newrec->connected);
		newrec->online = 0;
		strcpy(newrec->login_name,login);
		newrec->next = NULL;
		// select a client port
		#ifdef DEBUG
		printf("selecting port...");
		fflush(stdout);
		#endif
		y = 0;
		while (portlist[y] == 1 && y < MAX_PRIVATE_CLIENTS)
			y++;
		if (y == MAX_PRIVATE_CLIENTS)
		{
			error_logger.ap_log("ropchatd: out of client ports!");
                                // disconnect
			master_client.close_sock(1);
			delete newrec->port;
			free(newrec);
			return(0);
		}
		portlist[y]=1;          // allocate socket
		newrec->socket = y;     // save for de-allocation
		#ifdef DEBUG
		printf("...%d\n",y+CHAT_PRIVATE_CLIENT_BASE);
		fflush(stdout);
		#endif
		if (newrec->port->open_sock(NULL, CHAT_PRIVATE_CLIENT_BASE + y) != 0)
		{
			error_logger.ap_log("ropchatd: unable to open client socket");
			delete newrec->port;
			free(newrec);
			return(0);
		}
		// send back to client
		sprintf(tmpstr,"%d",CHAT_PRIVATE_CLIENT_BASE + y);
		master_client.send(tmpstr);
		messages_sent++;
		#ifdef DEBUG
		printf("sending port...%d\n",CHAT_PRIVATE_CLIENT_BASE+y);
		fflush(stdout);
		#endif
                                // disconnect
		master_client.close_sock(1);
		#ifdef DEBUG
		printf("send complete.  Closing master.\n");
		fflush(stdout);
		#endif
                                // connection?
		if (newrec->port->do_connect(0) == 0)
		{                       // cool.  quick. (or broken)
			#ifdef DEBUG
			printf("got connect on new socket.\n");
			fflush(stdout);
			#endif
			newrec->online = 1;
                                // connected
			clients_wo_connected--;
		}
		#ifdef DEBUG
		else
		{
			printf("client didn't connect.\n");
			fflush(stdout);
		}
		#endif

	}
	return(0);
}


// Function:   delete_client
// Purpose:    delete a client from the room list
// Input:  room - the room with the bad client
//     bad - the bad client
// Output: the client is deleted.  If the client was the last client in the room, the
//     room is marked as inactive.
// Author: Gregory Shaw
// Created:    4/10/95

void delete_client(RoomInfo *room, ClientInfo *bad)
{
	ClientInfo *tmp;            // tmp buffer
	char tmpstr[100];

	#ifdef DEBUG
	printf("room users: %d bad name: %s\n",room->users,bad->login_name);
	fflush(stdout);
	#endif
	// send a 'goodbye' message to people in room
	tmp = room->list;

	sprintf(tmpstr,HASLEFTROOM,bad->login_name);
	while (tmp != NULL)
	{
		if (tmp != bad)
		{
			tmp->port->send(tmpstr);
			messages_sent++;
		}
		tmp = tmp->next;
	}
	// now delete him from the list
	if (room->list == bad)      // trivial case - start of list
	{
		room->list = bad->next;
	}
	else
	{                           // look for the guy
		tmp = room->list;
		while (tmp->next != bad)
			tmp = tmp->next;
		tmp->next = bad->next;
	}
	// now free storage
	portlist[bad->socket] = 0;  // deallocate socket
	bad->port->close_sock(1);   // close sockets
	bad->port->close_sock(0);
	delete bad->port;           // delete ipc obj
	free(bad);                  // free record area
	if (room->users  == 1)      // last guy?
	{
		#ifdef DEBUG
		printf("Deleting room %d.\n",room->room);
		fflush(stdout);
		#endif
		// then mark room empty
		room->room = -1;
	}
	else
		room->users--;
	clients--;
};


// Function:   getout
// Purpose:    get out of the program.  Quick and dirty
// Author: Greg Shaw
// Created:    8/16/93

void getout(int sig)
{
	char   tmpstr[255];         // temporary string
	static int inalready = 0;   // if already in, don't do again
	int    x;

	if (!inalready)
	{
		inalready++;
                                // close for exit
		master_client.close_sock(0);
                                // close for exit
		master_info_admin.close_sock(0);
                                // close for exit
		master_client.close_sock(1);
                                // close for exit
		master_info_admin.close_sock(1);
		// now close the clients
		for (x=0; x<MAX_CHAT_ROOMS; x++)
		{
			if (roomlist[x].list != NULL)
			{
				while (roomlist[x].list != NULL)
				{
					delete_client(&roomlist[x],roomlist[x].list);
				}
			}
		}
		sprintf(tmpstr,"ropchatd: Got signal %d.  Exiting.",sig);
		error_logger.ap_log(tmpstr);
		exit(0);                // exit to OS
	}
}


// Function:   init_data
// Purpose:    initialize the data structures used by the program
// Input:  none
// Output: the sockets are created or the program exits
// Author: Gregory Shaw
// Created:    4/9/95

void init_data(void)
{
	int x;

	connections = 0;
	messages_sent = 0;
	messages_received = 0;
	room_number = 0;            // dynamic room number
	time(&boot);
	// init data structures
	for (x=0; x< MAX_CHAT_ROOMS; x++)
	{

		roomlist[x].room = -1;  // not in use
		roomlist[x].list = NULL;// no clients yet
                                // no target yet
		roomlist[x].target[0] = 0;
		roomlist[x].users = 0;  // no users
                                // not used...
		roomlist[x].movement = 0;
		roomlist[x].avg_participants = 0.0;
		roomlist[x].avg_lurkers = 0.0;
		roomlist[x].output_message = NULL;
	}
}


// Function:   init_masters
// Purpose:    initialize (create) the master sockets for the rochat daemon
// Input:  none
// Output: the sockets are created or the program exits
// Author: Gregory Shaw
// Created:    4/9/95

void init_masters(void)
{
	// create master sockets
	if (master_client.open_sock (NULL, CHAT_PRIVATE_MASTER_PORT) != 0)
	{
		error_logger.ap_log("ropchatd: Unable to open master client server socket.");
		exit(0);
	}
	if (master_info_admin.open_sock (NULL, CHAT_PRIVATE_ADMIN_MASTER_PORT) != 0)
	{
		error_logger.ap_log("ropchatd: Unable to open admin server socket.");
		exit (0);
	}
}


// Function:   shuffle_data
// Purpose:    look for message from clients and send messages to other clients
// Input:  messages are received from the clients
// Output: messages are sent to other clients in the same room(s)
// Author: Gregory Shaw
// Created:    4/9/95

int shuffle_data(void)
{
	ClientInfo *tmp,*snd;       // tmp buffer
	char   tmpstr[255];
	int    x;


	if (clients>0)
	{
		for (x=0; x< MAX_CHAT_ROOMS; x++)
		{
                                // room in use?
			if (roomlist[x].room != -1)
			{
                                // empty?
				if (roomlist[x].list != NULL)
				{
					tmp = roomlist[x].list;
					while (tmp != NULL)
					{
                                // message available?
						if (tmp->port->msg_avail(0))
						{
							messages_received++;
							if (tmp->port->receive(tmpstr) < 0)
							{   // port closed
								#ifdef DEBUG
								printf("Connection closed.\n");
								fflush(stdout);
								#endif
								delete_client(&roomlist[x],tmp);
							}
							else// got a message
							{   // send to everybody in group
								#ifdef DEBUG
								printf("Got %s.\n",tmpstr);
								fflush(stdout);
								#endif
								snd = roomlist[x].list;
								while (snd != NULL)
								{
									snd->port->send(tmpstr);
									messages_sent++;
									snd = snd->next;
								}
							}
						}
						tmp = tmp->next;
					}
				}
			}
		}
	}
	return(0);
}


// Function:   main
// Purpose:        the main calling routine for the error logger daemon
// Input:      socket connections will connect with program.
// Output:     all data sent to this daemon will be logged to a generic file
// Author:     Greg Shaw
// Created:        7/11/93

main()                          // no arguments
{
	int    x;
	struct timeval waittime;



	waittime.tv_sec = 0;
	#ifdef DEBUG
	printf("ropchatd init...\r\n");
	fflush(stdout);
	#endif
	for (x=1; x<15; x++)
		signal(x,&getout);
	for (x=0; x< MAX_PRIVATE_CLIENTS; x++)
		portlist[x] = 0;        // port not used yet

	// announce startup
	time(&boot);
	error_logger.ap_log("ropchatd (re) started");

	init_masters();             // initialize master ports
	init_data();                // clear data structures

	// Ok.  Data clean.  Let's start this beast!
	while (1)                   // a shutdown or other signal will kill this beast
	{
		client_master_connect();// check for incoming clients
		check_allocated();      // check for allocated (but not used used) connections
		check_admin_connect();  // check for admin information connect
		shuffle_data();         // look for messages and shuffle to clients
                if (clients)    // don't use CPU if no clients
                        waittime.tv_usec = 50;     // 50msec
                else
                        waittime.tv_usec = 800;     // 800msec
                select(0,NULL,NULL,NULL,&waittime);
	}
}


#endif







