// Filename:   moncon.C
// Contents:   the methods for the monitor connection object
// Author: Greg Shaw
// Created:    8/26/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _MONCON_C_
#define _MONCON_C_

#include "bbshdr.h"


// Method: constructor
// Purpose:    initialize all variables and attempt to connect to sysop process
// Input:  none
// Output: none
// Author: Greg Shaw
// Created:    8/26/93

moncon::moncon()
{

	cstart = cend = 0;          // setup buffer pointers
	costart = coend = 0;
	watch_on = 0;               // not watching
	mon_connected = 0;
	last_attempt = 0L;          // last attempt = 0;
};


// Method: connected
// Purpose:    return true if monitor connected
// Input:  none
// Output: internal object status is updated
// Author: Greg Shaw
// Created:    8/26/93

int moncon::connected(void)
{
	return(mon_connected);
};


// Method: connect_mon
// Purpose:    try to connect to the sysop monitor program
// Input:  none
// Output: internal object status is updated
// Author: Greg Shaw
// Created:    8/26/93

int moncon::connect_mon(void)
{
	char   watchname[30];
	char   msg[30];
	time_t now;

	if (!mon_connected)         // only connect if not connected
	{
		time(&now);             // set for next attempt
		// space tries to 10 seconds minimum
		if (abs(now - last_attempt) > 10)
		{
			time(&last_attempt);
			if (strcpy(watchname,watchhost()), watchname != NULL)
				// try to open socket
				if (open_sock(watchname,MONITOR_PORT) == 0)
			{
				mon_connected = 1;
				last_attempt = 0;
			}
		}
	}
	if (mon_connected)
	{
		if (receive(msg), strcmp(msg,MONITOR_ON) == 0)
		{
			watch_on = 1;
		}
	}
	return(0);
};


// Method: send_monitor
// Purpose:    send a string to the monitor process
// Input:  none
// Output: none
// Author: Greg Shaw
// Created:    8/26/93

int moncon::send_monitor(char *str)
{
	if (watch_on)
		if (send(str) != 0)     // error?
	{                           // yes - close socket
		watch_on = 0;
		mon_connected = 0;
	}
	return(0);
};


// Method: watching
// Purpose:    return true if sysop monitor is watching this BBS process
// Input:  none
// Output: none
// Author: Greg Shaw
// Created:    8/26/93

int moncon::watching(void)
{
	return(watch_on);
};


// Method: get_command
// Purpose:    attempt to get a command from the monitor process
// Input:  none
// Output: a character, should one be found
// Author: Greg Shaw
// Created:    8/27/93

int moncon::get_command(void)
{
	char   msg[255];            // message from monitor
	int    x;

	if (watch_on)
	{
		if (costart != coend)   // empty buffer first
		{
			x = costart++;
			costart %= COMMANDS_MAX;
			return(commands[x]);
		}
		if (msg_avail(0))
		{
			// look for command string
			if (receive(msg) == -1)
			{                   // receive failed - socket closed
				mon_connected = 0;
				watch_on = 0;
			} else
			{
				if (strstr(msg,COMMAND_MSG) != NULL)
				{               // command message
					return(msg[4]);
				}
				else
				{
					for (x=0; x<strlen(msg); x++)
					{
						cbuf[cend++] = msg[x];
						cend %= CBUF_MAX;
						// full?
						if (cstart == cend)
							// disregard
							break;
						//rest of message
					}
				}
			}
		}
	}
	return(0);
};


// Method: get_char
// Purpose:    attempt to get a character from the monitor process
// Input:  none
// Output: a character, should one be found
// Author: Greg Shaw
// Created:    8/27/93

char moncon::get_char(void)
{
	char   msg[255];            // message from monitor
	int    x;

	if (watch_on)
	{
		if (cstart != cend)     // empty buffer first
		{
			x = cstart++;
			cstart %= CBUF_MAX;
			return(cbuf[cstart++]);
		}
		if (msg_avail(1))
		{
			// look for command string
			if (receive(msg) == -1)
			{                   // receive failed - socket closed
				mon_connected = 0;
				watch_on = 0;
			} else
			{
				if (strstr(msg,COMMAND_MSG) != NULL)
				{               // command message
					// 5th byte
					commands[coend++] = msg[4];
					coend %= COMMANDS_MAX;
				}
				else
				{
					for (x=0; x<strlen(msg); x++)
					{
						cbuf[cend++] = msg[x];
						cend %= CBUF_MAX;
						// full?
						if (cstart == cend)
							// disregard
							break;
						//rest of message
					}
					x = cstart++;
					cstart %= CBUF_MAX;
					return(cbuf[x]);
				}
			}
		}
	}
	return(0);
};


#endif                          // _MONCON_C_








