// Filename:   filelock.C
// Contents:   the methods for the filelock (file locking) object
// Author:     Greg Shaw
// Created:        7/23/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _FLOCK_C_
#define _FLOCK_C_

#include "bbshdr.h"

// Function:   set_lock
// Purpose:        set a lock on a file
// Inputs:     type - type of file lock.  0 for read, 1 for write
//             fptr - the pointer to a file
// Output:     returns non-zero if error
// Author:     Greg Shaw
// Created:        7/24/93

int filelock::set_lock(char type, int fptr)
{
	struct flock   flockinfo;
	int    tries;

	flockinfo.l_start = 0L;     // start at byte 0
	flockinfo.l_len = 0L;       // through end of file
	flockinfo.l_whence = 0;     // set to start of file

	if (type)
	{                           // write lock
		// write lock
		flockinfo.l_type = F_WRLCK;
		tries = 0;
		while (tries < MAX_LOCK_TRIES)
		{
			if (fcntl(fptr,F_SETLKW,&flockinfo) != 0)
			{
				sleep(1);
				tries++;
				if (tries == MAX_LOCK_TRIES)
					return(-1);
			}
			else
				// worked, exit
				tries = MAX_LOCK_TRIES;
		}
	}
	else
	{                           // read lock
		// read lock
		flockinfo.l_type = F_RDLCK;
		tries = 0;
		while (tries < MAX_LOCK_TRIES)
		{
			if (fcntl(fptr,F_SETLKW,&flockinfo) != 0)
			{
				sleep(1);
				tries++;
				if (tries == MAX_LOCK_TRIES)
					return(-1);
			}
			else
				// worked, exit
				tries = MAX_LOCK_TRIES;
		}
	}
	return(0);
};


// Function:   clear_lock
// Purpose:        clear a lock on a file
// Inputs:     fptr - the pointer to a file
// Output:     returns non-zero if error
// Author:     Greg Shaw
// Created:        7/24/93

int filelock::clear_lock(int fptr)
{
	struct flock   flockinfo;

	flockinfo.l_start = 0;      // start at byte 0
	flockinfo.l_len = 0;        // through end of file
	flockinfo.l_whence = 0;     // set to start of file

	flockinfo.l_type = F_UNLCK; // read lock
	if (fcntl(fptr,F_SETLKW,&flockinfo) != 0)
	{
		return(-1);
	}
	return(0);
};


// Function:   bopen
// Purpose:        open a file, set locks as appropriate for mode
// Inputs:     path - path to file
//             mode - the mode of the file
// Output:     returns valid (open) file descriptor or NULL if error
// Author:     Greg Shaw
// Created:        7/24/93

FILE *filelock::bopen(char *path, char *mode)
{
	FILE   *fileptr;            // pointer to new opened file
	char   tmpstr[255];
	int x;

	if (fileptr = fopen(path,mode), fileptr == NULL)
	{
		return(NULL);
	}
	if (strchr(mode,'w') != NULL || strchr(mode,'a') != NULL)
	{
		if (set_lock(1,fileno(fileptr)) != 0)
		{
			sprintf(tmpstr,"bopen: able to open file, but unable to set lock %s",path);
			ap_log(tmpstr);
			fclose(fileptr);
			return(NULL);
		}
	}
	// read only
	else if (strchr(mode,'r') != NULL)
	{
		if (set_lock(0,fileno(fileptr)) != 0)
		{
			sprintf(tmpstr,"bopen: able to open file, but unable to set lock %s",path);
			ap_log(tmpstr);
			fclose(fileptr);
			return(NULL);
		}
	}
	else
	{
		sprintf(tmpstr,"filelock: Unable to determine read/write of file %s, mode %s",path,mode);
		ap_log(tmpstr);
		fclose(fileptr);
		return(NULL);
	}
	x = 0;
	while (file_list[x] != NULL && x < MAX_FILES)
		x++;
	if (x < MAX_FILES)
		file_list[x] = fileptr;
	else
	{
		ap_log("filelock: Out of file storage.  Please increase MAX_FILES");
	}
	return(fileptr);
};


// Function:   bclose
// Purpose:    close a file, clear locks
// Inputs: fptr - the pointer to the file
// Output: returns non zero for failure
// Author: Greg Shaw
// Created:    7/24/93

int filelock::bclose(FILE *fptr)
{
	int x;

	clear_lock(fileno(fptr));
	x=0;
	while (file_list[x] != fptr && x < MAX_FILES)
		x++;
	if (x < MAX_FILES)
		file_list[x] = NULL;
	else
	{
		ap_log("filelock: Unable to find file in open file list.");
	}
	fclose(fptr);
	return(0);
};


// Function:   constructor
// Purpose:    initialize the lists
// Inputs: none
// Output: none
// Author: Greg Shaw
// Created:    4/23/94

filelock::filelock()
{
	int x;

	for (x=0; x<MAX_FILES; x++)
		file_list[x] = NULL;
};


// Function:   destructor
// Purpose:    close all open files and clear locks
// Inputs: none
// Output: none
// Author: Greg Shaw
// Created:    4/23/94

filelock::~filelock()
{
	int x;

	for (x = 0; x< MAX_FILES; x++)
		if (file_list[x] != NULL)
			bclose(file_list[x]);

};


#endif                          // _FLOCK_C_







