// Filename:   bbsinfo.C
// Contents:   the methods for the bbsinfo object
// Author:     Greg Shaw
// Created:        7/11/93

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#ifndef _BBSINFO_C_
#define _BBSINFO_C_

#include "bbshdr.h"

// Function:   findline
// Purpose:        find a line containing a particular word from the bbsinfo
//             file.
// Input:      word - a string containing the word to look for
// Output:     line - the line (if found)
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::findline(char *line, char *word)
{
	FILE   *infile;             // bbs info file
	int        cnt;             // line byte counter
	char   c;                   // char
	char   in_line[255];        // a line from the file
	char   found;               // loop boolean
	char   *bbsdir;             // bbsdir string (from environment)
	char   tmpstr[255];         // temporary string

	// get bbs dir
	if (bbsdir = getenv("BBSDIR"), bbsdir == NULL)
		return(-1);
	strcpy(tmpstr,bbsdir);
	// get filename
	strcat(tmpstr,"/config/bbsinfo");
	if (infile = fopen(tmpstr,"r"), infile == NULL)
		return(-1);
	// now look for word in file
	found = 0;
	while (!found && !feof(infile))
	{
		cnt = 0;
		while (c = fgetc(infile), c != '\n' && c != '\r' && !feof(infile))
			in_line[cnt++] = c;
		in_line[cnt] = 0;
		if (in_line[0] == '#' || strlen(in_line) < 5)
			continue;
		// gotcha!
		if (strstr(in_line,word) != NULL)
		{
			strcpy(line,in_line);
			found++;
		}
	}
	fclose(infile);
	if (!found)
		return(-1);
	else
		return(0);
};


// Function:   cardinfo
// Purpose:        return the card information relating to a particular card
// Input:      type - the type of card:
//             0 - red
//             1 - blue
//             2 - green
//             3 - white
//             4 - grey
//             5 - pink
//             6 - yellow
//             7 - black
// Output:     a CardRec structure pointer filled in with information from
//             bbsinfo file.  (or NULL for error)
// Author:     Greg Shaw
// Created:        7/27/93

CardRec *bbsinfo::cardinfo(int type)
{
	char   line[255];
	char   dumy[25];
	CardRec    *card;

	if (card = (CardRec *)malloc(sizeof(CardRec)), card == NULL)
		return(NULL);
	switch(type)
	{
		case 0:
			if (findline(line,"REDCARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"red");
			return(card);
			break;
		case 1:
			if (findline(line,"BLUECARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"blue");
			return(card);
			break;
		case 2:
			if (findline(line,"GREENCARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"green");
			return(card);
			break;
		case 3:
			if (findline(line,"WHITECARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"white");
			return(card);
			break;
		case 4:
			if (findline(line,"GREYCARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"grey");
			return(card);
			break;
		case 5:
			if (findline(line,"PINKCARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"pink");
			return(card);
			break;
		case 6:
			if (findline(line,"YELLOWCARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"yellow");
			return(card);
			break;
		case 7:
			if (findline(line,"BLACKCARD") != 0)
				return(NULL);
			if (sscanf(line,"%s %d %d %x %d %d %d %d",dumy,&card->acl, &card->tl,
				&card->flags,
			&card->kbytes,&card->timelimit_days,
			&card->timelimit_period,
			&card->account_valid_months) != 8)
			return(NULL);
			strcpy(card->colr,"black");
			return(card);
			break;
		default:
			return(NULL);
	}
};


// Function:   sysop
// Purpose:        return the name of the BBS main sysop
// Input:      none
// Output:     a string that contains the 'pager' program
// Notes:      it should be possible to pass in some variable to tell
//             which sysop the user 'wants' in a multiple sysop system
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::sysop(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *sysopn;

	if (findline(line,"SYSOP") != 0)
		return(NULL);
	if (sysopn = (char *) malloc(20), sysopn == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,sysopn) != 2)
		return(NULL);
	return(sysopn);
};


// Function:   bbs_uid
// Purpose:    return the uid of the bbs administrator
// Input:  none
// Output: the bbs administrator's uid
// Author: Greg Shaw
// Created:    4/25/94

uid_t bbsinfo::bbs_uid(void)
{
	struct passwd *bbspw;

	if (bbspw = getpwnam("bbs"), bbspw == NULL)
		return(1);
	return(bbspw->pw_uid);
};


// Function:   bbs_gid
// Purpose:    return the gid of the bbs administrator
// Input:  none
// Output: the bbs administrator's gid
// Author: Greg Shaw
// Created:    4/25/94

gid_t bbsinfo::bbs_gid(void)
{
	struct passwd *bbspw;

	if (bbspw = getpwnam("bbs"), bbspw == NULL)
		return(1);
	return(bbspw->pw_gid);
};


// Function:   mailspool
// Purpose:    return the mail spool file directory for the system
// Input:      none
// Output:     a string that contains the mail spool dir
// Author:     Greg Shaw
// Created:        8/10/93

char *bbsinfo::mailspool(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *mspool;

	if (findline(line,"MAILSPOOL") != 0)
		return(NULL);
	if (mspool = (char *) malloc(20), mspool == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,mspool) != 2)
		return(NULL);
	return(mspool);
};


// Function:   checkmail
// Purpose:    return true if the BBS is to check for mail
// Input:  none
// Output: 1 for check for mail, 0 otherwise
// Author: Greg Shaw
// Created:    8/10/93

int bbsinfo::checkmail(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    msgonce;

	if (findline(line,"CHECKMAIL") != 0)
		return(0);
	if (sscanf(line,"%s%d",dummy,&msgonce) != 2)
		return(0);
	return(msgonce);
};


// Function:   mailchecktime
// Purpose:    return the number of seconds that should elapse between
//     mail checks
// Input:  none
// Output: a string that contains the mail spool dir
// Author: Greg Shaw
// Created:    4/24/94

int bbsinfo::mailchecktime(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int  msgonce;

	if (findline(line,"MAILCHECK") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&msgonce) != 2)
		return(0);
	return(msgonce);
};


// Function:   sys_pager
// Purpose:        return the default 'pager' program to be used by the
//             bbs to display files to the user
// Input:      none
// Output:     a string that contains the 'pager' program
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::sys_pager(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *pager;

	if (findline(line,"SYSTEMPAGER") != 0)
		return(NULL);
	if (pager = (char *) malloc(20), pager == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,pager) != 2)
		return(NULL);
	return(pager);
};


// Function:   chathost
// Purpose:    return the host where the chat daemon is running
// Input:      none
// Output:     a string that contains the chat host is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:    4/9/95

char *bbsinfo::chathost(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *chathst;

	if (findline(line,"CHATHOST") != 0)
		return(NULL);
	if (chathst = (char *) malloc(30), chathst == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,chathst) != 2)
		return(NULL);
	return(chathst);
};


// Function:   loghost
// Purpose:        return the host where the error logger is running
// Input:      none
// Output:     a string that contains the loghost is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::loghost(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *loghst;

	if (findline(line,"LOGHOST") != 0)
		return(NULL);
	if (loghst = (char *) malloc(30), loghst == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,loghst) != 2)
		return(NULL);
	return(loghst);
};


// Function:   talkprog
// Purpose:    return the 'talk' program for the system
// Input:  none
// Output: a string that contains the talkprog is returned
//         note: it should be freed on exit.
// Author: Greg Shaw
// Created:    7/11/93

char *bbsinfo::talkprog(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *talkprog;

	if (findline(line,"TALKPROG") != 0)
		return(NULL);
	if (talkprog  = (char *) malloc(30), talkprog == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,talkprog) != 2)
		return(NULL);
	return(talkprog);
};


// Function:   watchost
// Purpose:        return the host where the sysop 'watch' program runs
// Input:      none
// Output:     a string that contains the watchhost is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::watchhost(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *wathst;

	if (findline(line,"WATCHHOST") != 0)
		return(NULL);
	if (wathst = (char *) malloc(30), wathst == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,wathst) != 2)
		return(NULL);
	return(wathst);
};


// Function:   ratio
// Purpose:        return the upload to download ratio
// Input:      none
// Output:     a string that contains the loghost is returned
// Author:     Greg Shaw
// Created:        7/11/93

float bbsinfo::ratio(void)
{
	char line[255];
	char dummy[255];            // dummy string
	float rato;

	if (findline(line,"RATIO") != 0)
		return(0.0);
	if (sscanf(line,"%s%f",dummy,&rato) != 2)
		return(0.0);
	return(rato);
};


// Function:   sys_msg_once
// Purpose:        return true if the system message in question is only to be
//             displayed once
// Input:      num - the number of the file to check
// Output:     true if message to be shown once only
// Author:     Greg Shaw
// Created:        7/24/93

int bbsinfo::sys_msg_once(int num)
{
	char line[255];
	char dummy[255];            // dummy string
	int    msgonce;

	switch(num)
	{
		case 1:                 // first one
			if (findline(line,"SYSTEM1MSGONCE") != 0)
				return(0);
			break;
		case 2:                 // second one
			if (findline(line,"SYSTEM2MSGONCE") != 0)
				return(0);
			break;
		case 3:                 // second one
			if (findline(line,"SYSTEM3MSGONCE") != 0)
				return(0);
			break;
		default:
			return(0);
	}
	if (sscanf(line,"%s%d",dummy,&msgonce) != 2)
		return(0);
	return(msgonce);
};


// Function:   chat_avail
// Purpose:        return true if sysop is available for chat
// Inputs:     tim - the hour and time currently
// Outputs:        true if available
// Author:     Greg Shaw
// Created:        7/25/93

int bbsinfo::chat_avail(int tim)
{
	char line[255];
	char dummy[255];            // dummy string
	int    start;
	int end;

	if (findline(line,"CHATON") != 0)
		return(0);
	if (sscanf(line,"%s%d",dummy,&start) != 2)
		return(0);
	if (findline(line,"CHATOFF") != 0)
		return(0);
	if (sscanf(line,"%s%d",dummy,&end) != 2)
		return(0);
	if (tim > start && tim < end)
		return(1);
	else
		return(0);
};


// Function:   def_card
// Purpose:        return the default card color for the user
// Input:      none
// Output:     the default acl
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_card(void)
{
	char line[255];

	if (findline(line,"DEFCARD") != 0)
		return(0);
	// gotcha!
	if (strstr(line,"red") != NULL)
		return(0);
	// gotcha!
	if (strstr(line,"blue") != NULL)
		return(1);
	// gotcha!
	if (strstr(line,"green") != NULL)
		return(2);
	// gotcha!
	if (strstr(line,"white") != NULL)
		return(3);
	// gotcha!
	if (strstr(line,"grey") != NULL)
		return(4);
	// gotcha!
	if (strstr(line,"pink") != NULL)
		return(5);
	// gotcha!
	if (strstr(line,"yellow") != NULL)
		return(6);
	// gotcha!
	if (strstr(line,"black") != NULL)
		return(7);
	return(-1);
};


// Function:   def_attribute
// Purpose:    return the default attribute
// Input:      none
// Output:     a string that contains the default termtype is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_attribute(void)
{
	char line[255];

	if (findline(line,"DEFATTRIBUTE") != 0)
		return(0);
	// gotcha!
	if (strstr(line,"normal") != NULL)
		return(1);
	// gotcha!
	if (strstr(line,"bold") != NULL)
		return(2);
	// gotcha!
	if (strstr(line,"blink") != NULL)
		return(3);
	// gotcha!
	if (strstr(line,"inverse") != NULL)
		return(4);
	return(-1);
};


// Function:   def_foreground
// Purpose:    return the default foreground color to use
// Input:      none
// Output:     a string that contains the default foreground is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_foreground(void)
{
	char line[255];

	if (findline(line,"DEFFOREGROUND") != 0)
		return(0);
	// gotcha!
	if (strstr(line,"black") != NULL)
		return(1);
	// gotcha!
	if (strstr(line,"red") != NULL)
		return(2);
	// gotcha!
	if (strstr(line,"green") != NULL)
		return(3);
	// gotcha!
	if (strstr(line,"yellow") != NULL)
		return(4);
	// gotcha!
	if (strstr(line,"blue") != NULL)
		return(5);
	// gotcha!
	if (strstr(line,"magenta") != NULL)
		return(6);
	// gotcha!
	if (strstr(line,"cyan") != NULL)
		return(7);
	// gotcha!
	if (strstr(line,"white") != NULL)
		return(8);
	return(-1);
};


// Function:   def_background
// Purpose:    return the default background color to use
// Input:      none
// Output:     a string that contains the default background is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::def_background(void)
{
	char line[255];

	if (findline(line,"DEFBACKGROUND") != 0)
		return(0);
	// gotcha!
	if (strstr(line,"black") != NULL)
		return(1);
	// gotcha!
	if (strstr(line,"red") != NULL)
		return(2);
	// gotcha!
	if (strstr(line,"green") != NULL)
		return(3);
	// gotcha!
	if (strstr(line,"yellow") != NULL)
		return(4);
	// gotcha!
	if (strstr(line,"blue") != NULL)
		return(5);
	// gotcha!
	if (strstr(line,"magenta") != NULL)
		return(6);
	// gotcha!
	if (strstr(line,"cyan") != NULL)
		return(7);
	// gotcha!
	if (strstr(line,"white") != NULL)
		return(8);
	return(-1);
};


// Function:   def_termtype
// Purpose:        return the default terminal type of a new user
// Input:      none
// Output:     a string that contains the default termtype is returned
//             note: it should be freed on exit.
// Author:     Greg Shaw
// Created:        7/11/93

char *bbsinfo::def_termtype(void)
{
	char line[255];
	char dummy[255];            // dummy string
	char *dt;

	if (findline(line,"DEFTERM") != 0)
		return(NULL);
	if (dt = (char *) malloc(30), dt == NULL)
		return(NULL);
	if (sscanf(line,"%s%s",dummy,dt) != 2)
		return(NULL);
	return(dt);
};


// Function:   maxk
// Purpose:    return the maximum number of Kbytes the user can download at
//     one time.
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    8/24/93

int bbsinfo::maxk(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"MAXK") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(-1);
	return(dt);
};


// Function:   log_externals
// Purpose:    return whether to log external calls
// Input:      none
// Output:     1 for log, 0 for no
// Author:     Greg Shaw
// Created:    12/31/94

int bbsinfo::log_externals(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"LOG_EXTERNALS") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(0);
	return(dt);
};


// Function:   log_menu_accesses
// Purpose:    return whether to log menu movement
// Input:      none
// Output:     1 for log, 0 for no
// Author:     Greg Shaw
// Created:    12/31/94

int bbsinfo::log_menu_accesses(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"LOG_MENU_ACCESSES") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(0);
	return(dt);
};


// Function:   timelimit_fudge
// Purpose:    return the amount of additional time (percentage) to give a
//     user when his time has expired and he is downloading
// Input:  none
// Output: int (percent)
// Author: Greg Shaw
// Created:    10/17/94

int bbsinfo::timelimit_fudge(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"FUDGETIMELIMIT") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(0);
	return(dt);
};


// Function:   inactivity_timeout
// Purpose:    return the time to wait before logging out an inactive user
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    8/24/93

int bbsinfo::inactivity_timeout(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"INACTIVITY") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(-1);
	return(dt);
};


// Function:   creditchat
// Purpose:        return true if chat time with sysop is credited
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/11/93

int bbsinfo::creditchat(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"CREDITCHAT") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(-1);
	return(dt);
};


// Function:   credituploads
// Purpose:        return true if upload time is credited
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/13/93

int bbsinfo::credituploads(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"CREDITUPLOADS") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(-1);
	return(dt);
};


// Function:   userprompt
// Purpose:        return the amount of time between calls allowed (hours)
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/13/93

char *bbsinfo::userprompt(void)
{
	char line[255];
	char dum0[25];              // dummy string
	char dum1[25];              // dummy string
	char dum2[25];              // dummy string
	char dum3[25];              // dummy string
	char *dt;                   // return string

	dum0[0] = dum1[0] = dum2[0] = dum3[0] = 0;
	if (findline(line,"USERPROMPT") != 0)
		return(NULL);
	sscanf(line,"%s%s%s%s",dum0,dum1,dum2,dum3);
	strcpy(line,dum1);
	strcat(line,dum2);
	strcat(line,dum3);
	strcat(line," ");
	if (dt = (char *) malloc(strlen(line)), dt == NULL)
		return(NULL);
	strcpy(dt,line);
	return(dt);
};


// Function:   showtime
// Purpose:        return true if command prompt should show the amount of
//             time user has left.
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/27/93

int bbsinfo::showtime(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"SHOWTIMELEFT") != 0)
		return(0);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(0);
	return(dt);
};


// Function:   showfortune
// Purpose:    return true if the bbs should give the user a 'fortune cookie'
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    8/22/93

int bbsinfo::showfortune(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"SHOWFORTUNE") != 0)
		return(0);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(0);
	return(dt);
};


// Function:   showcoms
// Purpose:        return true if command prompt should show the keys for
//             the valid commands
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/27/93

int bbsinfo::showcoms(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"SHOWVALIDKEYS") != 0)
		return(0);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(0);
	return(dt);
};


// Function:   waittime
// Purpose:        return the amount of time between calls allowed (hours)
// Input:      none
// Output:     true/false
// Author:     Greg Shaw
// Created:        7/13/93

int bbsinfo::waittime(void)
{
	char line[255];
	char dummy[255];            // dummy string
	int    dt;

	if (findline(line,"WAITTIME") != 0)
		return(-1);
	if (sscanf(line,"%s%d",dummy,&dt) != 2)
		return(-1);
	if (dt<0)
		dt=0;
	return(dt);
};


// Function:   username
// Purpose:    return the login name of the current user
// Input:  none
// Output: true/false
// Author: Greg Shaw
// Created:    7/13/93

char *bbsinfo::username(void)
{
	struct passwd *thisuser;

	if (thisuser = getpwuid(geteuid()), thisuser == NULL)
		return(NULL);
	else
		return(thisuser->pw_name);
};


#endif                          // _BBSINFO_C_







