#ident "@(#)wjjr@csn.net:  nisttime.c 5.3, 4/15/95"

/*LINTLIBRARY*/

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#ifdef __USLC__        /* unixware */
#include <fcntl.h>
#include <sys/rtc.h>
#endif __USLC__
#include <sys/wait.h>
#include <unistd.h>

#ifdef __linux__
#define CLOCK     "/sbin/clock"        /* clock(1) */
#endif __linux__
#define CU        "/usr/bin/cu"        /* cu(1) */

/*
 * The Julian date is the number of days since 12:00 noon GMT on January 1st,
 * 4713 B.C.  The Modified Julian date is found by subtracting 2,400,000.5 
 * from the Julian date.  So the Julian date at 12:00 midnight GMT on January 
 * 1st, 1970 A.D. was 2,440,587.5 and the Modified Julian date was 40587.
 */
#define EPOCH     40587                /* modified julian date for 01/01/1970 */
#define HANGUP    "~.\r\n"	       /* hangup command, see cu(1) */
#define HOUR      (60 * 60)            /* seconds in an hour */
#define ONEDAY    (HOUR * 24)          /* seconds in a day */
#define OTM       '*'                  /* on time mark */
#define READ      0                    /* read file descriptor */
#define WRITE     1                    /* write file descriptor */
#define NISTFMT   "%d%*s%2d:%2d:%2d"   /* sscanf(3) format string for NIST */
#define USNOFMT   "%d%*d%2d%2d%2d"     /* sscanf(3) format string for USNO */

#define DELTA(x,y,z) fprintf (stdout, \
    "%s: the system clock is %ld second(s) %s.\n", x, y, z)

#define ONLYROOT(x) fprintf (stderr, \
    "%s: only root may set the system clock.\n", x)

#ifdef __linux__
#define USAGE(x) fprintf (stderr, "Usage:  %s [-dsv] [-u]\n", x)
#else
#define USAGE(x) fprintf (stderr, "Usage:  %s [-dsv]\n", x)
#endif __linux__

#ifndef __USLC__ 
typedef enum boolean {B_FALSE, B_TRUE} boolean_t;
#endif

static char *Sccsid =  "@(#)wjjr@csn.net:  nisttime.c 5.3, 4/15/95";

/*
 * returns the basename of pathname.
 */
static char *
BaseName (pathname)
char *pathname;
{
    char *basename;

    if ((basename = strrchr (pathname, '/')) == NULL) {
        basename = pathname;
    }
    else {
        basename++;
    }

    return basename;
}

/* 
 * returns the number of seconds since Midnight UTC on 01/01/1970.
 */
static long
Since1970 (Mjd, Hour, Min, Sec)
{
    return (((Mjd - EPOCH) * ONEDAY) + (Hour * HOUR) + (Min * 60) + Sec);
}

/* 
 * sets the CMOS clock to localtime on USL unixware PC's.
 */
#ifdef __USLC__    /* unixware */
static int 
WriteClk ()    
{
    int          clk;
    struct rtc_t sysclk;
    struct tm    *dt;
    time_t       ticks;

    if ((clk = open ("/dev/rtc", O_RDWR)) == -1) {
	perror ("/dev/rtc");
	return -1;
    }
    if (ioctl (clk, RTCRTIME, &sysclk) == -1) {
	perror ("ioctl()");
	return -2;
    }
    (void) time (&ticks);
    dt = localtime (&ticks);
    if (sysclk.rtc_statusb & RTC_DM) {    /* binary mode */
        sysclk.rtc_sec = dt->tm_sec;
        sysclk.rtc_min = dt->tm_min;
        sysclk.rtc_hr  = dt->tm_hour;
        sysclk.rtc_dow = dt->tm_wday;
        sysclk.rtc_dom = dt->tm_mday;
        sysclk.rtc_mon = dt->tm_mon+1;
        sysclk.rtc_yr  = dt->tm_year;
    }
    else {    /* BCD mode */
	sysclk.rtc_sec = (((dt->tm_sec / 10) * 16) + (dt->tm_sec % 10));
	sysclk.rtc_min = (((dt->tm_min / 10) * 16) + (dt->tm_min % 10));
        sysclk.rtc_hr  = (((dt->tm_hour / 10) * 16) + (dt->tm_hour % 10));
        sysclk.rtc_dow = dt->tm_wday;
        sysclk.rtc_dom = (((dt->tm_mday / 10) * 16) + (dt->tm_mday % 10));
        sysclk.rtc_mon = ((((dt->tm_mon+1) / 10) * 16) + ((dt->tm_mon+1) % 10));
        sysclk.rtc_yr = ((((dt->tm_year) / 10) * 16) + ((dt->tm_year) % 10));
    }
    if (ioctl (clk, RTCSTIME, &sysclk) == -1) {
	perror ("ioctl()");
	return -3;
    }
    (void) close (clk);

    return 0;
}
#endif __USLC__

main (argc, argv)
int  argc;
char **argv;
{
    FILE           *fp;                  /* pipe file descriptor */
    boolean_t      clkset = B_FALSE,     /* system clock was reset */
                   complete = B_FALSE,   /* finished receiving data */
                   dflg = B_FALSE,       /* select -d option */
                   errors = B_FALSE,     /* option errors */
                   no_data = B_TRUE,     /* no data received from time svc. */
                   sflg = B_FALSE,       /* select -s option */
#ifdef __linux__   
                   uflg = B_FALSE,       /* -u option for Linux */
#endif __linux__
                   vflg = B_FALSE,       /* select -v option */
                   usnoflg = B_FALSE;    /* call the USNO time service */
    char           line[BUFSIZ],         /* received data buffer */
                   *prog;                /* basename of program */
    extern   int   optind;
    register int   c,                    /* option character and loop ctr. */
                   lc = 0;               /* loop counter */
    int            Hour,                 /* hours, 0 to 23 */
                   Min,                  /* minutes, 0 to 60 */
                   Mjd,                  /* modified julian date */
                   Sec,                  /* seconds, 0 to 61 */
                   status,               /* child process status */
                   pd1[2],               /* pipe descriptor */
                   pd2[2];               /* pipe descriptor */
    long           delta = 0;            /* delta between system and UTC */
    pid_t          pid;                  /* child process pid */
    struct timeval ntv,                  /* UTC time structure */
                   otv;                  /* system time structure */


    /* basename determines which time-source is called */
    prog = BaseName (argv[0]);
    if (strncmp (prog, "usnotime", 8) == 0) {
        usnoflg = B_TRUE;
    }

#ifdef __linux__
    while ((c = getopt (argc, argv, "dsvu")) != -1) {
#else
    while ((c = getopt (argc, argv, "dsv")) != -1) {
#endif __linux__
        switch (c) {
            case 'd':  dflg = B_TRUE;
                       break;
            case 's':  if (getuid () == 0) {
                           sflg = B_TRUE;
                       }
                       else {
                           ONLYROOT(prog);
                           errors = B_TRUE;
                       } 
                       break;
#ifdef __linux__
            case 'u':  uflg = B_TRUE;
                       break;
#endif __linux__
            case 'v':  if (setvbuf (stdout, (char *) NULL, _IONBF, 0) != 0) {
                           perror ("setvbuf ()");
                       }
                       else {
                           vflg = B_TRUE;
                       }
                       break;
            case '?':  errors = B_TRUE;
                       break;
        }
    }
    if (argc > optind || errors) {
        USAGE(prog);
        exit (1);
    }

    /* create a pipeline and fork a child */
    if (pipe (pd1) < 0 || pipe (pd2) < 0) {
        fprintf (stderr, "%s: unable to create a pipeline, ", prog);
        perror ("pipe ()");
        exit (2);
    }
    if ((pid = fork ()) == -1) {
        fprintf (stderr, "%s: child not forked, ", prog);
        perror ("fork ()");
        exit (3);
    }

    /* request high cpu scheduling, parent and child */
#ifdef __USLC__
    (void) nice (0);
#else
    (void) nice (-20);
#endif __USLC__

    /* child process executes cu(1) */
    if (pid == 0) {
        static char *args[4];

        /* read half of pipeline */
        (void) close (READ);
        (void) dup (pd1[READ]);
        (void) close (pd1[READ]);
        (void) close (pd1[WRITE]);

        /* write half of pipeline */
        (void) close (WRITE);
        (void) dup (pd2[WRITE]);
        (void) close (pd2[WRITE]);
        (void) close (pd2[READ]);

        if (setvbuf (stdout, (char *) NULL, _IONBF, 0) != 0) {
            perror ("setvbuf()");
            exit (4);
        }

        args[0] = BaseName (CU);
        if (dflg) {
            args[1] = "-d";
            args[2] = prog;
            args[3] = (char *) NULL;
        }
        else {
            args[1] = prog;
            args[2] = (char *) NULL;
        }
        
        (void) execv (CU, args);

        _exit (0);
    }

    /* parent process */
    else {
        printf ("%d %d\nCalling, please standby ...\n", getpid (), pid);

        /* set up parents half of the pipeline */
        (void) close (pd1[READ]);
        (void) close (pd2[WRITE]);

        if ((fp = fdopen (pd2[READ], "r")) == NULL) {
            perror ("fdopen()");
            (void) kill (pid, SIGKILL);
            exit (5);
        }
        
        while (!complete && fgets (line, BUFSIZ-1, fp) != NULL) {
            /* Naval observatory */
            if (usnoflg) {
                if (line[0] == OTM) {
                    c++;
                }
                else {
                    (void) sscanf (line, USNOFMT, &Mjd, &Hour, &Min, &Sec);
                    if (c > 5) {
                        complete = B_TRUE;
                    }
                }
            }
            /* National Institute of Standards and Technology */
            else {
                (void) sscanf (line, NISTFMT, &Mjd, &Hour, &Min, &Sec);
                if (line[49] == OTM && c++ > 5) {
                    complete = B_TRUE;
                }
            }
            if (complete) {
                ntv.tv_sec = Since1970 (Mjd, Hour, Min, Sec);
                if (gettimeofday (&otv, (struct timezone *) NULL) != 0) {
                    perror ("gettimeofday()");
                    (void) kill (pid, SIGKILL);
                    exit (6);
                }
                delta = abs ((ntv.tv_sec - otv.tv_sec));
                if (delta > ONEDAY || delta == 0) {
                    sflg = B_FALSE;
                }
                if (sflg) {
                    if (settimeofday (&ntv,(struct timezone *)NULL) != 0) { 
                        perror ("settimeofday()");
			(void) kill (pid, SIGKILL);
                        exit (8);
                    }
                    else {
                        clkset = B_TRUE;
                    }
                }
                no_data = B_FALSE;
            }         
            if (vflg) {
                fprintf (stdout, "%s", line);
            }
            if (lc++ > 20 && !complete) {
                complete = no_data = B_TRUE;
            }
        }
        write (pd1[WRITE], HANGUP, strlen (HANGUP));
        wait (&status);
        fprintf (stdout, "\nCall complete.\n\n");
        if (no_data) {
            fprintf (stderr, "%s: no valid data received.\n", prog);
            exit (9);
        }
        if (ntv.tv_sec > otv.tv_sec) {
            DELTA(prog, delta, "slow");
        }
        if (ntv.tv_sec < otv.tv_sec) {
            DELTA(prog, delta, "fast");
        }
        if (ntv.tv_sec == otv.tv_sec) {
            fprintf (stdout, "%s: the system clock is on time.\n", prog);
        }
        if (clkset) {
            fprintf (stdout, "%s: reset the system clock.\n", prog);
        }

#ifdef __USLC__    /* unixware */
        if (sflg && WriteClk () == 0) {
            fprintf (stdout, "%s: reset the CMOS clock.\n", prog);
        }
#endif __USLC__

#ifdef __linux__
        if (sflg) {
            if ((pid = fork ()) == -1) {
                fprintf (stderr, "%s: child not forked, ", prog);
                perror ("fork ()");
                exit (10);
            }
            if (pid == 0) {
                static char *args[4];
    
                args[0] = BaseName (CLOCK);

                if (uflg) {
                    args[1] = "-u";
                    args[2] = "-w";
                    args[3] = (char *) NULL;
                }
                else {
                    args[1] = "-w";
                    args[2] = (char *) NULL;
                }
    
                execv (CLOCK, args);
    
                _exit (0);
            }
            else if (pid > 0) {
                wait (&status);
                if (WIFEXITED(status)) {
                    fprintf (stdout, "%s: reset the CMOS clock.\n", prog);
                }
            }
        }
#endif __linux__
    }
    return 0;
}
