/*
 * piece.c
 *
 * Function definitions for functions defined in the Piece class.
 */
/*

    3Dc, a game of 3-Dimensional Chess
    Copyright (C) 1995  Paul Hicks

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    E-Mail: P.Hicks@net-cs.ucd.ie
*/

#include <malloc.h>
#include "machine.h"
#include "3Dc.h"

static INLINE Boolean kingMayMove(Piece *,
                                  const File, const Rank, const Level);
static INLINE Boolean queenMayMove(Piece *,
                                   const File, const Rank, const Level);
static INLINE Boolean bishopMayMove(Piece *,
                                    const File, const Rank, const Level);
static INLINE Boolean knightMayMove(Piece *,
                                    const File, const Rank, const Level);
static INLINE Boolean rookMayMove(Piece *,
                                  const File, const Rank, const Level);
static INLINE Boolean princeMayMove(Piece *,
                                    const File, const Rank, const Level);
static INLINE Boolean princessMayMove(Piece *,
                                      const File, const Rank, const Level);
static INLINE Boolean abbeyMayMove(Piece *,
                                   const File, const Rank, const Level);
static INLINE Boolean cannonMayMove(Piece *,
                                    const File, const Rank, const Level);
static INLINE Boolean galleyMayMove(Piece *,
                                    const File, const Rank, const Level);
static INLINE Boolean pawnMayMove(Piece *,
                                  const File, const Rank, const Level);

#ifndef ABS
#define ABS(a) ((a) < 0 ? -(a) : (a))
#endif /* ABS */

extern Piece *pieceNew(const Title nType,
                       const File x, const Rank y, const Level z,
                       const Colour col)
{
  Piece *piece;

  piece = (Piece *)malloc(sizeof(Piece));

  if (!piece)
    return NULL;

  piece->xyzPos.xFile = x;
  piece->xyzPos.yRank = y;
  piece->xyzPos.zLevel = z;

  piece->bwSide = col;
  piece->nName = nType;
  piece->bVisible = True;
  piece->bHasMoved = False;
  piece->bInCheck = False;

  return piece;
}

extern void pieceDelete(Piece *piece)
{
  free(piece);
  piece = NULL;
}

extern Boolean pieceMayMove(Piece *piece,
                            const File xNew, const Rank yNew, const Level zNew)
{
  Boolean retval;

  if (!piece || !piece->bVisible)
    {
      n3DcErr = E3DcINVIS;
      return False;
    }

  /* Do bits which are the same for all pieces first */
  if (xNew == piece->xyzPos.xFile &&
      yNew == piece->xyzPos.yRank &&
      zNew == piece->xyzPos.zLevel)
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  if (Board[zNew][yNew][xNew] != NULL &&
      Board[zNew][yNew][xNew]->bwSide == piece->bwSide)
    {
      n3DcErr = E3DcBLOCK;
      return False;  /* Can't take a piece on your team */
    }

  switch (piece->nName)
    {
    case king:
      retval = kingMayMove(piece, xNew, yNew, zNew);
      break;
    case queen:
      retval = queenMayMove(piece, xNew, yNew, zNew);
      break;
    case bishop:
      retval = bishopMayMove(piece, xNew, yNew, zNew);
      break;
    case knight:
      retval = knightMayMove(piece, xNew, yNew, zNew);
      break;
    case rook:
      retval = rookMayMove(piece, xNew, yNew, zNew);
      break;
    case prince:
      retval = princeMayMove(piece, xNew, yNew, zNew);
      break;
    case princess:
      retval = princessMayMove(piece, xNew, yNew, zNew);
      break;
    case abbey:
      retval = abbeyMayMove(piece, xNew, yNew, zNew);
      break;
    case cannon:
      retval = cannonMayMove(piece, xNew, yNew, zNew);
      break;
    case galley:
      retval = galleyMayMove(piece, xNew, yNew, zNew);
      break;
    case pawn:
      retval = pawnMayMove(piece, xNew, yNew, zNew);
      break;
    default:
      retval = False;
      n3DcErr = E3DcSIMPLE;
    }

  return retval;
}

/*
 * Execute the move
 */
extern Boolean pieceMove(Piece *piece,
                         const File xNew, const Rank yNew, const Level zNew)
{
  Move thisMove;
  Boolean moveType; /* Not quite Boolean... */

  if (!(moveType = pieceMayMove(piece, xNew, yNew, zNew)))
    return False;

  /*
   * Keep record of move
   */
  thisMove.xyzBefore.xFile = piece->xyzPos.xFile;
  thisMove.xyzBefore.yRank = piece->xyzPos.yRank;
  thisMove.xyzBefore.zLevel = piece->xyzPos.zLevel;
  thisMove.xyzAfter.xFile = xNew;
  thisMove.xyzAfter.yRank = yNew;
  thisMove.xyzAfter.zLevel = zNew;
  thisMove.bHadMoved = piece->bHasMoved; /* For king, rook, pawn */

  if (moveType == EnPassant)
    {
      thisMove.pVictim = Board[zNew][yNew + (piece->bwSide == white ?
                                             -1 : 1)][xNew];
    }
  else if (moveType == Castle)
    {
      thisMove.pVictim = Board[1][yNew][xNew < 3 ? 0 : 7];
    }
  else if (moveType == Promote)
    {
      thisMove.pVictim = (Piece *)pawn;
    }
  else
    thisMove.pVictim = Board[zNew][yNew][xNew];

  stackPush(MoveStack, &thisMove);

  piece->bHasMoved = True;
  pieceDisplay(piece, False);
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;

  if (Board[zNew][yNew][xNew]) /* Kill victim */
    {
      pieceDisplay(Board[zNew][yNew][xNew], False);
      Board[zNew][yNew][xNew]->bVisible = False;
    }

  Board[zNew][yNew][xNew] = piece;
  piece->xyzPos.xFile = xNew;
  piece->xyzPos.yRank = yNew;
  piece->xyzPos.zLevel = zNew;
  pieceDisplay(piece, True);

  /* Now move any special pieces */
  if (moveType == Castle)
    {
      pieceDisplay(Board[1][yNew][xNew < 3 ? 0 : 7], False);

      Board[1][yNew][xNew < 3 ? xNew +1 : xNew -1] =
        Board[1][yNew][xNew < 3 ? 0 : 7];
      Board[1][yNew][xNew < 3 ? 0 : 7] = NULL;

      Board[1][yNew][xNew < 3 ? xNew +1 : xNew -1]->xyzPos.xFile =
        (xNew < 3 ? xNew+1 : xNew-1);

      pieceDisplay(Board[1][yNew][xNew < 3 ? xNew +1 : xNew -1], True);
    }
  else if (moveType == EnPassant)
    {
      pieceDisplay(Board[zNew][yNew == 5 ? 4 : 3][xNew], False);
      Board[zNew][yNew == 5 ? 4 : 3][xNew] = NULL;
    }
  else if (moveType == Promote)
    {
      pieceDisplay(piece, False);
      piecePromote(piece); /* This function asks for promotion type, etc. */
    }

  return True;
}

/*
 * Undo the move
 */
extern Boolean pieceUndo(void)
{
  Move *move;
  Colour bwMoved, bwTaken;
  Coord src, dest;

  move = stackPop(MoveStack);
  if (!move)
    {
      Err3Dc("Nothing to undo!", False);
      return False;
    }

  src = move->xyzAfter;
  dest = move->xyzBefore;

  bwMoved = Board[src.zLevel][src.yRank][src.xFile]->bwSide;
  bwTaken = (bwMoved == white ? black : white);

  /* Clear the "moved-to" square */
  pieceDisplay(Board[src.zLevel][src.yRank][src.xFile], False);

  /* Move the "moved" piece back */
  Board[dest.zLevel][dest.yRank][dest.xFile] =
    Board[src.zLevel][src.yRank][src.xFile];
  (Board[dest.zLevel][dest.yRank][dest.xFile])->xyzPos = dest;
  (Board[dest.zLevel][dest.yRank][dest.xFile])->bHasMoved = move->bHadMoved;

  if (move->pVictim == (Piece *)pawn &&
      ((src.yRank == 7 && bwMoved == white) ||
       (src.yRank == 0 && bwMoved == black)))
    {
      /* This piece was promoted from a pawn: demote it */
      Board[dest.zLevel][dest.yRank][dest.xFile]->nName = pawn;
      move->pVictim = NULL; /* To avoid complications later */
    }

  /* Draw the piece in its original space */
  pieceDisplay(Board[dest.zLevel][dest.yRank][dest.xFile], True);

  if (move->pVictim)
    {
      /* Put the taken piece back */
      if (move->pVictim->bwSide != bwMoved)
        {
          Board[src.zLevel][src.yRank][src.xFile] = move->pVictim;
          Board[src.zLevel][src.yRank][src.xFile]->bVisible = True;
          pieceDisplay(Board[src.zLevel][src.yRank][src.xFile], True);
        }
      else
        {
          /* The move undone was a castle */
          /* The king is back in the right place; now
           * fix the rook */
          if (src.xFile < dest.xFile)
            {
              /* Castled to a smaller-id square (Queen's side for white,
               * King's side for black */
              pieceDisplay(Board[1][dest.yRank][dest.xFile -1], False);
              Board[1][dest.yRank][0] = Board[dest.xFile -1][dest.yRank][1];
              Board[1][dest.yRank][dest.xFile -1] = NULL;
              (Board[1][dest.yRank][0])->xyzPos.xFile = 0;
              (Board[1][dest.yRank][0])->xyzPos.yRank = dest.yRank;
              (Board[1][dest.yRank][0])->xyzPos.zLevel = 1;
              pieceDisplay(Board[1][dest.yRank][0], True);
            }
          else
            {
              /* Castled to a larger-id square (Queen's side for black,
               * King's side for white */
              pieceDisplay(Board[1][dest.yRank][dest.xFile +1], False);
              Board[1][dest.yRank][7] = Board[1][dest.yRank][dest.xFile +1];
              Board[1][dest.yRank][dest.xFile +1] = NULL;
              (Board[1][dest.yRank][7])->xyzPos.xFile = 7;
              (Board[1][dest.yRank][7])->xyzPos.yRank = dest.yRank;
              (Board[1][dest.yRank][7])->xyzPos.zLevel = 1;
              pieceDisplay(Board[1][dest.yRank][7], True);
            }
        }
    }

  free(move);

  return True;
}

/*
 * Here down are the specific piece-movement functions
 */

static INLINE Boolean kingMayMove(Piece *piece,
                                  const File xNew,
                                  const Rank yNew,
                                  const zNew)
{
  File xDiff, xCur, xInc;
  Rank yDiff;
  Level zDiff;

  xDiff = xNew - piece->xyzPos.xFile;
  yDiff = yNew - piece->xyzPos.yRank;
  zDiff = zNew - piece->xyzPos.zLevel;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);
  zDiff = ABS(zDiff);

  if (xDiff > 2 || yDiff > 1 || zDiff > 1) /* Not allowed move more than 1 except when castling */
    {
      n3DcErr = E3DcDIST;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  if (pieceThreatened(piece, xNew, yNew, zNew) ||
      (xDiff == 2 && pieceThreatened(piece, ((xNew + piece->xyzPos.xFile)/2), yNew, zNew)))
    {
      n3DcErr = E3DcCHECK;
      return False; /* Can't move into or through check */
    }

  if (xDiff == 2)
    { /* Castling */
      File xRook;

      if (yDiff || zDiff)
        {
          n3DcErr = E3DcSIMPLE;
          return False;
        }

      /*
       * Determine x-pos of castling rook
       */
      if (xNew > piece->xyzPos.xFile)
        xRook = 7;
      else
        xRook = 0;

      if (piece->bInCheck)
        {
          n3DcErr = E3DcSIMPLE;
          return False;
        }
      else if (piece->bHasMoved ||
               Board[1][yNew][xRook]->bHasMoved)
        {
          n3DcErr = E3DcMOVED;
          return False;
        }
      else if (!Board[1][yNew][xRook])
        {
          n3DcErr = E3DcSIMPLE;
          return False;
        }

      if (xRook)
        xInc = 1;
      else
        xInc = -1;

      for (xCur = piece->xyzPos.xFile + xInc; xCur != xRook; xCur += xInc)
        {  /* Is the castle blocked? */
          if (Board[1][yNew][xCur])
            {
              n3DcErr = E3DcBLOCK;
              return False;
            }
        }

      return Castle;
    }

  return True;
}

static INLINE Boolean queenMayMove(Piece *piece,
                                   const File xNew,
                                   const Rank yNew,
                                   const zNew)
{
  File xDiff, xInc, xCur;
  Rank yDiff, yInc, yCur;
  Level zDiff, zInc, zCur;
  Coord xyzKing;
  Piece *pVictim;

  xInc = xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;
  zInc = zDiff = zNew - piece->xyzPos.zLevel;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);
  zDiff = ABS(zDiff);

  xInc = (xDiff ? xInc / xDiff : 0);
  yInc = (yDiff ? yInc / yDiff : 0);
  zInc = (zDiff ? zInc / zDiff : 0);

  if ((xDiff && yDiff && xDiff != yDiff) ||
      (xDiff && zDiff && xDiff != zDiff) ||
      (yDiff && zDiff && yDiff != zDiff))
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * 3D non-diagonal movement is not allowed either, but is catered for
   * with the previous 3 lines.
   */

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  for (xCur = piece->xyzPos.xFile + xInc,
       yCur = piece->xyzPos.yRank + yInc,
       zCur = piece->xyzPos.zLevel + zInc;
       xCur != xNew && yCur != yNew && zCur != zNew;
       xCur += xInc, yCur += yInc, zCur += zInc)
    {
      if (Board[piece->xyzPos.zLevel][yCur][xCur])
        {
          n3DcErr = E3DcBLOCK;
          return False;
        }
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  /* Undo faked move */
  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;

  return True;
}

static INLINE Boolean bishopMayMove(Piece *piece,
                                    const File xNew,
                                    const Rank yNew,
                                    const zNew)
{
  File xDiff, xInc, xCur;
  Rank yDiff, yInc, yCur;
  Level zDiff, zInc, zCur;
  Coord xyzKing;
  Piece *pVictim;

  xInc = xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;
  zInc = zDiff = zNew - piece->xyzPos.zLevel;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);
  zDiff = ABS(zDiff);

  xInc = (xDiff ? xInc / xDiff : 0);
  yInc = (yDiff ? yInc / yDiff : 0);
  zInc = (zDiff ? zInc / zDiff : 0);

  if (((xDiff && !yDiff && !zDiff) ||
      (!xDiff && yDiff && !zDiff) ||
      (!xDiff && !yDiff && zDiff)) ||/* Must move diagonally */
      (xDiff && yDiff && xDiff != yDiff) ||/* 2D non-diag movt not allowed */
      (xDiff && zDiff && xDiff != zDiff) ||/* 2D non-diag movt not allowed */
      (yDiff && zDiff && yDiff != zDiff))  /* 2D non-diag movt not allowed */
      {
        n3DcErr = E3DcSIMPLE;
        return False;
      }

  /*
   * 3D non-diagonal movement is not allowed either, but is catered for
   * with the previous 3 lines.
   */

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  for (xCur = piece->xyzPos.xFile + xInc,
       yCur = piece->xyzPos.yRank + yInc,
       zCur = piece->xyzPos.zLevel + zInc;
       xCur != xNew;
       xCur += xInc, yCur += yInc, zCur += zInc)
    {
      if (Board[piece->xyzPos.zLevel][yCur][xCur])
        {
          n3DcErr = E3DcBLOCK;
          return False;
        }
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;

  return True;
}

static INLINE Boolean knightMayMove(Piece *piece,
                                    const File xNew,
                                    const Rank yNew,
                                    const zNew)
{
  File xDiff;
  Rank yDiff;
  Coord xyzKing;
  Piece *pVictim;

  if (zNew != 1)
    {
      n3DcErr = E3DcLEVEL;
      return False; /* Knights may not change level */
    }

  xDiff = xNew - piece->xyzPos.xFile;
  yDiff = yNew - piece->xyzPos.yRank;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);


  if ((xDiff + yDiff) != 3)
    return False;

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;

  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean rookMayMove(Piece *piece,
                                  const File xNew,
                                  const Rank yNew,
                                  const zNew)
{
  File xDiff, xInc, xCur;
  Rank yDiff, yInc, yCur;
  Level zDiff, zInc, zCur;
  Coord xyzKing;
  Piece *pVictim;

  xInc = xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;
  zInc = zDiff = zNew - piece->xyzPos.zLevel;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);
  zDiff = ABS(zDiff);

  xInc = (xDiff ? xInc / xDiff : 0);
  yInc = (yDiff ? yInc / yDiff : 0);
  zInc = (zDiff ? zInc / zDiff : 0);

  if ((xDiff && yDiff) ||/* Must move linearly */
      (xDiff && zDiff && xDiff != zDiff) || /* 2D non-diag movt not allowed */
      (yDiff && zDiff && yDiff != zDiff))   /* 2D non-diag movt not allowed */
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  for (xCur = piece->xyzPos.xFile + xInc,
       yCur = piece->xyzPos.yRank + yInc,
       zCur = piece->xyzPos.zLevel + zInc;
       xCur != xNew;
       xCur += xInc, yCur += yInc, zCur += zInc)
    {
      if (Board[piece->xyzPos.zLevel][yCur][xCur])
        {
          n3DcErr = E3DcBLOCK;
          return False;
        }
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean princeMayMove(Piece *piece,
                                    const File xNew,
                                    const Rank yNew,
                                    const zNew)
{
  File xDiff;
  Rank yDiff;
  Coord xyzKing;
  Piece *pVictim;

  if (zNew != piece->xyzPos.zLevel)
    {
      n3DcErr = E3DcLEVEL;
      return False; /* Princes may not change level */
    }

  xDiff = xNew - piece->xyzPos.xFile;
  yDiff = yNew - piece->xyzPos.yRank;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);

  if (xDiff > 1 || yDiff > 1) /* Not allowed move more than 1 */
    {
      n3DcErr = E3DcDIST;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean princessMayMove(Piece *piece,
                                      const File xNew,
                                      const Rank yNew,
                                      const zNew)
{
  File xDiff, xInc, xCur;
  Rank yDiff, yInc, yCur;
  Coord xyzKing;
  Piece *pVictim;

  if (zNew != piece->xyzPos.zLevel)
    {
      n3DcErr = E3DcLEVEL;
      return False; /* Princesses may not change level */
    }

  xInc = xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);

  xInc = (xDiff ? xInc / xDiff : 0);
  yInc = (yDiff ? yInc / yDiff : 0);

  if (xDiff && yDiff && xDiff != yDiff) /* 2D non-diagonal movt not allowed */
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  for (xCur = piece->xyzPos.xFile + xInc,
       yCur = piece->xyzPos.yRank + yInc;
       xCur != xNew;
       xCur += xInc, yCur += yInc)
    {
      if (Board[piece->xyzPos.zLevel][yCur][xCur])
        {
          n3DcErr = E3DcBLOCK;
          return False;
        }
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean abbeyMayMove(Piece *piece,
                                   const File xNew,
                                   const Rank yNew,
                                   const zNew)
{
  File xDiff, xInc, xCur;
  Rank yDiff, yInc, yCur;
  Coord xyzKing;
  Piece *pVictim;

  if (zNew != piece->xyzPos.zLevel)
    {
      n3DcErr = E3DcLEVEL;
      return False; /* Abbies may not change level */
    }

  xInc = xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);

  xInc = (xDiff ? xInc / xDiff : 0);
  yInc = (yDiff ? yInc / yDiff : 0);

  if (xDiff != yDiff) /* Non-diagonal movt not allowed */
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  for (xCur = piece->xyzPos.xFile + xInc,
       yCur = piece->xyzPos.yRank + yInc;
       xCur != xNew;
       xCur += xInc, yCur += yInc)
    {
      if (Board[piece->xyzPos.zLevel][yCur][xCur])
        {
          n3DcErr = E3DcBLOCK;
          return False;
        }
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean cannonMayMove(Piece *piece,
                                    const File xNew,
                                    const Rank yNew,
                                    const zNew)
{
  File xDiff;
  Rank yDiff;
  Level zDiff;
  Coord xyzKing;
  Piece *pVictim;

  xDiff = xNew - piece->xyzPos.xFile;
  yDiff = yNew - piece->xyzPos.yRank;
  zDiff = zNew - piece->xyzPos.zLevel;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);
  zDiff = ABS(zDiff);

  if (((xDiff + yDiff + zDiff) != 6) ||
      ((xDiff != 3) && (yDiff != 3)) ||
      ((xDiff != 2) && (yDiff != 2) && (zDiff != 2)))
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean galleyMayMove(Piece *piece,
                                    const File xNew,
                                    const Rank yNew,
                                    const zNew)
{
  File xDiff, xInc, xCur;
  Rank yDiff, yInc, yCur;
  Coord xyzKing;
  Piece *pVictim;

  if (zNew != piece->xyzPos.zLevel)
    {
      n3DcErr = E3DcLEVEL;
      return False; /* Gallies may not change level */
    }

  xInc = xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);

  xInc = (xDiff ? xInc / xDiff : 0);
  yInc = (yDiff ? yInc / yDiff : 0);

  if (xDiff && yDiff) /* 2D non-linear movt not allowed */
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * At this stage, we have determined that, given an empty board,
   * the move is legal.  Now take other pieces into account.
   */
  for (xCur = piece->xyzPos.xFile + xInc,
       yCur = piece->xyzPos.yRank + yInc;
       xCur != xNew;
       xCur += xInc, yCur += yInc)
    {
      if (Board[piece->xyzPos.zLevel][yCur][xCur])
        return False;
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;
  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel))
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
  return True;
}

static INLINE Boolean pawnMayMove(Piece *piece,
                                  const File xNew, const Rank yNew, const zNew)
{
  File xDiff;
  Rank yDiff, yInc;
  Coord xyzKing;
  Piece *pVictim;

  if (zNew != piece->xyzPos.zLevel)
    {
      n3DcErr = E3DcLEVEL;
      return False; /* Pawns may not change level */
    }

  xDiff = xNew - piece->xyzPos.xFile;
  yInc = yDiff = yNew - piece->xyzPos.yRank;

  xDiff = ABS(xDiff);
  yDiff = ABS(yDiff);

  /*
   * Pawns must move at least 1 forward
   */
  if (!yDiff ||
      (((yInc /= yDiff) < 0) && piece->bwSide == white) ||
      (yInc > 0 && piece->bwSide == black)) /* Moving backwards */
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /* Under no circumstances can you move more than one horizontally */
  if (xDiff > 1)
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  /*
   * It is difficult to cater for 'en passant' in the middle of a
   * conditional.  So, against all convention laid out in other
   * rules functions, I am checking a move and returning true if it
   * is valid, rather than returning False if it is invalid.
   */
   /*
    * TODO:
    *  Each piece must have an identifier; either its memory location
    *  or its offset into the Muster.  That way this can be used as the
    *  4th line of this conditional.
    */
#if 0
  (  stackPeek(MoveStack, 1)->nId == Board[zNew][yNew - yInc][xNew]->nId &&
    !(stackPeek(MoveStack, 1)->bHadMoved) &&
      Board[zNew][yNew - yInc][xNew]->bHasMoved /* Moved only once */
   )
#endif
    if (xDiff == 1 && yDiff == 1 && !Board[zNew][yNew][xNew])
      { /* En passant? */
        if (Board[zNew][yNew - yInc][xNew] && /* 'Takable' piece */
            Board[zNew][yNew - yInc][xNew]->nName == pawn && /* Is pawn */
            Board[zNew][yNew - yInc][xNew]->bwSide != piece->bwSide && /* Is enemy */
            1) /* Dummy line to reduce no. of changes */
          {
            return EnPassant;
          }
        else
          {
            n3DcErr = E3DcSIMPLE;
            return False;
          }
      }

  /*
   * Pawns can not move forward under these conditions:
   *  They move more than 2
   *  They move more than 1 and they have already moved
   *  They attempt to take any piece (catered for in next conditional)
   */
  if (yDiff > 2 || /* Move too far */
      (piece->bHasMoved && yDiff == 2)) /* Move too far */
    {
      n3DcErr = E3DcDIST;
      return False;
    }

  /*
   * Pawns may not take anything under these conditions:
   *  They do not move diagonally forward one space
   *  The victim is an ally
   */
  if (Board[zNew][yNew][xNew]  && /* Taking something */
      ((xDiff != 1 || yDiff != 1) || /* Not moving diagonally */
       Board[zNew][yNew][xNew]->bwSide == piece->bwSide))
    {
      n3DcErr = E3DcSIMPLE;
      return False;
    }

  pVictim = Board[zNew][yNew][xNew];
  if (pVictim) pVictim->bVisible = False;
  Board[zNew][yNew][xNew] = piece; /* Imitate the movement */
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = NULL;
  xyzKing = Muster[piece->bwSide][idx(king, 0)]->xyzPos;

  if (pieceThreatened(Muster[piece->bwSide][idx(king, 0)],
                      xyzKing.xFile, xyzKing.yRank, xyzKing.zLevel)
      != NULL)
    {
      Board[zNew][yNew][xNew] = pVictim;
      if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
      Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;
      n3DcErr = E3DcCHECK;
      return False;  /* Move is illegal if it puts king in check */
    }

  Board[zNew][yNew][xNew] = pVictim;
  if (Board[zNew][yNew][xNew]) Board[zNew][yNew][xNew]->bVisible = True;
  Board[piece->xyzPos.zLevel][piece->xyzPos.yRank][piece->xyzPos.xFile] = piece;

  if ((yNew == 7 && piece->bwSide == white) ||
      (yNew == 0 && piece->bwSide == black))
    return Promote;

  return True;
}
