#ifndef __debuglib_h
#define __debuglib_h

/*
 * The debugging library is intended to allow users to set up debugging on
 * the basis of complete packages, or subgroups of the packages.  It also
 * lets each of the different subgroups have a different value which allows
 * finer gradation among available debugging choices.
 *
 * Debugging values are specified on the command line in the form
 * -D<debug-name>[=value].  If no value is specified, it defaults to 1.
 * A debug name is a colon separated list of packages and subgroups in those
 * packages.  For example, to set debugging on all of tcplib, you could use
 * -Deutl:tcplib
 * To set debugging on only sending messages in tcplib, you could use
 * -Deutl:tcplib:sendmsg
 * The debugging library will attempt to match debugging queries against the
 * most specific value.  If that fails, then more and more general debugging
 * values will be set, e.g. if while executing your program, you check to see
 * if eutl:tcplib:sendmsg debugging is set on.  If that value was not 
 * explicitly set, but the value for eutl:tcplib was, then that value will
 * be returned.
 * Each of the different packages should document what values can be set
 * for debugging purposes.  Failing this, you can set debugging flags on the
 * debugging library which will tell you about the different debugging 
 * values being checked.
 * 
 */

/* This function will destructively update argc, argv removing all arguments
   of the form -D<debug-name>[=value] */
void GetDebuggingArguments(int *argc,char **argv);
int GetDebuggingValue(char *debugname);
void SetDebuggingValue(char *debugname,int value);

/* The DEBUG macro allows you to easily set up debugging statements.
   Because of the bizarre way in which macros with arbitrary numbers of 
   arguments work, you need to put parenthesis around the format stuff.
   For example, this could be used as
   DEBUG("eutl:debuglib:check",1,
          ("Checking the debugging value of %s",debugname));
   Which would print out the message in the inner parenthesis along with
   information about what the debugvalue and the line and file were to 
   stdout if the debugging value "eutl:debuglib:check" is >= 1.
   There is also a version of this macro called QDEBUG which will not print
   out the information about the debugvalue line and file, it will just
   print what you specified in the format */

#define DEBUG(debugname,minval,format) {\
   if (__debuglib_debugging_on &&\
       GetDebuggingValue(debugname)>=(minval)) {\
     printf("DEBUG(debugname=%s,minlevel=%d,curlevel=%d,line=%d,file=%s)\n",\
	    debugname,GetDebuggingValue(debugname),(minval),\
	    __LINE__, __FILE__);\
     printf format;\
   }\
 }		      

/* Quiet Debug */
#define QDEBUG(debugname,minval,format) {\
   if (__debuglib_debugging_on &&\
       GetDebuggingValue(debugname)>=(minval)) {\
     printf format;\
   }\
 }		      
				    
/* Test Debugging On 
   used like:
   if (TDEBUG("eutl:debuglib:check",1)) 
      CallDumpDataStructureFunction(arg1,arg2,arg3);
*/
#define TDEBUG(debugname,minval) \
 (__debuglib_debugging_on && GetDebuggingValue(debugname) >= (minval))


/* exceptions are used, see errlib for a better description of how to 
   use them */

extern char *debuglib_packagever;

extern char *debuglib_Ecantcreatetable;
extern char *debuglib_Egeterror; /* Error getting stored value */
/* this will be set to one if there is any debugging currently set.
   this lets you have a faster test without suffering the overhead of a 
   procedure call in the normal case.  (As if that much overhead is really
   that much, but some people worry about this -- anyway, the complexity can
   be hidden in a macro) */
extern int __debuglib_debugging_on;

#endif

