/*
 * static char *rcs_treasure_c =
 *   "$Id: treasure.c,v 1.3 1993/04/25 16:08:34 frankj Exp $";
 */

/*
    CrossFire, A Multiplayer game for X-windows

    Copyright (C) 1992 Frank Tore Johansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be reached via e-mail to frankj@ifi.uio.no.
*/

#include <global.h>
#include <treasure.h>
#include <spellist.h>
#include <funcpoint.h>

/*
 * Allocate and return the pointer to an empty treasurelist structure.
 */

treasurelist *get_empty_treasurelist() {
  treasurelist *tl = (treasurelist *) malloc(sizeof(treasurelist));
  if(tl==NULL)
    fatal(OUT_OF_MEMORY);
  tl->name=NULL;
  tl->next=NULL;
  tl->items=NULL;
  return tl;
}

/*
 * Allocate and return the pointer to an empty treasure structure.
 */

treasure *get_empty_treasure() {
  treasure *t = (treasure *) malloc(sizeof(treasure));
  if(t==NULL)
    fatal(OUT_OF_MEMORY);
  t->item=NULL;
  t->next=NULL;
  t->next_yes=NULL;
  t->next_no=NULL;
  t->chance=100;
  t->nrof=0;
  t->magic=0;
  return t;
}

/*
 * Reads the lib/treasures file from disk, and parses the contents
 * into an internal treasure structure (very linked lists)
 */

treasure *load_treasure(FILE *fp) {
  char buf[MAX_BUF], *cp, variable[MAX_BUF];
  treasure *t=get_empty_treasure();
  int value;

  nroftreasures++;
  while(fgets(buf,MAX_BUF,fp)!=NULL) {
    if(*buf=='#')
      continue;
    if((cp=strchr(buf,'\n'))!=NULL)
      *cp='\0';
    cp=buf;
    while(*cp==' ') /* Skip blanks */
      cp++;
    if(sscanf(cp,"arch %s",variable)) {
      if((t->item=find_archetype(variable))==NULL)
        LOG(llevError,"Treasure lacks archetype: %s\n",variable);
    } else if(sscanf(cp,"chance %d",&value))
      t->chance=(unsigned char) value;
    else if(sscanf(cp,"nrof %d",&value))
      t->nrof=(unsigned short) value;
    else if(sscanf(cp,"magic %d",&value))
      t->magic=(unsigned char) value;
    else if(!strcmp(cp,"yes"))
      t->next_yes=load_treasure(fp);
    else if(!strcmp(cp,"no"))
      t->next_no=load_treasure(fp);
    else if(!strcmp(cp,"end"))
      return t;
    else if(!strcmp(cp,"more")) {
      t->next=load_treasure(fp);
      return t;
    } else
      LOG(llevError,"Unknown treasure-command: %s\n",cp);
  }
  LOG(llevError,"treasure lacks 'end'.\n");
  return t;
}

/*
 * Opens LIBDIR/treasure and reads all treasure-declarations from it.
 * Each treasure is parsed with the help of load_treasure().
 */

void load_treasures() {
  FILE *fp;
  char filename[MAX_BUF], buf[MAX_BUF], name[MAX_BUF];
  treasurelist *previous=NULL;

  sprintf(filename,"%s/%s",LibDir,Treasures);
  if((fp=open_and_uncompress(filename,0))==NULL) {
    LOG(llevError,"Can't open treasure file.\n");
    return;
  }
  while(fgets(buf,MAX_BUF,fp)!=NULL) {
    if(*buf=='#')
      continue;
    if(sscanf(buf,"treasure %s\n",name)) {
      treasurelist *tl=get_empty_treasurelist();
      tl->name=add_string(name);
      if(previous==NULL)
        first_treasurelist=tl;
      else
        previous->next=tl;
      previous=tl;
      tl->items=load_treasure(fp);
    } else
      LOG(llevError,"Treasure-list didn't understand: %s\n",buf);
  }
  close_and_delete(fp);
}

/*
 * Searches for the given treasurelist in the globally linked list
 * of treasurelists which has been built by load_treasures().
 */

treasurelist *find_treasurelist(char *name) {
  char *tmp=find_string(name);
  treasurelist *tl;
  if(tmp!=NULL)
    for(tl=first_treasurelist;tl!=NULL;tl=tl->next)
      if(tmp==tl->name)
        return tl;
  LOG(llevError,"Couldn't find treasurelist %s\n",name);
  return NULL;
}

/*
 * Searches, with the help of find_treasurelist, for the given
 * treasurelist.  Returns the first of the treasures within
 * the linked list of treasures in the found list.
 * Returns NULL on failure.
 */

treasure *find_treasure(char *name) {
  treasurelist *tl=find_treasurelist(name);
  if(tl!=NULL)
    return tl->items;
  else
    return NULL;
}

/*
 * Generates the objects specified by the given treasure.
 * It goes recursively through the rest of the linked list.
 * If there is a certain percental chance for a treasure to be generated,
 * this is taken into consideration.
 * The second argument specifies for which object the treasure is
 * being generated.
 * If flag isn't GT_ENVIRONMENT, monster_check_apply() is called.
 * If flag is GT_INVISIBLE, only invisible objects are generated (ie, only
 * abilities.  This is used by summon spells, thus no summoned monsters
 * start with equipment, but only their abilities).
 */

void create_treasure(treasure *t, object *op, int flag, int difficulty) {
  object *tmp;

  if(t->chance>=100||RANDOM()%100+1<t->chance) {
    if(t->item->clone.invisible != 0 || flag != GT_INVISIBLE) {
      tmp=arch_to_object(t->item);
      fix_generated_item(tmp,difficulty,t->magic);
      if(t->nrof&&tmp->nrof<=1)
        tmp->nrof=RANDOM()%t->nrof+1;
      if(flag != GT_ENVIRONMENT) {
        insert_ob_in_ob(tmp,op);
        (*monster_check_apply_func)(op,tmp);
      } else {
        tmp->x=op->x,tmp->y=op->y;
        insert_ob_in_map(tmp,op->map);
      }
    }
    if(t->next_yes!=NULL)
      create_treasure(t->next_yes,op,flag,difficulty);
  } else
    if(t->next_no!=NULL)
      create_treasure(t->next_no,op,flag,difficulty);
  if(t->next!=NULL)
    create_treasure(t->next,op,flag,difficulty);
}

/*
 * This is a new way of calculating the chance for an item to have
 * a specific magical bonus.
 * The array has two arguments, the difficulty of the level, and the
 * magical bonus "wanted".
 */

int difftomagic_list[DIFFLEVELS][MAXMAGIC+1] =
{
/*chance of magic    difficulty*/
/* +0  +1 +2 +3 +4 */
  { 95, 2, 2, 1, 0 }, /*1*/
  { 92, 5, 2, 1, 0 }, /*2*/
  { 85,10, 4, 1, 0 }, /*3*/
  { 80,14, 4, 2, 0 }, /*4*/
  { 75,17, 5, 2, 1 }, /*5*/
  { 70,18, 8, 3, 1 }, /*6*/
  { 65,21,10, 3, 1 }, /*7*/
  { 60,22,12, 4, 2 }, /*8*/
  { 55,25,14, 4, 2 }, /*9*/
  { 50,27,16, 5, 2 }, /*10*/
  { 45,28,18, 6, 3 }, /*11*/
  { 42,28,20, 7, 3 }, /*12*/
  { 40,27,21, 8, 4 }, /*13*/
  { 38,25,22,10, 5 }, /*14*/
};

/*
 * Based upon the specified difficulty and upon the difftomagic_list array,
 * a random magical bonus is returned.  This is used when determine
 * the magical bonus created on specific maps.
 */

int magic_from_difficulty(int difficulty)
{
  int percent,loop,chance;

  difficulty = difficulty-1;
  if(difficulty<0)
    difficulty=0;

  percent = RANDOM()%100;

  if (difficulty>=DIFFLEVELS)
    difficulty=DIFFLEVELS-1;

  for(loop=0;loop<(MAXMAGIC+1);++loop) {
    chance = difftomagic_list[difficulty][loop];
    if (percent<chance) {
      break;
    } else {
      percent -= chance;
    }
  }
  if (loop==(MAXMAGIC+1)) {
    LOG(llevError,"Warning, table for difficulty %d bad.\n",difficulty);
    loop=0;
  }
/*  printf("Chose magic %d for difficulty %d\n",loop,difficulty);*/
  return loop;
}

/*
 * Sets magical bonus in an object, and recalculates the effect on
 * the armour variable, and the effect on speed of armour.
 * This function doesn't work properly, should add use of archetypes
 * to make it truly absolute.
 */

void set_abs_magic(object *op, int magic) {
  if(!magic)
    return;
  op->magic=magic;
  op->weight=(op->weight*(100-magic*10))/100;
  if(op->armour)
    op->armour+=(op->armour*magic)/20;
  if(op->type==ARMOUR)
    ARMOUR_SPEED(op)=(ARMOUR_SPEED(op)*(100+magic*10))/100;
}

/*
 * Sets a random magical bonus in the given object based upon
 * the given difficulty, and the given max possible bonus.
 */

void set_magic(int difficulty,object *op, int max_magic) {
  int i;
  i = magic_from_difficulty(difficulty);
  if(i > max_magic)
    i = max_magic;
  set_abs_magic(op,i);
}

/*
 * Randomly adds one magical ability to the given object.
 */

void set_ring_bonus(object *op,int bonus) {
  int r=RANDOM()%(bonus>0?24:10);
  if(op->type==AMULET)
    if(!(RANDOM()%21))
      r=19+RANDOM()%2;
    else if(RANDOM()&2)
      r=9;
    else
      r=10+RANDOM()%9;
  switch(r) {
  case 0:
    if(op->stats.Str)
      return;
    op->stats.Str=bonus;
    break;
  case 1:
    if(op->stats.Int)
      return;
    op->stats.Int=bonus;
    break;
  case 2:
    if(op->stats.Con)
      return;
    op->stats.Con=bonus;
    break;
  case 3:
    if(op->stats.Dex)
      return;
    op->stats.Dex=bonus;
    break;
  case 4:
    if(op->stats.Cha)
      return;
    op->stats.Cha=bonus;
    break;
  case 5:
    if(op->stats.Wis)
      return;
    op->stats.Wis=bonus;
    break;
  case 6:
    if(op->stats.dam)
      return;
    op->stats.dam=bonus;
    break;
  case 7:
    if(op->stats.wc)
      return;
    op->stats.wc=bonus;
    break;
  case 8:
    if(op->stats.food)
      return;
    op->stats.food=bonus; /* hunger/sustenance */
    break;
  case 9:
    if(op->stats.ac)
      return;
    op->stats.ac=bonus;
    break;
  case 10:
    if(bonus==2) { /* Maybe make an artifact of this? */
      op->immune|=AT_PARALYZE;
      op->immune|=AT_SLOW;
      op->stats.exp+=1;
      op->value*=5;
    } else
      op->protected|=AT_PARALYZE;
    break;
  case 11:
    op->protected|=AT_MAGIC;
    op->value*=2;
    break;
  case 12:
    op->protected|=AT_ELECTRICITY;
    break;
  case 13:
    op->protected|=AT_FIRE;
    op->vulnerable|=AT_COLD;
    op->value=(op->value*3)/2;
    break;
  case 14:
    op->protected|=AT_DRAIN;
    break;
  case 15:
    op->protected|=AT_SLOW;
    break;
  case 16:
    op->protected|=AT_COLD;
    op->vulnerable|=AT_FIRE;
    break;
  case 17:
    op->protected|=AT_POISON;
    break;
  case 18:
    op->protected|=AT_FEAR;
    break;
  case 19:
    if(op->type==AMULET) {
      SET_REFL_SPELL(op);
      op->value*=11;
    } else {
      op->stats.hp=1; /* regenerate hit points */
      op->value*=4;
    }
    break;
  case 20:
    if(op->type==AMULET) {
      SET_REFL_MISSILE(op);
      op->value*=9;
    } else {
      op->stats.sp=1; /* regenerate spell points */
      op->value*=3;
    }
    break;
  case 21:
    if(op->stats.exp)
      return;
    op->stats.exp=bonus; /* Speed! */
    op->value=(op->value*2)/3;
    break;
  }
  if(bonus>0)
    op->value*=2*bonus;
  else
    op->value= -(op->value*2*bonus)/3;
}

/*
 * get_magic(diff) will return a random number between 0 and 4.
 * diff can be any value above 2.  The higher the diff-variable, the
 * higher is the chance of returning a low number.
 * It is only used in fix_generated_treasure() to set bonuses on
 * rings and amulets.
 * Another scheme is used to calculate the magic of weapons and armours.
 */

int get_magic(int diff) {
  int i;
  if(diff<3)
    diff=3;
  for(i=0;i<4;i++)
    if(RANDOM()%diff) return i;
  return 4;
}

#define DICE2	(get_magic(2)==2?2:1)
#define DICESPELL (RANDOM()%3+RANDOM()%3+RANDOM()%3+RANDOM()%3+RANDOM()%3)

/*
 * generate_treasure() creates an object based upon the given list of
 * objectnames (or new objectlists) and the given difficulty.
 * The lists are specified in the include/treasure.h file.
 * NULL is returned on failure.
 */

object *generate_treasure(int list,int difficulty) {
  int max_tries=1,idifficulty=difficulty;

  if(difficulty==-1)
    difficulty = 5;/* Store difficulty */
  if(list<0)
    return NULL;
  if(list>NROF_T) {
    LOG(llevError,"Illegal treasure list: %d\n",list);
    return NULL;
  }
  while(max_tries++<100){
    int roll=RANDOM()%20;
    object *op;
    if(!strcmp("unused",treasure_list[list][roll].name))
      continue;
    if (treasure_list[list][roll].nrof== -1)
      if(idifficulty!= -1)
        continue;

    if(!strncmp("list",treasure_list[list][roll].name,4)) {
      int newlist;

      sscanf(treasure_list[list][roll].name,"list %d",&newlist);
      if (difficulty>=treasure_list[list][roll].magic) {
/*	if (treasure_list[list][roll].magic)
	  printf("Took limited list transition %d,%d.\n",
		 difficulty,treasure_list[list][roll].magic);*/
	list = newlist;
	continue;
      } else {
	if (treasure_list[list][roll].nrof) {
/*	  printf("Taking unused special.\n");*/
	  continue;
	} else {
/*	  printf("Taking nothing special.\n");*/
	  return NULL;
	}
      }
    }
    if(!strcmp("nothing",treasure_list[list][roll].name))
      return NULL;
    op=get_archetype(treasure_list[list][roll].name);
    if(op==NULL) {
      LOG(llevError,"Illegal archetype: %s list %d roll %d\n",
              treasure_list[list][roll].name, list, roll);
      return NULL;
    }
    if(treasure_list[list][roll].nrof)
      op->nrof=RANDOM()%treasure_list[list][roll].nrof+1;
    fix_generated_item(op,difficulty,treasure_list[list][roll].magic);
    return op;
  }
  LOG(llevError,"Recursive error in treasure list %d.\n",list);
  return NULL;
}

/*
 * fix_generated_item():  This is called after an item is generated, in
 * order to set it up right.  This produced magical bonuses, puts spells
 * into scrolls/books/wands, etc.
 */

void fix_generated_item(object *op,int difficulty, int max_magic) {
  if(!op->magic)
    set_magic(difficulty,op,max_magic);
  switch(op->type) {
  case BRACERS:
    if(!RANDOM()%8) {
      set_ring_bonus(op,DICE2);
      op->value*=2;
      if(!RANDOM()%6) {
        int d=RANDOM()%2?DICE2:-DICE2;
        if(d>0)
          op->value*=5;
        set_ring_bonus(op,d);
      }
    }
    break;
  case AMULET:
    if(amulet_arch==NULL)
      amulet_arch=find_archetype("amulet");
    if(op->arch==amulet_arch)
      op->value*=5; /* Since it's not just decoration */
  case RING:
    if(ring_arch==NULL)
      ring_arch=find_archetype("ring");
    if(op->arch==NULL) {
      remove_ob(op);
      free_object(op);
      op=NULL;
      break;
    }
    if(op->arch!=ring_arch&&op->arch!=amulet_arch) /* It's a special artefact!*/
      break;
    set_ring_bonus(op,DICE2);
    if(op->type!=RING)
      break;
    if(!(RANDOM()%4)) {
      int d=RANDOM()%2?DICE2:-DICE2;
      if(d>0)
        op->value*=3;
      set_ring_bonus(op,d);
      if(!(RANDOM()%4)) {
        int d=RANDOM()%3?-DICE2:DICE2;
        if(d>0)
          op->value*=5;
        set_ring_bonus(op,d);
      }
    }
    if(op->arch->animations)
      op->face.number=op->arch->faces[RANDOM()%op->arch->animations];
    break;
  case SPELLBOOK:
    do
      op->stats.sp=RANDOM()%NROFREALSPELLS;
    while (RANDOM()%10>=spells[op->stats.sp].books);
    op->value=(op->value*spells[op->stats.sp].level)/
               (spells[op->stats.sp].level+4);
    break;
  case WAND:
    do 
      op->stats.sp=RANDOM()%NROFREALSPELLS;
    while (!spells[op->stats.sp].charges||
           spells[op->stats.sp].level>DICESPELL);
    if (spells[op->stats.sp].cleric)
    { /* Make the wand into a staff */
      short i = op->stats.sp;
      if (staff_arch == NULL)
        staff_arch = find_archetype("staff");
      copy_object(&staff_arch->clone, op);
      op->stats.sp = i;
    }
    op->stats.food=RANDOM()%spells[op->stats.sp].charges+1;
    op->value=(op->value*spells[op->stats.sp].level)/
               (spells[op->stats.sp].level+4);
    break;
  case SCROLL:
    do
      op->stats.sp=RANDOM()%NROFREALSPELLS;
    while (!spells[op->stats.sp].scrolls||
           spells[op->stats.sp].scroll_chance<=RANDOM()%10);
    op->nrof=RANDOM()%spells[op->stats.sp].scrolls+1;
    op->value=(op->value*spells[op->stats.sp].level)/
               (spells[op->stats.sp].level+4);
    break;
  }
}
