/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include <sys/types.h>
#include <dirent.h>
#include "life.h"


/*
 * Structure for command macros.
 */
typedef	struct	{
	UCHAR *	begptr;		/* beginning of data (NULL if none) */
	UCHAR *	endptr;		/* end of data */
} MACRO;


static	int	TtyChar PROTO((INPUT *));
static	void	TtyTerm PROTO((INPUT *));

static	int	FileChar PROTO((INPUT *));
static	void	FileTerm PROTO((INPUT *));

static	int	LoopChar PROTO((INPUT *));
static	void	LoopTerm PROTO((INPUT *));

static	int	MacroChar PROTO((INPUT *));
static	void	MacroTerm PROTO((INPUT *));

static	FILE *	OpenLibraryFile PROTO((char *, char *));
static	BOOL	MatchFile PROTO((char *, char *));
static	BOOL	SpaceWait PROTO((BOOL));
static	int	SortFunc PROTO((char **, char **));
static	void	AddFile PROTO((LIST *, char *));
static	MACRO *	FindMacro PROTO((int));


static	int	helprow;		/* current row of help text */
static	BOOL	helpbusy;		/* true if doing help display */
static	BOOL	helpabort;		/* true if help is aborted */
static	MACRO	macros[26 * 2 + 9];	/* macros a-z, A-Z, and 1-9 */


/*
 * Read the next character from the appropriate input source.
 * EOF is never returned by this routine (longjmps are done instead).
 * This routine is usually called indirectly by ScanChar.
 */
int
ReadChar()
{
	INPUT *	ip;
	int	ch;

	while (TRUE) {
		ip = curinput;
		ch = (*ip->getchar)(ip);

		if (ch != EOF)
			break;

		(*ip->term)(ip);
	}

	return ch;
}


/*
 * Determine if the next input character to be read is from the terminal.
 * Returns TRUE if so.
 */
BOOL
TtyIsInput()
{
	INPUT *	ip;

	ip = curinput;

	while ((ip->type == INP_LOOP) && (ip->first))
		ip--;

	return ip->type == INP_TTY;
}


/*
 * Set up to read commands from the terminal.  The lowest level of input
 * never quits, and is unusual in that it doesn't usually block waiting for
 * input.  Returns FALSE if unsuccessful.
 */
BOOL
SetTty()
{
	INPUT *	ip;

	ip = curinput + 1;

	if (ip >= &inputs[MAXINPUT])
		return FALSE;

	ip->getchar = TtyChar;
	ip->term = TtyTerm;
	ip->type = INP_TTY;
	curobj->update |= U_STAT;
	curinput = ip;

	return TRUE;
}


/*
 * Read next character from the terminal if it is ready.  If nothing is
 * going on we will wait for it anyway, to prevent excessive runtimes.
 * If something is going on, we will longjmp away if input is not ready.
 * We set the interactive flag to indicate we are talking to the user.
 */
static int
TtyChar(ip)
	INPUT *	ip;
{
	int	ch;
	int	wait;

	interact = TRUE;

	wait = dowait || !(curobj->update || (genleft > 0));

	if (wait) {
		if (setjmp(intjmp)) {
			intjmpok = FALSE;

			return '\0';
		}

		intjmpok = TRUE;
	}

	ch = (*dev->readchar)(dev, wait);

	intjmpok = FALSE;

	if ((ch == EOF) && !wait)
		ScanEof();

	if (errorstring) {	/* disable error message */
		errorstring = NULL;
		curobj->update |= U_STAT;
	}

	return ch;
}


/*
 * Terminate reading from the terminal.  If we are reading from the lowest
 * level of terminal input, this is a no-op, since that level can never return.
 */
static void
TtyTerm(ip)
	INPUT *	ip;
{
	if (ip > inputs)
		ip--;

	curinput = ip;
	curobj->update |= U_STAT;
}


/*
 * Set up to read commands from a given file name.  When the file is complete,
 * some parameters (like the current cursor position) will be restored to their
 * original value.  Returns FALSE if cannot set up to read the file.
 */
BOOL
SetFile(name)
	char *	name;
{
	INPUT *	ip;
	FILE *	fp;
	OBJECT *obj;
	int	ch;

	ip = curinput + 1;

	if (ip >= &inputs[MAXINPUT])
		return FALSE;

	fp = OpenLibraryFile(name, "r");

	if (fp == NULL)
		return FALSE;

	obj = curobj;

	ip->file = fp;
	ip->getchar = FileChar;
	ip->term = FileTerm;
	ip->type = INP_FILE;
	ip->obj = obj;
	ip->row = obj->currow;
	ip->col = obj->curcol;
	ip->origrow = obj->origrow;
	ip->origcol = obj->origcol;

	obj->origrow = obj->currow;
	obj->origcol = obj->curcol;
	obj->update |= U_STAT;

	curinput = ip;

	/*
	 * Now peek at the first character of the file to determine what type of
	 * format it is.  It can be either our own format, or else rle format,
	 * or else xlife format.  For our format, the contents will be read
	 * after we return.  But for the other formats, we must read in the
	 * complete file here.
	 */
	ch = fgetc(fp);
	if (ch != EOF)
		ungetc(ch, fp);

	switch (ch) {
		case 'x':
			ReadRle(fp, obj);
			(*ip->term)(ip);
			break;

		case '#':
			ReadXlife(fp, obj);
			(*ip->term)(ip);
			break;

		case '!':
			break;

		default:
			Error("Unknown file format");
	}

	return TRUE;
}


/*
 * Open a life library file, trying various transformed names if necessary.
 * The order of the transformations which are tried is:
 *	1.  Name exactly as given.
 *	2.  Name with LIFEEXT appended to it.
 *	3.  Name in any of the user's library directories.
 *	4.  Name with LIFEEXT appended to it in the user's libraries.
 *	5   Name in the system's library directory.
 *	6.  Name with LIFEEXT appended to it in the system library.
 * Returns the file handle if the open is successful, or NULL if all the
 * open attempts failed.
 */
static FILE *
OpenLibraryFile(name, mode)
	char *	name;
	char *	mode;
{
	FILE *	fp;
	int	i;
	char	buf[MAXPATH];

	/*
	 * Start by looking in the current directory, or the absolute
	 * path name if that was specified.
	 */
	fp = fopen(name, mode);

	if (fp)
		return fp;

	strcpy(buf, name);
	strcat(buf, LIFEEXT);

	fp = fopen(buf, mode);

	if (fp)
		return fp;

	/*
	 * If the name was absolute, then we fail now.
	 */
	if (*name == '/')
		return NULL;

	/*
	 * The name is relative so we can look elsewhere.
	 * Search all of the user's specified libraries in order.
	 */
	for (i = 0; i < userlibcount; i++) {
		strcpy(buf, userlibs[i]);
		strcat(buf, "/");
		strcat(buf, name);

		fp = fopen(buf, mode);

		if (fp)
			return fp;

		strcat(buf, LIFEEXT);

		fp = fopen(buf, mode);

		if (fp)
			return fp;
	}

	/*
	 * Finally look in our system default library.
	 */
	strcpy(buf, LIFELIB);
	strcat(buf, "/");
	strcat(buf, name);

	fp = fopen(buf, mode);

	if (fp)
		return fp;

	strcat(buf, LIFEEXT);

	return fopen(buf, mode);
}


/*
 * Here to read next character from a file.  Called by ReadChar when
 * input source is a file.  Returns EOF on end of file.
 */
static int
FileChar(ip)
	INPUT *	ip;
{
	return fgetc(ip->file);
}


/*
 * Here on end of file or error to close the input file and restore some
 * of the previous state such as the cursor location.
 */
static void
FileTerm(ip)
	INPUT *	ip;
{
	(void) fclose(ip->file);

	curobj = ip->obj;
	curobj->currow = ip->row;
	curobj->curcol = ip->col;
	curobj->origrow = ip->origrow;
	curobj->origcol = ip->origcol;
	curobj->update |= U_STAT;
	curinput = (ip - 1);
}


/*
 * Set up for execution of a loop.  This remembers the initial and final
 * loop values, and sets up to remember commands as they are given.
 * This is also used for defining a macro command.  The given character
 * will be assigned the string defined by the loop.
 * Returns FALSE if failed.
 */
BOOL
SetLoop(begval, endval, ch)
	VALUE	begval;
	VALUE	endval;
	int	ch;
{
	INPUT *	ip;

	ip = curinput + 1;

	if (ip >= &inputs[MAXINPUT])
		return FALSE;

	ip->begptr = malloc(LOOPSIZE);

	if (ip->begptr == NULL)
		return FALSE;

	ip->endptr = ip->begptr + LOOPSIZE;
	ip->curptr = ip->begptr;
	ip->first = TRUE;
	ip->getchar = LoopChar;
	ip->term = LoopTerm;
	ip->type = INP_LOOP;
	ip->curval = begval;
	ip->endval = endval;
	ip->macro = ch;
	curobj->update |= U_STAT;
	curinput = ip;

	return TRUE;
}


/*
 * End the range of the currently defined loop.  At this point, all of
 * the characters of the loop have been read in and saved, and we can
 * just proceed to iterate over them.  The next read will find out that
 * the first iteration of the loop is over.
 */
void
EndLoop()
{
	INPUT *	ip;

	ip = curinput;

	if (ip->type != INP_LOOP)
		Error("Loop not being defined");

	ip->endptr = ip->curptr - 1;	/* end before loop term cmd */
	ip->first = FALSE;
}


/*
 * Read one character from a loop buffer.  If at the end of the buffer, the
 * pointer is reset so that the buffer is reread.  When enough iterations
 * have been processed, EOF is returned.  A special case exists the first time
 * through the loop at the first nesting level, in that we don't yet have
 * the characters necessary for the loop, and so we have to read them by
 * ourself.
 */
static int
LoopChar(ip)
	INPUT *	ip;
{
	int	ch;

	if (ip->first) {	/* collecting input chars */
		if (ip->curptr >= ip->endptr)
			Error("Loop too long");

		ch = (*ip[-1].getchar)(ip - 1);	/* char from previous level */

		if (ch == EOF)
			Error("End of file in loop");

		*ip->curptr++ = ch;

		return ch;
	}

	if (ip->curptr >= ip->endptr) {	/* done with one iteration */
		if (ip->curval == ip->endval)
			return EOF;

		ip->curval += ((ip->curval < ip->endval) ? 1 : -1);
		ip->curptr = ip->begptr;
	}

	return *ip->curptr++;
}


/*
 * Terminate reading from a loop buffer.  If this was the definition of
 * a macro character, remember it for later.
 */
static void
LoopTerm(ip)
	INPUT *	ip;
{
	MACRO *	mp;

	mp = FindMacro(ip->macro);

	if (mp) {
		if (mp->begptr)
			free(mp->begptr);

		mp->begptr = ip->begptr;
		mp->endptr = ip->endptr;
	} else
		free(ip->begptr);	/* or free buffer */

	curobj->update |= U_STAT;
	curinput = (ip - 1);
}


/*
 * Find the macro structure for a macro character.
 * These are all upper and lower case letters and the digits 1 to 9.
 * Returns NULL if the macro character is invalid.
 */
MACRO *
FindMacro(ch)
	int	ch;
{
	if ((ch >= 'a') && (ch <= 'z'))
		return &macros[ch - 'a'];

	if ((ch >= 'A') && (ch <= 'Z'))
		return &macros[ch - 'A' + 26];

	if ((ch >= '1') && (ch <= '9'))
		return &macros[ch - '1' + 26*2];

	return NULL;
}


/*
 * Routine to return whether a macro character is valid.
 */
BOOL
MacroIsValid(ch)
	int	ch;
{
	return (FindMacro(ch) != NULL);
}


/*
 * Set up to read a defined macro command.  Returns FALSE if failed.
 */
BOOL
SetMacro(arg1, arg2, ch)
	VALUE	arg1;
	VALUE	arg2;
	int	ch;
{
	INPUT *	ip;		/* current input */
	MACRO *	mp;		/* macro command */

	mp = FindMacro(ch);

	if ((mp == NULL) || (mp->begptr == NULL))
		return FALSE;

	ip = curinput + 1;

	if (ip >= &inputs[MAXINPUT])
		return FALSE;

	ip->getchar = MacroChar;
	ip->term = MacroTerm;
	ip->begptr = mp->begptr;
	ip->curptr = mp->begptr;
	ip->endptr = mp->endptr;
	ip->type = INP_MACRO;
	ip->macro = ch;
	ip->curval = arg1;
	ip->endval = arg2;
	curobj->update |= U_STAT;
	curinput = ip;

	return TRUE;
}


/*
 * Read next character from macro definition.
 * Returns EOF when the macro is finished.
 */
static int
MacroChar(ip)
	INPUT *	ip;
{
	if (ip->curptr >= ip->endptr)
		return EOF;

	return *ip->curptr++;
}


/*
 * Terminate reading from a macro definition.
 */
static void
MacroTerm(ip)
	INPUT *	ip;
{
	curinput = (ip - 1);
	curobj->update |= U_STAT;
}


/*
 * Read a line from the user terminated by a newline (which is removed).
 * Editing of the input line is fully handled.  The prompt string and the
 * input line are only visible if the current input is from the terminal.
 * Returns a pointer to the null-terminated string.
 */
char *
ReadString(prompt)
	char *	prompt;
{
	ScanReset();			/* no more scan interference */

	if (prompt == NULL)
		prompt = "";		/* set prompt if given NULL */

	if (!TtyIsInput())
		prompt = NULL;		/* no window stuff if not tty */

	dowait = TRUE;			/* must wait for tty chars */

	ReadLine(prompt, stringbuf, STRINGSIZE, 0);

	dowait = FALSE;

	if (prompt)
		curobj->update |= U_STAT;

	if (stop)
		Error("Command aborted");

	return stringbuf;
}


/*
 * Read some input while possibly showing it on the status line.
 * If the prompt string is NULL, then editing is performed without
 * any display activity (useful when reading commands from scripts).
 * Otherwise, the prompt is shown in the window along with any input.
 * A trailing null character is inserted in the buffer.  The buffer may
 * contain initial data which will initially appear in the window, and
 * which can be edited by the user.  Initcount specifies the number of
 * initial characters in the buffer, so that a value of zero indicates
 * that there is no such data.  Editing of the input is handled.  If the
 * buffer fills up, the user is warned with beeps and further input is
 * ignored.  Returns number of bytes of data read.
 */
int
ReadLine(prompt, buf, count, initcount)
	char *	prompt;			/* prompt string (if any) */
	char *	buf;			/* address of the storage buffer */
	int	count;			/* maximum number of bytes allowed */
	int	initcount;		/* number of bytes already in buffer */
{
	int	ch;			/* character which was read */
	char *	bp;			/* current buffer pointer location */
	char *	endbp;			/* end of buffer */
	BOOL	redraw;			/* need to redisplay input */

	redraw = TRUE;
	bp = buf;
	endbp = bp + count - 1;

	if (initcount > 0) {
		if (initcount >= count)
			initcount = count - 1;

		bp = &buf[initcount];
	}

	while (TRUE) {
		if (prompt) {
			if (redraw) {
				*bp = '\0';
				(*dev->showstatus)(dev, prompt);
				(*dev->addstatus)(dev, buf);
				redraw = FALSE;
			}
			(*dev->update)(dev, TRUE);
		}

		ch = ReadChar();

		if (stop) {			/* interrupted */
			buf[0] = '\0';
			return 0;
		}

		if (ch == '\0')			/* ignore nulls */
			continue;

		if (ch == vlnext) {		/* literal input */
			ch = ReadChar();

			if (stop) {
				buf[0] = '\0';
				return 0;
			}

			goto store;
		}

		if (ch == '\n') {		/* end of line */
			*bp = '\0';

			return bp - buf;
		}

		if ((ch == verase1) || (ch == verase2)) {	/* char erase */
			if (bp <= buf)
				continue;
			bp--;
			redraw = TRUE;
			continue;
		}

		if (ch == vkill) {		/* line erase */
			bp = buf;
			redraw = TRUE;
			continue;
		}

		if (ch == vwerase) {		/* word erase */
			if (bp <= buf)
				continue;

			while ((bp > buf) && ((bp[-1] == '\n')
				|| isblank(bp[-1])))
					bp--;

			while ((bp > buf) && (bp[-1] != '\n')
				&& !isblank(bp[-1]))
			       		bp--;

			redraw = TRUE;
			continue;
		}

store:		if (bp >= endbp) {		/* buffer is full */
			Beep();
			continue;
		}

		*bp = ch;
		bp[1] = '\0';

		if (prompt) {
			(*dev->addstatus)(dev, bp);
			(*dev->update)(dev, TRUE);
		}

		bp++;
	}
}


/*
 * Routine called to wait until a space, newline, or escape character
 * is typed.  It is assumed that the screen contains a help display
 * that we can append our message to.  If the more flag is used, and
 * the user types escape instead of a space, then we return nonzero.
 */
static BOOL
SpaceWait(more)
	BOOL	more;
{
	int	ch;

	ScanReset();			/* throw out stored chars */

	if (more)
		(*dev->addhelp)(dev, "\n[Type <space> to continue or <esc> to return]");
	else
		(*dev->addhelp)(dev, "\n[Type <space> to return]");

	(*dev->update)(dev, FALSE);

	curobj->update |= U_ALL;
	dowait = TRUE;			/* must wait for chars */

	do {
		if (stop) {
			ch = ESC;
			break;
		}

		ch = ReadChar();
	} while ((ch != ' ') && (ch != ESC) && (ch != '\n'));

	dowait = FALSE;

	return (more && (ch == ESC));
}


/*
 * Write all defined macros to a the specifiled FILE handle so that
 * they can be read back in later.
 */
void
WriteMacros(fp)
	FILE *	fp;
{
	MACRO *	mp;
	UCHAR *	cp;
	int	ch;

	fputs("! macros\n", fp);

	for (ch = 0; ch < 256; ch++) {
		mp = FindMacro(ch);

		if ((mp == NULL) || (mp->begptr == NULL) ||
			(mp->begptr >= mp->endptr))
		{
			continue;
		}

		fprintf(fp, "<%c", ch);

		for (cp = mp->begptr; cp < mp->endptr; cp++)
			fputc(*cp, fp);

		fputs(">!\n", fp);
	}
}


/*
 * Collect all *.l files under a directory which contain the specified pattern.
 * The names are appended to the given list of strings.  If desired, the
 * list can have uninitialized pointers (indicated by max counts of zero).
 * For speed, the assumption is made that *.l files are valid life files,
 * and so are not directories.  The skip length is how many characters of
 * the given directory name are not to be displayed in the list of files.
 * The depth is how deep to allow the search to go.
 */
void
GetFiles(list, dirname, pattern, skiplen, depth)
	LIST *	list;
	char *	dirname;
	char *	pattern;
	int	skiplen;
	int	depth;
{
	DIR *		dirp;
	struct	dirent *dp;
	char *		name;
	char *		nameappend;
	int		len;
	char		fullname[MAXPATH];

	dirp = opendir(dirname);

	if (dirp == NULL)
		return;

	depth--;
	strcpy(fullname, dirname);
	strcat(fullname, "/");
	nameappend = fullname + strlen(fullname);

	while ((dp = readdir(dirp)) != NULL) {
		name = dp->d_name;
		len = strlen(name) - sizeof(LIFEEXT) + 1;
		strcpy(nameappend, name);

		if ((len <= 0) || strcmp(name + len, LIFEEXT)) {
			if (depth && strcmp(name, ".") && strcmp(name, ".."))
				GetFiles(list, fullname, pattern, skiplen, depth);

			fflush(stdout);

			continue;
		}

		nameappend[len] = '\0';

		if (!(*pattern) || MatchFile(fullname + skiplen, pattern))
			AddFile(list, fullname + skiplen);
	}

	closedir(dirp);
}


/*
 * Add a file name to the specified list.
 */
static void
AddFile(list, name)
	LIST *	list;
	char *	name;
{
	int	len;
	int	size;
	char *	mem;

	len = strlen(name);

	if (list->argc >= list->maxargc) {
		size = (list->maxargc + LISTARGSIZE) * sizeof(char *);

		if (list->maxargc)
			mem = realloc((char *) list->argv, size);
		else
			mem = malloc(size);

		if (mem == NULL)
			return;

		list->argv = (char **) mem;
		list->maxargc += LISTARGSIZE;
	}

	if (list->used + len >= list->maxused) {
		size = list->maxused + len + LISTBUFSIZE;

		if (list->maxused)
			mem = realloc(list->buf, size);
		else
			mem = malloc(size);

		if (mem == NULL)
			return;

		for (size = 0; size < list->argc; size++)
			list->argv[size] = mem +
				(list->argv[size] - list->buf);

		list->buf = mem;
		list->maxused += len + LISTBUFSIZE;
	}

	list->argv[list->argc] = &list->buf[list->used];
	strcpy(&list->buf[list->used], name);
	list->used += (len + 1);
	list->argc++;
}


/*
 * Sort and display a list of file names in the help window.
 * If many files are displayed, the user is asked to page through them all.
 */
void
ListFiles(list)
	LIST *	list;
{
	int	col;
	int	newcol;
	int	colwidth;
	int	argc;
	char **	argv;
	char *	cp;

	if (list->argc <= 0)
		return;

	qsort((char *) list->argv, list->argc, sizeof(char *), SortFunc);

	colwidth = MAXFILE - sizeof(LIFEEXT) + 2;
	col = 0;

	argv = list->argv;
	argc = list->argc;

	while (argc-- > 0) {
		cp = *argv++;

		newcol = col + strlen(cp) + 2;

		if ((col > 0) && (newcol > dev->textcols)) {
			ShowHelp("\n");
			newcol -= col;
			col = 0;
		}

		ShowHelp(cp);
		ShowHelp("  ");

		col = newcol;

		while (col % colwidth) {
			ShowHelp(" ");
			col++;
		}
	}

	if (col)
		ShowHelp("\n");

	EndHelp();
}


/*
 * Sort function for files.
 * This is a alphabetic-numeric sort, so numbers appear in a natural order.
 */
static int
SortFunc(cp1, cp2)
	char **	cp1;
	char **	cp2;
{
	char *	s1;
	char *	s2;
	int	c1;
	int	c2;
	long	n1;
	long	n2;

	s1 = *cp1;
	s2 = *cp2;

	while (TRUE) {
		c1 = *s1++;
		c2 = *s2++;

		if ((c1 == '\0') && (c2 == '\0'))
			return 0;

		if (!isdigit(c1) || !isdigit(c2)) {
			if (c1 == c2)
				continue;
			return c1 - c2;
		}

		n1 = c1 - '0';
		n2 = c2 - '0';

		while (isdigit(*s1))
			n1 = n1 * 10 + *s1++ - '0';

		while (isdigit(*s2))
			n2 = n2 * 10 + *s2++ - '0';

		if (n1 == n2)
			continue;

		return n1 - n2;
	}
}


/*
 * Add some text to the help screen.  The text may contain newlines,
 * but line wrapping is not done.  If the text goes to the bottom of
 * the screen, the output is stopped and the user is requested to type
 * a space to proceed, or escape to abort.  If the user aborts, then
 * all further help output is suppressed until the EndHelp call is made,
 * and this routine returns nonzero.  If this routine is called, then
 * EndHelp MUST also be called before non-help output occurs.
 */
BOOL
ShowHelp(str)
	char *	str;
{
	int	len;
	char *	cp;
	char	buf[80];

	if (helpbusy && helpabort)
		return TRUE;

	if (!helpbusy) {
		(*dev->showhelp)(dev);
		helprow = 0;
		helpabort = FALSE;
		helpbusy = TRUE;
	}

	while (*str) {
		if (helprow > dev->textrows - 3) {
			if (SpaceWait(TRUE)) {
				helpabort = TRUE;

				return TRUE;
			}

			(*dev->showhelp)(dev);
			helprow = 0;
		}

		cp = strchr(str, '\n');

		if ((cp == NULL) || (cp[1] == '\0')) {
			(*dev->addhelp)(dev, str);

			if (cp)
				helprow++;

			return FALSE;
		}

		/*
		 * Copy the line and null terminate it in order to display it.
		 */
		cp++;

		while (str != cp) {
			len = (cp - str);

			if (len >= sizeof(buf))
				len = sizeof(buf) - 1;

			memcpy(buf, str, len);
			buf[len] = '\0';
			(*dev->addhelp)(dev, buf);
			str += len;
		}
		helprow++;
	}

	return FALSE;
}


/*
 * End the help display, by waiting for the user to type a space.
 */
void
EndHelp()
{
	if (helpbusy && !helpabort)
		SpaceWait(FALSE);

	helpbusy = FALSE;
	helpabort = FALSE;
	helprow = 0;
}


/*
 * See if the specified file name is matched by the specified pattern.
 * For our purposes, matching means the pattern is contain somewhere
 * within the name.
 */
static BOOL
MatchFile(name, pattern)
	char *	name;
	char *	pattern;
{
	int	patlen;
	int	namelen;

	if (*pattern == '\0')
		return TRUE;

	namelen = strlen(name);
	patlen = strlen(pattern);

	while (namelen >= patlen) {
		if (memcmp(name, pattern, patlen) == 0)
			return TRUE;

		name++;
		namelen--;
	}

	return FALSE;
}

/* END CODE */
