/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"

#define	RLEMAX	70		/* maximum width of a line in rle format */


static	BOOL	ReadCoords PROTO((FILE *, VALUE *, VALUE *));
static	void	WritePicture PROTO((FILE *, OBJECT *, COORD, COORD,
			COORD, COORD, VALUE, VALUE));
static	void	WriteXlife PROTO((FILE *, OBJECT *, COORD, COORD,
			COORD, COORD));
static	void	WriteRle PROTO((FILE *, OBJECT *, COORD, COORD, COORD, COORD));
static	void	WriteRleItem PROTO((FILE *, int *, VALUE, int));


/*
 * Read in a run-length encoded description from an opened file.
 */
void
ReadRle(fp, obj)
	FILE *	fp;
	OBJECT *obj;
{
	COORD	mincol;
	VALUE	width;
	VALUE	height;
	VALUE	val;
	int	ch;
	char *	cp;
	char	buf[80];

	width = 0;
	height = 0;
	val = 0;

	do {
		ch = fgetc(fp);

		if ((ch == EOF) || (val >= sizeof(buf)))
			goto badhead;

		buf[val++] = ch;
	} while (ch != '\n');

	buf[val - 1] = '\0';

	cp = buf;

	if (*cp++ != 'x')
		goto badhead;

	while (*cp == ' ')
		cp++;

	if (*cp++ != '=')
		goto badhead;

	while (*cp == ' ')
		cp++;

	if (!isdigit(*cp))
		goto badhead;

	while (isdigit(*cp))
		width = width * 10 + *cp++ - '0';

	while (*cp == ' ')
		cp++;

	if (*cp++ != ',')
		goto badhead;

	while (*cp == ' ')
		cp++;

	if (*cp++ != 'y')
		goto badhead;

	while (*cp == ' ')
		cp++;

	if (*cp++ != '=')
		goto badhead;

	while (*cp == ' ')
		cp++;

	if (!isdigit(*cp))
		goto badhead;

	while (isdigit(*cp))
		height = height * 10 + *cp++ - '0';

	while (*cp == ' ')
		cp++;

	/*
	 * The rule string is optional.
	 */
	if (*cp == ',') {
		cp++;

		while (*cp == ' ')
			cp++;

		if (memcmp(cp, "rule", 4) != 0)
			goto badhead;

		cp += 4;

		while (*cp == ' ')
			cp++;

		if (*cp++ != '=')
			goto badhead;

		while (*cp == ' ')
			cp++;

		if (*cp != 'B')
			goto badhead;

		SetRule(cp);

		cp = "";
	}

	if (*cp || (width <= 0) || (height <= 0)) {
badhead:	Error("Bad header line in rle file");
	}

	obj->currow -= height / 2;
	obj->curcol -= width / 2;
	mincol = obj->curcol;

	while (TRUE) {
		val = 1;
		ch = fgetc(fp);

		if (isdigit(ch)) {
			val = 0;

			while (isdigit(ch)) {
				val = val * 10 + ch - '0';
				ch = fgetc(fp);
			}
		}

		switch (ch) {
			case 'b':
			case 'B':
				obj->curcol += val;
				break;

			case 'o':
			case 'O':
				while (val-- > 0)
					AddCell(obj, obj->currow,
						obj->curcol++);

				break;

			case '$':
				obj->currow += val;
				obj->curcol = mincol;
				break;

			case ' ':
			case '\t':
			case '\n':
				break;

			case '!':
				return;

			case EOF:
				Error("Unexpected end of file in rle file");

			default:
				Error("Illegal character in rle file");
		}
	}
}


/*
 * Read in an object in xlife format from an opened file.
 * Absolute coordinates are converted to relative ones.
 * However, we don't handle indirect files.
 */
void
ReadXlife(fp, obj)
	FILE *	fp;
	OBJECT *obj;
{
	VALUE	row;
	VALUE	col;
	BOOL	inheader;
	int	ch;

	inheader = TRUE;

	while (inheader) {
		ch = fgetc(fp);

		if (ch != '#')
			Error("Directive expected in xlife file");

		ch = fgetc(fp);

		switch (ch) {
			case 'C':
			case 'N':
			case 'O':
				while ((ch != EOF) && (ch != '\n'))
					ch = fgetc(fp);

				break;

			case 'I':
				Error("Indirect files not supported for xlife files");

			case 'A':
			case 'R':
				while ((ch != EOF) && (ch != '\n'))
					ch = fgetc(fp);

				while (ReadCoords(fp, &col, &row))
					AddCell(obj, obj->currow + row,
						obj->curcol + col);

				return;

			case 'P':
				if (!ReadCoords(fp, &col, &row))
					Error("Unexpected EOF in xlife file");

				obj->currow += row;
				obj->curcol += col;

				ReadPicture(fp, obj);

				return;

			default:
				Error("Unknown directive in xlife file");
		}
	}
}


/*
 * Read in a picture of a life object from an opened file.
 */
void
ReadPicture(fp, obj)
	FILE *	fp;
	OBJECT *obj;
{
	COORD	mincol;
	int	ch;

	mincol = obj->curcol;

	while (TRUE)
	{
		ch = fgetc(fp);

		switch (ch) {
			case '.':
			case ' ':
				obj->curcol++;
				break;

			case '*':
			case 'O':
			case 'o':
				AddCell(obj, obj->currow, obj->curcol++);
				break;

			case '\t':
				do {
					obj->curcol++;
				} while ((obj->curcol - mincol) % 8);
				break;

			case '\n':
				obj->curcol = mincol;
				obj->currow++;
				break;

			case EOF:
				return;

			default:
				Error("Illegal pictorial character in file");
		}
	}
}


/*
 * Read a pair of coordinates and the following end of line from a file.
 * The pair of values are returned indirectly through pointers.  Extra
 * blanks are allowed.  Returns TRUE if coordinates were read, or FALSE
 * if EOF was found.
 */
static BOOL
ReadCoords(fp, num1, num2)
	FILE *	fp;
	VALUE *	num1;
	VALUE *	num2;
{
	VALUE	val;
	BOOL	isneg;
	int	ch;
	char *	cp;
	char	buf[80];

	*num1 = 0;
	*num2 = 0;

	val = 0;

	do {
		ch = fgetc(fp);

		if (ch == EOF) {
			if (val == 0)
				return FALSE;

			goto badline;
		}

		if (val >= sizeof(buf))
			goto badline;

		buf[val++] = ch;
	} while (ch != '\n');

	cp = buf;

	while (isblank(*cp))
		cp++;

	isneg = (*cp == '-');

	if (isneg)
		cp++;

	val = 0;

	if (!isdigit(*cp))
		goto badline;

	while (isdigit(*cp))
		val = val * 10 + *cp++ - '0';

	if (isneg)
		val = -val;

	*num1 = val;

	if (!isblank(*cp))
		goto badline;

	while (isblank(*cp))
		cp++;

	isneg = (*cp == '-');

	if (isneg)
		cp++;

	val = 0;

	if (!isdigit(*cp))
		goto badline;

	while (isdigit(*cp))
		val = val * 10 + *cp++ - '0';

	if (isneg)
		val = -val;

	*num2 = val;

	while (isblank(*cp))
		cp++;

	if (*cp != '\n') {
badline:	Error("Invalid coordinate value in file");
	}

	return TRUE;
}


/*
 * Write an object out to a file in one of several different methods, which
 * is one of picture, xlife, or rle.  The maxrows and maxcols arguments are
 * used for limiting picture sizes.
 */
void
WriteObject(obj, method, name, maxrows, maxcols)
	OBJECT *obj;
	char *	method;
	char *	name;
	VALUE	maxrows;
	VALUE	maxcols;
{
	FILE *	fp;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;

	if (!IsAbbreviation(method, "picture") &&
		!IsAbbreviation(method, "rle") &&
		!IsAbbreviation(method, "xlife"))
	{
		Error("Unknown file format specified");
	}

	if (*name == '\0')
		Error("No filename specified");

	MinMax(obj, &minrow, &maxrow, &mincol, &maxcol);

	if (minrow > maxrow)
		Error("Null object");

	fp = fopen(name, "w");

	if (fp == NULL)
		Error("Cannot open output file");

	switch (*method) {
		case 'p':
			WritePicture(fp, obj, minrow, maxrow, mincol, maxcol,
				maxrows, maxcols);
			break;

		case 'x':
			WriteXlife(fp, obj, minrow, maxrow, mincol, maxcol);
			break;

		case 'r':
			WriteRle(fp, obj, minrow, maxrow, mincol, maxcol);
			break;
	}

	fflush(fp);

	if (ferror(fp)) {
		fclose(fp);
		Error("Write failed");
	}

	if (fclose(fp))
		Error("Close failed");

	if (stop)
		Error("Writing aborted");
}


/*
 * Write the current object out to the named file as a picture, with the
 * given maximum sizes for "pretty" output.  If the size is exceeded, the
 * output is compressed.  The file as written can be read in as commands
 * which will regenerate the object.
 */
static void
WritePicture(fp, obj, minrow, maxrow, mincol, maxcol, maxrows, maxcols)
	FILE *	fp;
	OBJECT *obj;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
	VALUE	maxrows;
	VALUE	maxcols;
{
	ROW *	rp;			/* current row structure */
	CELL *	cp;			/* current cell */
	COORD	row;			/* current row value */
	COORD	col;			/* current column value */
	ROW *	trp;			/* temporary row structure */
	COORD	curmin;			/* current minimum column */
	COORD	curmax;			/* current maximum column */
	COORD	testmin;		/* test minimum column of rows */
	COORD	testmax;		/* test maximum column of rows */

	MinMax(obj, &minrow, &maxrow, &mincol, &maxcol);

	if (minrow > maxrow) {
		fclose(fp);
		Error("Null object");
	}

	fprintf(fp, "! \"%s\" (cells %ld length %ld width %ld generation %ld)\n",
		obj->name, obj->count, maxrow - minrow + 1,
		maxcol - mincol + 1, obj->gen);

	if (strcmp(rulestring, DEFAULTRULE) != 0)
		fprintf(fp, ";rule %s\n", rulestring);

	if (obj->currow > minrow)
		fprintf(fp, "%ldk", obj->currow - minrow);

	if (obj->currow < minrow)
		fprintf(fp, "%ldj", minrow - obj->currow);

	if (obj->curcol > mincol)
		fprintf(fp, "%ldh", obj->curcol - mincol);

	if (obj->curcol < mincol)
		fprintf(fp, "%ldl", mincol - obj->curcol);

	fprintf(fp, "@!\n");

	curmin = INFINITY;
	curmax = -INFINITY;

	rp = obj->firstrow;
	row = minrow;

	for (; rp != termrow; rp = rp->next) {
		if (stop)
			return;

		/*
		 * Skip down to the next row with something in it.
		 */
		if (rp->firstcell == termcell)
			continue;

		if (rp->row > (row + maxrows)) {	/* skip to right row */
			fprintf(fp, "%ld\n", rp->row - row);
			row = rp->row;
		}

		while (rp->row > row) {
			fputs(".\n", fp);
			row++;
		}

		/*
		 * Output the current row, compressing if it is too wide.
		 */
		col = mincol;
		cp = rp->firstcell;

		if ((cp->col + maxcols) < rp->lastcell->col) {
			while (cp != termcell)
			{
				/*
				 * Write all adjacent blanks.
				 */
				if (cp->col > col + 2) {
					fprintf(fp, "%ld.", cp->col - col);
					col = cp->col;
				}

				while (cp->col > col) {
					fputc('.', fp);
					col++;
				}

				/*
				 * Write all adjacent cells.
				 */
				while ((cp->col + 1) == cp->next->col) {
					cp = cp->next;
				}

				if (cp->col >= col + 2) {
					fprintf(fp, "%ldO", cp->col - col + 1);
					col = cp->col + 1;
				}

				while (col <= cp->col) {
					fputc('O', fp);
					col++;
				}

				cp = cp->next;
			}

			fputc('\n', fp);

			row++;
			curmin = INFINITY;
			curmax = -INFINITY;

			continue;
		}

		/*
		 * Here if the row doesn't need compressing.  See if several
		 * rows from this one on can all fit in the same range.  If
		 * so, set things up so they will align themselves nicely.
		 */
		if ((cp->col < curmin) || (rp->lastcell->col > curmax)) {
			curmin = cp->col;
			curmax = rp->lastcell->col;
			trp = rp;

			for (; rp != termrow; rp = rp->next) {
				if (rp->firstcell == termcell)
					continue;

				testmin = rp->firstcell->col;

				if (testmin > curmin)
					testmin = curmin;

				testmax = rp->lastcell->col;

				if (testmax < curmax)
					testmax = curmax;

				if ((testmax - testmin) >= maxcols)
					break;

				curmin = testmin;
				curmax = testmax;
			}
			rp = trp;
		}

		/*
 		 * Type the row, with the initial shift if necessary.
		 */
		if (curmin != mincol) {
			fprintf(fp, "%ld.", curmin - mincol);
			col = curmin;
		}

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			while (cp->col > col) {
				fputc('.', fp);
				col++;
			}

			fputc('O', fp);
			col++;
		}

		fputc('\n', fp);
		row++;
	}
}


/*
 * Write an object out to a file in xlife format.
 * This is just a list of coordinates.
 */
static void
WriteXlife(fp, obj, minrow, maxrow, mincol, maxcol)
	FILE *	fp;
	OBJECT *obj;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
{
	ROW *	rp;
	CELL *	cp;

	fprintf(fp, "#C \"%s\" (cells %ld width %ld height %ld generation %ld)\n",
		obj->name, obj->count, maxcol - mincol + 1,
		maxrow - minrow + 1, obj->gen);

	fputs("#R\n", fp);

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		if (stop)
			return;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next)
			fprintf(fp, "%ld %ld\n", cp->col - obj->curcol,
				rp->row - obj->currow);
	}
}


/*
 * Write an object out to a file in rle format.
 */
static void
WriteRle(fp, obj, minrow, maxrow, mincol, maxcol)
	FILE *	fp;
	OBJECT *obj;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
{
	ROW *	rp;
	CELL *	cp;
	COORD	row;
	COORD	col;
	VALUE	count;
	int	pos;

	fprintf(fp, "x = %ld, y = %ld", maxcol - mincol + 1,
		maxrow - minrow + 1);

	if (strcmp(rulestring, DEFAULTRULE) != 0) {
		fputs(", rule = B", fp);

		for (pos = 0; pos < 9; pos++) {
			if (rule[pos])
				fputc('0' + pos, fp);
		}

		fputs("/S", fp);

		for (pos = 0; pos < 9; pos++) {
			if (rule[LIFE + pos])
				fputc('0' + pos, fp);
		}
	}

	fputc('\n', fp);

	row = minrow;
	col = mincol;
	pos = 0;

	for (rp = obj->firstrow; rp != termrow; rp = rp->next) {
		if (stop)
			return;

		if (rp->firstcell == termcell)
			continue;

		if (row != rp->row)
			WriteRleItem(fp, &pos, rp->row - row, '$');

		row = rp->row;

		col = mincol;

		for (cp = rp->firstcell; cp != termcell; cp = cp->next) {
			if (cp->col != col) {
				WriteRleItem(fp, &pos, cp->col - col, 'b');
				col = cp->col;
			}

			count = 1;

			while (cp->next->col == cp->col + 1) {
				cp = cp->next;
				count++;
			}

			WriteRleItem(fp, &pos, count, 'o');
			col = cp->col + 1;
		}
	}

	WriteRleItem(fp, &pos, (VALUE) 1, '!');
	fputc('\n', fp);
}


/*
 * Convert the specified count and action character to an rle string,
 * and write it to the specified file.  This adds the character count to
 * the specified line position, and if it would exceed RLEMAX, then inserts
 * a newline first.  The character position for the line is updated.
 */
static void
WriteRleItem(fp, pos, count, action)
	FILE *	fp;
	int *	pos;
	VALUE	count;
	int	action;
{
	int	len;
	char	buf[16];

	if (count == 0)
		return;

	if (count == 1) {
		buf[0] = action;
		buf[1] = '\0';
		len = 1;
	} else if (count == 2) {
		buf[0] = action;
		buf[1] = action;
		buf[2] = '\0';
		len = 2;
	} else if ((count < 0) || (count > 99)) {
		sprintf(buf, "%ld%c", count, action);
		len = strlen(buf);
	} else if (count < 10) {
		buf[0] = count + '0';
		buf[1] = action;
		buf[2] = '\0';
		len = 2;
	} else {
		buf[0] = (count / 10) + '0';
		buf[1] = (count % 10) + '0';
		buf[2] = action;
		buf[3] = '\0';
		len = 3;
	}

	if (*pos + len > RLEMAX) {
		fputc('\n', fp);
		*pos = 0;
	}

	fputs(buf, fp);
	*pos += len;
}

/* END CODE */
