/*
 * Copyright (c) 1995 David I. Bell
 * Permission is granted to use, distribute, or modify this source,
 * provided that this copyright notice remains intact.
 */

#include "life.h"


#define	ZERO	((VALUE) 0)

static	int	ReadValue PROTO((VALUE *, BOOL *));
static	VALUE	ScanExpression PROTO((void));
static	void	DoSelect PROTO((OBJECT *, int));
static	void	DoMove PROTO((OBJECT *, VALUE, VALUE));
static	void	DoShift PROTO((OBJECT *, VALUE, VALUE));
static	void	DoSearch PROTO((OBJECT *, VALUE));


/*
 * Read commands if available, and execute them.  Since we call ScanChar for
 * our characters, the code after the setjmp can be reentered many times in
 * order to finish any command.  This allows commands to be typed without
 * stopping the computation of generations of an object, and allows editing
 * of partially completed commands.
 */
void
DoCommand()
{
	int	ch;		/* character read */
	OBJECT *obj;		/* object being manipulated */
	VALUE	defarg;		/* first argument defaulted to one */
	VALUE	arg1;		/* first command argument */
	VALUE	arg2;		/* second command argument */
	BOOL	got1;		/* got first argument flag */
	BOOL	got2;		/* got second argument flag */

	switch (setjmp(ttyjmp)) {
		case SCAN_EOF:		/* not yet enough chars for a command */
			return;

		case SCAN_EDIT:		/* command edited before completion */
		case SCAN_ABORT:	/* normal command completion */
		default:		/* normal entry point */
			break;
	}

	if (stop)
		Error("Command aborted");

	CheckStartupFiles();

	obj = curobj;
	obj->mark = 0;
	arg2 = 0;
	got2 = FALSE;

	ch = ReadValue(&arg1, &got1);

	if (ch == ',')
		ch = ReadValue(&arg2, &got2);

	defarg = 1;

	if (got1)
		defarg = arg1;

	switch (ch) {
		case FF:			/* refresh screen */
			(*dev->refresh)(dev);
			obj->update |= U_REDRAW;
			break;

		case '\n':			/* move to next row */
			obj->currow += defarg;
			obj->curcol = obj->origcol;
			obj->update |= U_POS;
			break;

		case '\t':			/* move to next tab stop */
			while ((++obj->curcol - obj->origcol) % 8)
				;

			obj->update |= U_POS;
			break;

		case ESC:			/* execute a macro command */
			ch = ScanChar();

			if (ch == ESC)
				break;		/* ignore double escape */

			Backup();

			if (!SetMacro(arg1, arg2, ch))
				Error("Undefined macro");

			obj->update |= U_STAT;
			break;

		case ' ':			/* move to right */
		case '.':
			obj->curcol += defarg;
			obj->update |= U_POS;
			break;

		case ':':			/* execute line style command */
		case ';':
			DoLineCommand(arg1, arg2, got1, got2);
			break;

		case '<':			/* begin loop or macro definition */
			if (got1 || got2) {
				if (got2)
					SetLoop(defarg, arg2, 0);
				else
					SetLoop(1, defarg, 0);

				obj->update |= U_STAT;
				break;
			}

			ch = ScanChar();	/* defining macro */

			if (!MacroIsValid(ch))
				Error("Bad macro character");

			SetLoop(1, 1, ch);
			obj->update |= U_STAT;
			break;

		case '>':			/* end loop */
			EndLoop();
			break;

		case '+':			/* increment single char variable */
			ch = ScanChar();

			if (ch == '$')
				ch = ScanChar();

			SetVariable1(ch, GetVariable1(ch) + defarg);
			break;

		case 'a':			/* add coordinate to path */
			CheckRun();

			if (!AddPointToPath(obj, obj->currow, obj->curcol))
				Error("Too many coordinates in path");

			break;

		case 'A':			/* add closure to path */
			CheckRun();

			ClosePath(obj);
			break;

		case 'b':			/* move lower left with action */
			DoMove(obj, defarg, -defarg);
			break;

		case 'B':			/* shift to lower left */
			DoShift(obj, defarg, -defarg);
			break;

		case 'c':			/* pick cell as current location */
			obj->currow = obj->origrow + arg1;
			obj->curcol = obj->origcol + arg2;
			obj->update |= U_POS;
			break;

		case 'd':			/* delete selection */
			DoSelect(obj, ch);
			Backup();
			MoveMarkedObject(obj, deleteobject, MARK_CMD);
			obj->update |= U_ALL;
			break;

		case 'f':			/* flip selection */
			ch = ScanChar();

			if ((ch != 'r') && (ch != 'c'))
				Error("Bad flip axis");

			DoSelect(obj, 'f');
			Backup();
			obj->mark = MARK_USR;

			if (ch == 'r')
				FlipRowMarkedObject(obj, MARK_CMD);
			else
				FlipColumnMarkedObject(obj, MARK_CMD);

			obj->update |= U_ALL;
			break;

		case 'g':			/* compute generations */
			if (genleft <= 0)
				Backup();

			if (got1)
				genleft += arg1;
			else if (genleft) {
				genleft = 0;
				obj->update |= U_ALL;
			} else
				genleft = 1;

			freqcount = curobj->frequency;
			obj->update |= U_POS;
			break;

		case 'G':			/* compute infinite generations */
			if (genleft <= 0)
				Backup();

			genleft = INFINITY;
			freqcount = curobj->frequency;
			obj->update |= U_POS;
			break;

		case 'h':			/* move left with action */
			DoMove(obj, ZERO, -defarg);
			break;

		case 'H':			/* shift left lots */
			DoShift(obj, ZERO, -defarg);
			break;

		case 'j':			/* move down with action */
			DoMove(obj, defarg, ZERO);
			break;

		case 'J':			/* shift down lots */
			DoShift(obj, defarg, ZERO);
			break;

		case 'k':			/* move up with action */
			DoMove(obj, -defarg, ZERO);
			break;

		case 'K':			/* shift up lots */
			DoShift(obj, -defarg, ZERO);
			break;

		case 'l':			/* move right with action */
			DoMove(obj, ZERO, defarg);
			break;

		case 'L':			/* shift right lots */
			DoShift(obj, ZERO, defarg);
			break;

		case 'm':			/* mark current object */
			DoSelect(obj, ch);
			CopyMarks(obj, MARK_CMD, MARK_USR);
			obj->update |= U_ALL;
			break;

		case 'n':			/* move lower right with action */
			DoMove(obj, defarg, defarg);
			break;

		case 'N':			/* shift down and right */
			DoShift(obj, defarg, defarg);
			break;

		case 'o':			/* insert new cells */
		case 'O':
		case '*':
			CheckRun();
			Backup();

			while (!stop && (defarg-- > 0))
				AddCell(obj, obj->currow, obj->curcol++);

			break;

		case 'p':			/* place deleted object */
			CheckRun();
			Backup();
			obj->mark = MARK_USR;
			AddObject(deleteobject, obj, RELATIVE, FALSE);
			obj->update |= U_ALL;
			break;

		case 'r':			/* rotate selection */
			DoSelect(obj, ch);
			Backup();
			obj->mark = MARK_USR;
			RotateMarkedObject(obj, MARK_CMD);
			obj->update |= U_ALL;
			break;

		case 's':			/* set scale factor and center object */
			obj->autoscale = FALSE;

			if (!got1)
				arg1 = obj->scale;

			SetScale(obj, arg1);
			break;

		case 'S':			/* perform auto-scaling */
			if (!got1)
				arg1 = obj->scale;

			SetScale(obj, arg1);
			obj->autoscale = TRUE;
			obj->update |= U_ALL;
			break;

		case 't':			/* toggle current cell */
			CheckRun();
			Backup();

			if (!DeleteCell(obj, obj->currow, obj->curcol))
				AddCell(obj, obj->currow, obj->curcol);

			break;

		case 'u':			/* move upper right with action */
			DoMove(obj, -defarg, defarg);
			break;

		case 'U':			/* shift to upper right */
			DoShift(obj, -defarg, defarg);
			break;

		case 'w':			/* move to where remembered */
			ch = ScanChar();

			if ((ch < 'a') || (ch > 'z'))
				Error("Bad where letter");

			obj->currow = obj->wherelocs[ch - 'a'].row + arg1;
			obj->curcol = obj->wherelocs[ch - 'a'].col + arg2;

			obj->update |= U_POS;
			break;

		case 'W':			/* remember where we are */
			ch = ScanChar();

			if ((ch < 'a') || (ch > 'z'))
				Error("Bad where letter");

			obj->wherelocs[ch - 'a'].row = obj->currow;
			obj->wherelocs[ch - 'a'].col = obj->curcol;
			break;

		case 'x':			/* kill current cells */
			CheckRun();
			Backup();

			while (!stop && (defarg-- > 0))
				DeleteCell(obj, obj->currow, obj->curcol++);

			obj->update |= U_ALL;
			break;

		case 'y':			/* move upper left with action */
			DoMove(obj, -defarg, -defarg);
			break;

		case 'Y':			/* shift to upper left */
			DoShift(obj, -defarg, -defarg);
			break;

		case 'z':			/* clear or set generation number */
			CheckRun();
			obj->gen = arg1;
			obj->born = 0;
			obj->died = 0;
			obj->update |= U_STAT;
			break;

		case '/':			/* search for next object */
			DoSearch(obj, defarg);
			break;

		case '@':			/* point at current location */
			obj->origrow = obj->currow;
			obj->origcol = obj->curcol;
			break;

		case '!':			/* comment characters */
		case '#':
			while (ScanChar() != '\n')
				;

			break;

		case NULL_CMD:			/* null command */
			break;

		default:			/* unknown commands */
			if (ch == veof) {
				(*curinput->term)(curinput);
				break;
			}

			Error("Unknown command");
	}

	ScanAbort();				/* completed command */
}


/*
 * Read a numeric value (if any) to be used as an argument for a command.
 * Pointers to the returned value and returned flag are given.
 * Return value is the first non-argument character read.
 */
static int
ReadValue(valueptr, flagptr)
	VALUE *	valueptr;		/* pointer to returned value */
	BOOL *	flagptr;		/* pointer to got value flag */
{
	int	ch;			/* character being read */
	INPUT *	ip;			/* input structure */
	int	sign;			/* sign of result */

	*valueptr = 0;
	*flagptr = FALSE;

	sign = 1;

	ch = ScanChar();

	if (ch == '-') {			/* negative value */
		sign = -1;
		ch = ScanChar();
	}

	if (ch == '$') {			/* get variable value */
		*valueptr = sign * GetVariable1(ScanChar());
		*flagptr = TRUE;

		return ScanChar();
	}

	if (ch == '(') {			/* get expression */
		*valueptr = sign * ScanExpression();
		*flagptr = TRUE;

		return ScanChar();
	}

	while (isdigit(ch)) {			/* get numeric value */
		*valueptr = (*valueptr * 10) + ch - '0';
		*flagptr = TRUE;
		ch = ScanChar();
	}

	if (ch == '%') {			/* get loop value */
		ch = 1;

		if (*flagptr)
			ch = *valueptr;

		if (ch <= 0)
			Error("Bad nest value");

		ip = curinput + 1;

		while (ch > 0) {
			if (--ip < inputs)
				Error("Bad nest value");

			if (ip->type == INP_LOOP)
				ch--;
		}

		*valueptr = ip->curval;
		*flagptr = TRUE;
		ch = ScanChar();
	}

	*valueptr *= sign;

	return ch;
}


/*
 * Routine called from above to scan and evaluate a parenthesized expression.
 * This routines knows that one parenthesis has already been read.  Stops
 * reading on the matching parenthesis.
 */
static VALUE
ScanExpression()
{
	char *	cp;		/* current character */
	int	nest;		/* nesting depth */
	char	buf[100];	/* expression buffer */

	cp = buf;
	*cp++ = '(';			/* start with parenthesis */
	nest = 1;

	while (nest > 0) {
		if (cp >= &buf[sizeof(buf)-2])
			Error("expression too long");

		*cp = ScanChar();

		if (*cp == '(')
			nest++;

		if (*cp == ')')
			nest--;

		cp++;
	}

	*cp = '\0';

	return GetExpression(buf);
}


/*
 * Select a set of cells to be used for some command.  This involves reading
 * the next character and marking cells based on that character.  Repeating
 * the command character is equivilant to selecting the current object.  On a
 * successful return, exactly those cells specified are marked with MARK_CMD.
 * If no cells are found, an error is generated.
 */
static void
DoSelect(obj, cmd)
	OBJECT *obj;
	int	cmd;
{
	CELL *	cp;
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;
	int	ch;

	CheckRun();

	ch = ScanChar();

	if (ch == cmd)
		ch = 'o';		/* repeated char is connected object */

	minrow = -INFINITY;
	maxrow = -minrow;
	mincol = minrow;
	maxcol = maxrow;

	ClearMarks(obj, MARK_CMD);

	switch (ch) {
		case 'a':			/* all of object */
			break;

		case 'b':			/* below and left of cursor */
			minrow = obj->currow;
			maxcol = obj->curcol;
			break;

		case 'c':			/* current cell */
			cp = FindCell(obj, obj->currow, obj->curcol);

			if (cp == NULL)
				Error("No cell at current location");

			cp->marks |= MARK_CMD;
			return;

		case 'h':			/* left of cursor */
			maxcol = obj->curcol;
			break;

		case 'j':			/* below cursor */
			minrow = obj->currow;
			break;

		case 'k':			/* above cursor */
			maxrow = obj->currow;
			break;

		case 'l':			/* right of cursor */
			mincol = obj->curcol;
			break;

		case 'm':			/* marked cells */
			if (!CopyMarks(obj, MARK_USR, MARK_CMD))
				Error("No object marked");

			return;

		case 'n':			/* below and right of cursor */
			minrow = obj->currow;
			mincol = obj->curcol;
			break;

		case 'o':			/* connected object */
			if (!MarkObject(obj, obj->currow, obj->curcol,
				0, MARK_CMD))
			{
				Error("No object at current location");
			}

			return;

		case 'p':			/* mark within path */
			if (obj->path.count == 0)
				Error("No path defined for object");

			if (!MarkPath(obj, MARK_CMD))
				Error("No cells within defined path");

			ClearPath(obj);

			return;

		case 'e':			/* extended object */
			if (!MarkObject(obj, obj->currow, obj->curcol,
				1, MARK_CMD))
			{
				Error("No object at current location");
			}

			return;

		case 'r':			/* rectangle to some cell */
			ch = ScanChar();

			if (ch == '@') {
				minrow = obj->origrow;
				mincol = obj->origcol;
			} else if ((ch >= 'a') && (ch <= 'z')) {
				minrow = obj->wherelocs[ch - 'a'].row;
				mincol = obj->wherelocs[ch - 'a'].col;
			} else
				Error("Bad rectangle specification");

			maxrow = obj->currow;
			maxcol = obj->curcol;

			if (minrow > maxrow) {
				maxrow = minrow;
				minrow = obj->currow;
			}

			if (mincol > maxcol) {
				maxcol = mincol;
				mincol = obj->curcol;
			}

			break;

		case 'u':			/* above and right of cursor */
			maxrow = obj->currow;
			mincol = obj->curcol;
			break;

		case 'v':			/* things visible in window */
			minrow = obj->minrow;
			maxrow = obj->maxrow;
			mincol = obj->mincol;
			maxcol = obj->maxcol;
			break;

		case 'y':			/* above and left of cursor */
			maxrow = obj->currow;
			maxcol = obj->curcol;
			break;

		case 'i':			/* cells invisible in window */
			ch = MarkRegion(obj, MARK_CMD,
				obj->maxrow + 1, INFINITY,
				-INFINITY, INFINITY);

			ch += MarkRegion(obj, MARK_CMD,
				-INFINITY, obj->minrow - 1,
				-INFINITY, INFINITY);

			ch += MarkRegion(obj, MARK_CMD,
				-INFINITY, INFINITY,
				obj->maxcol + 1, INFINITY);

			ch += MarkRegion(obj, MARK_CMD,
				-INFINITY, INFINITY,
				-INFINITY, obj->mincol - 1);

			if (ch == 0)
				Error("No cells outside of visible region");

			return;

		default:			/* unknown */
			Error("Unknown selection command");
	}

	if (MarkRegion(obj, MARK_CMD, minrow, maxrow, mincol, maxcol) == 0)
		Error("No cells in region");
}


/*
 * Move the current position by the indicated deltas, performing the
 * current action to the configuration.  The movement is scaled by
 * the current scaling factor.
 */
static void
DoMove(obj, rowdelta, coldelta)
	OBJECT *obj;
	VALUE	rowdelta;	/* amount to change row by */
	VALUE	coldelta;	/* amount to change column by */
{
	COUNT	row1;		/* increment for row */
	COUNT	col1;		/* increment for column */

	if (obj->scale > 0) {
		rowdelta *= obj->scale;
		coldelta *= obj->scale;
	}

	if (mode == M_MOVE) {		/* just want to move */
		obj->currow += rowdelta;
		obj->curcol += coldelta;
		obj->update |= U_POS;

		return;
	}

	CheckRun();
	Backup();

	row1 = 0;			/* need to loop for insert or delete */
	col1 = 0;

	if (rowdelta > 0)
		row1 = 1;

	if (rowdelta < 0)
		row1 = -1;

	if (coldelta > 0)
		col1 = 1;

	if (coldelta < 0)
		col1 = -1;

	rowdelta += obj->currow;
	coldelta += obj->curcol;

	while (!stop && ((obj->currow != rowdelta) ||
		(obj->curcol != coldelta)))
	{
		obj->currow += row1;
		obj->curcol += col1;
		obj->update |= U_POS;

		switch (mode) {
			case M_INSERT:
				AddCell(obj, obj->currow, obj->curcol);
				break;

			case M_DELETE:
				DeleteCell(obj, obj->currow, obj->curcol);
		}
	}
}


/*
 * Shift the window lots in the indicated direction, and also shift the
 * cursor location the same amount so that the cursor location on the
 * screen doesn't change.  "Lots" is 1/5 of the screen width or height.
 * Special case: if both x and y are being shifted, we shift both by the
 * same amount, which is the minimum of the two shifts.
 */
static void
DoShift(obj, rowdelta, coldelta)
	OBJECT *obj;			/* current object */
	VALUE	rowdelta;		/* amount to change row by */
	VALUE	coldelta;		/* amount to change column by */
{
	int	rowsign;		/* sign of row */
	int	colsign;		/* sign of column */

	rowdelta *= (obj->viewrows / 5);
	coldelta *= (obj->viewcols / 5);

	/*
	 * If diagonal movement is required, then take the minimums of
	 * the absolute values of the row and column deltas.
	 */
	if (rowdelta && coldelta) {
		rowsign = 1;
		colsign = 1;

		if (rowdelta < 0) {
			rowsign = -1;
			rowdelta = -rowdelta;
		}

		if (coldelta < 0) {
			colsign = -1;
			coldelta = -coldelta;
		}

		if (rowdelta > coldelta)
			rowdelta = coldelta;

		if (coldelta > rowdelta)
			coldelta = rowdelta;

		rowdelta *= rowsign;
		coldelta *= colsign;
	}

	obj->currow += rowdelta;
	obj->minrow += rowdelta;
	obj->maxrow += rowdelta;
	obj->curcol += coldelta;
	obj->mincol += coldelta;
	obj->maxcol += coldelta;

	obj->update |= U_ALL;
}


/*
 * Search forwards for the next cell of the object.
 * This tries to find each connected object only once.
 */
static void
DoSearch(obj, defarg)
	OBJECT *obj;
	VALUE	defarg;
{
	COORD	minrow;
	COORD	maxrow;
	COORD	mincol;
	COORD	maxcol;

	CheckRun();

	if (!SearchObject(obj, defarg, 0))
		Error("Empty object");

	ClearMarks(obj, MARK_CMD);
	MarkObject(obj, obj->currow, obj->curcol, 1, MARK_CMD);
	MarkMinMax(obj, MARK_CMD, &minrow, &maxrow, &mincol, &maxcol);

	PositionView(minrow, maxrow, mincol, maxcol);

	obj->update |= U_POS;
}


/*
 * Perform a backup of the current object.  This is only done if reading
 * from the terminal, since it is from the point of view of the user.
 */
void
Backup()
{
	if (curinput->type == INP_TTY)
		CopyObject(curobj, backupobject);
}


/*
 * Check to see that generations are not being computed before proceeding
 * with the current command.
 */
void
CheckRun()
{
	if (genleft > 0)
		Error("Illegal while running");
}

/* END CODE */
