
/* x11window.c
 * 
 * Kevin P. Smith  6/11/89 Much modified by Jerry Frain and Joe Young */

#include "config.h"
#include <stdio.h>
#include INC_SYS_SELECT
#include INC_STRINGS
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>

#ifdef FUNCTION_KEYS
#include <X11/keysym.h>
#endif

#include <assert.h>
#include <time.h>
#include <sys/time.h>
#include INC_SYS_TIMEB
#include "Wlib.h"
#include "defs.h"
#include "struct.h"
#include "data.h"
#include "teams.bitmap"
#include "mapcursor.bitmap"
#include "localcursor.bitmap"

#ifdef VMS
void    vms_event_window(void);

#endif

/* XFIX speedup */
#define MAXCACHE	128

/* changes too good to risk leaving out, by Richard Caley (rjc@cstr.ed.ac.uk) */
/* Was #ifdef RJC, but now they're just part of the code                    */

#define FOURPLANEFIX

#ifdef SMALL_SCREEN
#define NORMAL_FONT	"5x7"
#define BOLD_FONT	"5x7"
#define ITALIC_FONT	"5x7"
#define IND_FONT	"5x7"
#else
#define NORMAL_FONT	"6x10"
#define BOLD_FONT	"-*-clean-bold-r-normal--10-100-75-75-c-60-*"
#define ITALIC_FONT	"-*-clean-bold-r-normal--10-100-75-75-c-60-*"
#define IND_FONT        "-*-clean-bold-r-normal--10-100-75-75-c-60-*"
#endif

#define BIG_FONT	"-*-lucidatypewriter-*-*-*-*-40-*-*-*-*-*-*-*"

#ifdef BEEPLITE
#define TTS_FONT        "-misc-fixed-bold-r-normal--15-*"
extern void init_tts(void);
int     forceMono = 0;

#endif

#define G_SET_WIDTH     0x1
#define G_SET_HEIGHT    0x2
#define G_SET_X         0x4
#define G_SET_Y         0x8


static char *_nfonts[] =
{
  NORMAL_FONT,
  "-*-clean-medium-r-normal--10-100-75-75-c-60-*",
  "fixed",
  NULL,
};
static char *_bfonts[] =
{
  BOLD_FONT,
  "-*-clean-bold-r-normal--10-100-75-75-c-60-*",
  "fixed",
  NULL,
};
static char *_ifonts[] =
{
  ITALIC_FONT,
  "-*-clean-bold-r-normal--10-100-75-75-c-60-*",
  "fixed",
  NULL,
};
static char *_bgfonts[] =
{
  BIG_FONT,
  "fixed",
  "fixed",
  NULL,
};
XFontStruct *find_font(char *oldf, char **fonts);

#define FONTS 4
#define BITGC 4

#define WHITE   0
#define BLACK   1
#define RED     2
#define GREEN   3
#define YELLOW  4
#define CYAN    5
#define GREY	6

#ifdef RACE_COLORS
#define C_ROM	7
#define C_KLI	8
#define C_FED	9
#define C_ORI	10
#define C_IND	11
#endif

#ifdef RACE_COLORS
#define COLORS  16
#define PLANES  4
#else
#define COLORS  8
#define PLANES  3
#endif

#define RaceDefaultOffset (C_ROM - RED)


static int zero = 0;
static int one = 1;
static int two = 2;
static int three = 3;


int     W_FastClear = 0;
Window  W_Root;
Colormap W_Colormap;
int     W_Screen;

#ifdef FOURPLANEFIX
Visual *W_Visual;

#endif
W_Font  W_BigFont = (W_Font) & zero, W_RegularFont = (W_Font) & one;
W_Font  W_HighlightFont = (W_Font) & two, W_UnderlineFont = (W_Font) & three;
Display *W_Display;
W_Color W_White = WHITE, W_Black = BLACK, W_Red = RED, W_Green = GREEN;
W_Color W_Yellow = YELLOW, W_Cyan = CYAN, W_Grey = GREY;

#ifdef RACE_COLORS
W_Color W_Ind = C_IND, W_Fed = C_FED, W_Rom = C_ROM, W_Kli = C_KLI, W_Ori = C_ORI;

#endif
int     W_Textwidth, W_Textheight;
char   *getdefault(char *str);

int     W_in_message = 0;			 /* jfy -- for Jerry's warp
						  * message hack */

/* TTS: moved this out so we can use the 8th color */

static unsigned long planes[PLANES];

extern W_Window baseWin;
static XClassHint class_hint =
{
  "netrek", "Netrek",
};

static XWMHints wm_hint =
{
  InputHint | StateHint,
  True,
  NormalState,
  None,
  None,
  0, 0,
  None,
  None,
};

static XSizeHints wm_size_hint;

static W_Event W_myevent;
static int W_isEvent = 0;

struct fontInfo
  {
    int     baseline;
  };

struct colors
  {
    char   *name;
    GC      contexts[FONTS + 1];
    GC      insens_contexts[FONTS + 1];
    Pixmap  pixmap;
    int     pixelValue;
  };

Pixmap  insens_tile;

struct icon
  {
    Window  window;
    Pixmap  bitmap;
    int     width, height;
    Pixmap  pixmap;
  };

#define WIN_GRAPH	1
#define WIN_TEXT	2
#define WIN_MENU	3
#define WIN_SCROLL	4

struct window
  {
    Window  window;
    int     type;
    char   *data;
    int     mapped;
    int     width, height;
    char   *name;
    W_Callback handle_keydown;
    W_Callback handle_keyup;
    W_Callback handle_button;
    W_Callback handle_expose;

#ifdef SHORT_PACKETS
    int     insensitive;
#endif
    Cursor  cursor;
  };

static void configureScrolling(struct window *win, int width, int height);
static int checkGeometry(char *name, int *x, int *y, int *width, int *height);

struct stringList
  {
    char   *string;
    W_Color color;
    struct stringList *next;
  };

struct menuItem
  {
    char   *string;
    W_Color color;
  };

struct colors colortable[] =
{
  {"white"},
  {"black"},
  {"red"},
  {"green"},
  {"yellow"},
  {"cyan"},
  {"light grey"}

#ifdef RACE_COLORS
  ,
  {"Rom"},
  {"Kli"},
  {"Fed"},
  {"Ori"},
  {"Ind"}
#endif

};

struct windowlist
  {
    struct window *window;
    struct windowlist *next;
  };

#define HASHSIZE 101
#define hash(x) (((int) (x)) % HASHSIZE)

struct windowlist *hashtable[HASHSIZE];
struct fontInfo fonts[FONTS];

struct window *newWindow(Window window, int type);
struct window *findWindow(Window window);

/* char *malloc (size_t); */
short  *x11tox10bits();

struct window myroot;

#define NCOLORS (sizeof(colortable)/sizeof(colortable[0]))
#define W_Void2Window(win) ((win) ? ((struct window *) (win)) : (&myroot))
#define W_Window2Void(window) ((W_Window) (window))
#define W_Void2Icon(bit) ((struct icon *) (bit))
#define W_Icon2Void(bit) ((W_Icon) (bit))
#define fontNum(font) (*((int *) font))
#define TILESIDE 16

#define WIN_EDGE 5				 /* border on l/r edges of
						  * text windows */
#define MENU_PAD 4				 /* border on t/b edges of
						  * text windows */
#define MENU_BAR 4				 /* width of menu bar */

static char gray[] =
{
  0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
  0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
  0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55,
  0xaa, 0xaa, 0x55, 0x55, 0xaa, 0xaa, 0x55, 0x55
};

static char striped[] =
{
  0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff,
  0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f,
  0xff, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00,
  0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0, 0x0f, 0xf0
};

static char solid[] =
{
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
  0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff
};

/* X debugging */
int
        _myerror(Display * d, XErrorEvent * e)
{
  abort();
}

pastebuffer(void)
{
  int     nbytes, x;
  char   *buff, c;

  buff = XFetchBuffer(W_Display, &nbytes, 0);
  for (x = 0; x < nbytes; x++)
    {
      c = buff[x];
      smessage(c);
    }
}

static long
        WMXYHintMode_default(void)
{
  static int fetched = 0;
  static long WMXYHM_default;
  char   *hm_default_string;

  if (!fetched)
    {
      hm_default_string = getdefault("WMXYHintMode");
      if (!hm_default_string || strcmp(hm_default_string, "USPosition") == 0)
	WMXYHM_default = USPosition;
      else
	WMXYHM_default = PPosition;
      fetched = 1;
    }
  return WMXYHM_default;
}


void
        W_Initialize(char *str)
{
  int     i;

#ifdef DEBUG
  printf("Initializing...\n");
#endif

  for (i = 0; i < HASHSIZE; i++)
    {
      hashtable[i] = NULL;
    }
  if ((W_Display = XOpenDisplay(str)) == NULL)
    {
      fprintf(stderr, "I can't open your display, twink!\n");
      exit(1);
    }
  /* tmp */
  /* XSynchronize(W_Display, True); */
  /* XSetErrorHandler(_myerror); */

  W_Root = DefaultRootWindow(W_Display);

#ifdef FOURPLANEFIX
  W_Visual = DefaultVisual(W_Display, DefaultScreen(W_Display));
#endif

  W_Screen = DefaultScreen(W_Display);
  W_Colormap = DefaultColormap(W_Display, W_Screen);
  myroot.window = W_Root;
  myroot.type = WIN_GRAPH;
  GetFonts();
  GetColors();

#ifdef BEEPLITE
  init_tts();
#endif
}

/* Make sure the font will work, ie: that it fits in the 6x10 character cell
 * that we expect. */
checkFont(XFontStruct * fontinfo, char *fontname)
{

#ifndef SMALL_SCREEN
  if (fontinfo->max_bounds.width != 6 ||
      fontinfo->min_bounds.width != 6 ||
      fontinfo->descent + fontinfo->ascent != 10 ||
      fontinfo->min_bounds.lbearing < 0 ||
      fontinfo->max_bounds.rbearing > 6 ||
      fontinfo->max_bounds.ascent > 8 ||
      fontinfo->max_bounds.descent > 2)
    {
      fprintf(stderr, "Warning: font '%s'\ndoes not conform to 6x10 character cell rules.\n", fontname);
    }
#endif
}

GetFonts(void)
{
  Font    regular, italic, bold, big, ind;
  int     i, j;
  XGCValues values;
  XFontStruct *fontinfo;
  char   *fontname;
  int     black, white;

  fontname = getdefault("font");
  if (fontname == NULL)
    fontname = NORMAL_FONT;
  fontinfo = XLoadQueryFont(W_Display, fontname);
  if (fontinfo == NULL)
    {
      fontinfo = find_font(fontname, _nfonts);
    }
  if (fontinfo == NULL)
    {
      printf("netrek: Can't find any fonts!\n");
      exit(1);
    }
  checkFont(fontinfo, fontname);
  regular = fontinfo->fid;
  W_Textwidth = fontinfo->max_bounds.width;
  W_Textheight = fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent;
  fonts[1].baseline = fontinfo->max_bounds.ascent;

  fontname = getdefault("boldfont");
  if (fontname == NULL)
    fontname = BOLD_FONT;
  fontinfo = XLoadQueryFont(W_Display, fontname);
  if (fontinfo == NULL)
    {
      fontinfo = find_font(fontname, _bfonts);
    }
  if (fontinfo == NULL)
    {
      bold = regular;
      fonts[2].baseline = fonts[1].baseline;
    }
  else
    {
      checkFont(fontinfo, fontname);
      bold = fontinfo->fid;
      fonts[2].baseline = fontinfo->max_bounds.ascent;
      if (fontinfo->max_bounds.width > W_Textwidth)
	W_Textwidth = fontinfo->max_bounds.width;
      if (fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent > W_Textheight)
	W_Textheight = fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent;
    }

  fontname = getdefault("italicfont");
  if (fontname == NULL)
    fontname = ITALIC_FONT;
  fontinfo = XLoadQueryFont(W_Display, fontname);
  if (fontinfo == NULL)
    {
      fontinfo = find_font(fontname, _ifonts);
    }
  if (fontinfo == NULL)
    {
      italic = regular;
      fonts[3].baseline = fonts[1].baseline;
    }
  else
    {
      checkFont(fontinfo, fontname);
      italic = fontinfo->fid;
      fonts[3].baseline = fontinfo->max_bounds.ascent;
      if (fontinfo->max_bounds.width > W_Textwidth)
	W_Textwidth = fontinfo->max_bounds.width;
      if (fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent > W_Textheight)
	W_Textheight = fontinfo->max_bounds.descent + fontinfo->max_bounds.ascent;
    }

  fontname = getdefault("bigfont");
  if (fontname == NULL)
    fontname = BIG_FONT;
  fontinfo = XLoadQueryFont(W_Display, fontname);
  if (fontinfo == NULL)
    {
      fontinfo = find_font(fontname, _bgfonts);
    }
  if (fontinfo == NULL)
    {
      big = regular;
      fonts[0].baseline = fonts[1].baseline;
    }
  else
    {
      big = fontinfo->fid;
      fonts[0].baseline = fontinfo->max_bounds.ascent;
    }
  white = WhitePixel(W_Display, W_Screen);
  black = BlackPixel(W_Display, W_Screen);
  insens_tile = XCreatePixmapFromBitmapData(W_Display, W_Root, gray,
       TILESIDE, TILESIDE, black, white, DefaultDepth(W_Display, W_Screen));
  for (i = 0; i < NCOLORS; i++)
    {
      values.font = big;
      colortable[i].contexts[0] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[0], False);
      values.font = regular;
      colortable[i].contexts[1] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[1], False);

      values.fill_style = FillTiled;
      values.tile = insens_tile;
      colortable[i].insens_contexts[1] = XCreateGC(W_Display, W_Root,
				    GCFont | GCFillStyle | GCTile, &values);
      XSetGraphicsExposures(W_Display, colortable[i].insens_contexts[1],
			    False);
      values.font = bold;
      colortable[i].contexts[2] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[2], False);
      values.font = italic;
      colortable[i].contexts[3] = XCreateGC(W_Display, W_Root, GCFont, &values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[3], False);
      {
	static char dl[] =
	{1, 8};

	XSetLineAttributes(W_Display, colortable[i].contexts[3],
			   0, LineOnOffDash, CapButt, JoinMiter);
	XSetDashes(W_Display, colortable[i].contexts[3], 0, dl, 2);
      }
      values.function = GXor;
      colortable[i].contexts[BITGC] = XCreateGC(W_Display, W_Root, GCFunction,
						&values);
      XSetGraphicsExposures(W_Display, colortable[i].contexts[BITGC], False);
    }
}
XFontStruct *
        find_font(char *oldf, char **fonts)
{
  XFontStruct *fi;
  char  **f;

  fprintf(stderr, "netrek: Can't find font %s.  Trying others...\n",
	  oldf);
  for (f = fonts; *f; f++)
    {
      if (strcmp(*f, oldf) != 0)
	{
	  if ((fi = XLoadQueryFont(W_Display, *f)))
	    return fi;
	}
    }
  return (NULL);
}

#ifdef FOURPLANEFIX
static unsigned short extrared[COLORS] =
{0x00, 0x20, 0x40, 0x60, 0x80, 0xa0, 0xb0, 0xc0};
static unsigned short extragreen[COLORS] =
{0x40, 0x60, 0x80, 0xa0, 0xb0, 0xc0, 0x00, 0x20};
static unsigned short extrablue[COLORS] =
{0x80, 0xa0, 0xb0, 0xc0, 0x00, 0x20, 0x40, 0x60};

#endif

GetColors(void)
{
  int     i, j;
  XColor  foo, garbage;
  int     white, black;
  unsigned long pixel;
  char    defaultstring[100];
  char   *defaults;

#ifdef FOURPLANEFIX
  char   *colorname;
  unsigned long extracolors[COLORS];
  XColor  colordef;

#endif

  forceMono = booleanDefault("forcemono", forceMono);	/* 11/14/91 TC */

  if ((DisplayCells(W_Display, W_Screen) <= 2) || forceMono)
    {
      white = WhitePixel(W_Display, W_Screen);
      black = BlackPixel(W_Display, W_Screen);
      for (i = 0; i < NCOLORS; i++)
	{
	  if (i != W_Black)
	    {
	      colortable[i].pixelValue = white;
	    }
	  else
	    {
	      colortable[i].pixelValue = black;
	    }
	  if (i == W_Red)
	    {
	      colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
			  W_Root, striped, TILESIDE, TILESIDE, white, black,
					 DefaultDepth(W_Display, W_Screen));
	    }
	  else if (i == W_Yellow)
	    {
	      colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
			     W_Root, gray, TILESIDE, TILESIDE, white, black,
					 DefaultDepth(W_Display, W_Screen));
	    }
	  else
	    {
	      colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
					  W_Root, solid, TILESIDE, TILESIDE,
						   colortable[i].pixelValue,
						   colortable[i].pixelValue,
					 DefaultDepth(W_Display, W_Screen));
	    }
	  /* We assume white is 0 or 1, and black is 0 or 1. We adjust
	   * graphics function based upon who is who. */
	  if (white == 0)
	    {					 /* Black is 1 */
	      XSetFunction(W_Display, colortable[i].contexts[BITGC],
			   GXand);
	    }
	}
    }
  else if (DefaultDepth(W_Display, W_Screen) == 8)
    {

#ifdef FOURPLANEFIX

      if (!XAllocColorCells(W_Display, W_Colormap, False, planes, PLANES,
			    &pixel, 1))
	{
	  /* couldn't allocate 3 planes, make a new colormap */
	  W_Colormap = XCreateColormap(W_Display, W_Root, W_Visual, AllocNone);

	  if (!XAllocColorCells(W_Display, W_Colormap, False, planes, PLANES,
				&pixel, 1))
	    {
	      fprintf(stderr, "Cannot create new colormap\n");
	      exit(1);
	    }
	  /* and fill it with at least 8 more colors so when mouse is inside
	   * netrek windows, use might be able to see his other windows */
	  if (XAllocColorCells(W_Display, W_Colormap, False, NULL, 0,
			       extracolors, COLORS))
	  {
	    colordef.flags = DoRed | DoGreen | DoBlue;

	    for (i = 0; i < COLORS; i++)
	      {
		colordef.pixel = extracolors[i];
		colordef.red = extrared[i] << 8;
		colordef.green = extragreen[i] << 8;
		colordef.blue = extrablue[i] << 8;
		XStoreColor(W_Display, W_Colormap, &colordef);
	      }
	  }
	}
#else

      XAllocColorCells(W_Display, W_Colormap, False, planes, PLANES, &pixel, 1);

#endif

      for (i = 0; i < NCOLORS; i++)
	{
	  sprintf(defaultstring, "color.%s", colortable[i].name);
	  defaults = getdefault(defaultstring);

	  if (defaults == NULL)
	    {

#ifdef RACE_COLORS
	      if (i > GREY)
		{
		  /* The default colour from the ROMS is the colour defined
		   * to be RED and not the colour which is actually RED. */

		  sprintf(defaultstring, "color.%s",
			  colortable[i - RaceDefaultOffset].name);
		  defaults = getdefault(defaultstring);

		  if (defaults == NULL)
		    defaults = colortable[i - RaceDefaultOffset].name;
		}
	      else
#endif

		defaults = colortable[i].name;
	    }

	  XParseColor(W_Display, W_Colormap, defaults, &foo);
	  
	  switch (i)
	    {
#ifndef RACE_COLORS
	    case WHITE:
	      foo.pixel = pixel | planes[0] | planes[1] | planes[2];
	      break;
	    case BLACK:
	      foo.pixel = pixel;
	      break;
	    case RED:
	      foo.pixel = pixel | planes[0];
	      break;
	    case CYAN:
	      foo.pixel = pixel | planes[1];
	      break;
	    case YELLOW:
	      foo.pixel = pixel | planes[2];
	      break;
	    case GREY:
	      foo.pixel = pixel | planes[0] | planes[1];
	      break;
	    case GREEN:
	      foo.pixel = pixel | planes[1] | planes[2];
	      break;
#else
	    /*
	       Choose colors so that when two ships overlap, things
	       look ok.  When players overlab, the bits are ORed together.
	    */

	    /* Background color */
	    case BLACK:
	      foo.pixel = pixel;
	      break;
	      
	    /* Alert colors (sum to grey usually) */
	    case RED:
	      foo.pixel = pixel | planes[1] | planes[2];
	      break;
	    case CYAN:
	      foo.pixel = pixel | planes[1] | planes[3];
	      break;
	    case YELLOW:
	      foo.pixel = pixel | planes[2] | planes[3];
	      break;
	    case GREEN:
	      foo.pixel = pixel | planes[1];
	      break;
	    case GREY:
	      foo.pixel = pixel | planes[1] | planes[2] | planes[3];
	      break;
	      
	    /* Your color */
	    case WHITE:
	      foo.pixel = pixel | planes[0];
	      break;
	      
	    /* The color of other ships should dominate over your color
	     * and should sum to C_IND where possible. */
	    case C_FED:
	      foo.pixel = pixel | planes[0] | planes[1] | planes[2];
	      break;
	    case C_ROM:
	      foo.pixel = pixel | planes[0] | planes[1] | planes[3];
	      break;
	    case C_KLI:
	      foo.pixel = pixel | planes[0] | planes[2] | planes[3];
	      break;
	    case C_ORI:
	      foo.pixel = pixel | planes[0] | planes[1] ;
	      break;
	    case C_IND:
	      foo.pixel = pixel | planes[0] | planes[1] | planes[2] |planes[3];
	      break;
#endif
	    }
	  XStoreColor(W_Display, W_Colormap, &foo);
	  colortable[i].pixelValue = foo.pixel;
	  colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
		    W_Root, solid, TILESIDE, TILESIDE, foo.pixel, foo.pixel,
					 DefaultDepth(W_Display, W_Screen));
	}
    }
  else if (W_Visual->class == TrueColor)
    {
      /* Stuff added by sheldon@iastate.edu 5/28/93 This is supposed to
       * detect a TrueColor display, and then do a lookup of the colors in
       * default colormap, instead of creating new colormap. */
      for (i = 0; i < NCOLORS; i++)
	{
	  sprintf(defaultstring, "color.%s", colortable[i].name);

	  defaults = getdefault(defaultstring);
	  if (defaults == NULL)
	    {

#ifdef RACE_COLORS
	      if (i > GREY)
		{
		  /* The default color from the ROMS is the color defined to
		   * be RED and not the color which is actually RED. */

		  sprintf(defaultstring, "color.%s",
			  colortable[i - RaceDefaultOffset].name);
		  defaults = getdefault(defaultstring);

		  if (defaults == NULL)
		    defaults = colortable[i - RaceDefaultOffset].name;
		}
	      else
#endif

		defaults = colortable[i].name;
	    }


	  XParseColor(W_Display, W_Colormap, defaults, &foo);
	  XAllocColor(W_Display, W_Colormap, &foo);
	  colortable[i].pixelValue = foo.pixel;
	  colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
		    W_Root, solid, TILESIDE, TILESIDE, foo.pixel, foo.pixel,
					 DefaultDepth(W_Display, W_Screen));
	}
    }
  else
    {

#ifdef FOURPLANEFIX				 /* 25/6/93 still problems
						  * with 4 plains SK */

      if (!XAllocColorCells(W_Display, W_Colormap, False, planes, PLANES,
			    &pixel, 1))
	{
	  /* couldn't allocate 3 planes, make a new colormap */
	  W_Colormap = XCreateColormap(W_Display, W_Root, W_Visual, AllocNone);

	  if (!XAllocColorCells(W_Display, W_Colormap, False, planes, PLANES,
				&pixel, 1))
	    {
	      fprintf(stderr, "Cannot create new colormap\n");
	      exit(1);
	    }
	  /* and fill it with at least 8 more colors so when mouse is inside
	   * netrek windows, use might be able to see his other windows */
	  if (XAllocColorCells(W_Display, W_Colormap, False, NULL, 0,
			       extracolors, COLORS))
	  {
	    colordef.flags = DoRed | DoGreen | DoBlue;

	    for (i = 0; i < COLORS; i++)
	      {
		colordef.pixel = extracolors[i];
		colordef.red = extrared[i] << 8;
		colordef.green = extragreen[i] << 8;
		colordef.blue = extrablue[i] << 8;
		XStoreColor(W_Display, W_Colormap, &colordef);
	      }
	  }
	}
#endif

      for (i = 0; i < NCOLORS; i++)
	{
	  sprintf(defaultstring, "color.%s", colortable[i].name);

	  defaults = getdefault(defaultstring);
	  if (defaults == NULL)
	    defaults = colortable[i].name;
	  XParseColor(W_Display, W_Colormap, defaults, &foo);
	  /* Black must be the color with all the planes off. That is the
	   * only restriction I concerned myself with in the following case
	   * statement. */
	  switch (i)
	    {
	    case WHITE:
	      foo.pixel = pixel | planes[0] | planes[1] | planes[2];
	      break;
	    case BLACK:
	      foo.pixel = pixel;
	      break;
	    case RED:
	      foo.pixel = pixel | planes[0];
	      break;
	    case CYAN:
	      foo.pixel = pixel | planes[1];
	      break;
	    case YELLOW:
	      foo.pixel = pixel | planes[2];
	      break;
	    case GREY:
	      foo.pixel = pixel | planes[0] | planes[1];
	      break;
	    case GREEN:
	      foo.pixel = pixel | planes[1] | planes[2];
	      break;

#ifdef RACE_COLORS
	    case C_IND:
	      foo.pixel = pixel | planes[0] | planes[3];
	      break;
	    case C_FED:
	      foo.pixel = pixel | planes[1] | planes[3];
	      break;
	    case C_ROM:
	      foo.pixel = pixel | planes[2] | planes[3];
	      break;
	    case C_KLI:
	      foo.pixel = pixel | planes[0] | planes[1] | planes[3];
	      break;
	    case C_ORI:
	      foo.pixel = pixel | planes[1] | planes[2] | planes[3];
	      break;
#endif
	    }
	  XStoreColor(W_Display, W_Colormap, &foo);
	  colortable[i].pixelValue = foo.pixel;
	  colortable[i].pixmap = XCreatePixmapFromBitmapData(W_Display,
		    W_Root, solid, TILESIDE, TILESIDE, foo.pixel, foo.pixel,
					 DefaultDepth(W_Display, W_Screen));
	}
    }
  for (i = 0; i < NCOLORS; i++)
    {
      for (j = 0; j < FONTS + 1; j++)
	{
	  XSetForeground(W_Display, colortable[i].contexts[j],
			 colortable[i].pixelValue);
	  XSetBackground(W_Display, colortable[i].contexts[j],
			 colortable[W_Black].pixelValue);
	}
    }
}

W_Window
W_RenameWindow(struct window *window, char *str)
{
  XStoreName(W_Display, window->window, str);
}

W_Window
W_MakeWindow(char *name, int x, int y, int width, int height, W_Window parent, int border, W_Color color)
{
  struct window *newwin;
  Window  wparent;
  XSetWindowAttributes attrs;
  char   *window_title = "Netrek", title_buff[257];
  XSizeHints *sz_hints;
  int     gcheck_result;

#ifdef DEBUG
  printf("New window...\n");
#endif

  gcheck_result = checkGeometry(name, &x, &y, &width, &height);
  checkParent(name, &parent);
  wparent = W_Void2Window(parent)->window;
  attrs.border_pixel = colortable[color].pixelValue;
  attrs.event_mask = KeyPressMask | ButtonPressMask | ExposureMask | LeaveWindowMask;

#ifdef AUTOKEY
  attrs.event_mask |= KeyReleaseMask;
#endif /* AUTOKEY */

#ifdef MOTION_MOUSE
  if (motion_mouse)
    attrs.event_mask |= ButtonMotionMask;
#endif

  if (strcmp(name, "netrek_icon") == 0)		 /* icon should not select
						  * for input */
    attrs.event_mask = ExposureMask;
  attrs.background_pixel = colortable[W_Black].pixelValue;
  attrs.do_not_propagate_mask = KeyPressMask | ButtonPressMask | ExposureMask;
  newwin = newWindow(
	      XCreateWindow(W_Display, wparent, x, y, width, height, border,
			    CopyFromParent, InputOutput, CopyFromParent,
			    CWBackPixel | CWEventMask |
			    CWBorderPixel,
			    &attrs),
		      WIN_GRAPH);
  /* top window */
  sz_hints = XAllocSizeHints();
  if (strcmp(name, "netrek") == 0 || strcmp(name, "wait") == 0)
    {
      if (!title && serverName)
	{
	  if (strcmp(name, "wait") == 0)
	    strcpy(title_buff, serverName);
	  else
	    sprintf(title_buff, "Netrek  @  %s", serverName);
	  window_title = title_buff;
	}
      else
      /* but title on command line will override */ if (title)
	window_title = title;
      sz_hints->min_width = width;
      sz_hints->max_width = width;
      sz_hints->min_height = height;
      sz_hints->max_height = height;
      sz_hints->flags = PMinSize | PMaxSize;
      if (gcheck_result & G_SET_X || gcheck_result & G_SET_Y)
	sz_hints->flags |= WMXYHintMode_default();
    }
  else
    {
      window_title = name;
      if (gcheck_result & G_SET_X || gcheck_result & G_SET_Y)
	sz_hints->flags |= WMXYHintMode_default();
    }
  XStoreName(W_Display, newwin->window, window_title);
  XSetWMNormalHints(W_Display, newwin->window, sz_hints);
  class_hint.res_name = name;
  XSetClassHint(W_Display, newwin->window, &class_hint);
  XSetWMHints(W_Display, newwin->window, &wm_hint);
  if (wparent == W_Root && baseWin != NULL && strcmp(name, "wait") != 0)
    XSetTransientForHint(W_Display, newwin->window, W_Void2Window(baseWin)->window);
  newwin->name = strdup(name);
  newwin->width = width;
  newwin->height = height;
  if (wparent != W_Root)
    if (checkMapped(name))
      W_MapWindow(W_Window2Void(newwin));

#ifdef VMS
  XSelectAsyncInput(W_Display, newwin->window, attrs.event_mask, vms_event_window, 0);
#endif

#ifdef DEBUG
  printf("New graphics window %d, child of %d\n", newwin, parent);
#endif

#ifdef FOURPLANEFIX
  XSetWindowColormap(W_Display, newwin->window, W_Colormap);
#endif

  return (W_Window2Void(newwin));
}

void
        W_ChangeBorder(W_Window window, int color)
{

#ifdef DEBUG
  printf("Changing border of %d\n", window);
#endif

  /* fix inexplicable color bug */
  if ((DisplayCells(W_Display, W_Screen) <= 2) || forceMono)

    XSetWindowBorderPixmap(W_Display, W_Void2Window(window)->window,
			   colortable[color].pixmap);
  else
    XSetWindowBorder(W_Display, W_Void2Window(window)->window,
		     colortable[color].pixelValue);


}

void
        W_MapWindow(W_Window window)
{
  struct window *win;

#ifdef DEBUG
  printf("Mapping %d\n", window);
#endif

  win = W_Void2Window(window);
  if (win->mapped)
    return;
  win->mapped = 1;
  XMapRaised(W_Display, win->window);
}

void
        W_UnmapWindow(W_Window window)
{
  struct window *win;

#ifdef DEBUG
  printf("UnMapping %d\n", window);
#endif

  win = W_Void2Window(window);
  if (win->mapped == 0)
    return;
  win->mapped = 0;
  XUnmapWindow(W_Display, win->window);
}

int
        W_IsMapped(W_Window window)
{
  struct window *win;

  win = W_Void2Window(window);
  if (win == NULL)
    return (0);
  return (win->mapped);
}

void
        W_FillArea(W_Window window, int x, int y, int width, int height, W_Color color)
{
  struct window *win;

#ifdef DEBUG
  printf("Clearing (%d %d) x (%d %d) with %d on %d\n", x, y, width, height,
	 color, window);
#endif

  win = W_Void2Window(window);
  switch (win->type)
    {
    case WIN_GRAPH:
      XFillRectangle(W_Display, win->window, colortable[color].contexts[0],
		     x, y, width, height);
      break;
    default:
      XFillRectangle(W_Display, win->window, colortable[color].contexts[0],
		     WIN_EDGE + x * W_Textwidth, MENU_PAD + y * W_Textheight,
		     width * W_Textwidth, height * W_Textheight);
    }
}

/* XFIX */

static XRectangle _rcache[MAXCACHE];
static int _rcache_index;

static void
        FlushClearAreaCache(Window win)
{
  XFillRectangles(W_Display, win, colortable[backColor].contexts[0],
		  _rcache, _rcache_index);
  _rcache_index = 0;
}

/* local window only */
void
        W_CacheClearArea(W_Window window, int x, int y, int width, int height)
{
  Window  win = W_Void2Window(window)->window;
  register XRectangle *r;

  if (_rcache_index == MAXCACHE)
    FlushClearAreaCache(win);

  r = &_rcache[_rcache_index++];
  r->x = (short) x;
  r->y = (short) y;
  r->width = (unsigned short) width;
  r->height = (unsigned short) height;
}

void
        W_FlushClearAreaCache(W_Window window)
{
  Window  win = W_Void2Window(window)->window;

  if (_rcache_index)
    FlushClearAreaCache(win);
}

/* XFIX: clears now instead of filling. */
void
        W_ClearArea(W_Window window, int x, int y, int width, int height)
{
  struct window *win;

  win = W_Void2Window(window);
  switch (win->type)
    {
    case WIN_GRAPH:
      /* XFIX: changed */
      XClearArea(W_Display, win->window, x, y, width, height, False);
      break;
    default:
      /* XFIX: changed */
      XClearArea(W_Display, win->window, WIN_EDGE + x * W_Textwidth,
		 MENU_PAD + y * W_Textheight, width * W_Textwidth, height * W_Textheight, False);
      break;
    }
}

void
        W_ClearWindow(W_Window window)
{

#ifdef DEBUG
  printf("Clearing %d\n", window);
#endif

  XClearWindow(W_Display, W_Void2Window(window)->window);
}

int
        W_EventsPending(void)
{
  if (W_isEvent)
    return (1);
  while (XPending(W_Display))
    {
      if (W_SpNextEvent(&W_myevent))
	{
	  W_isEvent = 1;
	  return (1);
	}
    }
  return (0);
}

void
        W_NextEvent(W_Event * wevent)
{
  if (W_isEvent)
    {
      *wevent = W_myevent;
      W_isEvent = 0;
      return;
    }
  while (W_SpNextEvent(wevent) == 0);
}

int
        W_SpNextEvent(W_Event * wevent)
{
  XEvent  event;
  XKeyEvent *key;
  XButtonEvent *button;
  XExposeEvent *expose;
  XConfigureEvent *configure;

#ifdef MOTION_MOUSE
  XMotionEvent *motion;
  static int prev_x, prev_y;
  int     thresh;

#endif

#ifdef CONTROL_KEY
  int     control_key = 0;

#endif
  unsigned char ch;
  int     nchars, ind;
  struct window *win;

#ifdef FUNCTION_KEYS
  KeySym  sym;

#endif

#ifdef DEBUG
  printf("Getting an event...\n");
#endif

  key = (XKeyEvent *) & event;
  button = (XButtonEvent *) & event;
  expose = (XExposeEvent *) & event;
  configure = (XConfigureEvent *) & event;

#ifdef MOTION_MOUSE
  motion = (XMotionEvent *) & event;
#endif

  for (;;)
    {
      XNextEvent(W_Display, &event);

#ifdef VMS
      vms_event_window_done();
#endif

      /* printf("read an event %d\n", event.type); */
      win = findWindow(key->window);
      if (win == NULL)
	return (0);
      if (key->send_event == True)		 /* event sent by another
						  * client */
	return 0;
      if ((event.type == KeyPress || event.type == ButtonPress) &&
	  win->type == WIN_MENU)
	{
	  if (key->y % (W_Textheight + MENU_PAD * 2 + MENU_BAR) >=
	      W_Textheight + MENU_PAD * 2)
	    return (0);
	  key->y = key->y / (W_Textheight + MENU_PAD * 2 + MENU_BAR);
	}
      switch ((int) event.type)
	{
	case LeaveNotify:			 /* for message window -- jfy */
	  if (win == (struct window *) messagew)
	    {
	      W_in_message = 0;
	    }
	  return (0);
	  break;
	case KeyPress:

	  if ((key->state & LockMask) && !(key->state & ShiftMask) &&
	      (ignoreCaps))
	    {
	      printf("Got a capslock!\n");
	      key->state = key->state & ~LockMask;
	    }

#ifdef CONTROL_KEY
	  if (key->state & ControlMask && use_control_key)
	    {
	      control_key = 1;
	      key->state = key->state & ~ControlMask;
	    }
	  else
	    control_key = 0;
#endif

	  if (XLookupString(key, &ch, 1, NULL, NULL) > 0)
	    {

#ifdef MOUSE_AS_SHIFT
	      if (mouse_as_shift)
		{
		  if (key->state & Button1Mask)
		    {
		      wevent->modifier = W_LBUTTON;
		      wevent->type = W_EV_MKEY;
		    }
		  else if (key->state & Button2Mask)
		    {
		      wevent->modifier = W_MBUTTON;
		      wevent->type = W_EV_MKEY;
		    }
		  else if (key->state & Button3Mask)
		    {
		      wevent->modifier = W_RBUTTON;
		      wevent->type = W_EV_MKEY;
		    }
		  else
		    {
		      wevent->type = W_EV_KEY;
		    }
		}
	      else
		wevent->type = W_EV_KEY;
#else
	      wevent->type = W_EV_KEY;
#endif

	      wevent->Window = W_Window2Void(win);
	      wevent->x = key->x;
	      wevent->y = key->y;

#ifdef CONTROL_KEY
	      if (control_key)
		wevent->key = (unsigned char) (ch + 96);
	      else
		wevent->key = ch;
#else
	      wevent->key = ch;
#endif

	      return (1);
	    }

#ifdef FUNCTION_KEYS
	  if ((sym = XLookupKeysym(key, 0)) != NoSymbol)
	    switch (sym)
	      {
	      case XK_F1:
		printf("Function key1 \n");
		break;
	      case XK_F2:
		printf("Function key2\n");
		break;
	      case XK_F3:
		printf("Function key3\n");
		break;
	      case XK_F4:
		printf("Function key4\n");
		break;
	      case XK_F5:
		printf("Function key5\n");
		break;
	      case XK_F6:
		printf("Function key6\n");
		break;
	      case XK_F7:
		printf("Function key7\n");
		break;
	      case XK_F8:
		printf("Function key8\n");
		break;
	      case XK_F9:
		printf("Function key9\n");
		break;
	      case XK_F10:
		printf("Function key10\n");
		break;
	      }
#endif

	  return (0);
	  break;

#ifdef AUTOKEY
	case KeyRelease:
	  if (XLookupString(key, &ch, 1, NULL, NULL) > 0)
	    {
	      wevent->type = W_EV_KEY_OFF;
	      wevent->Window = W_Window2Void(win);
	      wevent->x = key->x;
	      wevent->y = key->y;
	      wevent->key = ch;
	      return (1);
	    }
	  return (0);
	  break;
#endif /* AUTOKEY */

	case ButtonPress:
	  wevent->type = W_EV_BUTTON;
	  wevent->Window = W_Window2Void(win);

#ifdef MOTION_MOUSE
	  prev_x = wevent->x = button->x;
	  prev_y = wevent->y = button->y;
#else
	  wevent->x = button->x;
	  wevent->y = button->y;
#endif

#ifdef SHIFTED_MOUSE
	  if (extended_mouse)
	    {
	      if (button->state & ControlMask && button->state & ShiftMask)
		{
		  switch (button->button & 0xf)
		    {
		    case Button3:
		      wevent->key = W_RBUTTON4;
		      break;
		    case Button1:
		      wevent->key = W_LBUTTON4;
		      break;
		    case Button2:
		      wevent->key = W_MBUTTON4;
		      break;
		    }
		  return (1);
		}

	      if (button->state & ShiftMask)
		{
		  switch (button->button & 0xf)
		    {
		    case Button3:
		      wevent->key = W_RBUTTON2;
		      break;
		    case Button1:
		      wevent->key = W_LBUTTON2;
		      break;
		    case Button2:
		      wevent->key = W_MBUTTON2;
		      break;
		    }
		  return (1);
		}

	      if (button->state & ControlMask)
		{
		  switch (button->button & 0xf)
		    {
		    case Button3:
		      wevent->key = W_RBUTTON3;
		      break;
		    case Button1:
		      wevent->key = W_LBUTTON3;
		      break;
		    case Button2:
		      wevent->key = W_MBUTTON3;
		      break;
		    }
		  return (1);
		}
	    }
#endif

#ifdef MOUSE_AS_SHIFT
	  if (mouse_as_shift &&
	      (wevent->Window == mapw || wevent->Window == w))
	    switch (button->button & 0xf)
	      {
	      case Button3:
		if (b3_as_shift)
		  return (0);
		break;
	      case Button1:
		if (b1_as_shift)
		  return (0);
		break;
	      case Button2:
		if (b2_as_shift)
		  return (0);
		break;
	      }
#endif

	  switch (button->button & 0xf)
	    {
	    case Button3:
	      wevent->key = W_RBUTTON;
	      break;
	    case Button1:
	      wevent->key = W_LBUTTON;
	      break;
	    case Button2:
	      wevent->key = W_MBUTTON;
	      break;
	    }
	  return (1);

#ifdef MOTION_MOUSE
	case MotionNotify:
	  if (!motion_mouse)
	    return (0);
	  wevent->type = W_EV_BUTTON;
	  wevent->Window = W_Window2Void(win);

	  thresh = abs(prev_x - motion->x) + abs(prev_y - motion->y);

	  if (thresh < user_motion_thresh)
	    return (0);

	  prev_x = wevent->x = motion->x;
	  prev_y = wevent->y = motion->y;

#ifdef SHIFTED_MOUSE
	  if (extended_mouse)
	    {

	      if (button->state & ControlMask && button->state & ShiftMask)
		{
		  switch (button->button & 0xf)
		    {
		    case Button3:
		      wevent->key = W_RBUTTON4;
		      break;
		    case Button1:
		      wevent->key = W_LBUTTON4;
		      break;
		    case Button2:
		      wevent->key = W_MBUTTON4;
		      break;
		    }
		  return (1);
		}

	      if (button->state & ShiftMask)
		{
		  switch (button->button & 0xf)
		    {
		    case Button3:
		      wevent->key = W_RBUTTON2;
		      break;
		    case Button1:
		      wevent->key = W_LBUTTON2;
		      break;
		    case Button2:
		      wevent->key = W_MBUTTON2;
		      break;
		    }
		  return (1);
		}

	      if (button->state & ControlMask)
		{
		  switch (button->button & 0xf)
		    {
		    case Button3:
		      wevent->key = W_RBUTTON3;
		      break;
		    case Button1:
		      wevent->key = W_LBUTTON3;
		      break;
		    case Button2:
		      wevent->key = W_MBUTTON3;
		      break;
		    }
		  return (1);
		}
	    }
#endif

#ifdef MOUSE_AS_SHIFT
	  if (mouse_as_shift &&
	      (wevent->Window == mapw || wevent->Window == w))
	    switch (button->button & 0xf)
	      {
	      case Button3:
		if (b3_as_shift)
		  return (0);
		break;
	      case Button1:
		if (b1_as_shift)
		  return (0);
		break;
	      case Button2:
		if (b2_as_shift)
		  return (0);
		break;
	      }
#endif

	  switch (motion->state & 0xf)
	    {
	    case Button3:
	      wevent->key = W_RBUTTON;
	      break;
	    case Button1:
	      wevent->key = W_LBUTTON;
	      break;
	    case Button2:
	      wevent->key = W_MBUTTON;
	      break;
	    }
	  return (1);
#endif

	case Expose:
	  if (expose->count != 0)
	    return (0);
	  if (win->type == WIN_SCROLL)
	    {
	      redrawScrolling(win);
	      return (0);
	    }
	  if (win->type == WIN_MENU)
	    {
	      redrawMenu(win);
	      return (0);
	    }

	  wevent->type = W_EV_EXPOSE;
	  wevent->Window = W_Window2Void(win);
	  return (1);
	case ConfigureNotify:
	  configureScrolling(win, configure->width, configure->height);
	  break;
	default:
	  return (0);
	  break;
	}
    }
}

void
        W_MakeLine(W_Window window, int x0, int y0, int x1, int y1, W_Color color)
{
  Window  win;

#ifdef DEBUG
  printf("Line on %d\n", window);
#endif

  win = W_Void2Window(window)->window;
  XDrawLine(W_Display, win, colortable[color].contexts[0], x0, y0, x1, y1);
}

/* XFIX */

static XSegment _lcache[NCOLORS][MAXCACHE];
static int _lcache_index[NCOLORS];

static void
        FlushLineCache(Window win, int color)
{
  XDrawSegments(W_Display, win, colortable[color].contexts[0],
		_lcache[color], _lcache_index[color]);
  _lcache_index[color] = 0;
}

/* for local window only */
void
        W_CacheLine(W_Window window, int x0, int y0, int x1, int y1, int color)
{
  Window  win = W_Void2Window(window)->window;
  register XSegment *s;

  if (_lcache_index[color] == MAXCACHE)
    FlushLineCache(win, color);

  s = &_lcache[color][_lcache_index[color]++];
  s->x1 = (short) x0;
  s->y1 = (short) y0;
  s->x2 = (short) x1;
  s->y2 = (short) y1;
}

void
        W_FlushLineCaches(W_Window window)
{
  Window  win = W_Void2Window(window)->window;
  register i;

  for (i = 0; i < NCOLORS; i++)
    {
      if (_lcache_index[i])
	FlushLineCache(win, i);
    }
}

void
        W_MakeTractLine(W_Window window, int x0, int y0, int x1, int y1, W_Color color)
{
  Window  win;

#ifdef DEBUG
  printf("Line on %d\n", window);
#endif

  win = W_Void2Window(window)->window;
  XDrawLine(W_Display, win, colortable[color].contexts[3], x0, y0, x1, y1);
}

void
        W_MakePhaserLine(W_Window window, int x0, int y0, int x1, int y1, W_Color color)
{
  Window  win;

#ifdef DEBUG
  printf("Line on %d\n", window);
#endif

  win = W_Void2Window(window)->window;
  XDrawLine(W_Display, win, colortable[color].contexts[1], x0, y0, x1, y1);
}

void
        W_WriteTriangle(W_Window window, int x, int y, int s, int t, W_Color color)
{
  struct window *win = W_Void2Window(window);
  XPoint  points[4];

  if (t == 0)
    {
      points[0].x = x;
      points[0].y = y;
      points[1].x = x + s;
      points[1].y = y - s;
      points[2].x = x - s;
      points[2].y = y - s;
      points[3].x = x;
      points[3].y = y;
    }
  else
    {
      points[0].x = x;
      points[0].y = y;
      points[1].x = x + s;
      points[1].y = y + s;
      points[2].x = x - s;
      points[2].y = y + s;
      points[3].x = x;
      points[3].y = y;
    }


  XDrawLines(W_Display, win->window, colortable[color].contexts[0],
	     points, 4, CoordModeOrigin);
}

void
        W_WriteText(W_Window window, int x, int y, W_Color color, char *str, int len, W_Font font)
{
  struct window *win;
  int     addr;

#ifdef DEBUG
  printf("Text for %d @ (%d, %d) in %d: [%s]\n", window, x, y, color, str);
#endif

  win = W_Void2Window(window);
  switch (win->type)
    {
    case WIN_GRAPH:
      addr = fonts[fontNum(font)].baseline;
      XDrawImageString(W_Display, win->window,

#ifdef SHORT_PACKETS
		       win->insensitive ?
		       colortable[color].insens_contexts[1] :
		       colortable[color].contexts[fontNum(font)],
#else
		       colortable[color].contexts[fontNum(font)],
#endif

		       x, y + addr, str, len);
      break;
    case WIN_SCROLL:
      XCopyArea(W_Display, win->window, win->window,
	 colortable[W_White].contexts[0], WIN_EDGE, MENU_PAD + W_Textheight,
		win->width * W_Textwidth, (win->height - 1) * W_Textheight,
		WIN_EDGE, MENU_PAD);
      XClearArea(W_Display, win->window,
		 WIN_EDGE, MENU_PAD + W_Textheight * (win->height - 1),
		 W_Textwidth * win->width, W_Textheight, False);
      XDrawImageString(W_Display, win->window,

#ifdef SHORT_PACKETS
		       win->insensitive ?
		       colortable[color].insens_contexts[1] :
		       colortable[color].contexts[1],
#else
		       colortable[color].contexts[1],
#endif

		       WIN_EDGE, MENU_PAD + W_Textheight * (win->height - 1) + fonts[1].baseline,
		       str, len);
      AddToScrolling(win, color, str, len);
      break;
    case WIN_MENU:
      changeMenuItem(win, y, str, len, color);
      break;
    default:
      addr = fonts[fontNum(font)].baseline;
      XDrawImageString(W_Display, win->window,

#ifdef SHORT_PACKETS
		       win->insensitive ?
		       colortable[color].insens_contexts[1] :
		       colortable[color].contexts[fontNum(font)],
#else
		       colortable[color].contexts[fontNum(font)],
#endif

	     x * W_Textwidth + WIN_EDGE, MENU_PAD + y * W_Textheight + addr,
		       str, len);
      break;
    }
}

void
        W_MaskText(W_Window window, int x, int y, W_Color color, char *str, int len, W_Font font)
{
  struct window *win;
  int     addr;

  addr = fonts[fontNum(font)].baseline;

#ifdef DEBUG
  printf("TextMask for %d @ (%d, %d) in %d: [%s]\n", window, x, y, color, str);
#endif

  win = W_Void2Window(window);
  XDrawString(W_Display, win->window,
	  colortable[color].contexts[fontNum(font)], x, y + addr, str, len);
}

W_Icon
W_StoreBitmap(int width, int height, char *data, W_Window window)
{
  struct icon *newicon;
  struct window *win;
  int     black, white;

#ifdef DEBUG
  printf("Storing bitmap for %d (%d x %d)\n", window, width, height);
  fflush(stdout);
#endif

  win = W_Void2Window(window);
  newicon = (struct icon *) malloc(sizeof(struct icon));

  newicon->width = width;
  newicon->height = height;
  newicon->bitmap = XCreateBitmapFromData(W_Display, win->window,
					  data, width, height);

#ifdef nodef
  /* XFIX: changed to Pixmap */
  white = WhitePixel(W_Display, W_Screen);
  black = BlackPixel(W_Display, W_Screen);
  newicon->bitmap = XCreatePixmapFromBitmapData(W_Display, W_Root, data,
						width, height, white, black,
						DefaultDepth(W_Display,
							     W_Screen));
#endif /* nodef */

  newicon->window = win->window;
  newicon->pixmap = 0;
  return (W_Icon2Void(newicon));
}

void
        W_WriteBitmap(int x, int y, W_Icon bit, W_Color color)
{
  struct icon *icon;

  icon = W_Void2Icon(bit);

#ifdef DEBUG
  printf("Writing bitmap to %d\n", icon->window);
#endif

  XCopyPlane(W_Display, icon->bitmap, icon->window,
	 colortable[color].contexts[BITGC], 0, 0, icon->width, icon->height,
	     x, y, 1);

#ifdef nodef
  /* XFIX : copyarea */
  XCopyArea(W_Display, icon->bitmap, icon->window,
	 colortable[color].contexts[BITGC], 0, 0, icon->width, icon->height,
	    x, y);
#endif
}


void
        W_TileWindow(W_Window window, W_Icon bit)
{
  Window  win;
  struct icon *icon;

#ifdef DEBUG
  printf("Tiling window %d\n", window);
#endif

  icon = W_Void2Icon(bit);
  win = W_Void2Window(window)->window;

  if (icon->pixmap == 0)
    {
      icon->pixmap = XCreatePixmap(W_Display, W_Root,
	      icon->width, icon->height, DefaultDepth(W_Display, W_Screen));
      XCopyPlane(W_Display, icon->bitmap, icon->pixmap,
	   colortable[W_White].contexts[0], 0, 0, icon->width, icon->height,
		 0, 0, 1);
    }
  XSetWindowBackgroundPixmap(W_Display, win, icon->pixmap);
  XClearWindow(W_Display, win);

  /* if (icon->pixmap==0) { icon->pixmap=XMakePixmap(icon->bitmap,
   * colortable[W_White].pixelValue, colortable[W_Black].pixelValue); }
   * XChangeBackground(win, icon->pixmap); XClear(win); */
}

void
        W_UnTileWindow(W_Window window)
{
  Window  win;

#ifdef DEBUG
  printf("Untiling window %d\n", window);
#endif

  win = W_Void2Window(window)->window;

  XSetWindowBackground(W_Display, win, colortable[W_Black].pixelValue);
  XClearWindow(W_Display, win);
}

W_Window
W_MakeTextWindow(char *name, int x, int y, int width, int height, W_Window parent, int border)
{
  struct window *newwin;
  Window  wparent;
  XSetWindowAttributes attrs;
  XSizeHints *sz_hints;
  int     gcheck_result;

#ifdef DEBUG
  printf("New window...\n");
#endif

  gcheck_result = checkGeometry(name, &x, &y, &width, &height);
  checkParent(name, &parent);
  attrs.border_pixel = colortable[W_White].pixelValue;
  attrs.event_mask = ExposureMask;

#ifdef AUTOKEY
  attrs.event_mask |= KeyReleaseMask;
#endif /* AUTOKEY */

#ifdef SHORT_PACKETS
  attrs.event_mask |= ButtonPressMask;
#endif

  attrs.background_pixel = colortable[W_Black].pixelValue;
  attrs.do_not_propagate_mask = ExposureMask;
  wparent = W_Void2Window(parent)->window;
  newwin = newWindow(
		      XCreateWindow(W_Display, wparent, x, y,
   width * W_Textwidth + WIN_EDGE * 2, MENU_PAD * 2 + height * W_Textheight,
			border, CopyFromParent, InputOutput, CopyFromParent,
				    CWBackPixel | CWEventMask |
				    CWBorderPixel,
				    &attrs),
		      WIN_TEXT);
  class_hint.res_name = name;
  sz_hints = XAllocSizeHints();
  sz_hints->min_width = WIN_EDGE * 2 + width * W_Textwidth;
  sz_hints->max_width = WIN_EDGE * 2 + width * W_Textwidth;
  sz_hints->base_width = WIN_EDGE * 2;
  sz_hints->width_inc = W_Textwidth;
  sz_hints->min_height = MENU_PAD * 2 + 3 * W_Textheight;
  sz_hints->max_height = MENU_PAD * 2 + height * W_Textheight;
  sz_hints->base_height = MENU_PAD * 2 + 2 * W_Textheight;
  sz_hints->height_inc = W_Textheight;
  sz_hints->flags = PResizeInc | PMinSize | PMaxSize | PBaseSize;
  if (gcheck_result & G_SET_X || gcheck_result & G_SET_Y)
    sz_hints->flags |= WMXYHintMode_default();
  XStoreName(W_Display, newwin->window, name);
  XSetWMNormalHints(W_Display, newwin->window, sz_hints);
  XSetClassHint(W_Display, newwin->window, &class_hint);
  XSetWMHints(W_Display, newwin->window, &wm_hint);
  if (wparent == W_Root && baseWin != NULL)
    XSetTransientForHint(W_Display, newwin->window, W_Void2Window(baseWin)->window);
  newwin->name = strdup(name);
  newwin->width = width;
  newwin->height = height;
  if (wparent != W_Root)
    if (checkMapped(name))
      W_MapWindow(W_Window2Void(newwin));

#ifdef VMS
  XSelectAsyncInput(W_Display, newwin->window, attrs.event_mask, vms_event_window, 0);
#endif

#ifdef DEBUG
  printf("New text window %d, child of %d\n", newwin, parent);
#endif

#ifdef FOURPLANEFIX
  XSetWindowColormap(W_Display, newwin->window, W_Colormap);
#endif

  return (W_Window2Void(newwin));
}

struct window *
        newWindow(Window window, int type)
{
  struct window *newwin;

  newwin = (struct window *) malloc(sizeof(struct window));

  newwin->window = window;
  newwin->type = type;
  newwin->mapped = 0;
  newwin->handle_keydown = 0;
  newwin->handle_keyup = 0;
  newwin->handle_button = 0;
  newwin->handle_expose = 0;
  addToHash(newwin);

#ifdef SHORT_PACKETS
  newwin->insensitive = 0;
#endif

  newwin->cursor = (Cursor) 0;			 /* about the best I can do
						  * -jw */
  return (newwin);
}

struct window *
        findWindow(Window window)
{
  struct windowlist *entry;

  entry = hashtable[hash(window)];
  while (entry != NULL)
    {
      if (entry->window->window == window)
	return (entry->window);
      entry = entry->next;
    }
  return (NULL);
}

addToHash(struct window * win)
{
  struct windowlist **new;

#ifdef DEBUG
  printf("Adding to %d\n", hash(win->window));
#endif

  new = &hashtable[hash(win->window)];
  while (*new != NULL)
    {
      new = &((*new)->next);
    }
  *new = (struct windowlist *) malloc(sizeof(struct windowlist));

  (*new)->next = NULL;
  (*new)->window = win;
}

W_Window
W_MakeScrollingWindow(char *name, int x, int y, int width, int height, W_Window parent, int border)
{
  struct window *newwin;
  Window  wparent;
  XSetWindowAttributes attrs;
  XSizeHints *sz_hints;
  int     gcheck_result;

#ifdef DEBUG
  printf("New window...\n");
#endif

  gcheck_result = checkGeometry(name, &x, &y, &width, &height);
  checkParent(name, &parent);
  wparent = W_Void2Window(parent)->window;
  attrs.border_pixel = colortable[W_White].pixelValue;
  attrs.event_mask = StructureNotifyMask | ExposureMask;

#ifdef AUTOKEY
  attrs.event_mask |= KeyReleaseMask;
#endif /* AUTOKEY */

#ifdef SHORT_PACKETS
  attrs.event_mask |= ButtonPressMask;
#endif

  attrs.background_pixel = colortable[W_Black].pixelValue;
  attrs.do_not_propagate_mask = ResizeRedirectMask | ExposureMask;
  newwin = newWindow(
		      XCreateWindow(W_Display, wparent, x, y,
   width * W_Textwidth + WIN_EDGE * 2, MENU_PAD * 2 + height * W_Textheight,
			border, CopyFromParent, InputOutput, CopyFromParent,
				    CWBackPixel | CWEventMask |
				    CWBorderPixel,
				    &attrs),
		      WIN_SCROLL);
  class_hint.res_name = name;
  sz_hints = XAllocSizeHints();
  sz_hints->width_inc = W_Textwidth;
  sz_hints->height_inc = W_Textheight;
  sz_hints->min_width = WIN_EDGE * 2 + W_Textwidth;
  sz_hints->min_height = MENU_PAD * 2 + W_Textheight;
  sz_hints->base_width = WIN_EDGE * 2;
  sz_hints->base_height = MENU_PAD * 2;
  sz_hints->flags = PResizeInc | PMinSize | PBaseSize;
  if (gcheck_result & G_SET_X || gcheck_result & G_SET_Y)
    sz_hints->flags |= WMXYHintMode_default();
  XStoreName(W_Display, newwin->window, name);
  XSetWMNormalHints(W_Display, newwin->window, sz_hints);
  XSetClassHint(W_Display, newwin->window, &class_hint);
  XSetWMHints(W_Display, newwin->window, &wm_hint);
  if (wparent == W_Root && baseWin != NULL)
    XSetTransientForHint(W_Display, newwin->window, W_Void2Window(baseWin)->window);
  newwin->name = strdup(name);
  newwin->data = NULL;
  newwin->width = width;
  newwin->height = height;
  if (wparent != W_Root)
    if (checkMapped(name))
      W_MapWindow(W_Window2Void(newwin));

#ifdef VMS
  XSelectAsyncInput(W_Display, newwin->window, attrs.event_mask, vms_event_window, 0);
#endif

#ifdef DEBUG
  printf("New scroll window %d, child of %d\n", newwin, parent);
#endif

#ifdef FOURPLANEFIX
  XSetWindowColormap(W_Display, newwin->window, W_Colormap);
#endif

  return (W_Window2Void(newwin));
}

/* Add a string to the string list of the scrolling window. */
AddToScrolling(struct window * win, W_Color color, char *str, int len)
{
  struct stringList **strings;
  char   *newstring;
  int     count;

  /* printf("adding  color %d\n", color); */

  strings = (struct stringList **) &(win->data);
  count = 0;
  while ((*strings) != NULL)
    {
      count++;
      strings = &((*strings)->next);
    }
  (*strings) = (struct stringList *) malloc(sizeof(struct stringList));

  (*strings)->next = NULL;
  (*strings)->color = color;
  newstring = (char *) malloc(len + 1);
  STRNCPY(newstring, str, len + 1);
  newstring[len] = 0;
  (*strings)->string = newstring;
  if (count >= 100)
    {						 /* Arbitrary large size. */
      struct stringList *temp;

      temp = (struct stringList *) win->data;
      free(temp->string);
      temp = temp->next;
      free((char *) win->data);
      win->data = (char *) temp;
    }
}

#ifdef SHORT_PACKETS
W_SetSensitive(W_Window w, int v)
{
  struct window *win = W_Void2Window(w);

  win->insensitive = !v;

  if (win->type == WIN_SCROLL)
    redrawScrolling(win);
}

#endif

redrawScrolling(struct window *win)
{
  int     count;
  struct stringList *list;
  int     y;

  XClearWindow(W_Display, win->window);
  count = 0;
  list = (struct stringList *) win->data;
  while (list != NULL)
    {
      list = list->next;
      count++;
    }
  list = (struct stringList *) win->data;
  while (count > win->height)
    {
      list = list->next;
      count--;
    }
  y = (win->height - count) * W_Textheight + fonts[1].baseline;
  if (count == 0)
    return;
  while (list != NULL)
    {
      XDrawImageString(W_Display, win->window,

#ifdef SHORT_PACKETS
		       win->insensitive ?
		       colortable[list->color].insens_contexts[1] :
		       colortable[list->color].contexts[1],
#else
		       colortable[list->color].contexts[1],
#endif

		WIN_EDGE, MENU_PAD + y, list->string, strlen(list->string));
      list = list->next;
      y = y + W_Textheight;
    }
}

static void
        configureScrolling(struct window *win, int width, int height)
{
  int     new_text_width = (width - WIN_EDGE * 2) / W_Textwidth;
  int     new_text_height = (height - MENU_PAD * 2) / W_Textheight;
  int     new_real_width, new_real_height;

  if (new_text_width <= 0)
    new_text_width = 1;
  if (new_text_height <= 0)
    new_text_height = 1;
  new_real_width = new_text_width * W_Textwidth + WIN_EDGE * 2;
  new_real_height = new_text_height * W_Textheight + MENU_PAD * 2;
  if (new_real_height != height || new_real_width != width)
    {
      XResizeWindow(W_Display, win->window, new_real_width, new_real_height);
      return;
    }
  win->width = new_text_width;
  win->height = new_text_height;
}


W_Window
W_MakeMenu(char *name, int x, int y, int width, int height, W_Window parent, int border)
{
  struct window *newwin;
  struct menuItem *items;
  Window  wparent;
  int     i;
  XSetWindowAttributes attrs;

#ifdef DEBUG
  printf("New window...\n");
#endif

  checkGeometry(name, &x, &y, &width, &height);
  checkParent(name, &parent);
  wparent = W_Void2Window(parent)->window;
  attrs.border_pixel = colortable[W_White].pixelValue;
  attrs.event_mask = KeyPressMask | ButtonPressMask | ExposureMask;

#ifdef AUTOKEY
  attrs.event_mask |= KeyReleaseMask;
#endif /* AUTOKEY */

  attrs.background_pixel = colortable[W_Black].pixelValue;
  attrs.do_not_propagate_mask = KeyPressMask | ButtonPressMask | ExposureMask;
  newwin = newWindow(
		      XCreateWindow(W_Display, wparent, x, y,
				    width * W_Textwidth + WIN_EDGE * 2,
   height * (W_Textheight + MENU_PAD * 2) + (height - 1) * MENU_BAR, border,
				CopyFromParent, InputOutput, CopyFromParent,
				    CWBackPixel | CWEventMask |
				    CWBorderPixel,
				    &attrs),
		      WIN_MENU);
  class_hint.res_name = name;
  XSetClassHint(W_Display, newwin->window, &class_hint);
  XSetWMHints(W_Display, newwin->window, &wm_hint);
  if (wparent == W_Root && baseWin != NULL)
    XSetTransientForHint(W_Display, newwin->window, W_Void2Window(baseWin)->window);
  XStoreName(W_Display, newwin->window, name);
  newwin->name = strdup(name);
  items = (struct menuItem *) malloc(height * sizeof(struct menuItem));

  for (i = 0; i < height; i++)
    {
      items[i].string = NULL;
      items[i].color = W_White;
    }
  newwin->data = (char *) items;
  newwin->width = width;
  newwin->height = height;
  if (wparent != W_Root)
    if (checkMapped(name))
      W_MapWindow(W_Window2Void(newwin));

#ifdef VMS
  XSelectAsyncInput(W_Display, newwin->window, attrs.event_mask, vms_event_window, 0);
#endif

#ifdef DEBUG
  printf("New menu window %d, child of %d\n", newwin, parent);
#endif

#ifdef FOURPLANEFIX
  XSetWindowColormap(W_Display, newwin->window, W_Colormap);
#endif

  return (W_Window2Void(newwin));
}

redrawMenu(struct window * win)
{
  int     count;

  for (count = 1; count < win->height; count++)
    {
      XFillRectangle(W_Display, win->window,
		     colortable[W_White].contexts[0],
	  0, count * (W_Textheight + MENU_PAD * 2) + (count - 1) * MENU_BAR,
		     win->width * W_Textwidth + WIN_EDGE * 2, MENU_BAR);
    }
  for (count = 0; count < win->height; count++)
    {
      redrawMenuItem(win, count);
    }
}

redrawMenuItem(struct window *win, int n)
{
  struct menuItem *items;

  items = (struct menuItem *) win->data;
  XFillRectangle(W_Display, win->window,
		 colortable[W_Black].contexts[0],
	  WIN_EDGE, n * (W_Textheight + MENU_PAD * 2 + MENU_BAR) + MENU_PAD,
		 win->width * W_Textwidth, W_Textheight);
  if (items[n].string)
    {
      XDrawImageString(W_Display, win->window,
		       colortable[items[n].color].contexts[1],
		       WIN_EDGE,
		       n * (W_Textheight + MENU_PAD * 2 + MENU_BAR) + MENU_PAD + fonts[1].baseline,
		       items[n].string, strlen(items[n].string));
    }
}

changeMenuItem(struct window *win, int n, char *str, int len, W_Color color)
{
  struct menuItem *items;
  char   *news;

  items = (struct menuItem *) win->data;
  if (items[n].string)
    {
      free(items[n].string);
    }
  news = (char *) malloc(len + 1);
  STRNCPY(news, str, len + 1);
  news[len] = 0;
  items[n].string = news;
  items[n].color = color;
  redrawMenuItem(win, n);
}

void
        W_DefineMapcursor(W_Window window)
{
  Cursor  new;
  Pixmap  mapcursmaskbit;
  Pixmap  mapcursbit;
  struct window *win = W_Void2Window(window);
  char   *path;
  static
  XColor  f, b;
  int     w, h, xh, yh;

  xh = yh = 5;
  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  mapcursbit = XCreateBitmapFromData(W_Display, win->window, mapcursor_bits,
				     mapcursor_width, mapcursor_height);

  if ((path = getdefault("mapCursorDef")))
    {

      if (W_LoadBitmap(window, path, &mapcursmaskbit, &w, &h, &xh, &yh) != 1)
	{
	  mapcursmaskbit = XCreateBitmapFromData(W_Display, win->window,
			       mapmask_bits, mapmask_width, mapmask_height);
	  xh = yh = 5;
	}
      else
	{
	  mapcursbit = XCreatePixmap(W_Display, win->window, w, h, 1);
	  XFillRectangle(W_Display, mapcursbit,
			 colortable[W_White].contexts[0], 0, 0, w, h);
	}
    }

  else
    mapcursmaskbit = XCreateBitmapFromData(W_Display, win->window,
			       mapmask_bits, mapmask_width, mapmask_height);

  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreatePixmapCursor(W_Display, mapcursbit, mapcursmaskbit,
			    &b, &f, xh, yh);
  XRecolorCursor(W_Display, new, &b, &f);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineLocalcursor(W_Window window)
{
  Cursor  new;
  Pixmap  localcursmaskbit;
  Pixmap  localcursbit;
  struct window *win = W_Void2Window(window);
  char   *path;
  static
  XColor  f, b;
  int     w, h, xh, yh;

  xh = yh = 5;
  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  localcursbit = XCreateBitmapFromData(W_Display, win->window,
		   localcursor_bits, localcursor_width, localcursor_height);

  if ((path = getdefault("localCursorDef")))
    {
      if (W_LoadBitmap(window, path, &localcursmaskbit, &w, &h, &xh, &yh) != 1)
	{
	  localcursmaskbit = XCreateBitmapFromData(W_Display, win->window,
			 localmask_bits, localmask_width, localmask_height);
	  xh = yh = 5;
	}
      else
	{
	  localcursbit = XCreatePixmap(W_Display, win->window, w, h, 1);
	  XFillRectangle(W_Display, localcursbit,
			 colortable[W_White].contexts[0], 0, 0, w, h);
	}
    }
  else
    localcursmaskbit = XCreateBitmapFromData(W_Display, win->window,
			 localmask_bits, localmask_width, localmask_height);

  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreatePixmapCursor(W_Display, localcursbit, localcursmaskbit,
			    &b, &f, xh, yh);
  XRecolorCursor(W_Display, new, &b, &f);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineFedCursor(W_Window window)
{
  Cursor  new;
  Pixmap  fedcursmaskbit;
  Pixmap  fedcursbit;
  struct window *win = W_Void2Window(window);

  static
  XColor  f, b;

  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  fedcursbit = XCreateBitmapFromData(W_Display, win->window, fed_cruiser_bits,
				     fed_cruiser_width, fed_cruiser_height);
  fedcursmaskbit = XCreateBitmapFromData(W_Display, win->window,
			    fed_mask_bits, fed_mask_width, fed_mask_height);
  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreatePixmapCursor(W_Display, fedcursmaskbit, fedcursbit,
			    &b, &f, 10, 10);
  XRecolorCursor(W_Display, new, &b, &f);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineRomCursor(W_Window window)
{
  Cursor  new;
  Pixmap  romcursmaskbit;
  Pixmap  romcursbit;
  struct window *win = W_Void2Window(window);

  static
  XColor  f, b;

  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  romcursbit = XCreateBitmapFromData(W_Display, win->window, rom_cruiser_bits,
				     rom_cruiser_width, rom_cruiser_height);
  romcursmaskbit = XCreateBitmapFromData(W_Display, win->window,
		      rom_mask_bits, rom_cruiser_width, rom_cruiser_height);
  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreatePixmapCursor(W_Display, romcursmaskbit, romcursbit,
			    &b, &f, 10, 10);
  XRecolorCursor(W_Display, new, &b, &f);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineKliCursor(W_Window window)
{
  Cursor  new;
  Pixmap  klicursmaskbit;
  Pixmap  klicursbit;
  struct window *win = W_Void2Window(window);

  static
  XColor  f, b;

  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  klicursbit = XCreateBitmapFromData(W_Display, win->window, kli_cruiser_bits,
				     kli_cruiser_width, kli_cruiser_height);
  klicursmaskbit = XCreateBitmapFromData(W_Display, win->window,
		      fed_mask_bits, kli_cruiser_width, kli_cruiser_height);
  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreatePixmapCursor(W_Display, klicursmaskbit, klicursbit,
			    &b, &f, 10, 10);
  XRecolorCursor(W_Display, new, &b, &f);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineOriCursor(W_Window window)
{
  Cursor  new;
  Pixmap  oricursmaskbit;
  Pixmap  oricursbit;
  struct window *win = W_Void2Window(window);

  static
  XColor  f, b;

  f.pixel = colortable[W_White].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  oricursbit = XCreateBitmapFromData(W_Display, win->window, ori_cruiser_bits,
				     ori_cruiser_width, ori_cruiser_height);
  oricursmaskbit = XCreateBitmapFromData(W_Display, win->window,
		      fed_mask_bits, ori_cruiser_width, ori_cruiser_height);
  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreatePixmapCursor(W_Display, oricursmaskbit, oricursbit,
			    &b, &f, 10, 10);
  XRecolorCursor(W_Display, new, &b, &f);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineTrekCursor(W_Window window)
{
  Cursor  new;
  struct window *win = W_Void2Window(window);
  XColor  f, b;
  char   *path;
  int     w, h, xh, yh, mw, mh, mxh, myh;

  f.pixel = colortable[W_Yellow].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  if ((path = getdefault("infoCursorDef")))
    {
      Pixmap  pm, mpm;

      if (W_LoadBitmap(window, path, &pm, &w, &h, &xh, &yh) != 1)
	new = XCreateFontCursor(W_Display, XC_trek);
      else
	{
	  char    mask_path[512];

	  strcpy(mask_path, path);
	  strcat(mask_path, ".mask");

	  if (W_LoadBitmap(window, mask_path, &mpm, &mw, &mh, &mxh, &myh) != 1)
	    {
	      mw = w;
	      mh = h;
	      mpm = XCreatePixmap(W_Display, win->window, w, h, 1);
	      XFillRectangle(W_Display, mpm,
			     colortable[W_White].contexts[0], 0, 0, w, h);
	    }

	  if ((w != mw) || (h != mh))
	    {
	      printf("Cursor and mask are not the same size. %s\n", path);
	      new = XCreateFontCursor(W_Display, XC_trek);
	    }
	  else
	    new = XCreatePixmapCursor(W_Display, pm, mpm,
				      &b, &f, xh, yh);
	}
    }
  else
    new = XCreateFontCursor(W_Display, XC_trek);

  XRecolorCursor(W_Display, new, &f, &b);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineWarningCursor(W_Window window)
{
  Cursor  new;
  struct window *win = W_Void2Window(window);
  XColor  f, b;

  f.pixel = colortable[W_Red].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  new = XCreateFontCursor(W_Display, XC_pirate);
  XRecolorCursor(W_Display, new, &f, &b);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineArrowCursor(W_Window window)
{
  Cursor  new;
  struct window *win = W_Void2Window(window);
  XColor  f, b;
  char   *path;
  int     w, h, xh, yh, mw, mh, mxh, myh;

  f.pixel = colortable[W_Black].pixelValue;
  b.pixel = colortable[W_White].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  if ((path = getdefault("arrowCursorDef")))
    {
      Pixmap  pm, mpm;

      if (W_LoadBitmap(window, path, &pm, &w, &h, &xh, &yh) != 1)
	new = XCreateFontCursor(W_Display, XC_left_ptr);
      else
	{
	  char    mask_path[512];

	  strcpy(mask_path, path);
	  strcat(mask_path, ".mask");

	  if (W_LoadBitmap(window, mask_path, &mpm, &mw, &mh, &mxh, &myh) != 1)
	    {
	      mw = w;
	      mh = h;
	      mpm = XCreatePixmap(W_Display, win->window, w, h, 1);
	      XFillRectangle(W_Display, mpm,
			     colortable[W_White].contexts[0], 0, 0, w, h);
	    }

	  if ((w != mw) || (h != mh))
	    {
	      printf("Cursor and mask are not the same size. %s\n", path);
	      new = XCreateFontCursor(W_Display, XC_left_ptr);
	    }
	  else
	    new = XCreatePixmapCursor(W_Display, pm, mpm,
				      &b, &f, xh, yh);
	}
    }
  else
    new = XCreateFontCursor(W_Display, XC_left_ptr);

  XRecolorCursor(W_Display, new, &f, &b);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineTextCursor(W_Window window)
{
  Cursor  new;
  struct window *win = W_Void2Window(window);
  XColor  f, b;
  char   *path;
  int     w, h, xh, yh, mw, mh, mxh, myh;

  f.pixel = colortable[W_Yellow].pixelValue;
  b.pixel = colortable[W_Black].pixelValue;

  XQueryColor(W_Display, W_Colormap, &f);
  XQueryColor(W_Display, W_Colormap, &b);

  if (win->cursor)
    XFreeCursor(W_Display, win->cursor);

  if ((path = getdefault("textCursorDef")))
    {
      Pixmap  pm, mpm;

      if (W_LoadBitmap(window, path, &pm, &w, &h, &xh, &yh) != 1)
	new = XCreateFontCursor(W_Display, XC_xterm);
      else
	{
	  char    mask_path[512];

	  strcpy(mask_path, path);
	  strcat(mask_path, ".mask");

	  if (W_LoadBitmap(window, mask_path, &mpm, &mw, &mh, &mxh, &myh) != 1)
	    {
	      mw = w;
	      mh = h;
	      mpm = XCreatePixmap(W_Display, win->window, w, h, 1);
	      XFillRectangle(W_Display, mpm,
			     colortable[W_White].contexts[0], 0, 0, w, h);
	    }

	  if ((w != mw) || (h != mh))
	    {
	      printf("Cursor and mask are not the same size. %s\n", path);
	      new = XCreateFontCursor(W_Display, XC_xterm);
	    }
	  else
	    new = XCreatePixmapCursor(W_Display, pm, mpm,
				      &b, &f, xh, yh);
	}
    }
  else
    new = XCreateFontCursor(W_Display, XC_xterm);

  XRecolorCursor(W_Display, new, &f, &b);
  XDefineCursor(W_Display, win->window, new);
  win->cursor = new;
}

void
        W_DefineCursor(W_Window window, int width, int height, char *bits, char *mask, int xhot, int yhot)
{
  static char *oldbits = NULL;
  static Cursor curs;
  Pixmap  cursbits;
  Pixmap  cursmask;
  short  *curdata;
  short  *maskdata;
  struct window *win;
  XColor  whiteCol, blackCol;

#ifdef DEBUG
  printf("Defining cursor for %d\n", window);
#endif

  win = W_Void2Window(window);
  whiteCol.pixel = colortable[W_White].pixelValue;
  XQueryColor(W_Display, W_Colormap, &whiteCol);
  blackCol.pixel = colortable[W_Black].pixelValue;
  XQueryColor(W_Display, W_Colormap, &blackCol);
  if (!oldbits || oldbits != bits)
    {
      cursbits = XCreateBitmapFromData(W_Display, win->window,
				       bits, width, height);
      cursmask = XCreateBitmapFromData(W_Display, win->window,
				       mask, width, height);
      oldbits = bits;
      curs = XCreatePixmapCursor(W_Display, cursbits, cursmask,
				 &whiteCol, &blackCol, xhot, yhot);
      XFreePixmap(W_Display, cursbits);
      XFreePixmap(W_Display, cursmask);
    }
  XDefineCursor(W_Display, win->window, curs);
}

int
        W_LoadBitmap(W_Window window, char *path, Pixmap * pixmap, int *width, int *height, int *x_hot, int *y_hot)
{
  int     status;
  struct window *win;

  win = W_Void2Window(window);

  status = XReadBitmapFile(W_Display, win->window, path, width,
			   height, pixmap, x_hot, y_hot);

  if (status == BitmapSuccess)
    {
      if (*x_hot < 0)
	{
	  *x_hot = *width / 2;
	  *y_hot = *height / 2;
	}

      return (1);
    }
  else
    return (0);
}

void
        W_Beep(void)
{
  XBell(W_Display, 0);
}

int
        W_WindowWidth(W_Window window)
{
  return (W_Void2Window(window)->width);
}

int
        W_WindowHeight(W_Window window)
{
  return (W_Void2Window(window)->height);
}

int
        W_Socket(void)
{
  return (ConnectionNumber(W_Display));
}

void
        W_DestroyWindow(W_Window window)
{
  struct window *win;

#ifdef DEBUG
  printf("Destroying %d\n", window);
#endif

  win = W_Void2Window(window);
  deleteWindow(win);
  XDestroyWindow(W_Display, win->window);
  free((char *) win);
}

/* NOT USED */
void
        W_SetTransientForHint(W_Window w, W_Window pw)
{
  XSetTransientForHint(W_Display, W_Void2Window(w)->window,
		       W_Void2Window(pw)->window);
}


deleteWindow(struct window *window)
{
  struct windowlist **rm;
  struct windowlist *temp;

  rm = &hashtable[hash(window->window)];
  while (*rm != NULL && (*rm)->window != window)
    {
      rm = &((*rm)->next);
    }
  if (*rm == NULL)
    {
      printf("Attempt to delete non-existent window!\n");
      return;
    }
  temp = *rm;
  *rm = temp->next;
  free((char *) temp);
}

void
        W_SetIconWindow(W_Window main, W_Window icon)
{
  XWMHints hints;

  XSetIconName(W_Display, W_Void2Window(icon)->window, W_Void2Window(main)->name);

  hints.flags = IconWindowHint;
  hints.icon_window = W_Void2Window(icon)->window;
  XSetWMHints(W_Display, W_Void2Window(main)->window, &hints);
}

/*****************************************************************************/
/* Looks up any default geometry specified in the defaults file and        */
/* returns the values found there.  Geometry should be of the form         */
/* [=][<width>x<height>][{+-}<xoffset>{+-}<yoffset>]                  */
/* */
/* The result returned indicates which values were set.                    */
/* XValue, YValue, WidthValue, HeightValue                                */
/* */
/*****************************************************************************/

static int
        checkGeometry(char *name, int *x, int *y, int *width, int *height)
{
  char    buf[80], *geom_default;

  sprintf(buf, "%s.geometry", name);
  geom_default = getdefault(buf);
  if (!geom_default)
    return 0;					 /* nothing set */

  return XParseGeometry(geom_default, x, y, width, height);
}

checkParent(char *name, W_Window * parent)
{
  char   *adefault;
  char    buf[100];
  int     i;
  struct windowlist *windows;

  sprintf(buf, "%s.parent", name);
  adefault = getdefault(buf);
  if (adefault == NULL)
    return;
  /* parent must be name of other window or "root" */
  if (strcmpi(adefault, "root") == 0)
    {
      *parent = W_Window2Void(&myroot);
      return;
    }
  for (i = 0; i < HASHSIZE; i++)
    {
      windows = hashtable[i];
      while (windows != NULL)
	{
	  if (strcmpi(adefault, windows->window->name) == 0)
	    {
	      *parent = W_Window2Void(windows->window);
	      return;
	    }
	  windows = windows->next;
	}
    }
}

checkMapped(char *name)
{
  char   *adefault;
  char    buf[100];

  sprintf(buf, "%s.mapped", name);
  return (booleanDefault(buf, 0));
}

checkMappedPref(char *name, int preferred)
{
  char   *adefault;
  char    buf[100];

  sprintf(buf, "%s.mapped", name);
  return (booleanDefault(buf, preferred));
}

void
        W_WarpPointer(W_Window window, int x, int y)
{
  static int warped_from_x = 0, warped_from_y = 0;

  if (window == NULL)
    {
      if (W_in_message)
	{
	  XWarpPointer(W_Display, None, W_Root, 0, 0, 0, 0, warped_from_x, warped_from_y);
	  W_in_message = 0;
	}
    }
  else
    {
      findMouse(&warped_from_x, &warped_from_y);
      XWarpPointer(W_Display, None, W_Void2Window(window)->window, 0, 0, 0, 0, 0, 0);
      W_in_message = 1;
    }
}

findMouse(int *x, int *y)
{
  Window  theRoot, theChild;
  int     wX, wY, rootX, rootY, status;
  unsigned int wButtons;

  status = XQueryPointer(W_Display, W_Root, &theRoot, &theChild, &rootX, &rootY, &wX, &wY, &wButtons);
  if (status == True)
    {
      *x = wX;
      *y = wY;
    }
  else
    {
      *x = 0;
      *y = 0;
    }
}

int
        findMouseInWin(int *x, int *y, W_Window w)
{
  Window  theRoot, theChild;
  int     wX, wY, rootX, rootY, status;
  unsigned int wButtons;
  struct window *win = W_Void2Window(w);
  Window  thisWin = win->window;

  status = XQueryPointer(W_Display, thisWin, &theRoot, &theChild,
			 &rootX, &rootY, &wX, &wY, &wButtons);
  if (status == True)
    {
      /* if it's in the window we specified then the values returned should
       * be within the with and height of the window */
      if (wX <= win->width && wY <= win->height)
	{
	  *x = wX;
	  *y = wY;
	  return 1;
	}
    }
  *x = 0;
  *y = 0;
  return 0;
}

W_Flush(void)
{
  XFlush(W_Display);
}

#define MAKE_WINDOW_GETTER(name, part) \
  W_Callback name(W_Window w) \
  { \
    return (W_Void2Window(w)->part); \
  }

#define MAKE_WINDOW_SETTER(name, part) \
  W_Callback name(W_Window w, W_Callback c) \
  { \
    W_Void2Window(w)->part = c; \
  }

MAKE_WINDOW_GETTER(W_GetWindowKeyDownHandler, handle_keydown)
MAKE_WINDOW_SETTER(W_SetWindowKeyDownHandler, handle_keydown)

MAKE_WINDOW_GETTER(W_GetWindowKeyUpHandler, handle_keyup)
MAKE_WINDOW_SETTER(W_SetWindowKeyUpHandler, handle_keyup)

MAKE_WINDOW_GETTER(W_GetWindowButtonHandler, handle_button)
MAKE_WINDOW_SETTER(W_SetWindowButtonHandler, handle_button)

MAKE_WINDOW_GETTER(W_GetWindowExposeHandler, handle_expose)
MAKE_WINDOW_SETTER(W_SetWindowExposeHandler, handle_expose)

void
        W_ResizeWindow(W_Window window, int neww, int newh)	/* TSH 2/93 */



{
  Window  win = W_Void2Window(window)->window;
  XSizeHints *sz_hints;

  sz_hints = XAllocSizeHints();
  sz_hints->min_width = (unsigned int) neww;
  sz_hints->max_width = (unsigned int) neww;
  sz_hints->min_height = (unsigned int) newh;
  sz_hints->max_height = (unsigned int) newh;
  sz_hints->flags = PMinSize | PMaxSize;
  XSetWMNormalHints(W_Display, win, sz_hints);
  XResizeWindow(W_Display, win, (unsigned int) neww, (unsigned int) newh);
}

void
        W_ResizeTextWindow(W_Window window, int neww, int newh)	/* TSH 2/93 */



{
  W_ResizeWindow(window, neww * W_Textwidth + WIN_EDGE * 2,
		 newh * W_Textheight + WIN_EDGE * 2);
}

int
        W_Mono(void)
{
  return ((DisplayCells(W_Display, W_Screen) <= 2) || forceMono);
}

int
        W_EventsQueued(void)
{
  int     v;

  return (XEventsQueued(W_Display, QueuedAlready));
}

W_ReadEvents(void)
{
  XEvent  event;
  static
  struct timeval timeout =
  {0, 0};
  fd_set  readfds;

  FD_ZERO(&readfds);
  FD_SET(ConnectionNumber(W_Display), &readfds);
  if (select(max_fd, &readfds, 0, 0, &timeout) == 1)
    {
      XPeekEvent(W_Display, &event);
      return 1;
    }
  return 0;
}

#ifdef BEEPLITE
void
        W_OverlayBitmap(int x, int y, W_Icon bit, W_Color color)
{
  struct icon *icon = W_Void2Icon(bit);

#ifdef DEBUG
  printf("Overlaying bitmap to %d\n", icon->window);
#endif

  XCopyPlane(W_Display, icon->bitmap, icon->window,
	     colortable[color].contexts[0], 0, 0, icon->width, icon->height,
	     x, y, 1);
}

static GC _tts_gc;
static XFontStruct *_tts_fontinfo;
static int _tts_th, _tts_tw;

int
        W_TTSTextHeight(void)
{
  return _tts_th;
}

int
        W_TTSTextWidth(char *s, int l)
{
  return XTextWidth(_tts_fontinfo, s, l);
}


void
        init_tts(void)
{
  char   *fontname;
  XGCValues values;
  char   *color;
  XColor  xc;

  fontname = getdefault("tts_font");
  if (!fontname)
    fontname = TTS_FONT;

#ifdef SHOW_DEFAULTS
  show_defaults("TTS", "tts_font", fontname, "TTS font.");
#endif

#ifdef nodef
  if (forceMono || DisplayCells(W_Display, W_Screen) <= 2)
    {
      /* this is not going to work at all for b/w */
      tts_time = 0;
      F_beeplite_flags &= ~LITE_TTS;
      return;
    }
#endif

  _tts_fontinfo = XLoadQueryFont(W_Display, fontname);
  if (!_tts_fontinfo)
    {
      fprintf(stderr, "netrek: Can't find font \"%s\".\n", fontname);
      _tts_fontinfo = XLoadQueryFont(W_Display, "fixed");
    }
  if (!_tts_fontinfo)
    {
      fprintf(stderr, "netrek: Can't find any fonts.\n");
      exit(1);
    }
  _tts_th = _tts_fontinfo->max_bounds.descent +
      _tts_fontinfo->max_bounds.ascent;
  _tts_tw = _tts_fontinfo->max_bounds.width;

  values.font = _tts_fontinfo->fid;

  if (forceMono || DisplayCells(W_Display, W_Screen) <= 2)
    {
      values.foreground = colortable[W_White].pixelValue;
      values.function = GXor;
    }

  else
    {

      color = getdefault("tts_color");
      if (!color)
	color = "#777";

#ifdef SHOW_DEFAULTS
      show_defaults("TTS", "tts_color", color, "TTS msg color.");
#endif

      if (!XParseColor(W_Display, W_Colormap, color, &xc))
	{
	  fprintf(stderr, "netrek: Unknown color \"%s\"\n", color);
	  (void) XParseColor(W_Display, W_Colormap, "#777", &xc);
	}
      /* using the 8th color allocated in GetColors() */
      xc.pixel = colortable[W_Black].pixelValue | planes[0] | planes[2];
      XStoreColor(W_Display, W_Colormap, &xc);
      values.foreground = xc.pixel;
      values.function = GXor;
    }

  _tts_gc = XCreateGC(W_Display, W_Root, GCFont | GCForeground | GCFunction,
		      &values);

  XSetGraphicsExposures(W_Display, _tts_gc, False);
}

void
        W_EraseTTSText(W_Window window, int max_width, int y, int width)
{
  struct window *win = W_Void2Window(window);
  register int x = (max_width - width) / 2;

  if (x < 0)
    x = 4;
  y -= W_TTSTextHeight();

  W_ClearArea(w, x, y, width, W_TTSTextHeight());
}

void
        W_WriteTTSText(W_Window window, int max_width, int y, int width, char *str, int len)


/* max_width of window */
/* y coordinate */
/* actual width */
/* string */
/* length of string */
{
  struct window *win = W_Void2Window(window);
  register int x = (max_width - width) / 2;

  if (x < 0)
    x = 4;

  y -= _tts_fontinfo->max_bounds.descent;

  /* y -= W_TTSTextHeight(); y += _tts_fontinfo->max_bounds.ascent; */

  XDrawString(W_Display, win->window, _tts_gc, x, y, str, len);
}

#endif
