/*
   Background Sound Player

   Copyright 1993 by Kurt Siegl <siegl@risc.uni-linz.ac.at>
   Permission to use, modify, copy and distribute this software without
   fee is hereby granted as long as this notice is left here.

   NOTE:  This is the modified version to run on SGI's that don't have the
   /dev/audio device.
   It will only work with 8kHz aiff files.  The SGI's have a utility called
   sfconvert that you can use to convert to aiff.
*/

#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <termio.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <audio.h>
#include "sndconfig.h"

#define TRUE 1
#define FALSE 0

static ALport audio_port;
static int play_sound;
/* static int audio; */
static int abuf_size, play_fd, play_speed;
static char *audiobuf;
static struct timeval will_play, sound_tag;

static void StopSoundI ();
static void StartSoundI ();
static void PlaySoundI ();

static struct shm_sound *shared_data;
static int returnpid;

static void play ();

static void
terminate (x)
     int x;
{
  switch (x)
    {
    case (0):
      kill (returnpid, SIGUSR1);
      break;
    case (1):
      kill (returnpid, SIGUSR2);
    default:
      exit (x);
    }
}

int
main (argc, argv)
     int argc;
     char **argv;
{

  int dsp_speed = DEFAULT_DSP_SPEED, dsp_stereo = STEREO;
  int samplesize = DEFAULT_SAMPLE_SIZE;
  struct itimerval udt;
  int shmid;
  ALconfig audio_port_config;
  long pvbuf[2];

  if (argc != 3)		/* Usage: bgsndplay shmID parrentPID */
    {
      fprintf (stderr, "This program requires 2 arguments.\n");
      fprintf (stderr, "But why are you running it anyway :->\n");
      terminate (-1);
    }

  shmid = atoi (argv[1]);
  returnpid = atoi (argv[2]);

  abuf_size = 4096;

  if ((audiobuf = malloc (abuf_size)) == NULL)
    {
      fprintf (stderr, "Unable to allocate input/output buffer\n");
      terminate (1);
    }

/*
	audio = open (AUDIO, O_WRONLY, 0);
  if (audio == -1)
    {
      perror (AUDIO);
      terminate (1);
    }
*/

	pvbuf[0] = AL_OUTPUT_RATE;
  pvbuf[1] = 8000;
  ALsetparams(AL_DEFAULT_DEVICE, pvbuf, 2);

  audio_port_config = ALnewconfig();
  ALsetwidth(audio_port_config, AL_SAMPLE_8);
  ALsetchannels(audio_port_config, 1);
  audio_port = ALopenport("bgsndplay", "w", audio_port_config);



  /*
  if ((shmid=shmget(shmKey,sizeof(struct shm_sound),0))==-1)
	{
	  perror ("SHMGET");
	  terminate(1);
	}
*/

  if ((shared_data = (struct shm_sound *) shmat (shmid, 0, 0)) == (void *) -1)
    {
      perror ("SHMAT");
      terminate (1);
    }

  (void) signal (SIGALRM, SIG_IGN);
  udt.it_interval.tv_sec = CHECK_SEC;
  udt.it_interval.tv_usec = CHECK_USEC;
  udt.it_value.tv_sec = CHECK_SEC;
  udt.it_value.tv_usec = CHECK_USEC;
  if (setitimer (ITIMER_REAL, &udt, 0) < 0)
    {
      perror ("setitimer");
      terminate (1);
    }

  play_sound = FALSE;
  /* Get playtime for 1 byte in usec */
  play_speed = (1000000 / ((samplesize / 8) * (dsp_stereo + 1) * dsp_speed));
  gettimeofday (&will_play, NULL);

  signal (SIGUSR1, StopSoundI);
  signal (SIGUSR2, StartSoundI);

  terminate (0);

  while (1)
    {
      pause ();
      signal (SIGUSR1, SIG_IGN);
      signal (SIGUSR2, SIG_IGN);
      signal (SIGALRM, SIG_IGN);
      play ();
      while (memcmp (&(shared_data->tag), &sound_tag, sizeof (sound_tag)))
	{			/* Lost some Interrupts */
	  StartSoundI ();
	  play ();
	}
      signal (SIGALRM, PlaySoundI);
      signal (SIGUSR2, StartSoundI);
      signal (SIGUSR1, StopSoundI);
    }

  printf ("Shouldn't happen\n");
  exit (-1);
}

static void
StopSoundI ()
{
  signal (SIGALRM, SIG_IGN);
  if (play_sound)
    {
      close (play_fd);
      play_sound = FALSE;
    }
  signal (SIGUSR1, StopSoundI);
}


static void
StartSoundI ()
{
  char *name;
  long offset;
  
  StopSoundI ();
  name = shared_data->name;
  sound_tag = shared_data->tag;
  if ((play_fd = open (name, O_RDONLY, 0)) == -1)
    {
      perror (name);
      return;
    }
  if(read(play_fd, audiobuf, 4) < 4) {
    fprintf(stderr, "\n%s: error reading first header\n", name);
    return;
  }
  if(read(play_fd, &offset, 4) < 4) {
    fprintf(stderr, "\n%s: error reading first header size\n", name);
    return;
  }
  if(strncmp(audiobuf, "FORM", 4)) {
    fprintf(stderr, "\n%s: couldn't find form chunk\n", name);
    return;
  }
  else fprintf(stderr, "\n%s: found FORM chunk\n", name);
  if(read(play_fd, audiobuf, 4) < 4) {
    fprintf(stderr, "\n%s: couldn't read form chunk\n", name);
    return;
  }
  audiobuf[4] = 0;
  fprintf(stderr, "\n%s: FORM chunk id: \"%s\"\n", name, audiobuf);
  
  for(;;) {
    if(read(play_fd, audiobuf, 4) < 4) { /* read header id */
      fprintf(stderr, "\n%s: couldn't find ssnd chunk\n", name);
      return;
    }
    if(read(play_fd, &offset, 4) < 4) { /* read header size */
      fprintf(stderr, "\n%s: error reading header size\n", name);
      return;
    }
    audiobuf[4] = 0;
    fprintf(stderr, "\n%s: checking header \"%s\"\n", name, audiobuf);
    if(!strncmp(audiobuf, "SSND", 4)) {
      if(read(play_fd, audiobuf, 4) < 4) { 
	fprintf(stderr, "\n%s: couldn't read ssnd offset\n", name);
	return;
      }
      if(read(play_fd, audiobuf, 4) < 4) {
	fprintf(stderr, "\n%s: couldn't read ssnd chunk blocksize\n", name);
	return;
      }
      offset = lseek(play_fd, 0L, SEEK_CUR);
      fprintf(stderr, "\n%s: starting offset: %ld\n", name, offset);
      break;
    }
    lseek(play_fd, offset, SEEK_CUR);
  }
  
  play_sound = TRUE;
  /*  play(); */
  signal (SIGUSR2, StartSoundI);
  return;
}

static void
PlaySoundI ()
{
  signal (SIGALRM, PlaySoundI);	/* Interupt the pause() in main */
}

inline long
timediff (newt, oldt)
     struct timeval newt;
     struct timeval oldt;
{
  return ((newt.tv_sec - oldt.tv_sec) * 1000000 +
	  newt.tv_usec - oldt.tv_usec);
}

inline void
addtime (curr, diff)
     struct timeval *curr;
     long int diff;
{
  ldiv_t t;

  t = ldiv (curr->tv_usec + diff, 1000000);
  curr->tv_sec = curr->tv_sec + t.quot;
  curr->tv_usec = t.rem;
}

static void
play ()
{
  int l;
  struct timeval t;
  long offset;

#ifdef DEBUG
  printf("Start\n");
#endif

  if (!play_sound)
    return;

  gettimeofday (&t, NULL);
  if (timediff (will_play, t) > PLAY_TIME)
    return;


  l = read(play_fd, audiobuf, abuf_size);
  offset = lseek(play_fd, 0L, SEEK_CUR);
  if(l > 0) {
    ALwritesamps(audio_port, audiobuf, l);
    fprintf(stderr, "\nWrote sound at offset %ld\n", offset);
    addtime(&will_play, play_speed * l);
  }
  else {
    if (l == -1) {
      perror ("Input file");
      return;
    }
    
    close (play_fd);
    play_sound = FALSE;	/* Stop */
    return;
  }
  
#ifdef DEBUG
  printf("End\n");
#endif
}
