
/*
   Background Sound Player

   Copyright 1993 by Kurt Siegl <siegl@risc.uni-linz.ac.at>
   Permission to use, modify, copy and distribute this software without
   fee is hereby granted as long as this notice is left here.
*/

#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include <fcntl.h>
#include <string.h>
#ifdef linux
#include <linux/soundcard.h>
#endif
#include <signal.h>
#include <termio.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "sndconfig.h"

#define TRUE 1
#define FALSE 0

static int play_sound;
static int audio, abuf_size, play_fd, play_speed;
static char *audiobuf;
static struct timeval will_play, sound_tag;

static void StopSoundI ();
static void StartSoundI ();
static void PlaySoundI ();

static struct shm_sound *shared_data;
static int returnpid;

static void play ();

static void 
terminate (x)
     int x;
{
  switch (x)
    {
    case (0):
      kill (returnpid, SIGUSR1);
      break;
    case (1):
      kill (returnpid, SIGUSR2);
    default:
      exit (x);
    }
}

int 
main (argc, argv)
     int argc;
     char **argv;
{

  int dsp_speed = DEFAULT_DSP_SPEED, dsp_stereo = STEREO;
  int samplesize = DEFAULT_SAMPLE_SIZE;
  struct itimerval udt;
  int shmid;

  if (argc != 3)		/* Usage: bgsndplay shmID parrentPID */
    {
      fprintf (stderr, "This program requires 2 arguments.\n");
      fprintf (stderr, "But why are you running it anyway :->\n");
      terminate (-1);
    }

  shmid = atoi (argv[1]);
  returnpid = atoi (argv[2]);

#ifdef linux
  audio = open ("/dev/dsp", O_WRONLY, 0);
  if (audio == -1)
    {
      perror ("/dev/dsp");
      terminate (1);
    }

  ioctl (audio, SNDCTL_DSP_GETBLKSIZE, &abuf_size);
  if (abuf_size < 4096 || abuf_size > 65536)
    {
      if (abuf_size == -1)
	perror (AUDIO);
      else
	fprintf (stderr, "Invalid audio buffers size %d\n", abuf_size);
      terminate (1);
    }
  close (audio);
#else
  abuf_size = 4096;
#endif

  if ((audiobuf = malloc (abuf_size)) == NULL)
    {
      fprintf (stderr, "Unable to allocate input/output buffer\n");
      terminate (1);
    }

  audio = open (AUDIO, O_WRONLY, 0);
  if (audio == -1)
    {
      perror (AUDIO);
      terminate (1);
    }

#ifdef linux
#ifdef nodef
  if (ioctl (audio, SNDCTL_DSP_SAMPLESIZE, &samplesize) == -1)
    {
      fprintf (stderr, "Unable to set the sample size\n");
      terminate (1);
    }

  dsp_stereo = ioctl (audio, SNDCTL_DSP_STEREO, &dsp_stereo);

  if ((dsp_speed = ioctl (audio, SNDCTL_DSP_SPEED, &dsp_speed)) == -1)
    {
      fprintf (stderr, "Unable to set audio speed\n");
      perror (AUDIO);
      terminate (1);
    }
#endif
#endif

  /*
  if ((shmid=shmget(shmKey,sizeof(struct shm_sound),0))==-1)
	{
	  perror ("SHMGET");
	  terminate(1);
	}
*/

  if ((shared_data = (struct shm_sound *) shmat (shmid, 0, 0)) == (void *) -1)
    {
      perror ("SHMAT");
      terminate (1);
    }

  (void) signal (SIGALRM, SIG_IGN);
  udt.it_interval.tv_sec = CHECK_SEC;
  udt.it_interval.tv_usec = CHECK_USEC;
  udt.it_value.tv_sec = CHECK_SEC;
  udt.it_value.tv_usec = CHECK_USEC;
  if (setitimer (ITIMER_REAL, &udt, 0) < 0)
    {
      perror ("setitimer");
      terminate (1);
    }

  play_sound = FALSE;
  /* Get playtime for 1 byte in usec */
  play_speed = (1000000 / ((samplesize / 8) * (dsp_stereo + 1) * dsp_speed));
  gettimeofday (&will_play, NULL);

  signal (SIGUSR1, StopSoundI);
  signal (SIGUSR2, StartSoundI);

  terminate (0);

  while (1)
    {
      pause ();
      signal (SIGUSR1, SIG_IGN);
      signal (SIGUSR2, SIG_IGN);
      signal (SIGALRM, SIG_IGN);
      play ();
      while (memcmp (&(shared_data->tag), &sound_tag, sizeof (sound_tag)))
	{			/* Lost some Interrupts */
	  StartSoundI ();
	  play ();
	}
      signal (SIGALRM, PlaySoundI);
      signal (SIGUSR2, StartSoundI);
      signal (SIGUSR1, StopSoundI);
    }

  printf ("Shouldn't happen\n");
  exit (-1);
}

static void 
StopSoundI ()
{
  signal (SIGALRM, SIG_IGN);
  if (play_sound)
    {
      close (play_fd);
      play_sound = FALSE;
    }
  signal (SIGUSR1, StopSoundI);
}

static void 
StartSoundI ()
{
  char *name;

  StopSoundI ();
  name = shared_data->name;
  sound_tag = shared_data->tag;
  if ((play_fd = open (name, O_RDONLY, 0)) == -1)
    {
      perror (name);
      return;
    }
  play_sound = TRUE;
  /*  play(); */
  signal (SIGUSR2, StartSoundI);
  return;
}

static void 
PlaySoundI ()
{
  signal (SIGALRM, PlaySoundI);	/* Interupt the pause() in main */
}

inline long 
timediff (newt, oldt)
     struct timeval newt;
     struct timeval oldt;
{
  return ((newt.tv_sec - oldt.tv_sec) * 1000000 +
	  newt.tv_usec - oldt.tv_usec);
}

inline void 
addtime (curr, diff)
     struct timeval *curr;
     long int diff;
{
  ldiv_t t;

  t = ldiv (curr->tv_usec + diff, 1000000);
  curr->tv_sec = curr->tv_sec + t.quot;
  curr->tv_usec = t.rem;
}

static void 
play ()
{
  int l;
  struct timeval t;

#ifdef DEBUG
  printf("Start\n");
#endif

  if (!play_sound)
    return;

  gettimeofday (&t, NULL);
  if (timediff (will_play, t) > PLAY_TIME)
    return;

  if ((l = read (play_fd, audiobuf, abuf_size))
      > 0)
    {
      if (write (audio, audiobuf, l) != l)
	{
	  perror (AUDIO);
	  return;
	}
      write (audio, (char *) 0, 0);
      addtime (&will_play, play_speed * l);
#ifdef linux
      if (l < abuf_size)
	{
	  ioctl (audio, SNDCTL_DSP_POST);
	}
#endif
    }
  else
    {
      if (l == -1)
	{
	  perror ("Input file");
	  return;
	}
      close (play_fd);
      play_sound = FALSE;	/* Stop */
      return;
    }
#ifdef DEBUG
  printf("End\n");
#endif
}
