/* $Id: command2.c,v 30000.24 1993/05/25 00:52:02 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/*********************************************************
 * Fugue command handlers, part 2                        *
 *                                                       *
 * Contents:                                             *
 * 1. Flags and numerical runtime registers              *
 * 2. Macro subsystem, including gags, triggers, hilites *
 *********************************************************/

#include <ctype.h>
#include "port.h"
#include "dstring.h"
#include "tf.h"
#include "util.h"
#include "macro.h"
#include "keyboard.h"
#include "output.h"
#include "command1.h"
#include "history.h"
#include "world.h"
#include "socket.h"
#include "signal.h"

#define ON (!cstrcmp(args, "on"))
#define OFF (!cstrcmp(args, "off"))

static int  FDECL(do_prefix,(char *args, int *flag, char *fname, char *name, char *desc));
static int  FDECL(do_watch,(char *args, char *name, int *wlines, int *wmatch, int *flag));
static void FDECL(split_args,(char *args));

void NDECL(init_prefixes);
int  FDECL(do_file_load,(char *args));
void FDECL(read_file_commands,(TFILE *file, int oldformat));

HANDLER (handle_bamf_command);
HANDLER (handle_beep_command);
HANDLER (handle_bind_command);
HANDLER (handle_cat_command);
HANDLER (handle_def_command);
HANDLER (handle_dokey_command);
HANDLER (handle_edit_command);
HANDLER (handle_gag_command);
HANDLER (handle_hilite_command);
HANDLER (handle_hook_command);
HANDLER (handle_kecho_command);
HANDLER (handle_list_command);
HANDLER (handle_load_command);
HANDLER (handle_log_command);
HANDLER (handle_mecho_command);
HANDLER (handle_nogag_command);
HANDLER (handle_nohilite_command);
HANDLER (handle_purge_command);
HANDLER (handle_qecho_command);
HANDLER (handle_save_command);
HANDLER (handle_sub_command);
HANDLER (handle_suspend_command);
HANDLER (handle_trig_command);
HANDLER (handle_trigc_command);
HANDLER (handle_trigp_command);
HANDLER (handle_trigpc_command);
HANDLER (handle_unbind_command);
HANDLER (handle_undef_command);
HANDLER (handle_undefn_command);
HANDLER (handle_undeft_command);
HANDLER (handle_unhook_command);
HANDLER (handle_untrig_command);
HANDLER (handle_watchdog_command);
HANDLER (handle_watchname_command);
HANDLER (handle_wrap_command);

int log_on = 0;
int sub = 0;
int concat = 0;

Stringp pattern, body;
int wnmatch = 4, wnlines = 5, wdmatch = 2, wdlines = 5;

void init_prefixes()
{
    Stringinit(pattern);
    Stringinit(body);
}

#ifdef DMALLOC
void free_prefixes()
{
    Stringfree(pattern);
    Stringfree(body);
}
#endif

/********************
 * Generic handlers *
 ********************/   

static int do_prefix(args, flag, fname, name, desc)
    int *flag;
    char *args, *fname, *name, *desc;
{
    if (!*args) {
        oprintf("%% Echoing of %s is %s", desc, *flag ? "on" : "off");
    } else if (OFF) {
        setvar(fname, "0", FALSE);
    } else if (ON) {
        setvar(fname, "1", FALSE);
    } else {
        setvar(fname, "1", FALSE);
        setvar(name, args, FALSE);
    }
    return 1;
}

void read_file_commands(file, oldformat)
    TFILE *file;
    int oldformat;    /* Accept tinytalk-style world defs at top of file? */
{
    Stringp line, cmd;
    char *p;

    Stringinit(line);
    Stringinit(cmd);
    while (tfgetS(line, file) != NULL) {
        if (line->len) {
            if (line->s[0] == ';') continue;
            if (line->s[0] == '/') oldformat = FALSE;
            for (p = line->s; isspace(*p); p++);
            Stringcat(cmd, p);
            if (line->s[line->len - 1] == '\\') {
                if (line->len < 2 || line->s[line->len - 2] != '%') {
                    Stringterm(cmd, cmd->len - 1);
                    continue;
                }
            }
        }
        if (!cmd->len) continue;
        if (oldformat) addworld(cmd->s); else handle_command(cmd->s);
        Stringterm(cmd, 0);
    }
    if (cmd->len)
        if (oldformat) addworld(cmd->s); else handle_command(cmd->s);
    Stringfree(line);
    Stringfree(cmd);
}

int do_file_load(args)
    char *args;
{
    register TFILE *cmdfile;

    if ((cmdfile = tfopen(tfname(args, NULL), "r")) == NULL) {
        do_hook(H_LOADFAIL, "%% %s: %s", "%s %s", args, STRERROR(errno));
        return FALSE;
    }
    do_hook(H_LOAD, "%% Loading commands from %s.", "%s", cmdfile->name);
    read_file_commands(cmdfile, FALSE);
    tfclose(cmdfile);
    return TRUE;
}

/*********************************************************************
 *                           F L A G S                               *
 *********************************************************************/

int handle_bamf_command(args)
    char *args;
{
    if (!*args) {
        if (!bamf) oputs("% Bamfing is disabled.");
        else if (bamf == 2) oputs("% Old-style bamfing is enabled.");
        else oputs("% Unter bamfing is enabled.");
    } else {
        setvar("bamf", args, FALSE);
        if (cstrncmp(args, "old", 3) == 0) bamf = 2;
    }
    return 1;
}

int handle_sub_command(args)
    char *args;
{
    if (!*args) {
        if (!sub) oputs("% No substitution on normal input.");
        else if (sub == 1) oputs("% Newline substitution on.");
        else if (sub == 2) oputs("% Full substitution on normal input.");
    } else {
        if (OFF) sub = 0;
        else if (ON) sub = 1;
        else if (cstrcmp(args, "full") == 0) sub = 2;
        else tfputs("% Valid states are ON, OFF, and FULL.", tferr);
    }
    return 1;
}


/**************************
 * Toggles with arguments *
 **************************/

int handle_wrap_command(args)
    char *args;
{
    int width;
    char *next = args;

    if (*args) {
        if (OFF) setvar("wrap", "0", FALSE);
        else if (ON) setvar("wrap", "1", FALSE);
        else {
            width = numarg(&next);
            if (!next || *next) {
                tfputs("% Invalid argument.", tferr);
                return 0;
            }
            setvar("wrapsize", args, FALSE);
            setvar("wrap", "1", FALSE);
        }
    }
    oputs(wrapflag ? "% Wordwrap is enabled." : "% Wordwrap is disabled.");
    return 1;
}

/* Actually a miscellaneous routine, but it looks a lot like a flag */

int handle_beep_command(args)
    char *args;
{
    int beeps = 0;

    if (!*args) beeps = 3;
    else if (ON) setivar("beep", 1, FALSE);
    else if (OFF) setivar("beep", 0, FALSE);
    else if (isdigit(*args)) beeps = atoi(args);
    else return 0;

    if (beeping) while (beeps--) putch('\007');
    return 1;
}

int handle_log_command(args)    /* Enable logging. */
    char *args;
{
    return do_log(args);
}

int handle_kecho_command(args)
    char *args;
{
    return do_prefix(args, &kecho, "kecho", "kprefix", "input");
}

int handle_mecho_command(args)
    char *args;
{
    return do_prefix(args, &mecho, "mecho", "mprefix", "macros");
}

int handle_qecho_command(args)
    char *args;
{
    return do_prefix(args, &qecho, "qecho", "qprefix", "quote text");
}

int handle_cat_command(args)
    char *args;
{
    concat = (*args == '%') ? 2 : 1;
    return 1;
}

static int do_watch(args, name, wlines, wmatch, flag)
    char *args, *name;
    int *wlines, *wmatch, *flag;
{
    int lines = 0, match = 0;
    char *ptr;

    if (!*args) {
        oprintf("%% %s %sabled.", name, *flag ? "en" : "dis");
        return 1;
    } else if (OFF) {
        setvar(name, "0", FALSE);
        oprintf("%% %s disabled.", name);
        return 1;
    }
    ptr = args;
    if (ON) for (ptr += 2; isspace(*ptr); ptr++);
    if ((match = numarg(&ptr))) *wmatch = match;
    if (ptr && (lines = numarg(&ptr))) *wlines = lines;
    setvar(name, "1", FALSE);
    oprintf("%% %s enabled, searching for %d out of %d lines",
      name, *wmatch, *wlines);
    return 1;
}

int handle_watchdog_command(args)
    char *args;
{
    return do_watch(args, "watchdog", &wdlines, &wdmatch, &watchdog);
}

int handle_watchname_command(args)
    char *args;
{
    return do_watch(args, "watchname", &wnlines, &wnmatch, &watchname);
}


/*************************************************************************
 *                     M A C R O   S U B S Y S T E M                     *
 *************************************************************************/


/**********
 * Macros *
 **********/

int handle_def_command(args)                /* Define a macro. */
    char *args;
{
    Macro *spec;
    if (!*args) return 0;
    return (spec = macro_spec(args)) && do_add(spec);
}

int handle_edit_command(args)
    char *args;
{
    Macro *spec;
    return (spec = macro_spec(args)) && do_edit(spec);
}

int handle_undef_command(args)              /* Undefine a macro. */
    char *args;
{
    return remove_macro(args, 0, 0);
}

int handle_undeft_command(args)
    char *args;
{
    return remove_macro(args, F_ALL, 0);
}

int handle_undefn_command(args)
    char *args;
{
    return remove_by_number(args);
}

int handle_purge_command(args)
    char *args;
{
    return purge_macro(macro_spec(args));
}

int handle_save_command(args)
    char *args;
{
#ifdef RESTRICT_FILE
    tfputs("% /save: restricted", tferr);
    return 0;
#else
    if (*args) return save_macros(args);
    tfputs("% No filename.", tferr);
    return 0;
#endif
}

int handle_list_command(args)
    char *args;
{
    return list_macros(args);
}

int handle_load_command(args)
    char *args;
{                   
#ifdef RESTRICT_FILE
    extern int configured;
    if (configured) {
        tfputs("% /load: restricted", tferr);
        return 0;
    }
#endif
    if (*args) return do_file_load(args);
    tfputs("% No filename", tferr);
    return FALSE;
}

/*
 * Generic utility to split arguments into pattern and body.
 * Note: I can get away with this only because none of the functions
 * that use it are reentrant.  Be careful.
 */

static void split_args(args)
    char *args;
{
    extern Stringp pattern, body;
    char *place;

    place = strchr(args, '=');
    if (place == NULL) {
        Stringcpy(pattern, args);
        Stringterm(body, 0);
    } else {
        Stringncpy(pattern, args, place - args);
        while (isspace(*++place));
        Stringcpy(body, place);
    }
    stripString(pattern);
    stripString(body);
}

/***********
 * Hilites *
 ***********/

int handle_hilite_command(args)
    char *args;
{
    if (!*args) {
        setvar("hilite", "1", FALSE);
        oputs("% Hilites are enabled.");
    } else {
        split_args(args);
        add_macro(new_macro("", pattern->s, "", 0, "", body->s, NULL,
            hpri, 100, F_HILITE, 0, 0));
    }
    return 1;
}

int handle_nohilite_command(args)
    char *args;
{
    if (!*args) {
        setvar("hilite", "0", FALSE);
        oputs("% Hilites are disabled.");
        return 1;
    } else {
        return remove_macro(args, F_HILITE, 0);
    }
}


/********
 * Gags *
 ********/

int handle_gag_command(args)
    char *args;
{
    if (!*args) {
        setvar("gag", "1", FALSE);
        oputs("% Gags are enabled.");
    } else {
        split_args(args);
        add_macro(new_macro("", pattern->s, "", 0, "", body->s, NULL,
            gpri, 100, F_GAG, 0, 0));
    }
    return 1;
}

int handle_nogag_command(args)
    char *args;
{
    if (!*args) {
        setvar("gag", "0", FALSE);
        oputs("% Gags are disabled.");
        return 1;
    } else {
        return remove_macro(args, F_GAG, 0);
    }
}


/************
 * Triggers *
 ************/

int handle_trig_command(args)
    char *args;
{
    split_args(args);
    add_macro(new_macro("", pattern->s, "", 0, "", body->s, NULL, 0, 100,
        F_NORM, 0, 0));
    return 1;
}

int handle_trigp_command(args)
    char *args;
{
    int pri;

    pri = numarg(&args);
    if (!args || !*args) {
        tfputs("% Bad /trigp format", tferr);
        return 0;
    }
    split_args(args);
    add_macro(new_macro("", pattern->s, "", 0, "", body->s, NULL, pri,
        100, F_NORM, 0, 0));
    return 1;
}

int handle_trigc_command(args)
    char *args;
{
    int prob;

    prob = numarg(&args);
    if (!args || !*args) {
        tfputs("% Bad /trigc format", tferr);
        return 0;
    }
    split_args(args);
    add_macro(new_macro("", pattern->s, "", 0, "", body->s, NULL, 0, prob,
        F_NORM, 0, 0));
    return 1;
}

int handle_trigpc_command(args)
    char *args;
{
    int pri, prob;

    pri = numarg(&args);
    if (!args || !*args) {
        tfputs("% Bad /trigpc format", tferr);
        return 0;
    }
    prob = numarg(&args);
    if (!args || !*args) {
        tfputs("% Bad /trigpc format", tferr);
        return 0;
    }
    split_args(args);
    add_macro(new_macro("", pattern->s, "", 0, "", body->s, NULL, pri,
        prob, F_NORM, 0, 0));
    return 1;
}

int handle_untrig_command(args)
    char *args;
{
    return remove_macro(args, F_NORM, 0);
}


/*********
 * Hooks *
 *********/

int handle_hook_command(args)
    char *args;
{
    if (!*args) oprintf("%% Hooks %sabled", hookflag ? "en" : "dis");
    else if (OFF) setvar("hook", "0", FALSE);
    else if (ON) setvar("hook", "1", FALSE);
    else {
        split_args(args);
        return add_hook(pattern->s, body->s);
    }
    return 1;
}


int handle_unhook_command(args)
    char *args;
{
    return remove_macro(args, 0, 1);
}


/********
 * Keys *
 ********/

int handle_unbind_command(args)
    char *args;
{
    Macro *macro;

    if (!*args) return 0;
    if ((macro = find_key(print_to_ascii(args)))) kill_macro(macro);
    else tfprintf(tferr, "%% No binding for %s", args);
    return macro ? 1 : 0;
}

int handle_bind_command(args)
    char *args;
{
    Macro *spec;

    if (!*args) return 0;
    split_args(args);
    spec = new_macro("", "", print_to_ascii(pattern->s), 0, "", body->s,
        NULL, 0, 0, 0, 0, 0);
    if (!install_bind(spec)) return 0;
    add_macro(spec);
    return 1;
}

int handle_dokey_command(args)
    char *args;
{
    NFunc *func;

    if ((func = find_efunc(args))) (*func)();
    else tfprintf(tferr, "%% No editing function %s", args); 
    return func ? 1 : 0;
}

/*****/

int handle_suspend_command(args)
    char *args;
{
    return suspend();
}
