
// This module only runs under Linux.
#if defined(HAVE_VGA) && defined(linux)

/* This module handles all the VGA functions, for text mode switching,
   font loading, etc.
*/
#include <sys/types.h>
#include <linux/vt.h>
#include <sys/ioctl.h>
#include <termio.h>
#include <stdio.h>

#include "graphics.h"

#include "8x8font.h"

#ifdef LOAD_REGISTERS
#include "50L_regs.h"
#endif

VGA_Graphics:: VGA_Graphics()
{
	keyboard = new TTY_Keyboard;

	/* Find out how big the console is */
	detect_vtsize(&saved_cols, &saved_rows);
	if ( saved_rows == 50 )
		lastmode=mode=TEXT50;
	else
		lastmode=mode=TEXT25;
	new_rows=saved_rows;
	new_cols=saved_cols;

	/* Okay, we're in a console, initialize 50-line mode */
	vga_disabledriverreport();
	vga_setchipset(VGA);		/* avoid SVGA detection */
	vga_init();
	vga_setmode(TEXT);
	vga_gettextfont(saved_font);
	vga_gettextmoderegs(saved_regs);
	memcpy(new_regs, saved_regs, MAX_REGS);
	if ( saved_rows != 50 ) {
		new_rows=50; new_cols=80;
#ifdef LOAD_REGISTERS
		memcpy(new_regs, reg_data, sizeof(reg_data));
#else
		to_50lines(new_regs);
#endif
	}
	// Initialize IBM Graphics
	printf("\033(U"); fflush(stdout);
}

virtual
VGA_Graphics:: ~VGA_Graphics()
{
	if ( mode != TEXT25 )
		set_grmode(TEXT25);

	// End IBM Graphics
	printf("\033(B"); fflush(stdout);
}

virtual void 
VGA_Graphics:: flush(void)
{
	if ( mode == GRAPHICS )
		gl_copyscreen(Screen);
	fflush(stdout);
}

virtual int
VGA_Graphics:: has_graphics(void)
{
	return(1);
}

virtual int
VGA_Graphics:: ibm_charset(void)
{
	return(1);
}

virtual void 
VGA_Graphics:: set_grmode(int newmode)
{
  checkmode:
	switch (newmode) {
		case TEXT25:	if ( mode == TEXT25 )
					break;
				if ( (mode == TEXT50) && (saved_rows == 50) )
					break;
				vga_settextmoderegs(saved_regs);
				vga_puttextfont(saved_font);
				vga_setmode(TEXT);
				resize_vt(saved_cols, saved_rows);
				keyboard->raw(keyboard->getstate());
				break;
		case TEXT50:	if ( mode == TEXT50 )
					break;
				vga_settextmoderegs(new_regs);
				vga_puttextfont(font_data);
				vga_setmode(TEXT);
				resize_vt(new_cols, new_rows);
				keyboard->raw(keyboard->getstate());
				break;
		case GRAPHICS:	if ( mode == GRAPHICS ) {
					clr_grscreen();
					break;
				}
				/* Set the video into graphics mode */
				vga_setmode(G320x200x256);
				/* Get the context for the virtual screen */
				gl_setcontextvgavirtual(G320x200x256);
				Virtual = gl_allocatecontext();
				gl_getcontext(Virtual);
				/* Get the context for the real screen */
				gl_setcontextvga(G320x200x256);
				Screen = gl_allocatecontext();
				gl_getcontext(Screen);
				// Now switch to the virtual context...
				gl_setcontext(Virtual);
				break;
		case LASTMODE:	// Switch to the previous mode.
				if ( lastmode == GRAPHICS )  {
					// Free graphics mode stuff...
					gl_freecontext(Virtual);
					gl_freecontext(Screen);
				}
				newmode=lastmode;
				goto checkmode;
				break;
		default:	// Huh? 
				fprintf(stderr,
			"Unknown graphics mode: 0x%.2x\n", newmode);
				return;
	}
	lastmode=mode;
	mode=newmode;
}

virtual void 
VGA_Graphics:: set_colormap(struct color colormap[256])
{
	for ( int i=0; i<256; ++i ) {
		gl_setpalettecolor(i, colormap[i].red/4,
				colormap[i].green/4, colormap[i].blue/4);
	}
}

virtual void 
VGA_Graphics:: clr_grscreen(void)
{
	int row, col;

	for ( row=0; row < 200; ++row ) {
		for ( col=0; col < 320; ++col ) {
			gl_setpixelrgb(col, row, 0, 0, 0);
		}
	}
}

virtual void 
VGA_Graphics:: put_graphics_txt(int x, int y, char *string, unsigned char color)
{
	unsigned char font8x8[256 * 8 * 8 * BYTESPERPIXEL];
	gl_setwritemode(WRITEMODE_MASKED);
	gl_expandfont(8, 8, color, gl_font8x8, font8x8);
	gl_setfont(8, 8, font8x8);
	gl_write(x, y, string);
}

virtual void
VGA_Graphics:: drawpoint(int x, int y, unsigned char color)
{
	gl_setpixel(x, y, color);
}

void 
VGA_Graphics:: detect_vtsize(int *cols, int *rows)
{
	struct winsize win;

	if ( ioctl(0, TIOCGWINSZ, &win) < 0 ) {
		*cols=0; *rows=0;
		return;
	}
	*cols = win.ws_col;
	*rows = win.ws_row;
}

int 
VGA_Graphics:: resize_vt(int cols, int rows)
{
	struct vt_sizes vtsize;

	vtsize.v_cols = cols;
	vtsize.v_rows = rows;
	vtsize.v_scrollsize = 0;

	return(ioctl(0, VT_RESIZE, &vtsize));
}

void 
VGA_Graphics:: to_50lines(char *regs)
{
	regs[9]  = 0x47;
	regs[10] = 0x06;
	regs[11] = 0x07;
	regs[12] = 0x00;
	regs[13] = 0x00;
}

void 
VGA_Graphics:: to_25lines(char *regs)
{
	regs[9]  = 0x4F;
	regs[10] = 0x0D;
	regs[11] = 0x0E;
	regs[12] = 0x03;
	regs[13] = 0x70;
}

#endif /* linux */
