/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: operating system start point, and parse command line arguments
 */

#include <stdio.h>
#include <ac/stddef.h>
#include <ac/string.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <error.h>
#include <help.h>
#include <preprocess.h>
#include <str.h>
#include <trace.h>
#include <version.h>


static void usage _((void));

static void
usage()
{
	fprintf(stderr, "usage: %s [ <option>... ][ <infile> [ <outfile> ]]\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	fprintf(stderr, "       %s -VERSion\n", progname);
	exit(1);
}


static void main_help _((void));

static void
main_help()
{
	static char *text[] =
	{
#include <../man1/roffpp.h>
	};

	help(text, SIZEOF(text), usage);
}


enum
{
	arglex_token_include
};

static arglex_table_ty argtab[] =
{
	{ "-\\I*",	(arglex_token_ty)arglex_token_include,	},
	{ "-Include",	(arglex_token_ty)arglex_token_include,	},
	{ 0, (arglex_token_ty)0, }, /* end marker */
};


int main _((int, char **));

int
main(argc, argv)
	int	argc;
	char	**argv;
{
	char	*infile;
	char	*outfile;

	arglex_init(argc, argv, argtab);
	str_initialize();

	switch (arglex())
	{
	case arglex_token_help:
		main_help();
		exit(0);

	case arglex_token_version:
		version();
		exit(0);

	default:
		break;
	}

	infile = 0;
	outfile = 0;
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			error
			(
				"misplaced \"%s\" command line argument",
				arglex_value.alv_string
			);
			usage();
	
		case arglex_token_string:
			if (!infile)
				infile = arglex_value.alv_string;
			else if (!outfile)
				outfile = arglex_value.alv_string;
			else
				fatal("too many file names specified");
			break;
	
		case arglex_token_stdio:
			if (!infile)
				infile = "";
			else if (!outfile)
				outfile = "";
			else
				fatal("too many file names specified");
			break;
	
		case arglex_token_include:
			if (arglex() != arglex_token_string)
				fatal("-Include requires a string argument");
			preprocess_include(arglex_value.alv_string);
			break;

#ifdef DEBUG
		case arglex_token_tracing:
			if (arglex() != arglex_token_string)
				fatal("-TRACIng requires ore or more string arguments");
			for (;;)
			{
				trace_enable(arglex_value.alv_string);
				if (arglex() != arglex_token_string);
					break;
			}
			continue;
#endif
		}
		arglex();
	}
	if (infile && !*infile)
		infile = 0;
	if (outfile && !*outfile)
		outfile = 0;
	preprocess(infile, outfile);
	exit(0);
	return 0;
}
