/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate rule statements
 */

#include <ctype.h>
#include <ac/string.h>

#include <emit.h>
#include <mem.h>
#include <stmt/rule.h>
#include <variable.h>
#include <word.h>

enum
{
	fake_export_all_variables = 1,
	fake_ignore,
	fake_phony,
	fake_precious,
	fake_silent,
	fake_suffixes
};

typedef struct stmt_rule_ty stmt_rule_ty;
struct stmt_rule_ty
{
	STMT
	blob_list_ty	*lhs;
	blob_list_ty	*rhs;
	blob_list_ty	*pred;
	int		op;
	blob_list_ty	*body;
	int		fake;
	string_ty	*archive_target;
	string_ty	*archive_member;
};

static wlist	phony;
static wlist	precious;
static wlist	suffix;
static int	suffix_initted;
static wlist	implict_rules_done;
int		stmt_rule_default_history;


static void suffix_init _((void));

static void
suffix_init()
{
	static char *table[] =
	{
		".a", ".C", ".c", ".cc", ".ch", ".def", ".dvi", ".el",
		".elc", ".F", ".f", ".h", ".info", ".l", ".ln", ".mod",
		".o", ".out", ".p", ".r", ".S", ".s", ".sh", ".sym",
		".tex", ".texi", ".texinfo", ".txinfo", ".w", ".web",
		".y",
	};

	size_t		j;
	string_ty	*s;

	if (suffix_initted)
		return;
	suffix_initted = 1;

	for (j = 0; j < SIZEOF(table); ++j)
	{
		s = str_from_c(table[j]);
		wl_append(&suffix, s);
		str_free(s);
	}
}


static void check_for_default _((stmt_rule_ty *));

static void
check_for_default(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_default;

	if (!dot_default)
		dot_default = str_from_c(".DEFAULT");
	if
	(
		this->lhs->length == 1
	&&
		str_equal(this->lhs->list[0]->text, dot_default)
	)
	{
		str_free(this->lhs->list[0]->text);
		this->lhs->list[0]->text = str_from_c("%");
	}
}


static void check_for_export_all_variables _((stmt_rule_ty *));

static void
check_for_export_all_variables(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_export;

	if (!dot_export)
		dot_export = str_from_c(".EXPORT_ALL_VARIABLES");
	if
	(
		this->lhs->length == 1
	&&
		str_equal(this->lhs->list[0]->text, dot_export)
	)
	{
		this->fake = fake_export_all_variables;
	}
}


static void check_for_ignore _((stmt_rule_ty *));

static void
check_for_ignore(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_ignore;

	if (!dot_ignore)
		dot_ignore = str_from_c(".IGNORE");
	if
	(
		this->lhs->length == 1
	&&
		str_equal(this->lhs->list[0]->text, dot_ignore)
	)
	{
		this->fake = fake_ignore;
	}
}


static void check_for_phony _((stmt_rule_ty *));

static void
check_for_phony(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_phony;
	long		j;

	if (!dot_phony)
		dot_phony = str_from_c(".PHONY");
	if
	(
		this->lhs->length == 1
	&&
		str_equal(this->lhs->list[0]->text, dot_phony)
	)
	{
		this->fake = fake_phony;
		for (j = 0; j < this->rhs->length; ++j)
			wl_append_unique(&phony, this->rhs->list[j]->text);
	}
}


static void check_for_precious _((stmt_rule_ty *));

static void
check_for_precious(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_precious;
	long		j;

	if (!dot_precious)
		dot_precious = str_from_c(".PRECIOUS");
	if
	(
		this->lhs->length == 1
	&&
		str_equal(this->lhs->list[0]->text, dot_precious)
	)
	{
		this->fake = fake_precious;
		for (j = 0; j < this->rhs->length; ++j)
			wl_append_unique(&precious, this->rhs->list[j]->text);
	}
}


static void check_for_silent _((stmt_rule_ty *));

static void
check_for_silent(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_silent;

	if (!dot_silent)
		dot_silent = str_from_c(".SILENT");
	if
	(
		this->lhs->length == 1
	&&
		str_equal(this->lhs->list[0]->text, dot_silent)
	)
	{
		this->fake = fake_silent;
	}
}


static void check_for_suffixes _((stmt_rule_ty *));

static void
check_for_suffixes(this)
	stmt_rule_ty	*this;
{
	static string_ty *dot_suffixes;
	long		j;

	if (!dot_suffixes)
		dot_suffixes = str_from_c(".SUFFIXES");
	if (this->lhs->length != 1)
		return;
	if (!str_equal(this->lhs->list[0]->text, dot_suffixes))
		return;

	this->fake = fake_suffixes;
	if (this->rhs->length == 0)
		wl_free(&suffix);
	else
		for (j = 0; j < this->rhs->length; ++j)
			wl_append_unique(&suffix, this->rhs->list[j]->text);
}


static int single_suffix _((string_ty *));

static int
single_suffix(in)
	string_ty	*in;
{
	return wl_member(&suffix, in);
}


static int double_suffix _((string_ty *, string_ty **, string_ty **));

static int
double_suffix(in, out1, out2)
	string_ty	*in;
	string_ty	**out1;
	string_ty	**out2;
{
	long		j, k;
	string_ty	*s;

	for (j = 0; j < suffix.wl_nwords; ++j)
	{
		for (k = 0; k < suffix.wl_nwords; ++k)
		{
			s = str_catenate(suffix.wl_word[j], suffix.wl_word[k]);
			if (str_equal(in, s))
			{
				str_free(s);
				*out1 = suffix.wl_word[j];
				*out2 = suffix.wl_word[k];
				return 1;
			}
			str_free(s);
		}
	}
	return 0;
}


static void constructor _((stmt_ty *));

static void
constructor(that)
	stmt_ty		*that;
{
	stmt_rule_ty	*this;

	this = (stmt_rule_ty *)that;
	this->lhs = 0;
	this->rhs = 0;
	this->op = 0;
	this->body = 0;
	this->fake = 0;
	this->archive_target = 0;
	this->archive_member = 0;
}


static void destructor _((stmt_ty *));

static void
destructor(that)
	stmt_ty		*that;
{
	stmt_rule_ty	*this;

	this = (stmt_rule_ty *)that;
	if (this->lhs)
		blob_list_free(this->lhs);
	if (this->rhs)
		blob_list_free(this->rhs);
	if (this->body)
		blob_list_free(this->body);
	if (this->archive_target)
		str_free(this->archive_target);
	if (this->archive_member)
		str_free(this->archive_member);
}


static void emit _((stmt_ty *));

static void
emit(that)
	stmt_ty		*that;
{
	stmt_rule_ty	*this;
	size_t		j, k;
	wlist		flag;
	string_ty	*s;
	blob_ty		*bp;

	this = (stmt_rule_ty *)that;
	switch (this->fake)
	{
	case fake_export_all_variables:
		return;

	case fake_ignore:
		bp = this->lhs->list[0];
		emit_line_number(bp->line_number, bp->file_name);
		emit_str("set errok;\n");
		return;

	case fake_phony:
	case fake_precious:
		return;

	case fake_silent:
		bp = this->lhs->list[0];
		emit_line_number(bp->line_number, bp->file_name);
		emit_str("set silent;\n");
		return;

	case fake_suffixes:
		return;
	}

	emit_set_file(this->lhs->list[0]->file_name);
	for (j = 0; j < this->lhs->length; ++j)
	{
		if (j)
			emit_char(' ');
		blob_emit(this->lhs->list[j]);
	}
	emit_char(':');
	
	for (j = 0; j < this->rhs->length; ++j)
	{
		emit_char(' ');
		blob_emit(this->rhs->list[j]);
	}

	if (this->pred)
	{
		emit_bol();
		emit_indent_more();
		emit_str("if [in [target]");
		for (j = 0; j < this->pred->length; ++j)
		{
			emit_char(' ');
			blob_emit(this->pred->list[j]);
		}
		emit_str("]");
		emit_indent_less();
	}

	/*
	 * see if the recipe should have any flags
	 */
	wl_zero(&flag);
	for (j = 0; j < this->lhs->length; ++j)
	{
		if (wl_member(&phony, this->lhs->list[0]->text))
		{
			s = str_from_c("force");
			wl_append_unique(&flag, s);
			str_free(s);
			break;
		}
		if (wl_member(&precious, this->lhs->list[0]->text))
		{
			s = str_from_c("precious");
			wl_append_unique(&flag, s);
			str_free(s);
			break;
		}
	}
	if (flag.wl_nwords)
	{
		emit_bol();
		emit_indent_more();
		emit_str("set");
		for (j = 0; j < flag.wl_nwords; ++j)
		{
			emit_char(' ');
			emit_string(flag.wl_word[j]);
		}
		emit_indent_less();
	}
	wl_free(&flag);

	/*
	 * emit the body of the recipe
	 */
	if (!this->body)
		emit_str(";\n");
	else
	{
		emit_str("\n{\n"/*}*/);
		emit_indent_more();
		for (j = 0; j < this->body->length; ++j)
		{
			char		*cp;

			bp = this->body->list[j];
			cp = bp->text->str_text;
			for (;;)
			{
				switch (*cp)
				{
				case '-':
					s = str_from_c("errok");
					wl_append(&flag, s);
					str_free(s);
					++cp;
					continue;

				case '@':
					s = str_from_c("silent");
					wl_append(&flag, s);
					str_free(s);
					++cp;
					continue;

				case '+':
					s = str_from_c("notouch");
					wl_append(&flag, s);
					str_free(s);
					++cp;
					continue;

				default:
					break;
				}
				break;
			}
			emit_line_number(bp->line_number, bp->file_name);
			emit_str(cp);

			if (flag.wl_nwords)
			{
				emit_bol();
				emit_indent_more();
				emit_str("set");
				for (k = 0; k < flag.wl_nwords; ++k)
				{
					emit_char(' ');
					emit_string(flag.wl_word[k]);
				}
				emit_indent_less();
			}
			wl_free(&flag);
			emit_str(";\n");
		}
		emit_indent_less();
		emit_str(/*{*/"}\n");
	}
}


static stmt_method_ty method =
{
	sizeof(stmt_rule_ty),
	"rule",
	constructor,
	destructor,
	emit,
};


stmt_ty *
stmt_rule_alloc(lhs, op, rhs, pred)
	blob_list_ty	*lhs;
	int		op;
	blob_list_ty	*rhs;
	blob_list_ty	*pred;
{
	stmt_rule_ty	*result;
	string_ty	*s1;
	string_ty	*s2;
	blob_list_ty	*lhs2;
	size_t		j;
	blob_list_ty	*rhs2;
	blob_list_ty	*pred2;
	static string_ty *dot_a;
	string_ty	*archive_target;
	string_ty	*archive_member;

	/*
	 * rewite horrible target rules
	 */
	archive_target = 0;
	archive_member = 0;
	if (!dot_a)
		dot_a = str_from_c(".a");
	suffix_init();
	if (lhs->length == 1 && single_suffix(lhs->list[0]->text))
	{
		wl_append(&implict_rules_done, lhs->list[0]->text);
		blob_list_prepend
		(
			rhs,
			blob_alloc
			(
				str_format("%%%S", lhs->list[0]->text),
				lhs->list[0]->file_name,
				lhs->list[0]->line_number
			)
		);
		str_free(lhs->list[0]->text);
		lhs->list[0]->text = str_from_c("%");
	}
	else if
	(
		lhs->length == 1
	&&
		double_suffix(lhs->list[0]->text, &s1, &s2)
	)
	{
		if (str_equal(s2, dot_a))
		{
			archive_target = str_format("%%1%S", s2);
			archive_member = str_format("%%%S", s1);
		}
		wl_append(&implict_rules_done, lhs->list[0]->text);
		blob_list_append
		(
			lhs,
			blob_alloc
			(
				(
					archive_target
				?
					str_format("%%1%S(%%%S)", s2, s1)
				:
					str_format("%%%S", s2)
				),
				lhs->list[0]->file_name,
				lhs->list[0]->line_number
			)
		);

		blob_list_prepend
		(
			rhs,
			blob_alloc
			(
				str_format("%%%S", s1),
				lhs->list[0]->file_name,
				lhs->list[0]->line_number
			)
		);

		/* drop old target */
		blob_list_delete(lhs, lhs->list[0]);
	}
	else if
	(
		lhs->length == 1
	&&
		lhs->list[0]->text->str_text[0] == '('/*)*/
	&&
		(
			lhs->list[0]->text->str_text
			[
				lhs->list[0]->text->str_length - 1
			]
		==
			/*(*/')'
		)
	)
	{
		archive_target = str_format("%%1", s2);
		archive_member =
			str_n_from_c
			(
				lhs->list[0]->text->str_text + 1,
				lhs->list[0]->text->str_length - 2
			);
		str_free(lhs->list[0]->text);
		lhs->list[0]->text =
			str_format("%S(%S)", archive_target, archive_member);
	}

	/*
	 * rewrite the variable names
	 */
	result = (stmt_rule_ty *)stmt_alloc(&method);
	lhs2 = blob_list_alloc();
	for (j = 0; j < lhs->length; ++j)
		variable_rename(lhs->list[j], lhs2, &result->ref);
	blob_list_free(lhs);
	rhs2 = blob_list_alloc();
	for (j = 0; j < rhs->length; ++j)
		variable_rename(rhs->list[j], rhs2, &result->ref);
	blob_list_free(rhs);
	if (pred)
	{
		pred2 = blob_list_alloc();
		for (j = 0; j < pred->length; ++j)
			variable_rename(pred->list[j], pred2, &result->ref);
		blob_list_free(pred);
	}
	else
		pred2 = 0;

	result->lhs = lhs2;
	result->op = op;
	result->rhs = rhs2;
	result->pred = pred2;
	result->archive_target = archive_target;
	result->archive_member = archive_member;
	check_for_default(result);
	check_for_export_all_variables(result);
	check_for_ignore(result);
	check_for_phony(result);
	check_for_precious(result);
	check_for_silent(result);
	check_for_suffixes(result);
	return (stmt_ty *)result;
}


void
stmt_rule_append(that, lp)
	stmt_ty		*that;
	blob_ty		*lp;
{
	stmt_rule_ty	*this;
	blob_list_ty	*blp;
	string_ty	*s;
	blob_ty		*lp2;
	size_t		j;

	/*
	 * replace the variable names
	 */
	this = (stmt_rule_ty *)that;
	blp = blob_list_alloc();
	if (this->archive_target)
		variable_archive(this->archive_target, this->archive_member);
	variable_rename(lp, blp, &this->rref);

	/*
	 * reconstruct as a single string
	 */
	s = blp->length ? str_copy(blp->list[0]->text) : str_from_c("");
	for (j = 1; j < blp->length; ++j)
	{
		string_ty	*s2;

		s2 = str_format("%S %S", s, blp->list[j]->text);
		str_free(s);
		s = s2;
	}
	lp2 = blob_alloc(s, lp->file_name, lp->line_number);
	blob_free(lp);
	blob_list_free(blp);

	/*
	 * append to the rule body
	 */
	if (!this->body)
		this->body = blob_list_alloc();
	blob_list_append(this->body, lp2);
}


static void split_by_lines _((char *, stmt_ty *, string_ty *, long));

static void
split_by_lines(s, sp, fn, ln)
	char		*s;
	stmt_ty		*sp;
	string_ty	*fn;
	long		ln;
{
	char		*ep;

	while (isspace(*s))
		++s;
	for (;;)
	{
		ep = strchr(s, '\n');
		if (!ep)
			ep = s + strlen(s);
		while (ep > s && isspace(ep[-1]))
			--ep;
		stmt_rule_append
		(
			sp,
			blob_alloc(str_n_from_c(s, ep - s), fn, ln)
		);

		s = ep;
		while (isspace(*s))
			++s;
		if (!*s)
			break;
	}
}


stmt_ty *
stmt_rule_default(n)
	int		n;
{
	typedef struct table_ty table_ty;
	struct table_ty
	{
		char	*lhs1;
		char	*lhs2;
		char	*body;
		int	history;
	};

	static table_ty table[] =
	{
		{ ".o", "", "$(LINK.o) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
    		{ ".s", "", "$(LINK.s) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
    		{ ".S", "", "$(LINK.S) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
    		{ ".c", "", "$(LINK.c) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
    		{ ".cc", "", "$(LINK.cc) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
    		{ ".C", "", "$(LINK.C) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
    		{ ".f", "", "$(LINK.f) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
		{ ".p", "", "$(LINK.p) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
		{ ".F", "", "$(LINK.F) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
		{ ".r", "", "$(LINK.r) $^ $(LOADLIBES) $(LDLIBS) -o $@", },
		{ ".mod", "", "$(COMPILE.mod) -o $@ -e $@ $^", },
		{ ".def", ".sym", "$(COMPILE.def) -o $@ $<", },
		{ ".sh", "", "cat $< >$@ \n chmod a+x $@", },
		{
			".s",
			".o",
#if !defined(M_XENIX) || defined(__GNUC__)
			"$(COMPILE.s) -o $@ $<",
#else	/* Xenix.  */
			"$(COMPILE.s) -o$@ $<",
#endif	/* Not Xenix.  */
		},
		{
			".S",
			".o",
#if !defined(M_XENIX) || defined(__GNUC__)
			"$(COMPILE.S) -o $@ $<",
#else	/* Xenix.  */
			"$(COMPILE.S) -o$@ $<",
#endif	/* Not Xenix.  */
		},
		{ ".c", ".o", "$(COMPILE.c) $<", },
		{ ".cc", ".o", "$(COMPILE.cc) $<", },
		{ ".C", ".o", "$(COMPILE.C) $<", },
		{ ".f", ".o", "$(COMPILE.f) $<", },
		{ ".p", ".o", "$(COMPILE.p) $<", },
		{ ".F", ".o", "$(COMPILE.F) $<", },
		{ ".r", ".o", "$(COMPILE.r) $<", },
		{ ".mod", ".o", "$(COMPILE.mod) -o $@ $<", },
		{ ".c", ".ln", "$(LINT.c) -C$* $<", },
		{
			".y",
			".ln",
			"$(YACC.y) $<\n\
			$(LINT.c) -C$* y.tab.c\n\
			$(RM) y.tab.c",
		},
		{
			".l",
			".ln",
			"@$(RM) $*.c\n\
			$(LEX.l) $< > $*.c\n\
			$(LINT.c) -i $*.c -o $@\n\
			$(RM) $*.c",
		},
		{
			".y",
			".c",
			"$(YACC.y) $<\n\
			mv -f y.tab.c $@",
		},
		{
			".l",
			".c",
			"@$(RM) $@\n\
			$(LEX.l) $< > $@",
		},
		{ ".F", ".f", "$(PREPROCESS.F) $<", },
		{ ".r", ".f", "$(PREPROCESS.r) $<", },
		{
			/*
			 * This might actually make lex.yy.c if there's
			 * no %R% directive in $*.l, but in that case
			 * why were you trying to make $*.r anyway?
			 */
			".l",
			".r",
			"$(LEX.l) $< > $@\n\
			mv -f lex.yy.r $@",
		},
		{ ".S", ".s", "$(PREPROCESS.S) $< > $@", },
		{ ".texinfo", ".info", "$(MAKEINFO) $< -o $@", },
		{ ".texi", ".info", "$(MAKEINFO) $< -o $@", },
		{ ".txinfo", ".info", "$(MAKEINFO) $< -o $@", },
		{ ".tex", ".dvi", "$(TEX) $<", },
		{ ".texinfo", ".dvi", "$(TEXI2DVI) $<", },
		{ ".texi", ".dvi", "$(TEXI2DVI) $<", },
		{ ".txinfo", ".dvi", "$(TEXI2DVI) $<", },
		{
			/* The `-' says there is no `.ch' file.  */
			".w", ".c", "$(CTANGLE) $< - $@",
		},
		{ ".web", ".p", "$(TANGLE) $<", },
		{
			/* The `-' says there is no `.ch' file.  */
			".w", ".tex", "$(CWEAVE) $< - $@",
		},
		{ ".web", ".tex", "$(WEAVE) $<", },

		/*
		 * the default archive rule
		 */
		{ ".o", ".a", "$(AR) $(ARFLAGS) $@ $<", },
	};

	static table_ty table2[] =
	{
		/*
		 * The X.out rules are only in BSD's default set
		 * because BSD Make has no null-suffix rules, so
		 * `foo.out' and `foo' are the same thing.
		 */
		{ "%.out", "%", "@rm -f $@\ncp $< $@", },

		/*
		 * tangle and weave, the C web rules
		 */
		{ "%.c", "%.w %.ch", "$(CTANGLE) $^ $@", },
		{ "%.tex", "%.w %.ch", "$(CWEAVE) $^ $@", },

		/*
		 * the GNU default archive rule
		 */
		{ "(%)", "%", "$(AR) $(ARFLAGS) $@ $<", },

		/*
		 * history retrieval rules
		 */
		{ "%", "%,v", "+$(CHECKOUT,v) $^ $@", 1, },
		{ "%", "RCS/%,v", "+$(CHECKOUT,v) $^ $@", 1, },
		{ "%", "s.%", "$(GET) $(GFLAGS) $^", 1, },
		{ "%", "SCCS/s.%", "$(GET) $(GFLAGS) $^", 1, },
	};

	static long	linum	= 1000;
	static string_ty *builtin;
	table_ty	*tp;

	if (!builtin)
		builtin = str_from_c("builtin");
	for (tp = table; tp < ENDOF(table); ++tp)
	{
		string_ty	*lhs1;
		string_ty	*lhs2;
		string_ty	*s;
		blob_list_ty	*r1;
		blob_list_ty	*r2;
		stmt_ty		*sp;

		if (!stmt_rule_default_history && tp->history)
			continue;
		lhs1 = str_from_c(tp->lhs1);
		if (!wl_member(&suffix, lhs1))
		{
			str_free(lhs1);
			continue;
		}
		lhs2 = str_from_c(tp->lhs2);
		if (lhs2->str_length && !wl_member(&suffix, lhs2))
		{
			str_free(lhs1);
			str_free(lhs2);
			continue;
		}
		s = str_catenate(lhs1, lhs2);
		str_free(lhs1);
		str_free(lhs2);
		if (wl_member(&implict_rules_done, s))
		{
			str_free(s);
			continue;
		}

		++linum;

		r1 = blob_list_alloc();
		blob_list_append(r1, blob_alloc(s, builtin, linum));
		r2 = blob_list_alloc();

		sp = stmt_rule_alloc(r1, 1, r2, (blob_list_ty *)0);

		split_by_lines(tp->body, sp, builtin, linum);
		return sp;
	}

	for (tp = table2; tp < ENDOF(table2); ++tp)
	{
		string_ty	*lhs1;
		string_ty	*lhs2;
		string_ty	*s;
		blob_list_ty	*r1;
		blob_list_ty	*r2;
		stmt_ty		*sp;

		if (!stmt_rule_default_history && tp->history)
			continue;
		lhs1 = str_from_c(tp->lhs1);
		lhs2 = str_from_c(tp->lhs2);

		/*
		 * make sure we inderstand the suffixes
		 */
		 if (!tp->history)
		 {
			wlist		wl;
			wlist		wl2;
			int		ok;
			int		j;
			string_ty	*s;

			wl_zero(&wl);
			str2wl(&wl2, lhs1, (char *)0, 0);
			for (j = 0; j < wl2.wl_nwords; ++j)
				wl_append_unique(&wl, wl2.wl_word[j]);
			wl_free(&wl2);
			str2wl(&wl2, lhs2, (char *)0, 0);
			for (j = 0; j < wl2.wl_nwords; ++j)
				wl_append_unique(&wl, wl2.wl_word[j]);
			wl_free(&wl2);

			ok = 1;
			for (j = 0; j < wl.wl_nwords; ++j)
			{
				s = wl.wl_word[j];
				if (s->str_text[0] == '%')
					s =
						str_n_from_c
						(
							s->str_text + 1,
							s->str_length - 1
						);
				else
					s = str_from_c(".a");
				if (s->str_length && !wl_member(&suffix, s))
					ok = 0;
				str_free(s);
			}
			wl_free(&wl);

			if (!ok)
			{
				str_free(lhs1);
				str_free(lhs2);
				continue;
			}
		 }

		/*
		 * remember we have done it
		 */
		s = str_format("%S:%S", lhs1, lhs2);
		if (wl_member(&implict_rules_done, s))
		{
			str_free(lhs1);
			str_free(lhs2);
			str_free(s);
			continue;
		}
		wl_append(&implict_rules_done, s);
		str_free(s);

		/*
		 * construct and instanciate the rule
		 */
		++linum;
		r1 = blob_list_alloc();
		blob_list_append(r1, blob_alloc(lhs1, builtin, linum));
		r2 = blob_list_alloc();
		blob_list_append(r2, blob_alloc(lhs2, builtin, linum));
		sp = stmt_rule_alloc(r1, 1, r2, (blob_list_ty *)0);
		split_by_lines(tp->body, sp, builtin, linum);
		return sp;
	}
	return 0;
}
