/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate comment statements
 */

#include <ctype.h>

#include <emit.h>
#include <mem.h>
#include <stmt/comment.h>

typedef struct stmt_comment_ty stmt_comment_ty;
struct stmt_comment_ty
{
	STMT
	blob_list_ty	*body;
};


static void constructor _((stmt_ty *));

static void
constructor(that)
	stmt_ty		*that;
{
	stmt_comment_ty	*this;

	this = (stmt_comment_ty *)that;
	this->body = blob_list_alloc();
}


static void destructor _((stmt_ty *));

static void
destructor(that)
	stmt_ty		*that;
{
	stmt_comment_ty	*this;

	this = (stmt_comment_ty *)that;
	blob_list_free(this->body);
}


static int blank _((char *));

static int
blank(s)
	char		*s;
{
	while (*s && isspace(*s))
		++s;
	while (*s == '#')
		++s;
	while (*s && isspace(*s))
		++s;
	return !*s;
}


static void emit _((stmt_ty *));

static void
emit(that)
	stmt_ty		*that;
{
	stmt_comment_ty	*this;
	blob_list_ty	*blp;
	long		min;
	long		max;
	long		j;

	this = (stmt_comment_ty *)that;
	blp = this->body;
	if (!blp->length)
		return;
	for (min = 0; min < blp->length; ++min)
		if (!blank(blp->list[min]->text->str_text))
			break;
	for (max = blp->length - 1; max >= 0; --max)
		if (!blank(blp->list[max]->text->str_text))
			break;
	if (min > max)
		return;
	emit_line_number
	(
		blp->list[min]->line_number - 1,
		blp->list[min]->file_name
	);
	emit_str("/*\n");
	for (j = min; j <= max; ++j)
	{
		static string_ty *t1;
		static string_ty *t2;
		string_ty	*tmp;
		char		*s;

		if (!t1)
		{
			t1 = str_from_c("*/");
			t2 = str_from_c("* /");
		}
		tmp = str_substitute(t1, t2, blp->list[j]->text);
		s = tmp->str_text;
		while (*s && isspace(*s))
			++s;
		while (*s == '#')
			++s;
		while (*s && isspace(*s))
			++s;

		emit_str(" *");
		if (*s)
		{
			emit_char(' ');
			emit_str(s);
		}
		emit_char('\n');
		str_free(tmp);
	}
	emit_str(" */\n");
}


static stmt_method_ty method =
{
	sizeof(stmt_comment_ty),
	"comment",
	constructor,
	destructor,
	emit,
	0, /* regroup */
	0, /* sort */
};


stmt_ty *
stmt_comment_alloc()
{
	stmt_ty		*result;

	result = stmt_alloc(&method);
	result->white_space = 1;
	return result;
}


void
stmt_comment_append(that, lp)
	stmt_ty		*that;
	blob_ty		*lp;
{
	stmt_comment_ty	*this;

	this = (stmt_comment_ty *)that;
	blob_list_append(this->body, lp);
}
