/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to emit the result
 */

#include <stdio.h>

#include <emit.h>
#include <error.h>

static string_ty *filename;
static FILE	*output;
int		emit_line_numbers;
int		col;
long		old_ln;
string_ty	*old_fn;
int		depth;


void
emit_open(s)
	char		*s;
{
	assert(!output);
	if (s)
	{
		output = fopen(s, "w");
		if (!output)
			nfatal("%s", s);
		filename = str_from_c(s);
	}
	else
	{
		filename = str_from_c("standard output");
		output = stdout;
	}
	col = 0;
}


void
emit_close()
{
	assert(output);
	emit_bol();
	if (output != stdout && fclose(output))
		nfatal("%s", filename->str_text);
	str_free(filename);
	filename = 0;
	output = 0;
	col = 0;
}


void
emit_bol()
{
	if (col)
		emit_char('\n');
}


void
emit_char(c)
	int		c;
{
	if (c != '\n' && !col && depth > 0)
	{
		int	j;

		for (j = 0; j < depth; ++j)
			putc('\t', output);
	}
	if (putc(c, output) == EOF)
		nfatal("%s", filename->str_text);
	if (c == '\n')
	{
		++old_ln;
		col = 0;
	}
	else
	{
		++col;
	}
}


void
emit_str(s)
	char		*s;
{
	while (*s)
		emit_char(*s++);
}


void
emit_string(s)
	string_ty	*s;
{
	emit_str(s->str_text);
}


void
emit_line_number(lino, fn)
	long		lino;
	string_ty	*fn;
{
	static string_ty *builtin;

	assert(output);
	if (!builtin)
		builtin = str_from_c("builtin");
	if (str_equal(fn, builtin))
		return;
	if (old_ln == lino && str_equal(old_fn, fn))
		return;
	
	/*
	 * for short distances in the same file,
	 * just throw newlines
	 */
	if (str_equal(old_fn, fn) && lino > old_ln && lino - old_ln < 10)
	{
		while (old_ln < lino)
			emit_char('\n');
		return;
	}

	if (emit_line_numbers)
	{
		emit_bol();
		fprintf(output, "#line %ld \"%s\"\n", lino, fn->str_text);
		if (ferror(output))
			nfatal("write %s", filename->str_text);
	}
	else if (!str_equal(old_fn, fn) || lino > (old_ln + !!col))
	{
		emit_bol();
		emit_char('\n');
	}

	old_ln = lino;
	if (old_fn)
		str_free(old_fn);
	old_fn = str_copy(fn);
}


void
emit_set_file(fn)
	string_ty	*fn;
{
	if (str_equal(old_fn, fn))
		return;
	emit_bol();
	if (old_fn)
	{
		str_free(old_fn);
		emit_char('\n');
	}
	old_fn = str_copy(fn);
	old_ln = 32767;
}


void
emit_indent_more()
{
	++depth;
}


void
emit_indent_less()
{
	--depth;
}
