/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: operating system start point, and command line argument parsing
 */

#include <stdio.h>
#include <ac/stdlib.h>

#include <arglex.h>
#include <error.h>
#include <fp/cksum.h>
#include <fp/cksum.h>
#include <fp/combined.h>
#include <fp/md5.h>
#include <fp/snefru.h>
#include <help.h>
#include <version.h>
#include <word.h>


enum
{
	arglex_token_cksum,
	arglex_token_md5,
	arglex_token_snefru
};

static arglex_table_ty argtab[] =
{
	{ "-Snefru",		arglex_token_snefru,	},
	{ "-Message_Digest",	arglex_token_md5,	},
	{ "-Checksum",		arglex_token_cksum,	},
	{ 0, 0, } /* end marker */
};


static void usage _((void));

static void
usage()
{
	fprintf(stderr, "Usage: %s [ <option>... ][ <filename>... ]\n", progname);
	fprintf(stderr, "       %s -Help\n", progname);
	exit(1);
}


static void main_help _((void));

static void
main_help()
{
	static char *text[] =
	{
#include <../man1/cookfp.h>
	};

	help(text, SIZEOF(text), usage);
}


int main _((int, char **));

int
main(argc, argv)
	int		argc;
	char		**argv;
{
	string_ty	*minus;
	wlist		file;
	long		j;
	fingerprint_methods_ty *method;
	string_ty	*s;

	arglex_init(argc, argv, argtab);
	str_initialize();
	switch (arglex())
	{
	case arglex_token_help:
		main_help();
		exit(0);
	
	case arglex_token_version:
		version();
		exit(0);
	
	default:
		break;
	}

	method = 0;
	wl_zero(&file);
	minus = str_from_c("-");
	while (arglex_token != arglex_token_eoln)
	{
		switch (arglex_token)
		{
		default:
			error
			(
				"misplaced \"%s\" command line argument",
				arglex_value.alv_string
			);
			usage();

		case arglex_token_snefru:
			if (method)
			{
				too_many_methods:
				error("too many methods specified");
				usage();
			}
			method = &fp_snefru;
			break;

		case arglex_token_md5:
			if (method)
				goto too_many_methods;
			method = &fp_md5;
			break;

		case arglex_token_cksum:
			if (method)
				goto too_many_methods;
			method = &fp_cksum;
			break;

		case arglex_token_string:
			s = str_from_c(arglex_value.alv_string);
			wl_append(&file, s);
			str_free(s);
			break;

		case arglex_token_stdio:
			if (wl_member(&file, minus))
			{
				error("may only name stdin once");
				usage();
			}
			wl_append(&file, minus);
			break;
		}
		arglex();
	}

	/*
	 * if no files named, read stdin
	 */
	if (!file.wl_nwords)
		wl_append(&file, minus);

	/*
	 * by default, use the fp_combined class
	 */
	if (!method)
		method = &fp_combined;

	/*
	 * read the named files
	 */
	for (j = 0; j < file.wl_nwords; ++j)
	{
		fingerprint_ty	*p;
		char		buf[1000];

		p = fingerprint_new(method);
		s = file.wl_word[j];
		if (str_equal(s, minus))
		{
			if (fingerprint_file_sum(p, (char *)0, buf))
				nfatal("standard input");
			printf("%s", buf);
			if (file.wl_nwords != 1)
				printf("\tstdin");
			printf("\n");
		}
		else
		{
			if (fingerprint_file_sum(p, s->str_text, buf))
				nfatal("%s", s->str_text);
			printf("%s\t%s\n", buf, s->str_text);
		}
		fingerprint_delete(p);
	}
	exit(0);
	return 0;
}
