/*
 *	cook - file construction tool
 *	Copyright (C) 1990, 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manage statement trees
 *
 * This file contains the functions for manipulating statement trees,
 * allocating, interpreting and releasing.
 */

#include <ac/stddef.h>
#include <ac/string.h>
#include <ac/time.h>

#include <cook.h>
#include <env.h>
#include <error.h>
#include <expr.h>
#include <id.h>
#include <main.h>
#include <match.h>
#include <mem.h>
#include <option.h>
#include <os.h>
#include <stmt.h>
#include <trace.h>
#include <word.h>


/*
 *  NAME
 *	stmt_alloc - allocate a statement structure
 *
 *  SYNOPSIS
 *	stmt *stmt_alloc(void);
 *
 *  DESCRIPTION
 *	Allocates a statement structure, initializing it to zeroes.
 *
 *  RETURNS
 *	A pointer to the dynamically allocated space is returned.
 *
 *  CAVEAT
 *	It is the responsibility of the caller to ensure that the space is
 *	freed when finished with, by a call to stmt_free().
 */

stmt *
stmt_alloc()
{
	stmt		*sp;

	trace(("stmt_alloc()\n{\n"/*}*/));
	sp = mem_alloc_clear(sizeof(stmt));
	sp->s_references = 1;
	trace(("return %08lX;\n", sp));
	trace((/*{*/"}\n"));
	return sp;
}


/*
 *  NAME
 *      stmt_copy - copy statement tree
 *
 *  SYNOPSIS
 *      stmt *stmt_copy(stmt *);
 *
 *  DESCRIPTION
 *      The stmt_copy function is used to make a copy of a statement tree.
 *
 *  RETURNS
 *      stmt* - pointer to the root of the copied statement tree in dynamic
 *      memory.
 *
 *  CAVEAT
 *      Use the stmt_free function to release the tree when finished with.
 */

stmt *
stmt_copy(sp)
	stmt		*sp;
{
	trace(("stmt_copy(sp = %08X)\n{\n"/*}*/, sp));
	sp->s_references++;
	trace(("return %08X;\n", sp));
	trace((/*{*/"}\n"));
	return sp;
}


/*
 *  NAME
 *	stmt_free - free a statement tree
 *
 *  SYNOPSIS
 *	void stmt_free(stmt *sp);
 *
 *  DESCRIPTION
 *	Frees a statement structure after it has been executed.
 *
 *  CAVEAT
 *	It is assumed that the statement tree is in dynamic memory.
 */

void
stmt_free(sp)
	stmt		*sp;
{
	trace(("stmt_free(sp = %08X)\n{\n"/*}*/, sp));
	assert(sp);
	sp->s_references--;
	if (sp->s_references > 0)
		goto ret;
	assert(sp->s_references == 0);
	switch ((int)sp->s_op)
	{
	default:
		fatal("bad statement selector %d (bug)", sp->s_op);

	case OP_ASSIGN:
		el_free(&sp->s_assign.a_name);
		el_free(&sp->s_assign.a_value);
		break;

	case OP_COMMAND:
		el_free(&sp->s_cmd.c_args);
		if (sp->s_cmd.c_input)
			expr_free(sp->s_cmd.c_input);
		break;

	case OP_TOUCH:
	case OP_UNSETENV:
		el_free(&sp->s_cmd.c_args);
		break;

	case OP_COMPOUND:
		sl_free(&sp->s_list);
		break;

	case OP_SET:
	case OP_LOOPSTOP:
	case OP_NOP:
	case OP_FAIL:
	case OP_FAIL_DK:
		break;

	case OP_IF:
		expr_free(sp->s_if.sif_cond);
		stmt_free(sp->s_if.sif_true);
		stmt_free(sp->s_if.sif_false);
		break;

	case OP_LOOP:
		stmt_free(sp->s_loop);
		break;

	case OP_RECIPE:
		el_free(&sp->s_recipe.sr_target);
		el_free(&sp->s_recipe.sr_need);
		el_free(&sp->s_recipe.sr_need2);
		if (sp->s_recipe.sr_precondition)
			expr_free(sp->s_recipe.sr_precondition);
		if (sp->s_recipe.sr_action)
			stmt_free(sp->s_recipe.sr_action);
		if (sp->s_recipe.sr_use_action)
			stmt_free(sp->s_recipe.sr_use_action);
		str_free(sp->s_recipe.sr_position.pos_name);
		break;
	}
	mem_free(sp);
ret:
	trace((/*{*/"}\n"));
}


static int echo_command _((wlist *));

static int
echo_command(wlp)
	wlist		*wlp;
{
	static string_ty *echo;
	string_ty	*tmp;
	char		*cp;

	if (!echo)
		echo = str_from_c("echo");
	if (wlp->wl_nwords < 1 || !str_equal(wlp->wl_word[0], echo))
		return 0;
	tmp = wl2str(wlp, 0, wlp->wl_nwords - 1, (char *)0);
	for (cp = tmp->str_text; *cp; ++cp)
	{
		if (strchr("^|>", *cp))
		{
			str_free(tmp);
			return 0;
		}
	}
	str_free(tmp);
	return 1;
}


/*
 *  NAME
 *	stmt_eval - evaluate a statement
 *
 *  SYNOPSIS
 *	int stmt_eval(stmt *sp);
 *
 *  DESCRIPTION
 *	Stmt_eval is used to evaluate a statement tree.
 *	It performs the actions so implied.
 *
 *  RETURNS
 *	The value returned indicates why the statement evaluation terminated.
 *	    STMT_OK	normal termination, success
 *	    STMT_LSTOP	a loopstop statement was encountered
 *	    STMT_ERROR	an execution error in a command was encountered
 *	There is also the posibility of internal subroutines;
 *	If an when this happens, an additional STMT_RET value could be returned.
 *
 *  CAVEAT
 *	The explicit and implicit recipes need to copy the statement
 *	bound to them.
 *
 *	The explicit and implicit recipes need to overwrite recipes with
 *	identical target and prerequisite lists. The best place to do this
 *	is in rl_append().
 */

int
stmt_eval(sp)
	stmt		*sp;
{
	int		status;

	trace(("stmt_eval(sp = %08X)\n{\n"/*}*/, sp));
	assert(sp);
	if (desist)
	{
		status = STMT_ERROR;
		goto ret;
	}
	status = STMT_OK;
	switch ((int)sp->s_op)
	{
	default:
		error("bad statement selector %d (bug)", sp->s_op);
		option_set_errors();
		status = STMT_ERROR;
		break;

	case OP_ASSIGN:
		{
			wlist		name;
			wlist		value;
			static string_ty	*setenv_word;
			static string_ty	*mtime_word;

			/*
			 * The grammar has the assignment as an expression list.
			 * Evaluate the expression list into a word list.
			 * This word list is the value to assign.
			 */
			if (!setenv_word)
				setenv_word = str_from_c("setenv");
			if (!mtime_word)
				mtime_word = str_from_c("mtime");
			wl_zero(&name);
			if (el2wl(&name, &sp->s_assign.a_name))
			{
				status = STMT_ERROR;
				break;
			}
			switch (name.wl_nwords)
			{
			case 0:
				error("lefthand side of assignment is empty");
				option_set_errors();
				status = STMT_ERROR;
				break;

			case 1:
				if (el2wl(&value, &sp->s_assign.a_value))
				{
					status = STMT_ERROR;
					break;
				}
				id_assign(name.wl_word[0], &value);
				wl_free(&value);
				break;

			case 2:
				if (str_equal(name.wl_word[0], setenv_word))
				{
					string_ty *s;

					if (el2wl(&value, &sp->s_assign.a_value))
					{
						status = STMT_ERROR;
						break;
					}
					s =
						wl2str
						(
							&value,
							0,
							value.wl_nwords - 1,
							(char *)0
						);
					wl_free(&value);
					env_set
					(
						name.wl_word[1]->str_text,
						s->str_text
					);
					str_free(s);
					break;
				}
#if 0
				if (str_equal(name.wl_word[0], mtime_word))
				{
					mtime_pattern
					(
						name.wl_word[1],
						&sp->s_assign.a_value
					);
					break;
				}
#endif
				/* fall through... */

			default:
				error
				(
			     "lefthand side of assignment is more than one word"
				);
				option_set_errors();
				status = STMT_ERROR;
				break;
			}
			wl_free(&name);
		}
		break;

	case OP_UNSETENV:
		{
			wlist		wl;
			int		j;

			wl_zero(&wl);
			if (el2wl(&wl, &sp->s_cmd.c_args))
			{
				status = STMT_ERROR;
				break;
			}
			if (!wl.wl_nwords)
			{
				error("unsetenv was given no words");
				option_set_errors();
				status = STMT_ERROR;
			}
			for (j = 0; j < wl.wl_nwords; ++j)
				env_unset(wl.wl_word[j]->str_text);
			wl_free(&wl);
		}
		break;

	case OP_COMMAND:
		{
			wlist		wl;
			int		metering;
			int		silent;
			int		errok;

			/*
			 * The grammar has the command as an expression list.
			 * Evaluate the expression list into a word list.
			 * This word list is the command to execute.
			 */
			if (el2wl(&wl, &sp->s_cmd.c_args))
			{
				status = STMT_ERROR;
				break;
			}
			cook_flags(sp->s_cmd.c_flags, OPTION_LEVEL_EXECUTE);
			silent = option_test(OPTION_SILENT);
			if (!silent)
			{
				string_ty *cp;

				/*
				 * If the command has not been silenced,
				 * form it into a string and echo it.
				 */
				cp =
					wl2str
					(
						&wl,
						0,
						wl.wl_nwords - 1,
						(char *)0
					);
				error("%s", cp->str_text);
				str_free(cp);
				metering = option_test(OPTION_METER);
			}
			else
			{
				/* no metering if silent */
				metering = 0;
			}
			if (option_test(OPTION_ACTION))
			{
				string_ty *s;

				if (sp->s_cmd.c_input)
				{
					wlist	 input;

					wl_zero(&input);
					if (expr_eval(&input, sp->s_cmd.c_input))
					{
						status = STMT_ERROR;
						break;
					}
					s =
						wl2str
						(
							&input,
							0,
							input.wl_nwords - 1,
							(char *)0
						);
					wl_free(&input);
				}
				else
					s = 0;
				if (metering)
					os_meter_begin();
				errok = option_test(OPTION_ERROK);
				if (silent && echo_command(&wl))
					star_eoln();
				if (os_execute(&wl, s, errok))
					status = STMT_ERROR;
				if (metering)
					os_meter_end();
				if (s)
					str_free(s);
			}
			if (!silent)
				star_sync();
			if (option_test(OPTION_INVALIDATE_STAT_CACHE))
			{
				int		j;

				for (j = 0; j < wl.wl_nwords; ++j)
					if (os_clear_stat(wl.wl_word[j]))
						status = STMT_ERROR;
			}
			wl_free(&wl);
			option_undo_level(OPTION_LEVEL_EXECUTE);
		}
		break;

	case OP_SET:
		cook_flags(sp->s_cmd.c_flags, OPTION_LEVEL_COOKBOOK);
		break;

	case OP_FAIL:
		status = STMT_ERROR;
		break;

	case OP_FAIL_DK:
		status = STMT_BACKTRACK;
		break;

	case OP_COMPOUND:
		{
			int		j;

			for (j = 0; j < sp->s_list.sl_nstmts; j++)
			{
				status = stmt_eval(sp->s_list.sl_stmt[j]);
				if (status != STMT_OK)
					break;
			}
		}
		break;

	case OP_IF:
		switch (expr_eval_condition(sp->s_if.sif_cond))
		{
		case -1:
			status = STMT_ERROR;
			break;

		case 0:
			status = stmt_eval(sp->s_if.sif_false);
			break;

		default:
			status = stmt_eval(sp->s_if.sif_true);
			break;
		}
		break;

	case OP_LOOP:
		do
			status = stmt_eval(sp->s_loop);
		while
			(status == STMT_OK);
		if (status == STMT_LSTOP)
			status = STMT_OK;
		break;

	case OP_LOOPSTOP:
		status = STMT_LSTOP;
		break;

	case OP_NOP:
		break;

	case OP_RECIPE:
		{
			recipe		r;
			int		j;
			int		imp;

			/*
			 * A recipe in the grammar has expression lists for
			 * both targets and prerequisites.  These must be
			 * evaluated into word lists when the recipes are
			 * instanciated.
			 */
			if (el2wl(&r.r_target, &sp->s_recipe.sr_target))
			{
				status = STMT_ERROR;
				break;
			}
			if (r.r_target.wl_nwords == 0)
			{
				error
				(
			"%s: %d: attempt to instanciate recipe with no targets",
					sp->s_recipe.sr_position.pos_name
						->str_text,
					sp->s_recipe.sr_position.pos_line
				);
				option_set_errors();
				status = STMT_ERROR;
				break;
			}
			el_copy(&r.r_need, &sp->s_recipe.sr_need);
			el_copy(&r.r_need2, &sp->s_recipe.sr_need2);
			r.r_precondition =
				(
					sp->s_recipe.sr_precondition
				?
					expr_copy(sp->s_recipe.sr_precondition)
				:
					(expr*)0
				);
			r.r_flags = sp->s_recipe.sr_flags;
			r.r_action =
				(
					sp->s_recipe.sr_action
				?
					stmt_copy(sp->s_recipe.sr_action)
				:
					(stmt*)0
				);
			r.r_use_action =
				(
					sp->s_recipe.sr_use_action
				?
					stmt_copy(sp->s_recipe.sr_use_action)
				:
					(stmt*)0
				);
			r.r_tag = recipe_tag();
			r.r_multiple = sp->s_recipe.sr_multiple;

			/*
			 * is it implicit or explicit?
			 */
			imp = 0;
			for (j = 0; j < r.r_target.wl_nwords; ++j)
			{
				if (strchr(r.r_target.wl_word[j]->str_text, MATCH_CHAR))
				{
					imp = 1;
					break;
				}
			}

			/*
			 * add it to the list
			 */
			if (imp)
			{
				rl_append(&implicit, &r);
#if 0
				if (!r.r_need.el_nexprs)
				{
					error
					(
				     "%s: %d: implicit recipe must have action",
						sp->s_recipe.sr_position
							.pos_name->str_text,
						sp->s_recipe.sr_position
							.pos_line
					);
					option_set_errors();
				}
#endif
			}
			else
				rl_append(&explicit, &r);

			/*
			 * emit trace information, if enabled
			 */
			if (option_test(OPTION_TRACE))
			{
				error
				(
				    "%s: %d: %s recipe %d instanciated (trace)",
					sp->s_recipe.sr_position.pos_name
						->str_text,
					sp->s_recipe.sr_position.pos_line,
					imp ? "implicit" : "explicit",
					r.r_tag
				);
			}
		}
		break;

	case OP_TOUCH:
		{
			wlist	 wl;

			/*
			 * The grammar has the command as an expression list.
			 * Evaluate the expression list into a word list.
			 * This word list is the command to execute.
			 */
			if (el2wl(&wl, &sp->s_cmd.c_args))
			{
				status = STMT_ERROR;
				break;
			}
			if (!option_test(OPTION_SILENT))
			{
				string_ty *s;

				/*
				 * If the command has not been silenced,
				 * form it into a string and echo it.
				 */
				s = wl2str(&wl, 0, wl.wl_nwords - 1, (char *)0);
				error("touch %s", s->str_text);
				str_free(s);
			}
			if (option_test(OPTION_ACTION))
			{
				int		j;

				for (j = 0; j < wl.wl_nwords; j++)
					if (os_touch(wl.wl_word[j]))
						status = STMT_ERROR;
			}
			wl_free(&wl);
		}
		break;
	}
ret:
	trace(("return %d;\n", status));
	trace((/*{*/"}\n"));
	return status;
}


/*
 *  NAME
 *	sl_append - append to a statement list
 *
 *  SYNOPSIS
 *	void sl_append(slist *slp, stmt *sp);
 *
 *  DESCRIPTION
 *	Sl_append is used to append a statement to a statement list.
 */

void
sl_append(sl, s)
	slist		*sl;
	stmt		*s;
{
	size_t		nbytes;

	trace(("sl_append(sl = %08X, s = %08X)\n{\n"/*}*/, sl, s));
	nbytes = (sl->sl_nstmts + 1) * sizeof(stmt *);
	sl->sl_stmt = mem_change_size(sl->sl_stmt, nbytes);
	sl->sl_stmt[sl->sl_nstmts++] = stmt_copy(s);
	trace((/*{*/"}\n"));
}


/*
 *  NAME
 *      sl_free - free statement list
 *
 *  SYNOPSIS
 *      void sl_free(slist *);
 *
 *  DESCRIPTION
 *      The sl_free function is used to free the list of statement trees.
 *
 *  RETURNS
 *      void
 */

void
sl_free(sl)
	slist		*sl;
{
	int		j;

	trace(("sl_free(sl = %08X)\n{\n"/*}*/, sl));
	for (j = 0; j < sl->sl_nstmts; ++j)
		stmt_free(sl->sl_stmt[j]);
	if (sl->sl_nstmts)
		mem_free(sl->sl_stmt);
	sl->sl_nstmts = 0;
	sl->sl_stmt = 0;
	trace((/*{*/"}\n"));
}

void
sl_zero(slp)
	slist		*slp;
{
	slp->sl_nstmts = 0;
	slp->sl_stmt = 0;
}
