/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate the stat cache
 */

#include <errno.h>
#include <ac/string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <archive.h>
#include <error.h>
#include <fngrprnt.h>
#include <mem.h>
#include <option.h>
#include <stat.cache.h>
#include <symtab.h>
#include <trace.h>


typedef struct cache_ty cache_ty;
struct cache_ty
{
	time_t	oldest;
	time_t	newest;
};

static symtab_ty *symtab;


static void init _((void));

static void
init()
{
	trace(("init()\n{\n"/*}*/));
	if (!symtab)
	{
		symtab = symtab_alloc(100);
		symtab->reap = mem_free;
	}
	trace((/*{*/"}\n"));
}


static cache_ty *mem_copy_cache _((cache_ty *));

static cache_ty *
mem_copy_cache(st)
	cache_ty	*st;
{
	cache_ty	*result;

	result = mem_alloc(sizeof(cache_ty));
	*result = *st;
	return result;
}


/*
 * NAME
 *	stat_cache - stat() with caching
 *
 * SYNOPSIS
 *	int stat_cache(string_ty *path, struct stat *result);
 *
 * DESCRIPTION
 *	The stat_cache function is used to perform the same as the stat()
 *	system function, but the results are cached to avoid too many probes
 *	into the file system.  Files which do not exist are indicated by
 *	filling the result structure with zeros.
 *
 * RETURNS
 *	int; -1 on error, 0 on success
 *
 * CAVEAT
 *	Errors, other than ENOENT, result in a fatal diagnostic.
 */

static int stat_cache _((string_ty *, cache_ty *));

static int
stat_cache(path, cp)
	string_ty	*path;
	cache_ty	*cp;
{
	cache_ty	*data;
	int		err;
	struct stat	st;

	/*
	 * if we have previously stat()ed this file,
	 * return old information
	 */
	trace(("stat_cache(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (!symtab)
		init();
	data = symtab_query(symtab, path);
	if (data)
	{
		*cp = *data;
		trace(("return 0;\n"));
		trace((/*{*/"}\n"));
		return 0;
	}

	/*
	 * new file, perform stat() for the first time
	 */
	trace(("stat(\"%s\")\n", path->str_text));
	err = stat(path->str_text, &st);
	if (err && errno == ENOENT)
		err = archive_stat(path, &st);
	if (err)
	{
		switch (errno)
		{
		case ENOENT:
		case ENOTDIR:
			/*
			 * ENOENT occurs when a path element does not exist
			 * ENOTDIR occurs when a path element (except the last)
			 *		is not a directory.
			 * Either way, the file being "stat"ed does not exist.
			 */
			break;

		default:
			nerror("stat(\"%s\")", path->str_text);
			trace(("return -1;\n"));
			trace((/*{*/"}\n"));
			return -1;
		}

		fp_delete(path);

		cp->newest = 0;
		cp->oldest = 0;
	}
	else
	{
		fp_ty	*fp;

		/*
		 * make sure the times of existing files
		 * are always positive
		 */
		if (st.st_mtime < 1)
			st.st_mtime = 1;
		cp->oldest = st.st_mtime;
		cp->newest = st.st_mtime;
		
		/*
		 * see if we have its fingerprint on file
		 */
		if (option_test(OPTION_FINGERPRINT))
		{
			fp = fp_search(path);
			if (fp)
			{
				/*
				 * we have seen this file before
				 */
				if (fp->newest != cp->newest)
				{
					fp_ty	data;

					/*
					 * but it has changed
					 * since we last saw it
					 */
					data.fingerprint = fp_fingerprint(path);
					if (!data.fingerprint)
						goto fp_not_useful;
					if (str_equal(fp->fingerprint, data.fingerprint))
					{
						/*
						 * the fingerprint is the same,
						 * so give the oldest mtime
						 * known
						 */
						data.oldest = fp->oldest;
						data.newest = cp->newest;
						cp->oldest = fp->oldest;
						if (option_test(OPTION_TRACE))
						{
							struct tm	*tm;

							tm = localtime(&data.newest);
							error
							(
  "mtime(\"%s\") was %4d/%02d/%02d.%02d:%02d:%02d until fingerprinting (trace)",
								path->str_text,
							     1900 + tm->tm_year,
								tm->tm_mon + 1,
								tm->tm_mday,
								tm->tm_hour,
								tm->tm_min,
								tm->tm_sec
							);
						}
					}
					else
					{
						/*
						 * the fingerprint differs
						 * do not lie about mtime
						 */
						data.oldest = cp->newest;
						data.newest = cp->newest;
					}
					fp_assign(path, &data);
					str_free(data.fingerprint);
				}
				else
				{
					/*
					 * file not modified since last seen
					 */
					cp->oldest = fp->oldest;
				}
			}
			else
			{
				fp_ty	data;

				/*
				 * never fingerprinted this file before
				 */
				data.oldest = cp->newest;
				data.newest = cp->newest;
				data.fingerprint = fp_fingerprint(path);
				if (!data.fingerprint)
				{
					fp_not_useful:
					fp_delete(path);
				}
				else
				{
					fp_assign(path, &data);
					str_free(data.fingerprint);
				}
			}
		}
	}

	/*
	 * remember the stat information
	 */
	symtab_assign(symtab, path, mem_copy_cache(cp));
	trace(("return 0;\n"));
	trace((/*{*/"}\n"));
	return 0;
}


time_t
stat_cache_newest(path)
	string_ty	*path;
{
	cache_ty	cache;

	if (stat_cache(path, &cache))
		return -1;

	/*
	 * trace the last-modified time
	 */
	if (option_test(OPTION_TRACE))
	{
		if (!cache.newest)
		{
			error
			(
				"mtime(\"%s\") == ENOENT (trace)",
				path->str_text
			);
		}
		else
		{
			struct tm	*tm;
			char		*relage;

			relage = "";
			if
			(
				option_test(OPTION_FINGERPRINT)
			&&
				cache.newest != cache.oldest
			)
				relage = "newest ";
			tm = localtime(&cache.newest);
			error
			(
		      "%smtime(\"%s\") == %4d/%02d/%02d.%02d:%02d:%02d (trace)",
				relage,
				path->str_text,
				1900 + tm->tm_year,
				tm->tm_mon + 1,
				tm->tm_mday,
				tm->tm_hour,
				tm->tm_min,
				tm->tm_sec
			);
		}
	}
	return cache.newest;
}


time_t
stat_cache_oldest(path)
	string_ty	*path;
{
	cache_ty	cache;

	if (stat_cache(path, &cache))
		return -1;

	/*
	 * trace the last-modified time
	 */
	if (option_test(OPTION_TRACE))
	{
		if (!cache.oldest)
		{
			error
			(
				"mtime(\"%s\") == ENOENT (trace)",
				path->str_text
			);
		}
		else
		{
			struct tm	*tm;
			char		*relage;

			relage = "";
			if
			(
				option_test(OPTION_FINGERPRINT)
			&&
				cache.newest != cache.oldest
			)
				relage = "oldest ";
			tm = localtime(&cache.oldest);
			error
			(
		      "%smtime(\"%s\") == %4d/%02d/%02d.%02d:%02d:%02d (trace)",
				relage,
				path->str_text,
				1900 + tm->tm_year,
				tm->tm_mon + 1,
				tm->tm_mday,
				tm->tm_hour,
				tm->tm_min,
				tm->tm_sec
			);
		}
	}
	return cache.oldest;
}


void
stat_cache_set(path, when, fp2)
	string_ty	*path;
	time_t		when;
	int		fp2;
{
	cache_ty	*data;
	cache_ty	cache;

	trace(("stat_cache_set(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (!symtab)
		init();
	data = symtab_query(symtab, path);
	if (data)
	{
		if (!data->oldest || !option_test(OPTION_FINGERPRINT))
			data->oldest = when;
		data->newest = when;
	}
	else
	{
		cache.oldest = when;
		cache.newest = when;
		symtab_assign(symtab, path, mem_copy_cache(&cache));
		data = &cache;
	}

	/*
	 * Update the fingerprint.
	 * (Important not to lie here, the fp2 flags
	 * says is immediately following a utime call.)
	 */
	if (fp2 && option_test(OPTION_FINGERPRINT))
	{
		fp_ty		*fp;

		fp = fp_search(path);
		if (fp)
		{
			fp_ty		data;

			data.oldest = fp->oldest;
			data.newest = when;
			data.fingerprint = str_copy(fp->fingerprint);
			fp_assign(path, &data);
			str_free(data.fingerprint);
		}
	}

	/*
	 * emit a trace
	 */
	if (option_test(OPTION_TRACE))
	{
		struct tm	*tm;
		char		*relage;

		relage = "";
		if
		(
			option_test(OPTION_FINGERPRINT)
		&&
			data->oldest != data->newest
		)
			relage = "newest ";
		tm = localtime(&when);
		error
		(
		       "%smtime(\"%s\") = %4d/%02d/%02d.%02d:%02d:%02d (trace)",
			relage,
			path->str_text,
			1900 + tm->tm_year,
			tm->tm_mon + 1,
			tm->tm_mday,
			tm->tm_hour,
			tm->tm_min,
			tm->tm_sec
		);
	}
	trace((/*{*/"}\n"));
}


void
stat_cache_clear(path)
	string_ty	*path;
{
	trace(("stat_cache_clear(path =\"%s\")\n{\n"/*}*/, path->str_text));
	if (!symtab)
		init();
	symtab_delete(symtab, path);
	trace((/*{*/"}\n"));
}
