/*
 *	cook - file construction tool
 *	Copyright (C) 1990, 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to parse cookbooks
 */

%token	CATENATE
%token	COLON
%token	DATA
%token	DATAEND
%token	ELSE
%token	EQUALS
%token	FAIL
%token	IF
%token	LBRACE
%token	LBRAK
%token	LOOP
%token	LOOPSTOP
%token	RBRACE
%token	RBRAK
%token	SEMICOLON
%token	SET
%token	THEN
%token	UNSETENV
%token	WORD

%right	ELSE
%left	CATENATE

%{
#include <ac/stddef.h>
#include <ac/stdlib.h>
#include <stdio.h>

#include <cook.h>
#include <error.h>
#include <expr.h>
#include <lex.h>
#include <mem.h>
#include <option.h>
#include <parse.h>
#include <stmt.h>
#include <symtab.h>
#include <trace.h>
#include <word.h>

#ifdef	DEBUG
#define YYDEBUG 1
#define printf trace_where(__FILE__, __LINE__), lex_trace
extern int yydebug;
#endif

static	stmt    *looptemp;
static symtab_ty *flags_symtab;


/*
 *  NAME
 *      parse_initialize - start up parser
 *
 *  SYNOPSIS
 *      void parse_initialize(void);
 *
 *  DESCRIPTION
 *      The parse_initialize function is used to add the set clause names to
 *      the symbol table.
 *
 *  RETURNS
 *      void
 *
 *  CAVEAT
 *      Must be called after symbol table is initialized.
 *      Must be called before the parse function is used.
 */

void
parse_initialize()
{
	typedef struct table_ty table_ty;
	struct table_ty
	{
		char	*t_name;
		int	t_mask;
	};

	static table_ty table[] =
	{
		{ "clearstat",		RF_CLEARSTAT,		},
		{ "noclearstat",	RF_CLEARSTAT_OFF,	},
		{ "default",		RF_DEFAULT,		},
		{ "nodefault",		RF_DEFAULT_OFF,		},
		{ "errok",		RF_ERROK,		},
		{ "noerrok",		RF_ERROK_OFF,		},
		{ "fingerprint",	RF_FINGERPRINT,		},
		{ "fingerprinting",	RF_FINGERPRINT,		},
		{ "nofingerprint",	RF_FINGERPRINT_OFF,	},
		{ "nofingerprinting",	RF_FINGERPRINT_OFF,	},
		{ "force",		RF_FORCE,		},
		{ "forced",		RF_FORCE,		},
		{ "noforce",		RF_FORCE_OFF,		},
		{ "noforced",		RF_FORCE_OFF,		},
		{ "meter",		RF_METER,		},
		{ "nometer",		RF_METER_OFF,		},
		{ "precious",		RF_PRECIOUS,		},
		{ "noprecious",		RF_PRECIOUS_OFF,	},
		{ "silent",		RF_SILENT,		},
		{ "nosilent",		RF_SILENT_OFF,		},
		{ "stripdot",		RF_STRIPDOT,		},
		{ "nostripdot",		RF_STRIPDOT_OFF,	},
		{ "update",		RF_UPDATE,		},
		{ "noupdate",		RF_UPDATE_OFF,		},
	};

	table_ty *tp;

	trace(("parse_initialize()\n{\n"/*}*/));
	if (!flags_symtab)
		flags_symtab = symtab_alloc(SIZEOF(table));
	for (tp = table; tp < ENDOF(table); ++tp)
	{
		string_ty *s;

		s = str_from_c(tp->t_name);
		symtab_assign(flags_symtab, s, &tp->t_mask);
		str_free(s);
	}
	trace((/*{*/"}\n"));
}


/*
 *  NAME
 *	parse - read and process a cookbook
 *
 *  SYNOPSIS
 *	void parse(string_ty *filename);
 *
 *  DESCRIPTION
 *	Parse reads and processes a cookbook.
 *
 *  CAVEAT
 *	If any errors are found, the user will be told,
 *	and this routine will not return.
 */

void
parse(filename)
	string_ty *filename;
{
	int yyparse _((void)); /* forward */

	trace(("parse(filename = %08lX)\n{\n"/*}*/, filename));
	trace_string(filename->str_text);
	lex_open(filename);
#if YYDEBUG
	yydebug = trace_pretest_;
#endif
	yyparse();
	lex_close();
	trace((/*{*/"}\n"));
}

#define yyerror lex_error

%}

%union
{
	expr		*lv_expr;
	stmt		*lv_stmt;
	elist		lv_elist;
	string_ty	*lv_word;
	int		lv_flag;
	position	lv_position;
}

%type	<lv_stmt>	statement compound_statement statements
%type	<lv_stmt>	command simple_command loop use_clause
%type	<lv_elist>	elist exprs
%type	<lv_word>	WORD
%type	<lv_expr>	expr if_clause
%type	<lv_flag>	set_clause data lbrak mult
%type	<lv_position>	COLON

%%

cook
	: /* empty */
	| cook statement
		{
			if (stmt_eval($2))
				yyerror("statement failed");
			stmt_free($2);
		}
	| cook error
		{
			lex_mode(LM_NORMAL);
		}
	;

statement
	: compound_statement
		{
			$$ = $1;
		}
	| UNSETENV exprs SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_UNSETENV;
			$$->s_cmd.c_args = $2;
		}
	| elist EQUALS exprs SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_ASSIGN;
			$$->s_assign.a_name = $1;
			$$->s_assign.a_value = $3;
		}
	| elist COLON mult exprs set_clause if_clause compound_statement
			use_clause
		{
			$$ = stmt_alloc();
			$$->s_op = OP_RECIPE;
			$$->s_recipe.sr_target = $1;
			$$->s_recipe.sr_position = $2;
			$$->s_recipe.sr_multiple = $3;
			$$->s_recipe.sr_need = $4;
			el_zero(&$$->s_recipe.sr_need2);
			$$->s_recipe.sr_flags = $5;
			$$->s_recipe.sr_precondition = $6;
			$$->s_recipe.sr_action = $7;
			$$->s_recipe.sr_use_action = $8;
		}
	| elist COLON mult exprs COLON exprs set_clause if_clause
			compound_statement use_clause
		{
			$$ = stmt_alloc();
			$$->s_op = OP_RECIPE;
			$$->s_recipe.sr_target = $1;
			$$->s_recipe.sr_position = $2;
			$$->s_recipe.sr_multiple = $3;
			$$->s_recipe.sr_need = $4;
			$$->s_recipe.sr_need2 = $6;
			$$->s_recipe.sr_flags = $7;
			$$->s_recipe.sr_precondition = $8;
			$$->s_recipe.sr_action = $9;
			$$->s_recipe.sr_use_action = $10;
			str_free($5.pos_name);
		}
	| elist COLON mult exprs set_clause if_clause SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_RECIPE;
			$$->s_recipe.sr_target = $1;
			$$->s_recipe.sr_position = $2;
			$$->s_recipe.sr_need = $4;
			el_zero(&$$->s_recipe.sr_need2);
			$$->s_recipe.sr_flags = $5;
			$$->s_recipe.sr_precondition = $6;
			$$->s_recipe.sr_action = 0;
			$$->s_recipe.sr_use_action = 0;
			$$->s_recipe.sr_multiple = $3;
		}
	;

mult
	: /* empty */
		{
			$$ = 0;
		}
	| COLON
		{
			$$ = 1;
			str_free($1.pos_name);
		}
	;

set_clause
	: /* empty */
		{
			$$ = 0;
		}
	| SET exprs
		{
			int	j;
			wlist	wl;

			wl_zero(&wl);
			if (el2wl(&wl, &$2))
			{
				/* error message already emitted */
				wl_free(&wl);
			}
			el_free(&$2);
			$$ = 0;
			if (!wl.wl_nwords)
				yyerror("set clause has no flags");
			for (j = 0; j < wl.wl_nwords; ++j)
			{
				int	*data;

				assert(flags_symtab);
				data = symtab_query(flags_symtab, wl.wl_word[j]);
				if (data)
					$$ |= *data;
				else
				{
					yyerror
					(
				   "set clause does not understand \"%s\" flag",
						wl.wl_word[j]->str_text
					);
				}
			}
			wl_free(&wl);

			if ((($$ & 0xAAAAAAAA) >> 1) & $$)
				yyerror("may not set both a flag and its negative");
		}
	;

if_clause
	: /* empty */
		{
			$$ = 0;
		}
	| IF expr
		{
			$$ = $2;
		}
	;

use_clause
	: /* empty */
		{
			$$ = 0;
		}
	| THEN compound_statement
		{
			$$ = $2;
		}
	;

statement
	: command
		{
			$$ = $1;
		}
	| IF expr THEN statement
		%prec ELSE
		{
			$$ = stmt_alloc();
			$$->s_op = OP_IF;
			$$->s_if.sif_cond = $2;
			$$->s_if.sif_true = $4;
			$$->s_if.sif_false = stmt_alloc();
			$$->s_if.sif_false->s_op = OP_NOP;
		}
	| IF expr THEN statement ELSE statement
		{
			$$ = stmt_alloc();
			$$->s_op = OP_IF;
			$$->s_if.sif_cond = $2;
			$$->s_if.sif_true = $4;
			$$->s_if.sif_false = $6;
		}
	| loop statement
		{
			$$ = looptemp;
			looptemp = $1;
			$$->s_op = OP_LOOP;
			$$->s_loop = $2;
		}
	| LOOPSTOP SEMICOLON
		{
			$$ = stmt_alloc();
			if (!looptemp)
			{
				yyerror("'loopstop' encountered outside a loop");
				$$->s_op = OP_NOP;
			}
			else
				$$->s_op = OP_LOOPSTOP;
		}
	| SET exprs SEMICOLON
		{
			int	j;
			wlist   wl;

			$$ = stmt_alloc();
			$$->s_op = OP_SET;
			$$->s_cmd.c_flags = 0;

			wl_zero(&wl);
			if (el2wl(&wl, &$2))
			{
				/* error message already emitted */
				wl_free(&wl);
			}
			el_free(&$2);
			if (!wl.wl_nwords)
				yyerror("set statement has no flags");
			for (j = 0; j < wl.wl_nwords; ++j)
			{
				int	*data;

				assert(flags_symtab);
				data = symtab_query(flags_symtab, wl.wl_word[j]);
				if (data)
					$$->s_cmd.c_flags |= *data;
				else
				{
					yyerror
					(
				"set statement does not understand \"%s\" flag",
						wl.wl_word[j]->str_text
					);
				}
			}
			wl_free(&wl);
		}
	| FAIL SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_FAIL;
		}
	| FAIL WORD SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_FAIL_DK;
		}
	| SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_NOP;
		}
	;

loop
	: LOOP
		{
			$$ = looptemp;
			looptemp = stmt_alloc();
		}
	;

compound_statement
	: LBRACE statements RBRACE
		{
			$$ = $2;
		}
	;

statements
	: /* empty */
		{
			$$ = stmt_alloc();
			$$->s_op = OP_COMPOUND;
		}
	| statements statement
		{
			$$ = $1;
			sl_append(&$$->s_list, $2);
			stmt_free($2);
		}
	| statements error
		{
			$$ = $1;
		}
	;

elist
	: expr
		{
			el_zero(&$$);
			el_append(&$$, $1);
			expr_free($1);
		}
	| elist expr
		{
			$$ = $1;
			el_append(&$$, $2);
			expr_free($2);
		}
	;

exprs
	: /* empty */
		{
			el_zero(&$$);
		}
	| exprs expr
		{
			$$ = $1;
			el_append(&$$, $2);
			expr_free($2);
		}
	;

expr
	: WORD
		{
			$$ = expr_alloc();
			$$->e_op = OP_WORD;
			$$->e_word = $1;
		}
	| lbrak elist RBRAK
		{
			lex_mode($1);
			$$ = expr_alloc();
			$$->e_op = OP_FUNC;
			$$->e_list = $2;
		}
	| expr CATENATE expr
		{
			$$ = expr_alloc();
			$$->e_op = OP_CAT;
			$$->e_left = $1;
			$$->e_right = $3;
		}
	;

command
	: simple_command
		{
			$$ = $1;
		}
	| simple_command data expr DATAEND
		{
			lex_mode($2);
			$$ = $1;
			$$->s_cmd.c_input = $3;
		}
	;

simple_command
	: elist set_clause SEMICOLON
		{
			$$ = stmt_alloc();
			$$->s_op = OP_COMMAND;
			$$->s_cmd.c_args = $1;
			$$->s_cmd.c_flags = $2;
		}
	;

data
	: DATA
		{
			$$ = lex_mode(LM_DATA);
		}
	;

lbrak
	: LBRAK
		{
			$$ = lex_mode(LM_NORMAL);
		}
	;
