/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to isolate operating system interface
 */

#include <ac/stddef.h>
#include <ac/string.h>
#include <ac/stdlib.h>
#include <ac/limits.h>
#include <errno.h>
#include <stdio.h>
#include <signal.h>
#include <ac/time.h>
#include <ac/unistd.h>

#include <archive.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <utime.h>

#include <error.h>
#include <mem.h>
#include <option.h>
#include <os.h>
#include <stat.cache.h>
#include <trace.h>
#include <word.h>

#ifdef HAVE_GETRUSAGE
#include <sys/resource.h>
#else
#include <sys/times.h>
#endif


/*
 * NAME
 *	os_mtime - return the last-modified time of a file
 *
 * SYNOPSIS
 *	time_t os_mtime(string_ty *path);
 *
 * DESCRIPTION
 *	Os_mtime returns the time the named file was last modified.
 *	It returns 0 if the file does not exist.
 *
 * CAVEAT
 *	Assumes time will be the UNIX format.
 */

time_t
os_mtime_oldest(path)
	string_ty	*path;
{
	return stat_cache_oldest(path);
}

time_t
os_mtime_newest(path)
	string_ty	*path;
{
	return stat_cache_newest(path);
}


/*
 * NAME
 *	os_mtime_adjust - indicate change
 *
 * SYNOPSIS
 *	int os_mtime_adjust(string_ty *path);
 *
 * DESCRIPTION
 *	The os_mtime_adjust function is used to adjust the value in the stat
 *	cache to indicate that a recipe has constructed a file, and thus
 *	changed it last-modified time.  No change to the actual file system
 *	occurs.
 *
 * RETURNS
 *	int; -1 on error, 0 on success
 */

int
os_mtime_adjust(path, min_age)
	string_ty	*path;
	time_t		min_age;
{
	time_t		mtime;
	int		err;

	if (option_test(OPTION_UPDATE))
	{
		stat_cache_clear(path);
		mtime = os_mtime_newest(path);
		if (mtime < 0)
			return -1;
		if (mtime)
		{
			if (mtime < min_age)
			{
				struct utimbuf	ut;

				if (!option_test(OPTION_SILENT))
				{
					long		nsec;

					nsec = min_age - mtime;
					error
					(
					"adjusting \"%s\" forward %ld second%s",
						path->str_text,
						nsec,
						(nsec == 1 ? "" : "s")
					);
				}
				ut.modtime = min_age;
				ut.actime = min_age;
				err = utime(path->str_text, &ut);
				if (err && errno == ENOENT)
					err = archive_utime(path, &ut);
				if (err)
				{
					nerror("utime(\"%s\")", path->str_text);
					option_set_errors();
					return -1;
				}
				stat_cache_set(path, min_age, 1);
			}
		}
		else
		{
			/*
			 * file was deleted (or was a dummy)
			 * so pretend it was changed "now"
			 */
			time(&mtime);
			if (mtime > min_age)
				min_age = mtime;
			stat_cache_set(path, min_age, 0);
		}
	}
	else
	{
		time(&mtime);
		if (mtime > min_age)
			min_age = mtime;
		stat_cache_set(path, min_age, 0);
	}
	return 0;
}


/*
 * NAME
 *	os_clear_stat - invalidate cache
 *
 * SYNOPSIS
 *	int os_clear_stat(string_ty *path);
 *
 * DESCRIPTION
 *	The os_clear_stat function is used to invalidate the the stat
 *	cache to indicate that a recipe has constructed a file, and thus
 *	changed it last-modified time.  No change to the actual file system
 *	occurs.
 *
 * RETURNS
 *	int; 0 on success, -1 on error
 *
 * CAVEAT
 *	This is used in situations where the recipe changes a file not named
 *	in the targets of the recipe.  This usually occurs around mkdir, rm
 *	and mv commands, used in conjunction with the [exists] builtin function.
 */

int
os_clear_stat(path)
	string_ty	*path;
{
	stat_cache_clear(path);
	return 0;
}


/*
 * NAME
 *	os_touch - update the modify time of the file
 *
 * SYNOPSIS
 *	int os_touch(string_ty *path);
 *
 * DESCRIPTION
 *	Os_touch updates the last-modified time of the file to the present.
 *	If the named file does not exist, then nothing is done.
 *
 * RETURNS
 *	int; 0 on success, -1 on error
 */

int
os_touch(path)
	string_ty	*path;
{
	struct utimbuf	ut;
	int		err;

	time(&ut.modtime);
	if (ut.modtime < 0)
		ut.modtime = 0;
	ut.actime = ut.modtime;
	err = utime(path->str_text, &ut);
	if (err && errno == ENOENT)
		err = archive_utime(path, &ut);
	if (err)
	{
		if (errno == ENOENT)
		{
			stat_cache_clear(path);
			return 0;
		}
		nerror("utime(\"%s\")", path->str_text);
		option_set_errors();
		return -1;
	}
	stat_cache_set(path, ut.modtime, 1);
	return 0;
}


/*
 * NAME
 *	os_delete - delete a file
 *
 * SYNOPSIS
 *	int os_delete(string_ty *path);
 *
 * DESCRIPTION
 *	Os_delete deletes the named file.
 *	If it does not exist, no error is given.
 *
 * RETURNS
 *	int; -1 on error, 0 on success
 */

int
os_delete(path)
	string_ty	*path;
{
	if (unlink(path->str_text) && errno != ENOENT)
	{
		nerror("unlink(\"%s\")", path->str_text);
		option_set_errors();
		return -1;
	}

	/*
	 * if the knew about the existence of the file before we deleted
	 * it, then we will have to adjust the stat cache.
	 */
	stat_cache_clear(path);
	return 0;
}


/*
 * NAME
 *	signal_name - find it
 *
 * SYNOPSIS
 *	char *signal_name(int n);
 *
 * DESCRIPTION
 *	The signal_name function is used to find the name of a signal from its
 *	number.
 *
 * RETURNS
 *	char *: pointer to the signal name.
 *
 * CAVEAT
 *	The signal name may not be written on.  Subsequent calls may alter the
 *	area pointed to.
 */

char *
signal_name(n)
	int		n;
{
	static char	buffer[16];

	switch (n)
	{
#ifdef SIGHUP
	case SIGHUP:
		return "hang up [SIGHUP]";
#endif /* SIGHUP */

#ifdef SIGINT
	case SIGINT:
		return "user interrupt [SIGINT]";
#endif /* SIGINT */

#ifdef SIGQUIT
	case SIGQUIT:
		return "user quit [SIGQUIT]";
#endif /* SIGQUIT */

#ifdef SIGILL
	case SIGILL:
		return "illegal instruction [SIGILL]";
#endif /* SIGILL */

#ifdef SIGTRAP
	case SIGTRAP:
		return "trace trap [SIGTRAP]";
#endif /* SIGTRAP */

#ifdef SIGIOT
	case SIGIOT:
		return "abort [SIGIOT]";
#endif /* SIGIOT */

#ifdef SIGEMT
	case SIGEMT:
		return "EMT instruction [SIGEMT]";
#endif /* SIGEMT */

#ifdef SIGFPE
	case SIGFPE:
		return "floating point exception [SIGFPE]";
#endif /* SIGFPE */

#ifdef SIGKILL
	case SIGKILL:
		return "kill [SIGKILL]";
#endif /* SIGKILL */

#ifdef SIGBUS
	case SIGBUS:
		return "bus error [SIGBUS]";
#endif /* SIGBUS */

#ifdef SIGSEGV
	case SIGSEGV:
		return "segmentation violation [SIGSEGV]";
#endif /* SIGSEGV */

#ifdef SIGSYS
	case SIGSYS:
		return "bad argument to system call [SIGSYS]";
#endif /* SIGSYS */

#ifdef SIGPIPE
	case SIGPIPE:
		return "write on a pipe with no one to read it [SIGPIPE]";
#endif /* SIGPIPE */

#ifdef SIGALRM
	case SIGALRM:
		return "alarm clock [SIGALRM]";
#endif /* SIGALRM */

#ifdef SIGTERM
	case SIGTERM:
		return "software termination [SIGTERM]";
#endif /* SIGTERM */

#ifdef SIGUSR1
	case SIGUSR1:
		return "user defined signal one [SIGUSR1]";
#endif /* SIGUSR1 */

#ifdef SIGUSR2
	case SIGUSR2:
		return "user defined signal two [SIGUSR2]";
#endif /* SIGUSR2 */

#ifdef SIGCLD
	case SIGCLD:
		return "death of child [SIGCLD]";
#endif /* SIGCLD */

#ifdef SIGPWR
	case SIGPWR:
		return "power failure [SIGPWR]";
#endif /* SIGPWR */

	default:
		sprintf(buffer, "signal %d", n);
		return buffer;
	}
}


/*
 * NAME
 *	exit_status - pretty print
 *
 * SYNOPSIS
 *	int exit_status(char *cmd, int status, int errok);
 *
 * DESCRIPTION
 *	The exit_status function is used to pretty print the meaning of the
 *	exit status of the given command.  No output is generated for normal
 *	(0) termination.
 *
 * RETURNS
 *	int: zero if the command succeeded, 1 if it failed.
 *
 * CAVEAT
 *	This function should be static, but func_collect (builtin.c) uses it.
 */

int
exit_status(cmd, status, errok)
	char		*cmd;
	int		status;
	int		errok;
{
	int		a, b, c;

	a = (status >> 8) & 0xFF;
	b = (status >> 7) & 1;
	c = status & 0x7F;
	switch (c)
	{
	case 0x7F:
		/*
		 * process was stopped,
		 * since we didn't do it, treat it as an error
		 */
		error("%s: stopped", cmd);
		return 1;

	case 0:
		/* normal termination */
		if (a)
		{
			error
			(
				"%s: exit status: %d%s",
				cmd,
				a,
				(errok ? " (ignored)" : "")
			);
			if (errok)
				return 0;
			return 1;
		}
		return 0;

	default:
		/*
		 * process dies from unhandled condition
		 */
		error
		(
			"%s: terminated by %s%s",
			cmd,
			signal_name(c),
			(b ? " (core dumped)" : "")
		);
		return 1;
	}
}


/*
 * NAME
 *	execute - do a command
 *
 * SYNOPSIS
 *	int execute(wlist *cmd, int fd, int errok);
 *
 * DESCRIPTION
 *	The execute function is used to execute the command in the word list.
 *	If the file descriptor is >= 0, it indicates a file to use as stdin to
 *	the command.
 *
 * RETURNS
 *	int: zero if the commands succeeds, nonzero if it fails.
 *
 * CAVEAT
 */

static int execute _((wlist *, int, int));

static int
execute(cmd, fd, errok)
	wlist		*cmd;
	int		fd;
	int		errok;
{
	int		child;
	int		pid;
	int		status;
	static char	**argv;
	static size_t	argvlen;
	int		j;

	if (!argv)
	{
		argvlen = cmd->wl_nwords + 1;
		argv = mem_alloc(argvlen * sizeof(char*));
	}
	else
	{
		if (argvlen < cmd->wl_nwords + 1)
		{
			argvlen = cmd->wl_nwords + 1;
			argv = mem_change_size(argv, argvlen * sizeof(char *));
		}
	}
	for (j = 0; j < cmd->wl_nwords; ++j)
		argv[j] = cmd->wl_word[j]->str_text;
	argv[cmd->wl_nwords] = 0;
	switch (child = fork())
	{
	case -1:
		nerror("fork()");
		return -1;

	case 0:
		if (fd >= 0)
		{
			if (close(0) && errno != EBADF)
				nfatal("close(0)");
			if (dup(fd) < 0)
				nfatal("dup()");
		}
		if (argv[0][0] == '/')
			execv(argv[0], argv);
		else
			execvp(argv[0], argv);
		nfatal("exec(\"%s\")", argv[0]);

	default:
		for (;;)
		{
			pid = wait(&status);
			if (pid == child)
				return exit_status(argv[0], status, errok);
			if (pid < 0 && errno != EINTR)
			{
				nerror("wait()");
				option_set_errors();
				return -1;
			}
		}
	}
}


/*
 * NAME
 *	os_execute - execute a command
 *
 * SYNOPSIS
 *	int os_execute(wlist *args, string_ty *input, int errok);
 *
 * DESCRIPTION
 *	Os_execute performs the given command.
 *	If the command succeeds 0 is returned.
 *	If the command fails a diagnostic message is given
 *	and 1 is returned.
 */

int
os_execute(args, input, errok)
	wlist		*args;
	string_ty	*input;
	int		errok;
{
	int		j;
	FILE		*fp;
	char		iname[L_tmpnam];
	int		retval;

	assert(args);
	assert(args->wl_nwords > 0);

	fp = 0;
	if (input)
	{
		/*
		 * He has given a string to be used as input to the command,
		 * so write it out to a file, and then redirect the input.
		 */
		tmpnam(iname);
		fp = fopen(iname, "w");
		if (!fp)
		{
			nerror("create %s", iname);
			exec_fails:
			option_set_errors();
			retval = -1;
			goto ret;
		}
		if (unlink(iname))
		{
			nerror("unlink(\"%s\")", iname);
			goto exec_fails;
		}
		fputs(input->str_text, fp);
		if (ferror(fp) || fseek(fp, 0L, 0))
		{
			nerror("write %s", iname);
			goto exec_fails;
		}
	}

	for (j = 1; j < args->wl_nwords; j++)
	{
		char	*s;

		s = args->wl_word[j]->str_text;
		while (*s)
		{
			if (strchr("#|=^();&<>*?[]:$`'\"\\\n", *s))
			{
				string_ty	*str;
				wlist		newcmd;
				char		*cp;

				cp = getenv("SHELL");
				if (!cp || !*cp)
					cp = CONF_SHELL;
				wl_zero(&newcmd);
				str = str_from_c(cp);
				wl_append(&newcmd, str);
				str_free(str);
				if (option_test(OPTION_ERROK))
					str = str_from_c("-c");
				else
					str = str_from_c("-ce");
				wl_append(&newcmd, str);
				str_free(str);
				str =
					wl2str
					(
						args,
						0,
						args->wl_nwords - 1,
						(char *)0
					);
				wl_append(&newcmd, str);
				str_free(str);
				retval =
					execute
					(
						&newcmd,
						(input ? fileno(fp) : -1),
						errok
					);
				wl_free(&newcmd);
				if (input)
					fclose(fp);
				return retval;
			}
			++s;
		}
	}
	retval = execute(args, (input ? fileno(fp) : -1), errok);
	if (input)
		fclose(fp);
ret:
	return retval;
}


/*
 * NAME
 *	os_exists - tests for the existence of a file
 *
 * SYNOPSIS
 *	int os_exists(string_ty *path);
 *
 * DESCRIPTION
 *	The os_exists function is used to determine the existence of a file.
 *
 * RETURNS
 *	int; 1 if the file exists, 0 if it does not.  -1 on error
 */

int
os_exists(path)
	string_ty	*path;
{
	time_t		mtime;

	mtime = stat_cache_newest(path);
	if (mtime < 0)
		return -1;
	return (mtime != 0);
}


/*
 * NAME
 *	os_accdir - return the directory path of the users account
 *
 * SYNOPSIS
 *	string_ty *os_accdir(void);
 *
 * DESCRIPTION
 *	The full pathname of the user's account is returned.
 *	The string will have been dynamically allocated.
 *
 * RETURNS
 *	A pointer to a string in dynamic memory is returned.
 *	A null pointer is returned on error.
 *
 * CAVEAT
 *	Use str_free() when you are done with the value returned.
 */

string_ty *
os_accdir()
{
	static char	home[] = "HOME";
	static string_ty *s;

	if (!s)
	{
		char		*cp;

		cp = getenv(home);
		if (!cp)
		{
			error("%s: environment variable not defined", home);
			option_set_errors();
			return 0;
		}
		s = str_from_c(cp);
	}
	return str_copy(s);
}


/*
 * NAME
 *	os_curdir - full current directory path
 *
 * SYNOPSIS
 *	string_ty *os_curdir(void);
 *
 * DESCRIPTION
 *	Os_curdir is used to determine the full pathname
 *	of the current directory.
 *
 * RETURNS
 *	A pointer to a string in dynamic memory is returned.
 *	A null pointer is returned on error.
 *
 * CAVEAT
 *	Use str_free() when you are done with the value returned.
 */

string_ty *
os_curdir()
{
	static string_ty	*s;

	if (!s)
	{
		char		buffer[1200];

		if (!getcwd(buffer, sizeof(buffer)))
		{
			nerror("getcwd");
			option_set_errors();
			return 0;
		}
		s = str_from_c(buffer);
	}
	return str_copy(s);
}


/*
 * NAME
 *	os_pathname - determine full file name
 *
 * SYNOPSIS
 *	string_ty *os_pathname(string_ty *path);
 *
 * DESCRIPTION
 *	Os_pathname is used to determine the full path name
 *	of a partial path given.
 *
 * RETURNS
 *	pointer to dynamically allocated string.
 *
 * CAVEAT
 *	Use str_free() when you are done with the value returned.
 */

string_ty *
os_pathname(path)
	string_ty	*path;
{
	static char	*tmp;
	static size_t	tmplen;
	static size_t	ipos;
	static size_t	opos;
	int		c;
	int		found;
#ifdef S_IFLNK
	char		link[2000];
	int		nbytes;
	wlist		loop;
	string_ty	*s;
#endif

	/*
	 * Change relative pathnames to absolute
	 */
	trace(("os_pathname(path = %08lX)\n{\n"/*}*/, path));
	trace_string(path->str_text);
	if (path->str_text[0] != '/')
	{
		string_ty	*cwd;

		cwd = os_curdir();
		assert(cwd);
		path = str_format("%S/%S", cwd, path);
		str_free(cwd);
	}
	else
		path = str_copy(path);
	if (!tmp)
	{
		tmplen = 200;
		tmp = mem_alloc(tmplen);
	}

	/*
	 * Take kinks out of the pathname
	 */
	ipos = 0;
	opos = 0;
	found = 0;
#ifdef S_IFLNK
	wl_zero(&loop);
#endif
	while (!found)
	{
		/*
		 * get the next character
		 */
		c = path->str_text[ipos];
		if (c)
			ipos++;
		else
		{
			found = 1;
			c = '/';
		}

		/*
		 * remember the normal characters
		 * until get to slash
		 */
		if (c != '/')
			goto remember;

		/*
		 * leave root alone
		 */
		if (!opos)
			goto remember;

		/*
		 * "/.." -> "/"
		 */
		if (opos == 3 && tmp[1] == '.' && tmp[2] == '.')
		{
			opos = 1;
			continue;
		}

		/*
		 * "a//" -> "a/"
		 */
		if (tmp[opos - 1] == '/')
			continue;

		/*
		 * "a/./" -> "a/"
		 */
		if (opos >= 2 && tmp[opos - 1] == '.' && tmp[opos - 2] == '/')
		{
			opos--;
			continue;
		}

		/*
		 * "a/b/../" -> "a/"
		 */
		if
		(
			opos > 3
		&&
			tmp[opos - 1] == '.'
		&&
			tmp[opos - 2] == '.'
		&&
			tmp[opos - 3] == '/'
		)
		{
			opos -= 4;
			assert(opos > 0);
			while (tmp[opos - 1] != '/')
				opos--;
			continue;
		}

		/*
		 * see if the path so far is a symbolic link
		 */
#ifdef S_IFLNK
		s = str_n_from_c(tmp, opos);
		if (wl_member(&loop, s))
		{
			fatal
			(
				"symbolic link loop \"%s\" detected",
				s->str_text
			);
		}
		nbytes = readlink(s->str_text, link, sizeof(link) - 1);
		if (nbytes < 0)
		{
			/*
			 * probably not a symbolic link
			 */
			if
			(
				errno != ENXIO
			&&
				errno != EINVAL
			&&
				errno != ENOENT
			)
				nfatal("readlink(\"%s\")", s->str_text);
			not_a_symlink:
			str_free(s);
		}
		else
		{
			string_ty	*newpath;

			if (nbytes == 0)
			{
				fatal
				(
					"readlink(\"%s\") returned \"\"",
					s->str_text
				);
			}
			link[nbytes] = 0;

			/*
			 * ignore auto-mounter temporary mount links,
			 * they will change with each mount.
			 */
			if (!strncmp(link, "/tmp_mnt/", 9))
				goto not_a_symlink;

			wl_append(&loop, s);
			str_free(s);
			if (link[0] == '/')
				tmp[1] = 0;
			else
			{
				while (tmp[opos - 1] != '/')
					opos--;
				tmp[opos] = 0;
			}
			newpath =
				str_format
				(
					"%s/%s/%s",
					tmp,
					link,
					path->str_text + ipos
				);
			str_free(path);
			path = newpath;
			ipos = 0;
			opos = 0;
			found = 0;
			continue;
		}
#endif

		/*
		 * keep the slash
		 */
		remember:
		if (opos >= tmplen)
		{
			tmplen += 100;
			tmp = mem_change_size(tmp, tmplen);
		}
		tmp[opos++] = c;
	}
#ifdef S_IFLNK
	wl_free(&loop);
#endif
	str_free(path);
	assert(opos >= 1);
	assert(tmp[0] == '/');
	assert(tmp[opos - 1] == '/');
	if (opos >= 2)
		opos--;
	path = str_n_from_c(tmp, opos);
	trace_string(path->str_text);
	trace((/*{*/"}\n"));
	return path;
}


/*
 * NAME
 *	os_entryname - take path apart
 *
 * SYNOPSIS
 *	string_ty *os_entryname(string_ty *path);
 *
 * DESCRIPTION
 *	Os_entryname is used to extract the entry part
 *	from a pathname.
 *
 * RETURNS
 *	pointer to dynamically allocated string.
 *
 * CAVEAT
 *	Use str_free() when you are done with the return value.
 */

string_ty *
os_entryname(path)
	string_ty	*path;
{
	char		*cp;

	trace(("os_entryname(path = %08lX) entry", path));
	trace_string(path->str_text);
	cp = strrchr(path->str_text, '/');
	if (cp)
		path = str_from_c(cp + 1);
	else
		path = str_copy(path);
	trace_string(path->str_text);
	trace(("return %08lX;\n", path));
	trace((/*{*/"}\n"));
	return path;
}


/*
 * NAME
 *	os_dirname - take path apart
 *
 * SYNOPSIS
 *	string_ty *os_dirname(string_ty *path);
 *
 * DESCRIPTION
 *	Os_dirname is used to extract the directory part
 *	of a pathname.
 *
 * RETURNS
 *	pointer to dynamically allocated string.
 *	A null pointer is returned on error.
 *
 * CAVEAT
 *	Use str_free() when you are done with the value returned.
 */

string_ty *
os_dirname(path)
	string_ty	*path;
{
	char		*cp;

	trace(("os_dirname(path = %08lX)\n{\n"/*}*/, path));
	trace_string(path->str_text);
	cp = strrchr(path->str_text, '/');
	if (cp)
	{
		if (cp > path->str_text)
			path = str_n_from_c(path->str_text, cp - path->str_text);
		else
			path = str_from_c("/");
	}
	else
		path = os_curdir();
	trace_string(path->str_text);
	trace(("return %08lX;\n", path));
	trace((/*{*/"}\n"));
	return path;
}


/*
 * NAME
 *	os_meter_grab - get metering snapshot
 *
 * SYNOPSIS
 *	void os_meter_grab(grab*);
 *
 * DESCRIPTION
 *	The os_meter_gram function is used to take a snapshot of metering
 *	information.
 *
 * RETURNS
 *	void
 */

typedef struct grab grab;
struct grab
{
	double	g_elapsed;
	double	g_cpu;
	double	g_io;
};

static void os_meter_grab _((grab *));

static void
os_meter_grab(gp)
	grab		*gp;
{
#ifdef HAVE_GETRUSAGE
	struct rusage	ru;
	struct timeval	tv;
	struct timezone	tz;

	int getrusage _((int, struct rusage *));
	int gettimeofday _((struct timeval *, struct timezone *));

	if (getrusage(RUSAGE_CHILDREN, &ru))
		nfatal("getrusage");
	if (gettimeofday(&tv, &tz))
		nfatal("gettimeofday");
	gp->g_cpu = ru.ru_utime.tv_sec + ru.ru_utime.tv_usec * 1.0e-6;
	gp->g_io = ru.ru_stime.tv_sec + ru.ru_stime.tv_usec * 1.0e-6;
	gp->g_elapsed = tv.tv_sec + tv.tv_usec * 1.0e-6;
#else
	struct tms	buffer;
	time_t		n;

	n = times(&buffer);
	gp->g_elapsed = n / (double)HZ;
	gp->g_cpu = buffer.tms_cutime / (double)HZ;
	gp->g_io = buffer.tms_cstime / (double)HZ;
#endif
}


/*
 * NAME
 *	os_meter_ptime - print timing info
 *
 * SYNOPSIS
 *	void os_meter_ptime(double t, char *s, double elapsed);
 *
 * DESCRIPTION
 *	The os_meter_ptime function is used to print e representation of the
 *	time, in seconds, given in the `t' argument.  If `elapsed' is >0, it is
 *	the elapsed time taken, for percentage figures.  The `s' argument is a
 *	title string.
 *
 * RETURNS
 *	void
 */

static void os_meter_ptime _((double, char *, double));

static void
os_meter_ptime(t,s,e)
	double	t;
	char	*s;
	double	e;
{
	char	buffer[32];
	char	temp[32];
	long	hour, min, sec, frac;

	frac = t * 1000 + 0.5;
	sec = frac / 1000;
	frac %= 1000;
	min = sec / 60;
	sec %= 60;
	hour = min / 60;
	min %= 60;
	if (e > 0.0)
	{
		sprintf(temp, "(%.2f%%)", 100. * t / e);
		sprintf(buffer, "%-4s%9s", s, temp);
		s = buffer;
	}
	fprintf(stderr, "%2ld:%02ld:%02ld.%03ld %s\n", hour, min, sec, frac, s);
}


/*
 * NAME
 *	os_meter_begin - start metering interval
 *
 * SYNOPSIS
 *	void os_meter_begin(void);
 *
 * DESCRIPTION
 *	The os_meter_begin function is used to start a metring interval.
 *
 * RETURNS
 *	void
 */

static	grab	os_meter_start;

void
os_meter_begin()
{
	os_meter_grab(&os_meter_start);
}


/*
 * NAME
 *	os_meter_end - end a metering interval
 *
 * SYNOPSIS
 *	void os_meter_end(void);
 *
 * DESCRIPTION
 *	The os_meter_end function is used to end a metering interval and print
 *	the metered information.
 *
 * RETURNS
 *	void
 */

void
os_meter_end()
{
	grab	end;
	double	elapsed;
	double	cpu;
	double	io;

	os_meter_grab(&end);

	elapsed = end.g_elapsed - os_meter_start.g_elapsed;
	os_meter_ptime(elapsed, "elapsed", 0.0);
	cpu = end.g_cpu - os_meter_start.g_cpu;
	os_meter_ptime(cpu, "usr", elapsed);
	io = end.g_io - os_meter_start.g_io;
	os_meter_ptime(io, "sys", elapsed);
}


static int os_pathconf_path_max _((char *));

static int
os_pathconf_path_max(path)
	char	*path;
{
	long	result;

#ifdef _PC_PATH_MAX
	result = pathconf(path, _PC_PATH_MAX);
	if (result < 0 && (errno == EINVAL || errno == ENOSYS))
	{
		/*
		 * probably NFSv2 mounted
		 * assume is same as root
		 */
		path = "/";
		result = pathconf(path, _PC_PATH_MAX);
		if (result < 0 && (errno == EINVAL || errno == ENOSYS))
			result = 1024;
	}
	if (result < 0)
		nfatal("pathconf(\"%s\", {PATH_MAX})", path);
#else
	result = 1024;
#endif
	return result;
}


static int os_pathconf_name_max _((char *));

static int
os_pathconf_name_max(path)
	char	*path;
{
	long	result;

#ifdef _PC_NAME_MAX
	result = pathconf(path, _PC_NAME_MAX);
	if (result < 0 && (errno == EINVAL || errno == ENOSYS))
	{
		/*
		 * probably NFSv2 mounted
		 * assume is same as root
		 */
		path = "/";
		result = pathconf(path, _PC_NAME_MAX);
		if (result < 0 && (errno == EINVAL || errno == ENOSYS))
		{
#ifdef HAVE_LONG_FILE_NAMES
			result = 255;
#else
			result = 14;
#endif
		}
	}
	if (result < 0)
		nfatal("pathconf(\"%s\", {NAME_MAX})", path);
#else
#ifdef HAVE_LONG_FILE_NAMES
	result = 255;
#else
	result = 14;
#endif
#endif
	return result;
}


/*
 * NAME
 *	os_legal_path - test if path is legal
 *
 * SYNOPSIS
 *	int os_legal_path(string_ty *path);
 *
 * DESCRIPTION
 *	The os_legal_path function is used to test if each of the components of
 *	the given path are legal to the underlying operating system.
 *
 * RETURNS
 *	int: zero if it is an illegal path, nonzero it is a legal path.
 */

int
os_legal_path(str)
	string_ty	*str;
{
	char		*s;
	char		*ep;
	int		max;

	max = os_pathconf_path_max(".");
	if (str->str_length < 1 || str->str_length > max)
		return 0;
	max = os_pathconf_name_max(".");
	s = str->str_text;
	for (;;)
	{
		ep = strchr(s, '/');
		if (!ep)
			return (strlen(s) <= max);
		if (ep - s > max)
			return 0;
		s = ep + 1;
	}
}


#ifdef TIMING

static long os_totals_baseline;

void
os_totals_start()
{
	struct tms	buffer;

	os_totals_baseline = times(&buffer);
}

void
os_totals()
{
	struct tms	buffer;
	time_t		n;
	double		elapsed;
	double		t;

	error("timing totals...");
	n = times(&buffer);
	elapsed = (n - os_totals_baseline) / (double)HZ;
	os_meter_ptime(elapsed, "elapsed", 0.0);
	t = buffer.tms_utime / (double)HZ;
	os_meter_ptime(t, "usr", elapsed);
	t = buffer.tms_stime / (double)HZ;
	os_meter_ptime(t, "sys", elapsed);
	t = buffer.tms_cutime / (double)HZ;
	os_meter_ptime(t, "cusr", elapsed);
	t = buffer.tms_cstime / (double)HZ;
	os_meter_ptime(t, "csys", elapsed);
}

#endif
