/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to parse #directive lines in cookbooks
 #
 * The hashline.y and parse.y parsers share the same lexer.
 * This means using the classic sed hack for yacc output.
 * Note that the expression grammars must be as similar as possible
 * in the two grammars.
 *
 * The state table in the condition frames is very simple:
 *	state 0: before #if
 *	state 1: after #if (and variants)
 *	state 2: after #elif
 *	state 3: after #else
 *	state 0: after #endif
 */

%{

#include <stdio.h>
#include <ac/stddef.h>
#include <ac/string.h>
#include <ac/time.h>
#include <ac/stdlib.h>

#include <cook.h>
#include <error.h>
#include <expr.h>
#include <hashline.h>
#include <lex.h>
#include <mem.h>
#include <option.h>
#include <os.h>
#include <trace.h>
#include <word.h>


static wlist done_once;


typedef struct cond cond;
struct cond
{
	int	pass;
	int	state;
	cond	*next;
};

static	cond	*stack;
static	cond	*cond_free_list;

#ifdef DEBUG
#define YYDEBUG 1
#define printf trace_where(__FILE__, __LINE__), lex_trace
extern int yydebug;
#endif


#define yyerror lex_error


/*
 * NAME
 *	open_include - open an include file
 *
 * SYNOPSIS
 *	void open_include(string_ty *filename);
 *
 * DESCRIPTION
 *	The open_include function is used to search for a given file name in
 *	the include path and lex_open it when found.
 *
 * RETURNS
 *	void
 */

static void open_include_once _((string_ty *));

static void
open_include_once(filename)
	string_ty	*filename;
{
	if (!wl_member(&done_once, filename))
		lex_open_include(filename);
}


void
hashline_reset()
{
	wl_free(&done_once);
}


static void open_include _((string_ty *, int));

static void
open_include(filename, local)
	string_ty	*filename;
	int		local;
{
	int		j;
	string_ty	*path;

	trace(("open_include(filename = %08lX, local = %d) entry",
		filename, local));
	trace_string(filename->str_text);
	if (filename->str_text[0] != '/')
	{
		if (local)
		{
			string_ty	*s;

			s = lex_cur_file();
			if (strchr(s->str_text, '/'))
			{
				s = os_dirname(s);
				if (!s)
				{
					bomb:
					yyerror("unable to construct include file name");
					goto ret;
				}
				path = str_format("%S/%S", s, filename);
				str_free(s);
			}
			else
				path = str_copy(filename);
			switch (os_exists(path))
			{
			case -1:
				str_free(path);
				goto bomb;

			case 1:
				open_include_once(path);
				str_free(path);
				goto ret;
			}
			str_free(path);
		}
		for (j = 0; j < option.o_search_path.wl_nwords; ++j)
		{
			path = str_format("%S/%S", option.o_search_path.wl_word[j], filename);
			switch (os_exists(path))
			{
			case -1:
				str_free(path);
				goto bomb;

			case 1:
				open_include_once(path);
				str_free(path);
				goto ret;
			}
			str_free(path);
		}
	}
	open_include_once(filename);
	ret:
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hashline - the # control line processor
 *
 * SYNOPSIS
 *	void hashline(void);
 *
 * DESCRIPTION
 *	The hashline function is used to process # control lines.
 *
 * RETURNS
 *	void
 */

void
hashline()
{
	int yyparse _((void)); /* forward */

	trace(("hashline()\n{\n"/*}*/));
#if YYDEBUG
	yydebug = trace_pretest_;
#endif
	yyparse();
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	cond_alloc - allocate a condition structure
 *
 * SYNOPSIS
 *	cond *cond_alloc(void);
 *
 * DESCRIPTION
 *	The cond_alloc function is used to allocate a condition structure
 *	from dynamic memory.
 *
 * RETURNS
 *	cond * - pointer to condition structure.
 *
 * CAVEAT
 *	A free list is maintained to avoid malloc overheads.
 */

static cond *cond_alloc _((void));

static cond *
cond_alloc()
{
	cond		*c;

	if (cond_free_list)
	{
		c = cond_free_list;
		cond_free_list = c->next;
	}
	else
		c = (cond *)mem_alloc(sizeof(cond));
	return c;
}


/*
 * NAME
 *	cond_free - free condition structure
 *
 * SYNOPSIS
 *	void cond_free(cond*);
 *
 * DESCRIPTION
 *	The cond_free function is used to indicate that a condition structure
 *	is finished with.
 *
 * RETURNS
 *	void
 *
 * CAVEAT
 *	A free list is maintained to avoid malloc overheads.
 */

static void cond_free _((cond *));

static void
cond_free(c)
	cond		*c;
{
	c->next = cond_free_list;
	cond_free_list = c;
}
 

/*
 * NAME
 *	hash_include - process #include directive
 *
 * SYNOPSIS
 *	void hash_include(expr *filename);
 *
 * DESCRIPTION
 *	The hash_include function is used to process #include directives.
 *
 * RETURNS
 *	void
 */

static void hash_include _((expr *));

static void
hash_include(ep)
	expr		*ep;
{
	wlist		result;
	string_ty	*s;

	if (stack && !stack->pass)
		return;
	wl_zero(&result);
	if (expr_eval(&result, ep))
	{
		wl_free(&result);
		hashline_error("include file name evaluation failed");
		return;
	}
	switch (result.wl_nwords)
	{
	case 0:
		yyerror("expression produces no file name to include");
		break;

	case 1:
		s = result.wl_word[0];
		if
		(
			s->str_length > 2
		&&
			s->str_text[0] == '<'
		&&
			s->str_text[s->str_length - 1] == '>'
		)
		{
			s = str_n_from_c(s->str_text + 1, s->str_length - 2);
			open_include(s, 0);
			str_free(s);
		}
		else
		{
			if (s->str_length)
				open_include(s, 1);
			else
				yyerror("expression produces null file name to include");
		}
		break;

	default:
		yyerror("expression produces more than one file name to include");
		break;
	}
	wl_free(&result);
}
 

/*
 * NAME
 *	hash_include - process #include-cooked directive
 *
 * SYNOPSIS
 *	void hash_include_cooked(elist *filename);
 *
 * DESCRIPTION
 *	The hash_include_cooked function is used to
 *	process #include-cooked directives.
 *
 * RETURNS
 *	void
 */

static void hash_include_cooked _((elist *, int));

static void
hash_include_cooked(elp, warn)
	elist		*elp;
	int		warn;
{
	wlist		result;
	string_ty	*s;
	long		j;
	wlist		actual;
	long		nerr;

	/*
	 * if conditional is false, don't do
	 */
	if (stack && !stack->pass)
		return;

	/*
	 * turn the expressions into words
	 */
	wl_zero(&result);
	for (j = 0; j < elp->el_nexprs; ++j)
	{
		if (expr_eval(&result, elp->el_expr[j]))
		{
			wl_free(&result);
			hashline_error("include file name evaluation failed");
			return;
		}
	}

	/*
	 * make sure we like the words they used
	 */
	nerr = 0;
	for (j = 0; j < result.wl_nwords; ++j)
	{
		s = result.wl_word[j];
		if
		(
			s->str_length > 2
		&&
			s->str_text[0] == '<'
		&&
			s->str_text[s->str_length - 1] == '>'
		)
		{
			yyerror("may not use angle brackets with #include-cooked");
			++nerr;
		}
		else if (!s->str_length)
		{
			yyerror("expression produces null file name to include");
			++nerr;
		}
	}
	if (nerr)
	{
		wl_free(&result);
		return;
	}

	/*
	 * append to the auto-cook list
	 *
	 * If any of the auto-cook list are out-of-date,
	 * they are recooked, and then cook starts over.
	 */
	cook_auto(&result);

	/*
	 * resolve the words into paths
	 */
	wl_zero(&actual);
	cook_mtime_resolve(&actual, &result, 0);
	wl_free(&result);

	/*
	 * include the resolved paths,
	 * warning if they do not exist
	 * (they will later, hopefully)
	 */
	for (j = 0; j < actual.wl_nwords; ++j)
	{
		s = actual.wl_word[j];
		if (os_exists(s))
			open_include_once(s);
		else if (warn)
		{
			lex_warning
			(
				"include cooked \"%s\": file not found",
				s->str_text
			);
		}
	}
	wl_free(&actual);
}


/*
 * NAME
 *	hash_if - process #if directive
 *
 * SYNOPSIS
 *	void hash_if(expr *);
 *
 * DESCRIPTION
 *	The hash_if function is used to process #if directives.
 *
 * RETURNS
 *	void
 */

static void hash_if _((expr *));

static void
hash_if(ep)
	expr		*ep;
{
	cond		*c;

	trace(("hash_if(ep = %08lX)\n{\n"/*}*/, ep));
	c = cond_alloc();
	c->next = stack;
	if (stack && !stack->pass)
	{
		c->pass = 0;
		c->state = 1;
		lex_passing(0);
	}
	else
	{
		switch (expr_eval_condition(ep))
		{
		case -1:
			yyerror("condition evaluation failed");
			/* fall through... */

		case 0:
			c->pass = 0;
			c->state = 2;
			lex_passing(0);
			break;

		default:
			c->pass = 1;
			c->state = 1;
			lex_passing(1);
			break;
		}
	}
	stack = c;
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hash_ifdef - process #ifdef directive
 *
 * SYNOPSIS
 *	void hash_ifdef(expr*);
 *
 * DESCRIPTION
 *	The hash_ifdef function is used to process #ifdef directives.
 *
 * RETURNS
 *	void
 */

static void hash_ifdef _((expr *));

static void
hash_ifdef(ep)
	expr		*ep;
{
	expr		*e1;
	expr		*e2;

	trace(("hash_ifdef(ep = %08lX)\n{\n"/*}*/, ep));
	e1 = expr_alloc();
	e1->e_op = OP_WORD;
	e1->e_word = str_from_c("defined");
	e2 = expr_alloc();
	e2->e_op = OP_FUNC;
	el_append(&e2->e_list, e1);
	el_append(&e2->e_list, ep);
	expr_free(e1);
	hash_if(e2);
	expr_free(e2);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hash_ifndef - process #ifndef directives
 *
 * SYNOPSIS
 *	void hash_ifndef(expr *);
 *
 * DESCRIPTION
 *	The hash_ifndef function is used to process #ifndef directives.
 *
 * RETURNS
 *	void
 */

static void hash_ifndef _((expr *));

static void
hash_ifndef(ep)
	expr		*ep;
{
	expr		*e1;
	expr		*e2;
	expr		*e3;

	trace(("hash_ifndef(ep = %08lX)\n{\n"/*}*/, ep));
	e1 = expr_alloc();
	e1->e_op = OP_WORD;
	e1->e_word = str_from_c("defined");
	e2 = expr_alloc();
	e2->e_op = OP_FUNC;
	el_append(&e2->e_list, e1);
	el_append(&e2->e_list, ep);
	expr_free(e1);

	e1 = expr_alloc();
	e1->e_op = OP_WORD;
	e1->e_word = str_from_c("not");
	e3 = expr_alloc();
	e3->e_op = OP_FUNC;
	el_append(&e3->e_list, e1);
	el_append(&e3->e_list, e2);
	expr_free(e1);
	expr_free(e2);

	hash_if(e3);
	expr_free(e3);
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hash_elif - process #elif directive
 *
 * SYNOPSIS
 *	void hash_elif(expr*);
 *
 * DESCRIPTION
 *	The hash_elif function is used to provess #elif directives.
 *
 * RETURNS
 *	void
 */

static void hash_elif _((expr *));

static void
hash_elif(ep)
	expr		*ep;
{
	trace(("hash_elif(ep = %08lX)\n{\n"/*}*/, ep));
	if (!stack)
		yyerror("#elif without matching #if");
	else
	{
		switch (stack->state)
		{
		case 1:
			stack->pass = 0;
			stack->state = 1;
			lex_passing(0);
			break;

		case 2:
			switch (expr_eval_condition(ep))
			{
			case -1:
				yyerror("condition evaluation failed");
				/* fall through... */

			case 0:
				stack->pass = 0;
				stack->state = 2;
				lex_passing(0);
				break;

			default:
				stack->pass = 1;
				stack->state = 1;
				lex_passing(1);
				break;
			}
			break;

		case 3:
			stack->pass = 0;
			stack->state = 3;
			yyerror("#elif after #else");
			lex_passing(0);
			break;
		}
	}
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hash_else - process #else directive
 *
 * SYNOPSIS
 *	void hash_else(void);
 *
 * DESCRIPTION
 *	The hash_else function is used to process #else directives.
 *
 * RETURNS
 *	void
 */

static void hash_else _((void));

static void
hash_else()
{
	trace(("hash_else()\n{\n"/*}*/));
	if (!stack)
		yyerror("#else without matching #if");
	else
	{
		switch (stack->state)
		{
		case 1:
			stack->pass = 0;
			stack->state = 3;
			lex_passing(0);
			break;

		case 2:
			stack->pass = 1;
			stack->state = 3;
			lex_passing(1);
			break;

		case 3:
			stack->pass = 0;
			stack->state = 3;
			yyerror("#else after #else");
			lex_passing(0);
			break;
		}
	}
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hash_endif - process #endif directive
 *
 * SYNOPSIS
 *	void hash_endif(void);
 *
 * DESCRIPTION
 *	The hash_endif function is used to process #endif directives.
 *
 * RETURNS
 *	void
 */

static void hash_endif _((void));

static void
hash_endif()
{
	trace(("hash_endif()\n{\n"/*}*/));
	if (!stack)
		yyerror("#endif without matching #if");
	else
	{
		cond	*c;

		c = stack;
		stack = c->next;
		cond_free(c);
		lex_passing(stack ? stack->pass : 1);
	}
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	hash_pragma - process #pragma directive
 *
 * SYNOPSIS
 *	void hash_pragma(elist *elp);
 *
 * DESCRIPTION
 *	The hash_pragma function is used to process #pragma directives.
 *
 * RETURNS
 *	void
 */

static void hash_pragma _((elist *));

static void
hash_pragma(elp)
	elist		*elp;
{
	static string_ty	*once;

	trace(("hash_if(elp = %08lX)\n{\n"/*}*/, elp));
	if (stack && !stack->pass)
		goto ret;

	/*
	 * see if it was "#pragma once"
	 */
	if (!once)
		once = str_from_c("once");
	if
	(
		elp->el_nexprs == 1
	&&
		elp->el_expr[0]->e_op == OP_WORD
	&&
		str_equal(elp->el_expr[0]->e_word, once)
	)
	{
		wl_append_unique(&done_once, lex_cur_file());
		goto ret;
	}

	/*
	 * add more pragma's here
	 */

	ret:
	trace((/*{*/"}\n"));
}

%}

/*
 * this list must be IDENTICAL to the list in parse.y
 */
%token	CATENATE
%token	COLON
%token	DATA
%token	DATAEND
%token	ELSE
%token	EQUALS
%token	FAIL
%token	IF
%token	LBRACE
%token	LBRAK
%token	LOOP
%token	LOOPSTOP
%token	RBRACE
%token	RBRAK
%token	SEMICOLON
%token	SET
%token	THEN
%token	UNSETENV
%token	WORD

/*
 * this list must not appear in parse.y
 */
%token	HASH_ELIF
%token	HASH_ELSE
%token	HASH_ENDIF
%token	HASH_IF
%token	HASH_IFDEF
%token	HASH_IFNDEF
%token	HASH_INCLUDE
%token	HASH_INCLUDE_COOKED
%token	HASH_INCLUDE_COOKED2
%token	HASH_PRAGMA

%left	CATENATE
%right	ELSE

%union
{
	expr		*lv_expr;
	elist		lv_elist;
	string_ty	*lv_word;
}

%type	<lv_elist>	elist
%type	<lv_word>	WORD
%type	<lv_expr>	expr

%%

/*
 * note that the grammar accepts a single line.
 * this means that 0 (end-of-input) must be sent on end-of-line.
 */

hashline
	: HASH_INCLUDE expr
		{
			hash_include($2);
			expr_free($2);
		}
	| HASH_INCLUDE_COOKED elist
		{
			hash_include_cooked(&$2, 1);
			el_free(&$2);
		}
	| HASH_INCLUDE_COOKED2 elist
		{
			hash_include_cooked(&$2, 0);
			el_free(&$2);
		}
	| HASH_IF expr
		{
			hash_if($2);
			expr_free($2);
		}
	| HASH_IFDEF expr
		{
			hash_ifdef($2);
			expr_free($2);
		}
	| HASH_IFNDEF expr
		{
			hash_ifndef($2);
			expr_free($2);
		}
	| HASH_ELIF expr 
		{
			hash_elif($2);
			expr_free($2);
		}
	| HASH_ELSE 
		{
			hash_else();
		}
	| HASH_ENDIF 
		{
			hash_endif();
		}
	| HASH_PRAGMA elist 
		{
			hash_pragma(&$2);
			el_free(&$2);
		}
	| error 
	;

/*
 * this expression form is the same as in parse.y
 * except that the lbrak processing is not necessary.
 */

expr
	: WORD
		{
			$$ = expr_alloc();
			$$->e_op = OP_WORD;
			$$->e_word = $1;
			$1 = 0;
		}
	| LBRAK elist RBRAK
		{
			$$ = expr_alloc();
			$$->e_op = OP_FUNC;
			$$->e_list = $2;
		}
	| expr CATENATE expr
		{
			$$ = expr_alloc();
			$$->e_op = OP_CAT;
			$$->e_left = $1;
			$$->e_right = $3;
		}
	;

elist
	: expr
		{
			el_zero(&$$);
			el_append(&$$, $1);
			expr_free($1);
		}
	| elist expr
		{
			$$ = $1;
			el_append(&$$, $2);
			expr_free($2);
		}
	;
