/*
 *	cook - file construction tool
 *	Copyright (C) 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to manipulate the persistent fingerprint cache
 */

%{
#include <errno.h>
#include <stdio.h>
#include <ac/stdlib.h>
#include <ac/unistd.h>

#include <archive.h>
#include <arglex.h>
#include <error.h>
#include <fngrprnt.h>
#include <fngrprnt_lex.h>
#include <fp/combined.h>
#include <mem.h>
#include <symtab.h>
#include <trace.h>
%}

%token	STRING
%token	JUNK
%token	NUMBER
%token	EQ
%token	LB
%token	RB

%union
{
	string_ty	*lv_string;
	long		lv_number;
}

%type	<lv_string>	STRING
%type	<lv_number>	NUMBER

%%

cache
	: /* empty */
	| cache entry
	;

entry
	: STRING EQ LB NUMBER NUMBER STRING RB
		{
			fp_ty	data;

			data.oldest = $4;
			data.newest = $5;
			data.fingerprint = $6;
			fp_assign($1, &data);
			str_free($1);
			str_free($6);
		}
	| STRING EQ LB NUMBER STRING RB
		{
			fp_ty	data;

			data.oldest = $4;
			data.newest = $4;
			data.fingerprint = $5;
			fp_assign($1, &data);
			str_free($1);
			str_free($5);
		}
	;

%%



static symtab_ty *symtab;
static int	dirty;
static int	inited;


static void reap _((void *));

static void
reap(p)
	void	*p;
{
	fp_ty	*fp;

	trace(("reap(p = %08lX)\n{\n"/*}*/, (long)p));
	fp = p;
	str_free(fp->fingerprint);
	mem_free(fp);
	trace((/*{*/"}\n"));
}


static char *fp_filename _((void));

static char *
fp_filename()
{
	static string_ty *s;

	if (!s)
		s = str_format(".%.10s.fp", progname);
	return s->str_text;
}


static void walk _((symtab_ty *, string_ty *, void *, void *));

static void
walk(sp, key, p, arg)
	symtab_ty	*sp;
	string_ty	*key;
	void		*p;
	void		*arg;
{
	fp_ty		*data;
	FILE		*fp;

	trace(("walk(sp = %08lX, key = \"%s\", p = %08lX, arg = %08lX)\n{\n"/*}*/, (long)sp, key->str_text, (long)p, (long)arg));
	data = p;
	fp = arg;
	if (data->oldest == data->newest)
		fprintf
		(
			fp,
			"\"%s\" = { %ld\n\"%s\" }\n",
			key->str_text,
			(long)data->newest,
			data->fingerprint->str_text
		);
	else
		fprintf
		(
			fp,
			"\"%s\" = { %ld %ld\n\"%s\" }\n",
			key->str_text,
			(long)data->oldest,
			(long)data->newest,
			data->fingerprint->str_text
		);
	trace((/*{*/"}\n"));
}


static void fp_close _((void));

static void
fp_close()
{
	char	*fn;
	FILE	*fp;

	if (!dirty)
		return;
	if (!inited)
		return;
	trace(("fp_close()\n{\n"/*}*/));
	fn = fp_filename();
	if (unlink(fn) && errno != ENOENT)
		nerror("warning: unlink %s", fn);
	trace(("open\n"));
	fp = fopen(fn, "w");
	if (!fp)
		nfatal("open %s", fn);
	trace(("walk\n"));
	trace(("symtab = %08lX;\n", (long)symtab));
	symtab_walk(symtab, walk, fp);
	trace(("close\n"));
	fflush(fp);
	if (ferror(fp))
	{
		int	err;

		err = errno;
		unlink(fn); /* ignore error */
		errno = err;
		nfatal("write %s", fn);
	}
	fclose(fp);
	dirty = 0;
	trace((/*{*/"}\n"));
}


static void init _((void));

static void
init()
{
	int yyparse _((void));

	if (inited)
		return;
	trace(("init()\n{\n"/*}*/));
	inited = 1;
	symtab = symtab_alloc(100);
	symtab->reap = reap;
	yylex_open(fp_filename());
	yyparse();
	yylex_close();
	dirty = 0;
	quit_handler(fp_close);
	trace((/*{*/"}\n"));
}


fp_ty *
fp_search(path)
	string_ty	*path;
{
	fp_ty		*fp;

	trace(("fp_search(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (!inited)
		init();
	fp = symtab_query(symtab, path);
	trace(("return %08lX;\n", (long)fp));
	trace((/*{*/"}\n"));
	return fp;
}


void
fp_assign(path, fp)
	string_ty	*path;
	fp_ty		*fp;
{
	fp_ty		*data;

	trace(("fp_assign(path = \"%s\", fp = %08lX)\n{\n"/*}*/,
		path->str_text, (long)fp));
	if (!inited)
		init();
	data = mem_alloc(sizeof(fp_ty));
	data->oldest = fp->oldest;
	data->newest = fp->newest;
	data->fingerprint = str_copy(fp->fingerprint);
	symtab_assign(symtab, path, data);
	dirty = 1;
	trace((/*{*/"}\n"));
}


void
fp_delete(path)
	string_ty	*path;
{
	if (!inited)
		return;
	trace(("fp_delete(path = \"%s\")\n{\n"/*}*/, path->str_text));
	if (symtab_query(symtab, path))
	{
		symtab_delete(symtab, path);
		dirty = 1;
	}
	trace((/*{*/"}\n"));
}


string_ty *
fp_fingerprint(path)
	string_ty	*path;
{
	fingerprint_ty	*fp;
	string_ty	*result;
	int		err;
	char		buffer[1000];

	trace(("fp_fingerprint(path = \"%s\")\n{\n"/*}*/, path->str_text));
	fp = fingerprint_new(&fp_combined);
	err = fingerprint_file_sum(fp, path->str_text, buffer);
	if (err && errno == ENOENT)
		err = archive_fingerprint(fp, path, buffer);
	if (err)
	{
		switch (errno)
		{
		case ENOTDIR:
		case EISDIR:
		case ENOENT:
			break;

		default:
			nfatal("fingerprint \"%s\"", path->str_text);
		}
		result = 0;
	}
	else
		result = str_from_c(buffer);
	trace(("return \"%s\";\n", result ? result->str_text : ""));
	trace((/*{*/"}\n"));
	return result;
}
