/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin stringset function
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <builtin/stringset.h>


#define STRINGSET_EOLN  1
#define STRINGSET_WORD  2
#define STRINGSET_MINUS 3
#define STRINGSET_STAR  4

static wlist    stringset_args;
static size_t   stringset_pos;
static int      stringset_token;
static string_ty *stringset_token_value;


static void stringset_lex _((void));

static void
stringset_lex()
{
	static string_ty *minus;
	static string_ty *star;

	if (!minus)
		minus = str_from_c("-");
	if (!star)
		star = str_from_c("*");
	if (stringset_pos >= stringset_args.wl_nwords)
	{
		stringset_token_value = str_false;
		stringset_token = STRINGSET_EOLN;
		return;
	}
	stringset_token_value = stringset_args.wl_word[stringset_pos++];
	if (str_equal(minus, stringset_token_value))
	{
		stringset_token = STRINGSET_MINUS;
		return;
	}
	if (str_equal(star, stringset_token_value))
	{
		stringset_token = STRINGSET_STAR;
		return;
	}
	stringset_token = STRINGSET_WORD;
}


static int stringset_three _((wlist *));

static int
stringset_three(result)
	wlist		*result;
{
	while (stringset_token == STRINGSET_WORD)
	{
		wl_append_unique(result, stringset_token_value);
		stringset_lex();
	}
	return 0;
}


static int stringset_two _((wlist *));

static int
stringset_two(result)
	wlist		*result;
{
	if (stringset_three(result))
		return -1;
	while (stringset_token == STRINGSET_STAR)
	{
		wlist		lhs;
		wlist		rhs;
		int		j;
		string_ty	*s;

		stringset_lex();
		wl_zero(&rhs);
		if (stringset_three(&rhs))
			return -1;
		lhs = *result;
		wl_zero(result);
		for (j = 0; j < rhs.wl_nwords; ++j)
		{
			s = rhs.wl_word[j];
			if (wl_member(&lhs, s))
				wl_append_unique(result, s);
		}
		wl_free(&lhs);
		wl_free(&rhs);
	}
	return 0;
}


static int stringset_one _((wlist *));

static int
stringset_one(result)
	wlist		*result;
{
	if (stringset_two(result))
		return -1;
	while (stringset_token == STRINGSET_MINUS)
	{
		wlist		wl;
		int		j;

		wl_zero(&wl);
		stringset_lex();
		if (stringset_two(&wl))
			return -1;
		for (j = 0; j < wl.wl_nwords; ++j)
			wl_delete(result, wl.wl_word[j]);
		wl_free(&wl);
	}
	return 0;
}


int
builtin_stringset(result, args)
	wlist		*result;
	wlist		*args;
{
	wlist		wl;
	int		j;

	stringset_args = *args;
	stringset_pos = 1;
	stringset_lex();

	wl_zero(&wl);
	if (stringset_one(&wl))
		return -1;
	for (j = 0; j < wl.wl_nwords; ++j)
		wl_append(result, wl.wl_word[j]);
	wl_free(&wl);
	return 0;
}
