/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin os function
 *
 * The builtin functions all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <sys/utsname.h>

#include <builtin/opsys.h>
#include <error.h>
#include <expr.h>


/*
 * NAME
 *	builtin_opsys - describe operating system
 *
 * SYNOPSIS
 *	int builtin_opsys(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Operating_system is a built-in function of cook, described as follows:
 *	This function must have zero or more arguments.
 *
 * RETURNS
 *	The resulting wordlist contains the values of various
 *	attributes of the operating system, as named in the arguments.
 *	If no attributes are named "name" is assumed.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_opsys(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	static string_ty *name;
	static string_ty *machine;
	static struct utsname uts;
	static char	uts_defined;
	static string_ty *system;
	static string_ty *node;
	static string_ty *release;
	static string_ty *version;

	if (!uts_defined)
	{
		uname(&uts);
		uts_defined = 1;
	}
	if (!name)
		name = str_from_c("name");
	if (!system)
		system = str_from_c("system");
	if (!node)
		node = str_from_c("node");
	if (!release)
		release = str_from_c("release");
	if (!version)
		version = str_from_c("version");
	if (!machine)
		machine = str_from_c("machine");
	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
		wl_append(args, name);
	for (j = 1; j < args->wl_nwords; j++)
	{
		string_ty	*x;

		x = args->wl_word[j];
		if (str_equal(name, x) || str_equal(system, x))
		{
			static string_ty *s;
			if (!s)
				s = str_from_c(uts.sysname);
			wl_append(result, s);
		}
		else if (str_equal(node, x))
		{
			static string_ty *s;
			if (!s)
				s = str_from_c(uts.nodename);
			wl_append(result, s);
		}
		else if (str_equal(release, x))
		{
			static string_ty *s;
			if (!s)
				s = str_from_c(uts.release);
			wl_append(result, s);
		}
		else if (str_equal(version, x))
		{
			static string_ty *s;
			if (!s)
				s = str_from_c(uts.version);
			wl_append(result, s);
		}
		else if (str_equal(machine, x))
		{
			static string_ty *s;
			if (!s)
				s = str_from_c(uts.machine);
			wl_append(result, s);
		}
		else
		{
			expr_error
			(
				"%s: unknown \"%s\" attribute",
				args->wl_word[0]->str_text,
				x->str_text
			);
			return -1;
		}
	}
	return 0;
}
