/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to perform shell-style file pattern matching
 */

#include <ac/stddef.h>
#include <string.h>
#include <errno.h>
#include <ac/stdlib.h>

#include <ac/dirent.h>

#include <builtin/glob.h>
#include <trace.h>
#include <mem.h>
#include <error.h>


static	char	*original;
static	wlist	*where;
static	char	*tmp;
static	size_t	tmp_max;
static	size_t	tmp_len;


/*
 * NAME
 *	tmp_char - append character
 *
 * SYNOPSIS
 *	void tmp_char(int c);
 *
 * DESCRIPTION
 *	The tmp_char function is used to append a character to the string being
 *	built by the builtin_glob function.
 *
 * RETURNS
 *	void
 */

static void tmp_char _((int));

static void
tmp_char(c)
	int		c;
{
	if (tmp_len >= tmp_max)
	{
		tmp_max += 128;
		tmp = mem_change_size(tmp, tmp_max);
	}
	tmp[tmp_len++] = c;
}


/*
 * NAME
 *	gmatch - match entryname pattern
 *
 * SYNOPSIS
 *	int gmatch(char *formal, char *formal_end, char *actual);
 *
 * DESCRIPTION
 *	The formal strings is used as a template to match the given actual
 *	string against.
 *
 *	The pattern elements understood are
 *	*	match zero or more of any character
 *	?	match any single character
 *	[^xxx]	match any single character not in the set given.
 *	[xxx]	match any single character not in the set given.
 *	        The - character is understood to be a range indicator.
 *	        If the ] character is the first of the set it is considered
 *	        as part of the set, not the terminator.
 *
 * RETURNS
 *	the gmatch function returns zero if they do not match,
 *	and nonzero if they do.  Returns -1 on error.
 *
 * CAVEAT
 *	This is a limited set of the sh(1) patterns.
 *	Assumes that the `original' global variable has been initialized, it is
 *	used for error reporting.
 */

static int gmatch _((char *, char *, char *));

static int
gmatch(formal, formal_end, actual)
	char		*formal;
	char		*formal_end;
	char		*actual;
{
	char		*cp;
	int		 result;

	trace(("gmatch(formal = %08lX, formal_end = %08lX, actual = %08lX)\n{\n"/*}*/, formal, formal_end, actual));
	while (formal < formal_end)
	{
		trace(("formal == \"%.*s\";\n", formal_end - formal, formal));
		trace(("actual = \"%s\";\n", actual));
		switch (*formal)
		{
		default:
			if (*actual++ != *formal++)
			{
				result = 0;
				goto ret;
			}
			break;

		case '?':
			if (!*actual++)
			{
				result = 0;
				goto ret;
			}
			++formal;
			break;

		case '*':
			cp = actual + strlen(actual);
			++formal;
			for (;;)
			{
				if (gmatch(formal, formal_end, cp))
				{
					result = 1;
					goto ret;
				}
				--cp;
				if (cp < actual)
				{
					result = 0;
					goto ret;
				}
			}

		case '[':
			++formal;
			if (*formal == '^')
			{
				++formal;
				for (;;)
				{
					if (formal >= formal_end)
					{
						no_close:
						error("pattern \"%s\" missing closing ']'", original);
						result = -1;
						goto ret;
					}
					/* note: this allows leading ']' elegantly */
					if
					(
						formal_end >= formal + 3
					&&
						formal[1] == '-'
					&&
						formal[2] != ']'
					)
					{
						char	c1;
						char	c2;

						c1 = formal[0];
						c2 = formal[2];
						formal += 3;
						if
						(
							c1 <= c2
						?
							(c1 <= *actual && *actual <= c2)
						:
							(c2 <= *actual && *actual <= c1)
						)
						{
							result = 0;
							goto ret;
						}
					}
					else
					if (*actual == *formal++)
					{
						result = 0;
						goto ret;
					}
					if (*formal == ']')
						break;
				}
				++formal;
			}
			else
			{
				for (;;)
				{
					if (formal >= formal_end)
						goto no_close;
					/* note: this allows leading ']' elegantly */
					trace(("formal == \"%.*s\";\n", formal_end - formal, formal));
					trace(("actual = \"%s\";\n", actual));
					if
					(
						formal_end >= formal + 3
					&&
						formal[1] == '-'
					&&
						formal[2] != ']'
					)
					{
						char	c1;
						char	c2;

						c1 = formal[0];
						c2 = formal[2];
						formal += 3;
						if
						(
							c1 <= c2
						?
							(c1 <= *actual && *actual <= c2)
						:
							(c2 <= *actual && *actual <= c1)
						)
							break;
					}
					else
					if (*actual == *formal++)
						break;
					if (*formal == ']')
					{
						result = 0;
						goto ret;
					}
				}
				for (;;)
				{
					if (formal >= formal_end)
						goto no_close;
					trace(("formal == \"%.*s\";\n", formal_end - formal, formal));
					trace(("actual = \"%s\";\n", actual));
					if (*formal++ == ']')
						break;
				}
			}
			++actual;
			break;
		}
	}
	result = (*actual == 0);
	ret:
	trace(("return %d;\n", result));
	trace((/*{*/"}\n"));
	return result;
}


/*
 * NAME
 *	globber - file name expander
 *
 * SYNOPSIS
 *	int globber(char *formal);
 *
 * DESCRIPTION
 *	The globber function is used to generate a list of file names from the
 *	given `formal' pattern.  Results are appended to the word list pointed
 *	to the global `where' variable.
 *
 * RETURNS
 *	int; 0 on success, -1 on error.
 *
 * CAVEAT
 *	Assumes that the `where' global variable has been initialized.
 */

static int globber _((char *));

static int
globber(formal)
	char		*formal;
{
	char		*formal_end;
	char		*cp;
	int		 retval;

	trace(("globber(formal = %08lX)\n{\n"/*}*/, formal));
	trace_string(formal);
	retval = 0;
	for (;;)
	{
		while (*formal == '/')
			tmp_char(*formal++);
		formal_end = strchr(formal, '/');
		if (!formal_end)
			formal_end = formal + strlen(formal);
		for (cp = formal; cp < formal_end; ++cp)
			if (strchr("[?*", *cp))
				break;
		if (cp >= formal_end)
		{
			/* nothing special */
			trace(("ordinary = \"%.*s\"", formal_end - formal, formal));
			for (cp = formal; cp < formal_end; ++cp)
				tmp_char(*cp);
			if (!*cp)
			{
				string_ty *s;

				s = str_n_from_c(tmp, tmp_len);
				wl_append(where, s);
				str_free(s);
				break;
			}
			formal = formal_end;
		}
		else
		{
			size_t		n;
			DIR		*dp;
			struct dirent	*dep;

			/* need to expand wild characters */
			trace(("expand = \"%.*s\"", formal_end - formal, formal));
			n = tmp_len;
			tmp_char(0);
			dp = opendir(tmp[0] ? tmp : ".");
			if (!dp)
			{
				if (errno == ENOTDIR)
					break;
				nerror("%s", tmp);
				retval = -1;
				goto ret;
			}
			tmp_len = n;
			for (;;)
			{
				char	*np;

				dep = readdir(dp);
				if (!dep)
					break;
				np = dep->d_name;
				if
				(
					np[0] == '.'
				&&
					(
						!np[1]
					||
						(np[1] == '.' && !np[2])
					)  
				)
					continue;
				switch (gmatch(formal, formal_end, np))
				{
				case 0:
					continue;

				case -1:
					retval = -1;
					goto ret;
				}
				for (cp = np; *cp; ++cp)
					tmp_char(*cp);
				if (!*formal_end)
				{
					string_ty *s;

					s = str_n_from_c(tmp, tmp_len);
					wl_append(where, s);
					str_free(s);
				}
				else
				{
					tmp_char('/');
					if (globber(formal_end + 1))
					{
						closedir(dp);
						retval = -1;
						goto ret;
					}
				}
				tmp_len = n;
			}
			closedir(dp);
			break;
		}
	}
	ret:
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}


/*
 * NAME
 *	cmp - compare strings
 *
 * SYNOPSIS
 *	int cmp(string_ty **, string_ty **);
 *
 * DESCRIPTION
 *	The cmp function is used to compare two strings.
 *
 * RETURNS
 *	int; <0 if a<b, 0 if a==b, >0 if a>b
 *
 * CAVEAT
 *	Intended for use by qsort.
 */

static int cmp _((const void *, const void *));

static int
cmp(va, vb)
	const void	*va;
	const void	*vb;
{
	string_ty	*a;
	string_ty	*b;

	a = *(string_ty **)va;
	b = *(string_ty **)vb;
	return strcmp(a->str_text, b->str_text);
}


/*
 * NAME
 *	builtin_glob - builtin function for expanding file names
 *
 * SYNOPSIS
 *	int builtin_glob(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	The builtin_glob function is used to implement the "glob" builtin
 *	function of cook to expand file name patterns.
 *
 * RETURNS
 *	int; 0 on success, -1 on any error
 *
 * CAVEAT
 *	This function is designed to be used as a "builtin" function.
 */

int
builtin_glob(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	int		start;
	int		retval;

	trace(("glob(result = %08X, args = %08X)\n{\n"/*}*/, result, args));
	retval = 0;
	where = result;
	for (j = 1; j < args->wl_nwords; ++j)
	{
		tmp_len = 0;
		original = args->wl_word[j]->str_text;
		start = result->wl_nwords;
		if (globber(original))
		{
			retval = -1;
			break;
		}
		qsort
		(
			result->wl_word + start,
			result->wl_nwords - start,
			sizeof(string_ty *),
			cmp
		);
	}
	trace(("return %d;\n", retval));
	trace((/*{*/"}\n"));
	return retval;
}
