/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 *
 * Only a limited set of this are candidates for builtin functions,
 * these are 
 *	- string manipulation [dirname, stringset, ect ]
 *	- environment manipulation [getenv(3), etc]
 *	- stat(3) related functions [exists, mtime, pathname, etc]
 *	- launching OS commands [execute, collect]
 * The above list is though to be exhaustive.
 *
 * This explicitly and forever excluded from being a builtin function
 * is anything which known or understands the format of some secific 
 * class of files.
 *
 * Access to stdio(3) has been thought of, and explicitly avoided.
 * Mostly because a specialist program used through [collect]
 * will almost always be far faster.
 */

#include <ac/stdlib.h>
#include <ac/string.h>

#include <builtin/find_command.h>
#include <error.h>
#include <expr.h>
#include <os.h>


/*
 * NAME
 *	builtin_find_command - find pathname commands
 *
 * SYNOPSIS
 *	int builtin_find_command(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Find_command is a built-in function of cook, described as follows:
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containing the expanded pathname of the named commands
 *	given as arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_find_command(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	string_ty	*paths;
	char		*cp;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	cp = getenv("PATH");
	if (!cp || !*cp)
	{
		expr_error
		(
			"%s: PATH environment variable not set",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	paths = str_from_c(cp);
	for (j = 1; j < args->wl_nwords; ++j)
	{
		string_ty	*s;
		string_ty	*s2;

		s = args->wl_word[j];
		if (strchr(s->str_text, '/'))
		{
			if (os_exists(s))
			{
				s2 = os_pathname(s);
				wl_append(result, s2);
				str_free(s2);
			}
			else
				wl_append(result, str_false);
		}
		else
		{
			string_ty	*f;
			int		n;

			for (n = 0; ; ++n)
			{
				f = str_field(paths, ':', n);
				if (!f)
				{
					wl_append(result, str_false);
					break;
				}
				s2 = str_format("%S/%S", f, s);
				str_free(f);
				switch (os_exists(s2))
				{
				case -1:
					str_free(s2);
					return -1;

				case 0:
					str_free(s2);
					continue;

				case 1:
					f = os_pathname(s2);
					str_free(s2);
					wl_append(result, f);
					str_free(f);
					break;
				}
				break;
			}
		}
	}
	str_free(paths);
	return 0;
}
