/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin collect functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <stdio.h>
#include <ac/string.h>
#include <errno.h>

#include <builtin/collect.h>
#include <error.h>
#include <expr.h>
#include <option.h>
#include <os.h>


/*
 * NAME
 *	builtin_collect - get output of a command
 *
 * SYNOPSIS
 *	int builtin_collect(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Collect is a built-in function of cook, described as follows:
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containing the values of the output lines of the
 *	program given in the arguments.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_collect(result, args)
	wlist		*result;
	wlist		*args;
{
	FILE		*fp;
	string_ty	*s;
	char		*delim;
	int		status;
	int		silent;
	int		errok;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	s = wl2str(args, 1, args->wl_nwords - 1, (char *)0);
	option_set(OPTION_SILENT, OPTION_LEVEL_EXECUTE, 1);
	option_set(OPTION_ERROK, OPTION_LEVEL_EXECUTE, 0);
	silent = option_test(OPTION_SILENT);
	errok = option_test(OPTION_ERROK);
	if (!silent)
		error("%s", s->str_text);
	fp = popen(s->str_text, "r");
	str_free(s);
	if (!fp)
	{
		expr_error
		(
			"%s: %s",
			args->wl_word[1]->str_text,
			strerror(errno)
		);
		return -1;
	}
	delim = strchr(args->wl_word[0]->str_text, '_') ? "\n" : "\n \t\f";
	for (;;)
	{
		char		buffer[1024];
		char		*cp;
		int		c;

		for (;;)
		{
			c = fgetc(fp);
			if (c == EOF || !strchr(delim, c))
				break;
		}
		if (c == EOF)
			break;
		cp = buffer;
		for (;;)
		{
			*cp++ = c;
			c = fgetc(fp);
			if (c == EOF || strchr(delim, c))
				break;
		}
		s = str_n_from_c(buffer, cp - buffer);
		wl_append(result, s);
		str_free(s);
		if (c == EOF)
			break;
	}
	if (ferror(fp))
	{
		expr_error
		(
			"%s: %s",
			args->wl_word[1]->str_text,
			strerror(errno)
		);
		option_undo_level(OPTION_LEVEL_EXECUTE);
		return -1;
	}
	if (!silent)
		star_sync();
	status = pclose(fp);
	status = exit_status(args->wl_word[0]->str_text, status, errok);
	option_undo_level(OPTION_LEVEL_EXECUTE);
	if (status)
		return -1;
	return 0;
}
