/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to implement the builtin functions
 *
 * The builtin function all append their results to the supplied
 * `result' word list.  The first word of the `args' word list
 * is the name of the function.
 *
 * all of the functions return 0 in success, or -1 on error.
 */

#include <builtin/boolean.h>
#include <error.h>
#include <expr.h>


/*
 * NAME
 *	builtin_if - conditional evaluation
 *
 * SYNOPSIS
 *	int builtin_if(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Defined is a built-in function of if, described as follows:
 *	This function requires one or more arguments.
 *	The condition before the 'then' keyword is evaluated,
 *	if true, the words between the 'then' and the 'else' are the result,
 *	otherwise the words between the 'else' and the end are the value.
 *	The else is optional.
 *
 * RETURNS
 *	Appropriate things, see above.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 *
 *	'then' and 'else' cant be escaped, sorry.
 */

int
builtin_if(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;
	int		cond;
	static string_ty	*str_then;
	static string_ty	*str_else;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	cond = 0;
	if (!str_then)
		str_then = str_from_c("then");
	if (!str_else)
		str_else = str_from_c("else");
	for
	(
		j = 1;
		j < args->wl_nwords && !str_equal(str_then, args->wl_word[j]);
		j++
	)
		cond |= str_bool(args->wl_word[j]);
	if (j >= args->wl_nwords)
	{
		expr_error("%s: no 'then' word", args->wl_word[0]->str_text);
		return -1;
	}
	j++;
	if (cond)
	{
		while
		(
			j < args->wl_nwords
		&&
			!str_equal(str_else, args->wl_word[j])
		)
		{
			wl_append(result, args->wl_word[j]);
			j++;
		}
	}
	else
	{
		while
		(
			j < args->wl_nwords
		&&
			!str_equal(str_else, args->wl_word[j])
		)
			j++;
		if (j < args->wl_nwords)
		{
			j++;
			while (j < args->wl_nwords)
			{
				wl_append(result, args->wl_word[j]);
				++j;
			}
		}
	}
	return 0;
}


/*
 * NAME
 *	builtin_not - logical negation
 *
 * SYNOPSIS
 *	int builtin_not(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Not is a built-in function of cook, described as follows:
 *	This function requires zero or more arguments,
 *	the value to be logically negated.
 *
 * RETURNS
 *	It returns "1" (true) if all of the arguments are "" (false), or there
 *	are no arguments; and returns "" (false) otherwise.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_not(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	for (j = 1; j < args->wl_nwords; j++)
	{
		if (str_bool(args->wl_word[j]))
		{
			wl_append(result, str_false);
			return 0;
		}
	}
	wl_append(result, str_true);
	return 0;
}


/*
 * NAME
 *	builtin_and - logical conjunction
 *
 * SYNOPSIS
 *	int builtin_and(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	And is a built-in function of cook, described as follows:
 *	This function requires at least two arguments,
 *	upon which it forms a logical conjunction.
 *
 * RETURNS
 *	The value returned is "1" (true) if none of the arguments
 *	are "" (false), otherwise "" (false) is returned.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_and(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 3)
	{
		expr_error
		(
			"%s: requires at least two arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		if (!str_bool(args->wl_word[j]))
		{
			wl_append(result, str_false);
			return 0;
		}
	}
	wl_append(result, str_true);
	return 0;
}


/*
 * NAME
 *	builtin_or - logical disjunction
 *
 * SYNOPSIS
 *	int builtin_or(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	Or is a built-in function of cook, described as follows:
 *	This function requires at least two arguments,
 *	upon which it forms a logical disjunction.
 *
 * RETURNS
 *	The value returned is "1" (true) if any one of the arguments is
 *	not "" (false), otherwise "" (false) is returned.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_or(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 3)
	{
		expr_error
		(
			"%s: requires at least two arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 1; j < args->wl_nwords; j++)
	{
		if (str_bool(args->wl_word[j]))
		{
			wl_append(result, str_true);
			return 0;
		}
	}
	wl_append(result, str_false);
	return 0;
}


/*
 * NAME
 *	builtin_in - test for set membership
 *
 * SYNOPSIS
 *	int builtin_in(wlist *result, wlist *args);
 *
 * DESCRIPTION
 *	In is a built-in function of cook, described as follows:
 *	This function requires one or more arguments.
 *
 * RETURNS
 *	A word list containg a single word: "1" (true) if the first argument
 *	is the same as any of the later ones; "" (false) if not.
 *
 * CAVEAT
 *	The returned result is in dynamic memory.
 *	It is the responsibility of the caller to dispose of
 *	the result when it is finished, with a wl_free() call.
 */

int
builtin_in(result, args)
	wlist		*result;
	wlist		*args;
{
	int		j;

	assert(result);
	assert(args);
	assert(args->wl_nwords);
	if (args->wl_nwords < 2)
	{
		expr_error
		(
			"%s: requires one or more arguments",
			args->wl_word[0]->str_text
		);
		return -1;
	}
	for (j = 2; j < args->wl_nwords; j++)
	{
		if (str_equal(args->wl_word[1], args->wl_word[j]))
		{
			wl_append(result, str_true);
			return 0;
		}
	}
	wl_append(result, str_false);
	return 0;
}
