/*
 *	cook - file construction tool
 *	Copyright (C) 1991, 1992, 1993, 1994 Peter Miller.
 *	All rights reserved.
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * MANIFEST: functions to scan source files looking for include files
 */

#include <ac/stddef.h>
#include <stdio.h>
#include <ac/string.h>
#include <errno.h>

#include <cache.h>
#include <error.h>
#include <mem.h>
#include <os.h>
#include <sniff.h>
#include <word.h>
#include <trace.h>

option_ty	option;
char		*prefix;
char		*suffix;
static long	pcount;
static wlist	srl;
static wlist	visited;
static sniff_ty	*lang;


void
sniff_language(lp)
	sniff_ty	*lp;
{
	trace(("sniff_language(lp = %08lX)\n{\n"/*}*/, lp));
	assert(lp);
	lang = lp;
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	sniff_include
 *
 * SYNOPSIS
 *	void sniff_include(string_ty *path);
 *
 * DESCRIPTION
 *	The sniff_include function is used to add to
 *	the standard include paths.
 *
 * ARGUMENTS
 *	path	- path to add
 */

void
sniff_include(path)
	char		*path;
{
	string_ty	*s;

	assert(path);
	trace(("sniff_include(path = \"%s\")\n{\n"/*}*/, path));
	s = str_from_c(path);
	wl_append_unique(&srl, s);
	str_free(s);
	trace((/*{*/"}\n"));
}


long
sniff_include_count()
{
	trace(("sniff_include_count()\n{\n"/*}*/));
	trace(("return %ld;\n", (long)srl.wl_nwords));
	trace((/*{*/"}\n"));
	return srl.wl_nwords;
}


/*
 * NAME
 *	sniff_prepare
 *
 * SYNOPSIS
 *	void sniff_prepare(void);
 *
 * DESCRIPTION
 *	The sniff_prepare function is used to append the standard
 *	search paths after the user-supplied search paths.
 */

void
sniff_prepare()
{
	trace(("sniff_prepare()\n{\n"/*}*/));
	assert(lang);
	lang->prepare();
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	flatten - remove kinks from paths
 *
 * SYNOPSIS
 *	void flatten(char *);
 *
 * DESCRIPTION
 *	The flatten function is used to remove kinks
 *	from unix path named.
 *
 * ARGUMENTS
 *	path	- path to flatten
 *
 * CAVEATS
 *	It doesn't understand symbolic links.
 */

static string_ty *flatten _((string_ty *));

static string_ty *
flatten(s)
	string_ty	*s;
{
	static size_t	tmplen;
	static char	*tmp;
	char		*in;
	char		*out;

	/*
	 * make sure tmp area is big enough
	 */
	trace(("flatten(s = \"%s\")\n{\n"/*}*/, s->str_text));
	if (s->str_length > tmplen)
	{
		tmplen = s->str_length;
		tmp = mem_change_size(tmp, tmplen + 1);
	}

	/*
	 * remove multiple '/'s
	 */
	out = tmp;
	for (in = s->str_text; *in; ++in)
	{
		if (in[0] != '/' || in[1] != '/')
			*out++ = *in;
	}
	*out = 0;

	/*
	 * remove . references
	 */
	in = out = tmp;
	if (*in == '/')
		*out++ = *in++;
	while (*in)
	{
		if (in[0] == '.' && !in[1])
			break;
		if (in[0] == '.' && in[1] == '/')
		{
			in += 2;
			continue;
		}
		while (*in && (*out++ = *in++) != '/')
			;
	}
	*out = 0;

	/*
	 * remove .. references
	 */
	in = out = tmp;
	if (*in == '/')
		*out++ = *in++;
	while (*in)
	{
		if (in[0] == '.' && in[1] == '.' && !in[2])
		{
			if (out == tmp)
			{
				/* ".." -> ".." */
				*out++ = *in++;
				*out++ = *in++;
				break;
			}
			if (out == tmp + 1)
			{
				/* "/.." -> "/" */
				break;
			}
			--out;
			while (out > tmp && out[-1] != '/')
				--out;
			break;
		}
		if (in[0] == '.' && in[1] == '.' && in[2] == '/')
		{
			if (out == tmp)
			{
				/* "../" -> "../" */
				*out++ = *in++;
				*out++ = *in++;
				*out++ = *in++;
				continue;
			}
			if (out == tmp + 1)
			{
				/* "/../" -> "/" */
				in += 3;
				continue;
			}
			in += 3;
			--out;
			while (out > tmp && out[-1] != '/')
				--out;
			continue;
		}
		while (*in && (*out++ = *in++) != '/')
			;
	}
	*out = 0;

	/*
	 * remove trailing '/'
	 */
	if (out > tmp + 1 && out[-1] == '/')
		*--out = 0;

	/*
	 * "" -> "."
	 */
	if (!*tmp)
	{
		tmp[0] = '.';
		tmp[1] = 0;
	}

	/*
	 * return a string
	 */
	trace(("return \"%s\";\n", tmp));
	trace((/*{*/"}\n"));
	return str_from_c(tmp);
}


/*
 * NAME
 *	resolve
 *
 * SYNOPSIS
 *	string_ty *resolve(string_ty *filename, string_ty *extra);
 *
 * DESCRIPTION
 *	The resolve function is used to resolve an include
 *	filename into the path of an existing file.
 *
 * ARGUMENTS
 *	filename - name to be resolved
 *	extra	- extra first search element, if not NULL
 *
 * RETURNS
 *	string_ty *; name of path, or NULL if unmentionable
 */

static string_ty *resolve _((string_ty *filename, string_ty *extra));

static string_ty *
resolve(filename, extra)
	string_ty	*filename;
	string_ty	*extra;
{
	string_ty	*s;
	size_t		j;
	string_ty	*result;

	/*
	 * If the name is absolute, irrespecitive of
	 * which style, we need look no further.
	 */
	trace(("resolve(filename = \"%s\", extra = \"%s\")\n{\n"/*}*/,
		filename->str_text, extra ? extra->str_text : "NULL"));
	if (filename->str_text[0] == '/')
	{
		result = flatten(filename);
		if (os_exists(result->str_text))
			goto done;
		goto dilema;
	}

	/*
	 * Includes of the form "filename" look in the directory
	 * of the parent file, an then the <filename> places.
	 */
	if (extra)
	{
		s = str_format("%S/%S", extra, filename);
		result = flatten(s);
		str_free(s);
		if (option.o_verbose)
			error("may need to look at \"%s\"", result->str_text);
		if (os_exists(result->str_text))
			goto done;
		str_free(result);
	}

	/*
	 * look in all the standard places
	 */
	for (j = 0; j < srl.wl_nwords; ++j)
	{
		s = str_format("%S/%S", srl.wl_word[j], filename);
		result = flatten(s);
		str_free(s);
		if (option.o_verbose)
			error("may need to look at \"%s\"", result->str_text);
		if (os_exists(result->str_text))
			goto done;
		str_free(result);
	}

	/*
	 * not found, must have been ifdef'ed out
	 * or needs to be built
	 */
	dilema:
	switch (extra ? option.o_absent_local : option.o_absent_system)
	{
	default:
		fatal("include file \"%s\" not found", filename->str_text);

	case absent_ignore:
		result = 0;
		break;

	case absent_mention:
		if (filename->str_text[0] == '/')
			result = str_copy(filename);
		else if (extra)
		{
			s = str_format("%S/%S", extra, filename);
			result = flatten(s);
			str_free(s);
		}
		else if (srl.wl_nwords)
		{
			s = str_format("%S/%S", srl.wl_word[0], filename);
			result = flatten(s);
			str_free(s);
		}
		else
			result = flatten(filename);
		break;
	}

	/*
	 * here for all exits
	 */
	done:
	trace(("return \"%s\";\n", result ? result->str_text : "NULL"));
	trace((/*{*/"}\n"));
	return result;
}


/*
 * NAME
 *	stat_equal - compare stat structures
 *
 * SYNOPSIS
 *	vint stat_equal(struct stat *, struct stat *);
 *
 * DESCRIPTION
 *	The stat_equal function is sued to compare two stat structures
 *	for equality.  Only this fields which would change if
 *	the file changed are examined.
 */

static int stat_equal _((struct stat *, struct stat *));

static int
stat_equal(st1, st2)
	struct stat	*st1;
	struct stat	*st2;
{
	return
	(
		st1->st_dev == st2->st_dev
	&&
		st1->st_ino == st2->st_ino
	&&
		st1->st_size == st2->st_size
	&&
		st1->st_mtime == st2->st_mtime
	&&
		st1->st_ctime == st2->st_ctime
	);
}


/*
 * NAME
 *	sniffer - search file for include dependencies
 *
 * SYNOPSIS
 *	void sniffer(string_ty *pathname);
 *
 * DESCRIPTION
 *	The sniffer function is used to walk a file looking
 *	for any files which it includes, and walking then also.
 *	The names of any include files encountered are printed onto
 *	the standard output.
 *
 * ARGUMENTS
 *	pathname	- pathname to read
 *	
 * CAVEATS
 *	Uses the cache where possible to speed things up.
 */

static void sniffer _((string_ty *, int));

static void
sniffer(filename, prnam)
	string_ty	*filename;
	int		prnam;
{
	FILE		*fp;
	cache_ty	*cp;
	struct stat	st;
	size_t		j;

	trace(("sniffer(filename = \"%s\")\n{\n"/*}*/, filename->str_text));
	if (prnam)
	{
		if (!pcount && prefix)
			printf("%s\n", prefix);
		printf("%s\n", filename->str_text);
		++pcount;
	}

	/*
	 * find the file in the cache
	 * (will be created if not already there)
	 */
	cp = cache_search(filename);
	assert(cp);
	if (stat(filename->str_text, &st) < 0)
	{
		/*
		 * here for failure to open/find a file
		 */
		absent:
		switch (errno)
		{
		case ENOENT:
			break;

		case ENOTDIR:
		case EACCES:
			nerror("warning: %s", filename->str_text);
			break;

		default:
			nfatal("%s", filename->str_text);
		}

		/*
		 * zap the stat info,
		 * and pretend the file was empty
		 */
		memset(&cp->st, 0, sizeof(cp->st));
		wl_free(&cp->ingredients);
		cache_update_notify();
		if (option.o_verbose)
			error("bogus empty \"%s\" file", filename->str_text);
		goto done;
	}

	/*
	 * if the stat in the cache is not the same
	 * as the state just obtained, reread the file.
	 */
	if (!stat_equal(&st, &cp->st))
	{
		wlist	type1;
		wlist	type2;

		cp->st = st;
		wl_free(&cp->ingredients);
		cache_update_notify();
		if (option.o_verbose)
			error("cache miss for \"%s\" file", filename->str_text);

		fp = fopen(filename->str_text, "r");
		if (!fp)
		{
			/*
			 * probably "permission denied" error,
			 * but file could have suddenly vanished
			 */
			goto absent;
		}
		wl_zero(&type1);
		wl_zero(&type2);
		if (lang->scan(fp, &type1, &type2) || fclose(fp))
			nfatal("%s", filename->str_text);

		/*
		 * type2 names have an implicit first element of the search path
		 * which is the directory of the including file
		 */
		if (type2.wl_nwords)
		{
			string_ty	*parent;
			char		*ep;
			char		*sp;

			sp = filename->str_text;
			ep = strrchr(sp, '/');
			if (ep)
				parent = str_n_from_c(sp, ep - sp);
			else
				parent = str_from_c(".");
			for (j = 0; j < type2.wl_nwords; ++j)
			{
				string_ty	*path;

				path = resolve(type2.wl_word[j], parent);
				if (path)
				{
					wl_append_unique(&cp->ingredients, path);
					str_free(path);
				}
			}
			str_free(parent);
		}

		/*
		 * type1 names scan the search path
		 */
		for (j = 0; j < type1.wl_nwords; ++j)
		{
			string_ty	*path;

			path = resolve(type1.wl_word[j], (string_ty *)0);
			if (path)
			{
				wl_append_unique(&cp->ingredients, path);
				str_free(path);
			}
		}

		/*
		 * let the lists go
		 */
		wl_free(&type1);
		wl_free(&type2);
	}

	/*
	 * work down the ingredients list
	 * to see if there are more dependencies
	 */
	wl_append_unique(&visited, filename);
	for (j = 0; j < cp->ingredients.wl_nwords; ++j)
	{
		string_ty	*s;

		s = cp->ingredients.wl_word[j];
		if (!wl_member(&visited, s))
			sniffer(s, 1);
	}

	/*
	 * here for all exits
	 */
	done:
	trace((/*{*/"}\n"));
}


/*
 * NAME
 *	sniff - search file for include dependencies
 *
 * SYNOPSIS
 *	void sniff(char *pathname);
 *
 * DESCRIPTION
 *	The sniff function is used to walk a file looking
 *	for any files which it includes, and walking then also.
 *	The names of any include files encountered are printed onto
 *	the standard output.
 *
 * ARGUMENTS
 *	pathname	- pathname to read
 */

void
sniff(filename)
	char		*filename;
{
	string_ty	*s;

	assert(filename);
	trace(("sniff(filename = \"%s\")\n{\n"/*}*/, filename));
	if (!os_exists(filename))
	{
		switch (option.o_absent_program)
		{
		case absent_error:
			fatal("%s: no such file", filename);
			break;

		case absent_mention:
			error("warning: %s: no such file", filename);
			break;

		default:
			break;
		}
		goto done;
	}
	s = str_from_c(filename);
	sniffer(s, 0);
	str_free(s);
	if (pcount && suffix)
		printf("%s\n", suffix);
	done:
	trace((/*{*/"}\n"));
}
