# ColorEd.tcl --
#
#	Implements the GL Widget "TixColorEdit"
#

tixWidgetClass tixColorEdit {
    -classname TixColorEdit
    -superclass tixPrimitive
    -method {
    }
    -flag {
	-adjustsize -background -bg -foreground -fg
	-rgb -command -state -disabledforeground
    }
    -configspec {
	{-adjustsize adjustsize Adjustsize 0}
	{-background background Background #ffe4c4}
	{-foreground foreground Foreground black}
	{-rgb rgb Rgb {1.0 1.0 1.0}}
	{-command command Command {}}
	{-state state State normal}
	{-disabledforeground -disabledForeground DisabledForeground #606060}
    }
    -alias {
	{-bg -background}
	{-fg -foreground}
    }
}

proc tixColorEdit::InitWidgetRec {w} {
    upvar #0 $w data

    tixChainMethod $w InitWidgetRec

    set data(flag)    none
    set data(active)  1
    set data(Rscale)  0
    set data(Gscale)  0
    set data(Bscale)  0
    set data(Hscale)  0
    set data(Sscale)  0
    set data(Vscale)  1

    set data(oldhsv)  {}
    tixColorEdit::RGBtoHSV  $w
}

#--------------------------
# Create Widget
#--------------------------
proc tixColorEdit::ConstructWidget {w} {
    upvar #0 $w data

    tixChainMethod $w ConstructWidget

    set data(active) 0

    set frame1 [tixColorEdit::MkFrame1 $w]
    set frame2 [tixColorEdit::MkFrame2 $w]
    set frame3 [tixColorEdit::MkFrame3 $w]

    pack $frame1 -in $w  -side top -fill x
    pack $frame2 -in $w  -side top -fill x
    pack $frame3 -in $w  -side top -fill x

    $data(menu1) invoke 0

    set data(active) 1
}

proc tixColorEdit::MkFrame1 {w} {
    upvar #0 $w data

    # 1. MENU
    # -------
    frame $w.menu -relief raised -borderwidth 2

    # OPTION MENU
    menubutton $w.menu.option -text "Option" -menu $w.menu.option.m
    menu $w.menu.option.m

    set data(scalemode) 1
    set var [format %s(scalemode) $w]
    $w.menu.option.m add radiobutton -label "Value" \
        -command "tixColorEdit::SetScales $w V"	     -variable $var -value 1
    $w.menu.option.m add radiobutton -label "HSV" \
	-command "tixColorEdit::SetScales $w HSV"    -variable $var -value 2
    $w.menu.option.m add radiobutton -label "RGB" \
        -command "tixColorEdit::SetScales $w RGB"    -variable $var -value 3
    $w.menu.option.m add radiobutton -label "RGB V" \
        -command "tixColorEdit::SetScales $w RGBV"   -variable $var -value 4
    $w.menu.option.m add radiobutton -label "RGBHSV" \
        -command "tixColorEdit::SetScales $w RGBHSV" -variable $var -value 5

    tixDoWhenIdle tixColorEdit::SetScales $w V

    pack $w.menu.option -side left

    tk_menuBar $w.menu $w.menu.option

    # SET DATA STRUCT
    #----------------
    set data(menu1) $w.menu.option.m

    return $w.menu
}

proc tixColorEdit::MkFrame2 {w} {
    upvar #0 $w data
    
    frame $w.f2  -borderwidth 0
    frame $w.f22 -relief raised -borderwidth 4
    frame $w.f23 -borderwidth 0

    # Make the left side : color block
    #--------------------------------
    tixLabelFrame $w.labf -label "Selection"
    set f [$w.labf subwidget frame]
    frame $f.f -relief sunken -borderwidth 4 -bg black
    tixColorBlock $f.cdemo
    $f.cdemo config -rgb $data(-rgb)

    pack $f.cdemo -in $f.f -fill both -expand yes
    pack $f.f -in $f -fill x -padx 15 -pady 5

    # Make the right side : color space
    #----------------------------------
    tixColorSpace $w.cspace -command "tixColorEdit::PickColor $w"
    $w.cspace config -hsv $data(hsv)
    pack $w.cspace -in $w.f22 -expand yes

    pack $w.labf -in $w.f2 -side left  -expand yes -fill both
    pack $w.f22  -in $w.f2 -side right -expand yes

    set data(cdemo)  $f.cdemo
    set data(cspace) $w.cspace
    return $w.f2
}

proc tixColorEdit::MkFrame3 {w} {
    upvar #0 $w data

    set f [frame $w.f3]
    set data(scale_frame) $f

    scale $f.scale1 -from 0.0 -to 1.000 -length 5c -orient horizontal \
	-command "after 0 tixColorEdit::ScaleChange $w R" -label "Red" \
	-bigincrement 5.0 -resolution 0.01
    scale $f.scale2 -from 0.0 -to 1.000 -length 5c -orient horizontal \
	-command "after 0 tixColorEdit::ScaleChange $w G" -label "Green" \
	-bigincrement 5.0 -resolution 0.01
    scale $f.scale3 -from 0.0 -to 1.000 -length 5c -orient horizontal \
	-command "after 0 tixColorEdit::ScaleChange $w B" -label "Blue" \
	-bigincrement 5.0 -resolution 0.01
    scale $f.scale4 -from 0.0 -to 1.000 -length 5c -orient horizontal \
	-command "after 0 tixColorEdit::ScaleChange $w H" -label "Hue" \
	-bigincrement 5.0 -resolution 0.01
    scale $f.scale5 -from 0.0 -to 1.000 -length 5c -orient horizontal \
	-command "after 0 tixColorEdit::ScaleChange $w S" -label "Satuation" \
	-bigincrement 5.0 -resolution 0.01
    scale $f.scale6 -from 0.0 -to 1.000 -length 5c -orient horizontal \
	-command "after 0 tixColorEdit::ScaleChange $w V" -label "Value" \
	-bigincrement 5.0 -resolution 0.01

    pack $f.scale1 -fill x
    pack $f.scale2 -fill x
    pack $f.scale3 -fill x
    pack $f.scale4 -fill x
    pack $f.scale5 -fill x
    pack $f.scale6 -fill x

    return $f
}

#----------------------------------------------------------------------
#                         Config Options
#----------------------------------------------------------------------
proc tixColorEdit::config-rgb {w arg} {
    upvar #0 $w data

    tixColorEdit::RGBtoHSV $w
    tixColorEdit::ResetScales $w
}

proc tixColorEdit::config-state {w arg} {
    upvar #0 $w data

    set f $w.f3

    if {$arg == "normal"} {
	$f.scale1 config -state normal -fg $data(-foreground)
	$f.scale2 config -state normal -fg $data(-foreground)
	$f.scale3 config -state normal -fg $data(-foreground)
	$f.scale4 config -state normal -fg $data(-foreground)
	$f.scale5 config -state normal -fg $data(-foreground)
	$f.scale6 config -state normal -fg $data(-foreground)
	$data(cspace) config -state normal
    } else {
	$f.scale1 config -state disabled -fg $data(-disabledforeground)
	$f.scale2 config -state disabled -fg $data(-disabledforeground)
	$f.scale3 config -state disabled -fg $data(-disabledforeground)
	$f.scale4 config -state disabled -fg $data(-disabledforeground)
	$f.scale5 config -state disabled -fg $data(-disabledforeground)
	$f.scale6 config -state disabled -fg $data(-disabledforeground)
	$data(cspace) config -state disabled
    }
}

#----------------------------------------------------------------------
#                         Internal commands
#----------------------------------------------------------------------
proc tixColorEdit::SetScales {w flag} {
    if {![winfo exists $w]} {
	return
    }

    upvar #0 $w data

    if {$data(flag) == $flag} {
	return
    } else {
	set data(flag) $flag
    }

    case $flag in {
	V {
	    set data(Rscale) 0
	    set data(Gscale) 0
	    set data(Bscale) 0
	    set data(Hscale) 0
	    set data(Sscale) 0
	    set data(Vscale) 1
	}
	RGB {
	    set data(Rscale) 1
	    set data(Gscale) 1
	    set data(Bscale) 1
	    set data(Hscale) 0
	    set data(Sscale) 0
	    set data(Vscale) 0

	}
	HSV {
	    set data(Rscale) 0
	    set data(Gscale) 0
	    set data(Bscale) 0
	    set data(Hscale) 1
	    set data(Sscale) 1
	    set data(Vscale) 1
	}
	RGBV {
	    set data(Rscale) 1
	    set data(Gscale) 1
	    set data(Bscale) 1
	    set data(Hscale) 0
	    set data(Sscale) 0
	    set data(Vscale) 1
	}
	RGBHSV {
	    set data(Rscale) 1
	    set data(Gscale) 1
	    set data(Bscale) 1
	    set data(Hscale) 1
	    set data(Sscale) 1
	    set data(Vscale) 1
	}
	default {
	    return
	}
    }
    tixColorEdit::ReplaceScales $w
}

proc tixColorEdit::ReplaceScales {w} {
    upvar #0 $w data

    set f $w.f3

    pack $f.scale1 -fill x
    pack $f.scale2 -fill x -after $f.scale1
    pack $f.scale3 -fill x -after $f.scale2
    pack $f.scale4 -fill x -after $f.scale3
    pack $f.scale5 -fill x -after $f.scale4
    pack $f.scale6 -fill x -after $f.scale5

    if {$data(Rscale) == "0"} {
	pack forget $f.scale1
    }
    if {$data(Gscale) == "0"} {
	pack forget $f.scale2
    }
    if {$data(Bscale) == "0"} {
	pack forget $f.scale3
    }
    if {$data(Hscale) == "0"} {
	pack forget $f.scale4
    }
    if {$data(Sscale) == "0"} {
	pack forget $f.scale5
    }
    if {$data(Vscale) == "0"} {
	pack forget $f.scale6
    }

    if {$data(-adjustsize)} {
	after 100 tixColorEdit::ResetSize $w
    }

    tixColorEdit::ResetScales $w
}

proc tixColorEdit::ResetSize {w} {
    upvar #0 $w data

    set top [winfo toplevel $w]
    set W [winfo reqwidth  $top]
    set H [winfo reqheight $top]
    if {$W != 1 && $H != 1} {
	wm geometry $top $W\x$H
    } else {
	tixWidgetDoWhenIdle tixColorEdit::ResetSize $w
    }
}

proc tixColorEdit::ResetScales {w} {
    upvar #0 $w data

    tixColorEdit::ResetRGB $w
    tixColorEdit::ResetHSV $w
    $data(cspace) config -hsv $data(hsv)
    $data(cdemo)  config -rgb $data(-rgb)
}

proc tixColorEdit::ResetRGB {w} {
    upvar #0 $w data

    set data(active) 0
    set f $data(scale_frame)

    if {$data(Rscale) == "1"} {
	$f.scale1 set [lindex $data(-rgb) 0]
    }
    if {$data(Gscale) == "1"} {
	$f.scale2 set [lindex $data(-rgb) 1]
    }
    if {$data(Bscale) == "1"} {
	$f.scale3 set [lindex $data(-rgb) 2]
    }

    set data(active) 1
}

proc tixColorEdit::ResetHSV {w} {
    upvar #0 $w data

    set data(active) 0
    set f $data(scale_frame)

    if {$data(Hscale) == "1"} {
	$f.scale4 set [lindex $data(hsv) 0]
    }
    if {$data(Sscale) == "1"} {
	$f.scale5 set [lindex $data(hsv) 1]
    }
    if {$data(Vscale) == "1"} {
	$f.scale6 set [lindex $data(hsv) 2]
    }
    set data(active) 1
}

proc tixColorEdit::ScaleChange {w flag value} {
    if {![winfo exists $w]} {
	return
    }

    upvar #0 $w data

    if {$data(active) == "0"} {
	return
    }
    set f $data(scale_frame)

    case $flag in {
	{R G B} {
	    set r [$f.scale1 get]
	    set g [$f.scale2 get]
	    set b [$f.scale3 get]
	    set data(-rgb) "$r $g $b"
	    tixColorEdit::RGBtoHSV $w
	    tixColorEdit::ResetHSV $w
	    $data(cspace) config -hsv $data(hsv)
 	    $data(cdemo)  config -rgb $data(-rgb)
	}
	{H S V} {
	    if {$data(Hscale) == "1"} {
		set h [$f.scale4 get]
		set s [$f.scale5 get]
	    } else {
		set h [lindex $data(hsv) 0]
		set s [lindex $data(hsv) 1]
	    }
	    set v [$f.scale6 get]

	    # The following code tries to remember the
	    # H and S of a color when V becomes 0.
	    #
	    # ToDo: it doesn't work well ..
	    if {$v == 0} {
		if {$data(oldhsv) == {}} {
		    set data(oldhsv) $data(hsv)
		}
	    } else {
		if {$data(oldhsv) != {}} {
		    set oldH [lindex $data(oldhsv) 0]
		    set oldS [lindex $data(oldhsv) 0]
		    set oldV [lindex $data(oldhsv) 0]

		    if {$oldV != 0} {
			set h $oldH
			set s $oldS
		    }
		}
		set data(oldhsv) {}
	    }
	    set data(hsv) "$h $s $v"

	    tixColorEdit::HSVtoRGB $w
	    tixColorEdit::ResetRGB $w
	    $data(cspace) config -hsv $data(hsv)
 	    $data(cdemo)  config -rgb $data(-rgb)
	}
    }

    if {$data(-command) != {}} {
	eval $data(-command) \
	    [lindex $data(-rgb) 0] \
	    [lindex $data(-rgb) 1] \
	    [lindex $data(-rgb) 2]
    }
}

proc tixColorEdit::PickColor {w h s v} {
    upvar #0 $w data

    set data(hsv) "$h $s $v"
    tixColorEdit::HSVtoRGB $w
    tixColorEdit::ResetRGB $w
    tixColorEdit::ResetHSV $w
    $data(cdemo) config -rgb $data(-rgb)

    if {$data(-command) != {}} {
	eval $data(-command) \
	    [lindex $data(-rgb) 0] \
	    [lindex $data(-rgb) 1] \
	    [lindex $data(-rgb) 2]
    }
}

proc tixColorEdit::RGBtoHSV {w} {
    upvar #0 $w data
    set data(hsv) [glwRgbToHsv \
       [lindex $data(-rgb) 0] \
       [lindex $data(-rgb) 1] \
       [lindex $data(-rgb) 2]]
}

proc tixColorEdit::HSVtoRGB {w} {
    upvar #0 $w data
    set data(-rgb) [glwHsvToRgb \
       [lindex $data(hsv) 0] \
       [lindex $data(hsv) 1] \
       [lindex $data(hsv) 2]]
}
