#ifndef lint
static char *RCSid = "$Id: variable.c,v 1.9 1992/04/05 19:46:55 anders Exp anders $";
#endif

/*
 * Copyright (C) 1992 Anders Christensen <anders@solan.unit.no>
 * Read file README for more information on copying
 */

/*
 * $Log: variable.c,v $
 * Revision 1.9  1992/04/05  19:46:55  anders
 * Added copyright notice.
 * Included stdio.h, since sun need it for assert.h (yuk!)
 * Implemented code for dropping variables.
 * Marked memory allocated for environement names
 * Swapped strcasecmp with my own routine, for compatibility
 *
 * Revision 1.8  1992/03/23  21:14:07  anders
 * Fixed bug that made symbol() return incorrect value, problem was in
 *    looking up variables, where upper/lower case letters were not
 *    properly treated.
 *
 * Revision 1.7  1992/03/22  18:51:23  anders
 * Fixed bug, subst_index() returned pointer to automatic allocated
 *    array of bytes.
 *
 * Revision 1.6  1992/03/22  01:05:58  anders
 * #include'd <strings.h> which is not included in rexx.h anymore
 * Changed isvariable to return the pointer to the variable if
 *    it found one, not just a boolean.
 *
 * Revision 1.5  1992/03/01  19:24:30  anders
 * Added support for memory management.
 *
 * Revision 1.4  1991/06/03  02:35:54  anders
 * Made a todo list
 * Added empty entries for dropping variables
 * Added empty entries for dumping datastructures, and printing
 *     statistics about the useage of memory etc
 * Created a new make_stem() routine
 * Let subst_index() get the datastructure by parameter, not as global
 * Added new routines for exposing variables.
 *
 * Revision 1.3  91/06/02  22:20:18  anders
 * Major changes to the implementation of variables.
 * Wrote a lot of inline documentation
 * Renamed swapvalue() to replace_value() ... more suitable name
 * Declared all only-local functions as 'static'
 * Removed bug in newbox (forgot to set ->index to NULL)
 * Inline newlev() into setvalue_compound()
 * Rewrote hashfunc(), used arrays instead of pointers, and changed 
 *    the functionallity slightly, so 2nd parameter mey be NULL
 * Devided getvalue() and setvalue() into three parts each, leaving 
 *    the two old routines just as entrylevel routines deciding 
 *    which spesific routine to call.
 * Extracted parts of ???value_compound() into subst_index()
 * Extracted parts of setvalue_stem() into kill_index()
 * Created expose_var() ... though it is still empty
 * Introduced bugs into handling of tracing
 * Still buggy if ptr->value==NULL
 * 
 * Revision 1.2  90/08/09  04:04:55  anders
 * Changed magic numbers with TRC_* macros
 * Put enough parameters in markall() calls
 * 
 * Revision 1.1  90/08/08  02:14:27  anders
 * Initial revision
 * 
 */

/* 
 * Concept: Each REXX procedure (the top - or main - by default) has
 *  an array of hash-pointers. Routines that don't keep local
 *  variables (i.e hasn't seen a PROCEDURE) are using the hashpointers
 *  of the routine above them. The size of the array is HASHTABLENGTH.
 *
 * Each cell in this array is a pointer to a single-linked list of
 *  boxes. In common for all these boxes is that their name returns
 *  the same value when given as parameter to the hashfunc() function.
 *  
 * Each of these boxes contains five variables: name, value, index,
 *  realbox and next. 'next' points to next box in the list. 
 * 
 * 'name' is the name of the variable, and 'value' is the value it
 *  contains. However, if 'realbox' is set, it points to another
 *  box which contains the real value of the variable. This mechanism
 *  gives support for EXPOSE'ing variables in PROCEDUREs. 
 *
 * The 'index' is a pointer to another hashtable, and gives support
 *  for compound variables. If a variable is compound, its 'index' is
 *  set to point at the hashtable, each entry in this table do also
 *  point at the start of a single linked list of variable boxes, but
 *  these boxes has the 'after-the-period' part of the compound name
 *  as 'name'. The 'realbox', but not the 'index' may be set in these
 *  boxes. 
 *
 * A variable is set when it exists in the datastructures, and the
 *  relevant 'value' pointer is non-NULL. When dropping a variable
 *  that is EXPOSE'ed, the 'value' is set to NULL. 
 *
 * The 'test' and the 'test.' variables have two different
 *  variableboxes, and 'index' is only set in the box for 'test.'. A
 *  'realbox' existing for 'test' makes it exposed. A 'realbox'
 *  'test.' make the whole "array" exposed. 
 * 
 * A 'value' existing for 'test.' denotes the default value. 
 *
 * Yet to do:
 *
 *    o the datastructure for the variables should be local, not global
 *    o must implement the code for dropping variables.
 *    o dont always handle ptr->value==NULL correct
 *    o tracing is incorrect
 */

#include "rexx.h"
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <stdio.h>   /* f*ck sun, they can't write a proper assert!!! */

static int foundflag=FALSE ;

#ifdef TRACEMEM
void markvariables()
{
   extern proclevel currlevel ;
   proclevel procptr ;
   variableptr vptr ;
   labelboxptr lptr ;
   paramboxptr pptr ;
   int i ;
   
   for(procptr=currlevel;procptr;procptr=procptr->prev) 
   {
     markmemory( procptr->environment, TRC_VARBOX ) ;
     markmemory( procptr->prev_env, TRC_VARBOX ) ;
      for(i=0;i<HASHTABLENGTH;i++) 
         for(vptr=(procptr->vars)[i];vptr;vptr=vptr->next) 
         {
            markmemory((char*)vptr,TRC_VARBOX) ;
            if (vptr->name)
               markmemory((char*)vptr->name,TRC_VARNAME) ;
            if (vptr->value)
               markmemory((char*)vptr->value,TRC_VARVALUE) ; 
         }
      markmemory((char*)procptr,TRC_PROCBOX) ;
      for (lptr=procptr->first; lptr; lptr=lptr->next)
	 markmemory((char*)lptr, TRC_LABEL) ;
      
      markmemory((char*)procptr->vars,TRC_HASHTAB) ; 
      if (procptr->args) 
      {
         for (pptr=procptr->args; pptr; pptr=pptr->next) {
            markmemory((char*) pptr, TRC_PROCARG) ;
            if (pptr->value) 
               markmemory((char*) pptr->value, TRC_PROCARG) ;
         }
      }
   }
}
#endif /* TRACEMEM */


static void replace_value( char *value, variableptr ptr )
{
   int length ;

   if (ptr->value)
      Free(ptr->value) ;

   ptr->value = Malloc(length=(strlen(value)+1)) ;
   memcpy(ptr->value,value,length) ; 
}

static void make_stem( char *name, char *value, variableptr *oldptr, int len )
{
   variableptr ptr ;

   ptr = (variableptr)Malloc(sizeof(variable)) ;
   ptr->value = value ;
   ptr->next = *oldptr ;
   ptr->realbox = NULL ;
   ptr->index = inithashtbl() ;
   strncpy(ptr->name=Malloc(len+1),name,len) ;
   ptr->name[len] = 0x00 ;
   *oldptr = ptr ; 
}


static variableptr newbox( char *name, char *value, variableptr *oldptr ) 
{
   int length ;
   variableptr newptr ;
   
   newptr = (variableptr)Malloc(sizeof(variable)) ;
   newptr->next = *oldptr ;
   newptr->realbox = NULL ;
   newptr->index = NULL ;

   *oldptr = newptr ;
   newptr->name = Malloc(length=(strlen(name)+1)) ;
   memcpy(newptr->name,name,length) ;

   if (value) {
      newptr->value = Malloc(length=(strlen(value)+1)) ;
      memcpy(newptr->value,value,length) ; }
   else
      newptr->value = NULL ;
   return newptr ;
}


static int hashfunc( char *name, char **ptr ) 
{
   int sum=0, idx=0, i ;
   
   for (i=0; (name[i]) && ((!ptr)||(i==0)||(name[i-1]!='.')) ;i++)
      if (('0'<=name[i]) && (name[i]<='9')) 
         idx = idx*10 + name[i] - '0' ;
      else if (('a'<=name[i])&&(name[i]<='z'))
      { 
         sum += abs(name[i]) - 0x20 + idx ;
         idx = 0 ;
      }
      else
      {
         sum += abs(name[i]) + idx ;
         idx = 0 ; 
      }
  
   if (ptr)
/*         *ptr = &(name[((name[i]) ? (i-1) : (i))]) ; */
   *ptr = &(name[i]) ;

   return( (sum + idx) % HASHTABLENGTH ) ;
}

variableptr *inithashtbl() 
{
   variableptr *ptr ;
   int i ;

   ptr = (variableptr *)Malloc( HASHTABLENGTH*sizeof(variableptr) ) ;
   for (i=0;i<HASHTABLENGTH;ptr[i++]=NULL) ;

   return(ptr) ;
}

char *isvariable( char *str )
{
   extern int foundflag ;
   char *ptr ;

   ptr = getvalue(str,FALSE) ;
   if (foundflag)
      return ptr ;

   Free( ptr ) ;
   return NULL ;
}


static char *subst_index( char *cptr, variableptr *vars ) 
{
   int index=0, length, hashval, i ;
   char *cptr2 ;
   static char indexstr[MAX_INDEX_LENGTH+1] ;
   variableptr nptr ;

   for (cptr2=cptr;(*cptr2);cptr=cptr2) {
      hashval = hashfunc(cptr,&cptr2) ;
      length = cptr2-cptr ;
      for (nptr=vars[hashval];
         (nptr)&&((strnccmp(nptr->name,cptr,length))||
         ((nptr->name)[length]));nptr=nptr->next) ;

      if (nptr)  {   /* remember to record that substitution took place */
         for (;(nptr->realbox);nptr=nptr->realbox) ;
         memcpy(&indexstr[index],nptr->value,length=strlen(nptr->value)) ; }
      else 
         for (i=0; i<length; i++)
            indexstr[index+i] = toupper(cptr[i]) ;

      indexstr[(index+=(length+1))-1] = '.' ; } 

   indexstr[(index)?(index-1):0] = '\000' ;
   return indexstr ;
}



static void kill_index( variableptr *array ) 
{
   variableptr ptr, tptr ;
   int i ;

   for (i=0;i<HASHTABLENGTH;array[i++]=NULL)
      if (tptr=array[i]) 
         for (;ptr=tptr;) {
            tptr = tptr->next ;
            Free(ptr->name) ;
            if (ptr->value)
               Free(ptr->value) ;
            Free((char*)ptr) ; }
}

 

static void setvalue_simple( char *name, char *value ) 
{
   extern proclevel currlevel ;
   variableptr ptr ;
   int hashval ;

   ptr = currlevel->vars[hashval=hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   if (ptr) {
      for (;(ptr->realbox);ptr=ptr->realbox) ;
      replace_value(value,ptr) ; }
   else 
      newbox( name, value, &((currlevel->vars)[hashval]) ) ; 
}



static void setvalue_stem( char *name, char *value ) 
{
   extern proclevel currlevel ;
   variableptr ptr ;
   int hashval ;

   ptr = currlevel->vars[hashval=hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;

   if (ptr) {
      for (;(ptr->realbox);ptr=ptr->realbox) ;
      replace_value(value,ptr) ;
      kill_index(ptr->index) ; }
   else {
      newbox(name,value,&currlevel->vars[hashval]) ;
      currlevel->vars[hashval]->index = inithashtbl() ; }
}  


static void setvalue_compound( char *name, char *value ) 
{
   extern proclevel currlevel ;
   int hashval, length; 
   variableptr ptr, nptr ;
   char *cptr, *indexstr ;

   ptr = currlevel->vars[hashval=hashfunc(name,&cptr)] ;
   length = cptr - name ;
   for (;(ptr)&&(memcmp(ptr->name,name,length));ptr=ptr->next) ;

   if (!ptr) { 
      ptr = (variableptr)Malloc(sizeof(variable)) ;
      ptr->value = NULL ;
      ptr->next = currlevel->vars[hashval] ;
      ptr->realbox = NULL ;
      ptr->index = inithashtbl() ;
      strncpy(ptr->name=Malloc(length+1),name,length) ;
      ptr->name[length] = 0x00 ;
      currlevel->vars[hashval] = ptr ; }

   for (;(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( cptr, currlevel->vars ) ;
   hashval = hashfunc(indexstr,NULL) ;

   nptr = ((variableptr *)(ptr->index))[hashval] ;
   for (;(nptr)&&(strcmp(nptr->name,indexstr));nptr=nptr->next) ;

   if (nptr) {
      for (;(nptr->realbox);nptr=nptr->realbox) ;
      replace_value(value,nptr) ; }
   else 
      newbox(indexstr,value,&(((variableptr *)(ptr->index))[hashval])) ; 
}




static void expose_simple( variableptr *table, char *name ) 
{
   extern proclevel currlevel ;
   int hashval ;
   variableptr ptr ;

   ptr = table[hashval=hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   if (ptr)  /* hey, you just exposed that one! */
      return ;   

   ptr = currlevel->vars[hashval] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

   if (!ptr) 
      newbox(name,NULL,&currlevel->vars[hashval]) ;
   newbox(name,NULL,&table[hashval]) ;
   table[hashval]->realbox = ((ptr) ? (ptr) : currlevel->vars[hashval]) ; 
}



static void expose_stem( variableptr *table, char *name ) 
{
   extern proclevel currlevel ;
   variableptr ptr, tptr ;
   int hashval ;

   ptr = table[hashval=hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   if ((ptr)&&(ptr->realbox))
      return ; /* once is enough !!! */

   tptr = currlevel->vars[hashval] ;
   for (;(tptr)&&(strccmp(tptr->name,name));tptr=tptr->next) ;

   if (!tptr) {
      newbox(name,NULL,&currlevel->vars[hashval]) ;
      (tptr=currlevel->vars[hashval])->index = inithashtbl() ; }

   if (ptr) {
      kill_index(ptr->index) ; 
      Free((char*)ptr->index) ;
      ptr->index = NULL ;
      if (ptr->realbox!=tptr)
         exiterror(ERR_INTERPRETER_FAILURE) ; } /* probably not needed ... */

   else {
      newbox(name,NULL,&table[hashval]) ;
      table[hashval]->realbox = tptr ; } /* dont need ->index */
}  



static void expose_compound( variableptr *table, char *name ) 
{
   extern proclevel currlevel ;
   int hashval, length, hashval2 ; 
   variableptr ptr, nptr, tptr ;
   char *cptr, *indexstr ;

   ptr = table[hashval=hashfunc(name,&cptr)] ;
   length = cptr - name ;
   for (;(ptr)&&(strnccmp(ptr->name,name,length));ptr=ptr->next) ;
   if ((ptr)&&(ptr->realbox))
      return ; /* whole array already exposed */

   indexstr = subst_index( cptr, table ) ;
   if (!ptr) {
      make_stem(name,NULL,&table[hashval],length) ;
      ptr = table[hashval] ; }

   nptr = (ptr->index)[hashval2=hashfunc(indexstr,NULL)] ;
   for (;(nptr)&&(strcmp(nptr->name,indexstr));nptr=nptr->next) ;
   if ((nptr)&&(nptr->realbox))
      return ; /* can't your remember *anything* !!! */
   else {
      newbox(indexstr,NULL,&ptr->index[hashval2]) ;
      nptr = ptr->index[hashval2] ; }

   tptr = currlevel->vars[hashval] ;
   for (;(tptr)&&(strnccmp(tptr->name,name,length));tptr=tptr->next) ;
   for (;(tptr->realbox);tptr=tptr->realbox) ;

   if (!tptr) {
      make_stem(name,NULL,&currlevel->vars[hashval],length) ;
      newbox(indexstr,NULL,&currlevel->vars[hashval]->index[hashval2]) ;
      nptr = currlevel->vars[hashval]->index[hashval2] ; }

   nptr->realbox = currlevel->vars[hashval]->index[hashval2] ; 
}



static char *getvalue_simple( char *name ) 
{
   extern proclevel currlevel ;
   extern int foundflag ;
   variableptr ptr ;
   char *value ;

   ptr = currlevel->vars[hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

   value = ((foundflag=((ptr)&&(ptr->value))) ? ptr->value : name ) ;
   value = strcpy(Malloc(strlen(value)+1),value) ;
/*    if (trace)   
         tracevalue(value,((ptr) ? 'V' : 'L')) ; } */

   return value ;
}


static char *getvalue_stem( char *name )
{
   return getvalue_simple( name ) ;
}


static char *getvalue_compound( char *name ) 
{
   int hashval, baselength, subst=FALSE ;
   variableptr ptr, nptr ;
   char *value, *cptr, *indexstr ;
   extern proclevel currlevel ;
   extern int foundflag ;

   ptr = currlevel->vars[hashval=hashfunc(name,&cptr)] ;
   baselength = cptr-name ;
   for (;(ptr)&&(strnccmp(ptr->name,name,baselength));ptr=ptr->next) ; 
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( cptr, currlevel->vars ) ;
   hashval = hashfunc(indexstr,NULL) ;

   if (subst)   /* trace it */
      tracecompound(name,baselength-1,indexstr) ;

   if (ptr) {   /* find spesific value */
      nptr = ((variableptr *)(ptr->index))[hashval] ;
      for (;(nptr)&&(strcmp(nptr->name,indexstr));nptr=nptr->next) ; 
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ; }

   if ((ptr)&&(!nptr))   /* find default value */
      nptr = ptr ;

   if (foundflag=((ptr)&&(nptr)&&(nptr->value)))
      strcpy(value=Malloc(strlen(nptr->value)+1), nptr->value) ; 
   else {
      memcpy(value=Malloc(strlen(indexstr)+baselength+1),name,baselength) ;
      memcpy(&value[baselength],indexstr,strlen(indexstr)+1) ; }

   return( value ) ;
} 



/* 
 * This is the entry-level routine that will take the parameters,
 *  decide what kind of variable it is (simple, stem or compound) and
 *  call the appropriate routine to do the dirty work
 */
void setvalue( char *name, char *value )
{
   int i ;

   for (i=0;(name[i])&&(name[i]!='.');i++) ;
   if (!name[i]) 
      setvalue_simple(name,value) ;
   else if (!name[i+1])
      setvalue_stem(name,value) ;
   else
      setvalue_compound(name,value) ;
}

     
void expose_var( char* name )
{
   int i ;
   static variableptr *table=NULL ;
   extern proclevel currlevel ;
   
   if (!table)
      table = inithashtbl() ;

   if (!name) {
      currlevel->vars = table ;
      currlevel->varflag = 1 ;
      table = NULL ; 
      return ; }

   for (i=0;(name[i])&&(name[i]!='.');i++) ;
   if (!name[i]) 
      expose_simple(table,name) ;
   else if (!name[i+1])
      expose_stem(table,name) ;
   else
      expose_compound(table,name) ;
}   


char *getvalue( char *name, int foobar )
{
   int i ;

   for (i=0;(name[i])&&(name[i]!='.');i++) ;
   if (!name[i]) 
      return getvalue_simple(name) ;
   else if (!name[i+1])
      return getvalue_stem(name) ;
   else
      return getvalue_compound(name) ;
}


void drop_var_simple( char *name ) 
{
   extern proclevel currlevel ;
   variableptr ptr ;

   ptr = currlevel->vars[hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

   if ((ptr)&&(ptr->value))
   {
      Free( ptr->value ) ;
      ptr->value = NULL ;
   }
}



void drop_var_stem( char *name ) 
{
   extern proclevel currlevel ;
   variableptr ptr, *tptr, ttptr, tttptr ;

   ptr = currlevel->vars[hashfunc(name,NULL)] ;
   for (;(ptr)&&(strccmp(ptr->name,name));ptr=ptr->next) ;
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;

   if (ptr)
   {
      if (ptr->value) /* fix the default value ... */
      {
         Free( ptr->value ) ;
         ptr->value = NULL ;
      }

      assert(ptr->index!=NULL) ;
      for (tptr=ptr->index; tptr; tptr++ ) 
      {
          for (ttptr=*tptr; ttptr; ttptr=ttptr->next )
             for (tttptr=ttptr; tttptr->realbox; tttptr=tttptr->realbox ) 
                if (tttptr->value)
                {
                   Free( tttptr->value ) ;
                   tttptr->value = NULL ;
                }
      }
   }
}



void drop_var_compound( char *name ) 
{
   int hashval, baselength, subst=FALSE ;
   variableptr ptr, nptr ;
   char *cptr, *indexstr ;
   extern proclevel currlevel ;

   ptr = currlevel->vars[hashval=hashfunc(name,&cptr)] ;
   baselength = cptr-name ;
   for (;(ptr)&&(strnccmp(ptr->name,name,baselength));ptr=ptr->next) ; 
   for (;(ptr)&&(ptr->realbox);ptr=ptr->realbox) ;
   indexstr = subst_index( cptr, currlevel->vars ) ;
   hashval = hashfunc(indexstr,NULL) ;

   if (subst)   /* trace it */
      tracecompound(name,baselength-1,indexstr) ;

   if (ptr) {   /* find spesific value */
      nptr = ((variableptr *)(ptr->index))[hashval] ;
      for (;(nptr)&&(strcmp(nptr->name,indexstr));nptr=nptr->next) ; 
      for (;(nptr)&&(nptr->realbox);nptr=nptr->realbox) ; }

   if ((nptr)&&(nptr->value))
   {
      Free( nptr->value ) ;
      nptr->value = NULL ;
   }
}


void drop_var( char *name ) 
{ 
   int i ;

   for (i=0; (name[i])&&(name[i]!='.'); i++ ) ;
   if (!name[i])
      drop_var_simple( name ) ;
   else if (!name[i+1])
      drop_var_stem( name ) ;
   else
      drop_var_compound( name ) ;   
}
