#ifndef lint
static char *RCSid = "$Id: stack.c,v 1.7 1992/04/05 19:53:07 anders Exp anders $";
#endif

/*
 * Copyright (C) 1992 Anders Christensen <anders@solan.unit.no>
 * Read file README for more information on copying
 */

/*
 * $Log: stack.c,v $
 * Revision 1.7  1992/04/05  19:53:07  anders
 * Added copyright notice
 *
 * Revision 1.6  1992/03/22  01:10:25  anders
 * #include'd stdio.h which is not included in rexx.h anymore
 * Fixed bug: lines in the stack was not really linked properly.
 *
 * Revision 1.5  1991/05/28  23:29:24  anders
 * Major changes and much more/better documentation.
 * Queue'ing strings now puts them in the start of the current buffer, not
 * the start of the stack.
 * Printing from buftype() now counts correctly
 * Stack_empty() is not fooled by a stack consisting of empty buffers
 * Destroy_buffer() does actually destroy the buffer, not just read
 * lines until stack_empty() returns true.
 * Drop_buffer() can take negative number to remove a number of buffers
 * counted from the top of the stack, or zero to remove the whole stack.
 * The accounting on how many buffermarks there are on the stack is
 * made rigid.
 *
 * Revision 1.4  91/04/05  23:27:35  anders
 * Fixed bug in stack_empty(), which returned inverse result
 * Put #ifdef's around mark_stack()
 * 
 * Revision 1.3  90/08/19  02:29:10  anders
 * Added support for buffers in the stack.
 * New static variable in stack.c, buffers, which have the current 
 * number of buffers in the stack.
 * Implemented code for makebuf, dropbuf, desbuf and buftype, which are
 * called from cmsfuncs.c
 * 
 * Revision 1.2  90/08/11  00:45:40  anders
 * Move in definitions of stack from types.h (to be hidden in stack.c)
 * Removed initstack(), using initialized static variables instead
 * Removed references to systeminfo, using local variables instead
 * Changes parametername for stack_lifo from 'test' to 'line'
 * Moved nullstringptr() to misc.c 
 * Added three new routines: stack_empty(), lines_in_stack(), mark_stack()
 * Free()'ed allocated memory in popline()
 * 
 * Revision 1.1  90/08/08  02:13:03  anders
 * Initial revision
 * 
 */

/*
 * Consept of implementation: 
 *
 * The stack has two pointers to a double-linked list, which contain 
 * just a pointer to char in addition to pointers to next and previous
 * box. Last box' next, and first box' prev is both NIL. The char pointer
 * points to the text of that line in the stack. 

 * If the char pointer is NIL, then that box in the stack-structure is
 * just symbolizing a stack buffer mark. The integer variable
 * 'buffers' contains the current number of buffermarks in the stack.
 * A buffer is the lines between to buffermarks. The first buffer mark
 * (for buffer 0) is not marked, but is implisit the bottom of the
 * stack. The last buffer is the lines between the uppermost
 * buffermark and the top of the stack.

 * Initialy all lines are put in Buffer 0. When a new buffer is
 * created, lines pushed lifo is put on top of that buffer, and lines
 * queued fifo are put in the bottom of that buffer (i.e. not in the
 * bottom of buffer 0.)

 * When reading lines from the stack, and all lines in the current
 * buffer has been read, the buffer-mark will be removed, and lines
 * are read from the buffer underneath.

 * When the whole stack is empty, lines are read from standard input.

 * Buffer 0 is contains the lines between the bottom of the stack, and
 * the first buffermark on the stack, the first buffer, is the lines
 * between buffermark 1 and buffer mark 2.

 * When creating a buffer, the value returned will be the number of
 * buffermarks in the stack after the new buffermark is created.
 * When destroying buffers, the parameter given will equal the number
 * of the lowest buffermark to be destroyed (i.e dropbuf 4) will leave
 * 3 buffermarks on the stack).

 * Possible extentions:
 *  o A new 'hard-buffer', that is not removed when a line is popped
 *    from it while it is empty
 *  o dropbuf should be able to take negative numbers, signifying the
 *    number of buffermarks to be destroyed, counted from the top of
 *    buffer. 
 *  o The decision that dropbuf only can take parameters in the range
 *    [1,buffer] is a political one.
 */


#include "rexx.h"
#include <stdio.h>

/*
 * Hide the definition of the stack from the rest of the program
 */
typedef struct stacklinestruct *stacklineptr ;

typedef struct stacklinestruct 
{ 
   stacklineptr next, prev ;
   char *contents ;
} stackline ;


/*
 * Pointers to first and last line in the stack.
 */
static stacklineptr lastline=NULL ;
static stacklineptr firstline=NULL ;


/*
 * The number of buffermarks in the stack
 */
static int buffers=0 ;


/*
 * Pushes 'line' onto the REXX stack, lifo, and sets 'lastline' to
 *    point to the new line. The line is put on top of the current
 *    buffer. 
 */
void stack_lifo( char *line ) 
{
   extern stacklineptr lastline, firstline ;
   stacklineptr newbox ;

   newbox = (stacklineptr)Malloc(sizeof(stackline)) ;
   if (lastline) {
      lastline->next = newbox ;
      newbox->prev = lastline ; }
   else {
      newbox->prev = NULL ;
      firstline = newbox ; }
   
   newbox->next = NULL ;
   newbox->contents = line ;
   lastline = newbox ; 
}


/* 
 * Puts 'line' on the REXX stack, fifo. This routine is similar to
 *    stack_lifo but the differences are big enough to have a separate
 *    routine. The line is put in the bottom of the current buffer
 */
void stack_fifo( char *line ) 
{
   extern stacklineptr firstline, lastline ;
   stacklineptr newbox, ptr ;

   if (!line)  /* can't happen */
      exiterror(ERR_INTERPRETER_FAILURE) ;

   /* Bug: inserts into bottom of stack, not bottom of current buffer */
   newbox = (stacklineptr)Malloc(sizeof(stackline)) ;
   newbox->prev = newbox->next = NULL ;
   newbox->contents = line ;

   for (ptr=lastline;(ptr)&&(ptr->contents);ptr=ptr->prev) ;

   if (ptr) {
      newbox->prev = ptr ;
      newbox->next = ptr->next ;
      if (ptr->next)
         ptr->next->prev = newbox ;
      else 
         lastline = newbox ;
      ptr->next = newbox ; }
   else {
      newbox->next = firstline ;
      firstline = newbox ; 
      if (newbox->next)
	 newbox->next->prev = newbox ;
      if (!lastline) 
         lastline = newbox ; }
}


/* 
 * Removes one (or several) buffers from the stack. The number of
 *    buffers to remove is decided by 'number', which is the number of
 *    buffers to remain on the stack. The global variable buffer
 *    contains the number of buffermarks currently in the stack.
 * 
 * When number==2, buffer 0 (the implisit), and buffer 1 (the first 
 *    buffer created by the user) is the only that remain. All lines
 *    from buffermark 2 (inclusive) and above is removed. Remember
 *    that buffer N is the lines following the N'th buffermark, until
 *    end-of-stack or another buffermark is reached.
 *
 * If number is less than zero, then abs(number) buffers is removed
 *    from the top of the stack (or until the stack is empty). If the
 *    number is zero, the stack is emptied. If a buffer with a number
 *    that is higher than the current buffer is spesified, errorcode
 *    is retured.
 */
int drop_buffer( int number ) 
{
   extern stacklineptr firstline, lastline ;
   stacklineptr ptr, tptr ;
   extern buffers ;

   if (number<0) 
      number = (buffers + number + 1) ;

   if (number<=0) {
      destroy_buffer() ;
      return 0 ; }

   if (number>buffers)
      return -2 ;

   buffers = number-1 ;
   for (ptr=firstline;(ptr);) {
      if ((number) && (!(ptr->contents)) && (!(--number))) {
         if (lastline=(ptr->prev))
            ptr->prev->next = NULL ; 
         else
            firstline = NULL ; }

      ptr = (tptr=ptr)->next ; 
      if (!number) {
         if (tptr->contents)
            Free(tptr->contents) ; 
         Free((char*)tptr) ; } ; }
         
   return 0 ;
}


/* 
 * Fetches a line from the top of the stack. If the current buffer
 *    does not contain any lines, it is removed and the second current
 *    buffer is search for a line etc. If there isn't any lines in the
 *    stack, a line is read from the standard input.
 */
char *popline( void ) 
{
   extern stacklineptr firstline, lastline ;
   extern int buffers ;
   char *contents ;
   stacklineptr line ;

   line = lastline ;
   if (line=lastline) {
      contents = line->contents ;
      lastline = line->prev ; 
      if (!line->prev) 
         firstline = NULL ;
      else
         line->prev->next = NULL ; 
      Free((char*)line) ;
      if (!contents) {
         buffers-- ;
         contents = popline() ; } ; }

   else 
      contents = readkbdline() ;   
    
   return contents ;
}


/* 
 * Counts the lines in the stack.
 */
int lines_in_stack() 
{
   extern stacklineptr firstline ;
   stacklineptr ptr ;
   int lines ;

   ptr = firstline ;
   for (lines=0;ptr;ptr=ptr->next) 
      if (ptr->contents)
         lines++ ;

   return lines ;
}


/* 
 * Returns boolean expression: is the stack empty?
 */
int stack_empty()
{
   extern stacklineptr firstline ;
   stacklineptr ptr ;

   for (ptr=firstline;ptr;ptr=ptr->next) 
      if (ptr->contents)
         return 1 ;

   return 0 ;
}


#ifdef TRACEMEM
/* 
 * Marks all chunks of dynamic allocated memory
 */
void mark_stack()
{
   extern stacklineptr firstline ;
   stacklineptr ptr ;

   for (ptr=firstline;ptr;ptr=ptr->next) {
      markmemory((char*)ptr,TRC_STACKBOX) ;
      if (ptr->contents)
         markmemory(ptr->contents,TRC_STACKLINE) ; }
}
#endif


/* 
 * Creates a new buffer, by putting a buffer mark at the top of the
 *    stack 
 */
int make_buffer() 
{
   extern int buffers ;

   stack_lifo( (char*)NULL ) ;
   return ++buffers ;
}


/* 
 * Removes all buffers and buffermarks
 */
int destroy_buffer()
{
   extern int buffers ;
   extern stacklineptr firstline, lastline ;
   stacklineptr ptr, tptr ;

   buffers = 0 ;
   for (ptr=firstline;ptr;) {
      ptr = (tptr=ptr)->next ; 
      if (tptr->contents)
         Free(tptr->contents) ;
      Free((char*)tptr) ; }

   firstline = lastline = NULL ;
   return 0 ;
}


/* 
 * Dumps the contents of the stack to standard error. Buffers are
 *    indicated in the printout.
 */
void type_buffer()
{
   extern stacklineptr lastline ;
   extern int buffers ;
   stacklineptr ptr=lastline ;
   int counter=buffers ;

   fprintf(stderr,"==> Lines: %d\n", lines_in_stack()) ;
   fprintf(stderr,"==> Buffer: %d\n", counter) ;
   for (;ptr;ptr=ptr->prev) {
      if (ptr->contents)
         fprintf(stderr,"\"%s\"\n",ptr->contents) ;
      else
         fprintf(stderr,"==> Buffer: %d\n",--counter) ; }

   fprintf(stderr,"==> End of Stack\n") ;
}
