#ifndef lint
static char *RCSid = "$Id: memory.c,v 1.9 1992/04/05 20:17:20 anders Exp anders $";
#endif

/*
 * Copyright (C) 1992 Anders Christensen <anders@solan.unit.no>
 * Read file README for more information on copying
 */

/*
 * $Log: memory.c,v $
 * Revision 1.9  1992/04/05  20:17:20  anders
 * Added copyright notice
 * Added magic cookie for memory management
 * Fixed some formatting in printout.
 *
 * Revision 1.8  1992/03/22  01:42:16  anders
 * Reimplementation of memory handling, uses a better way to store
 *    the info when traceing memory.
 *
 * Revision 1.7  1992/03/01  04:02:55  anders
 * Added support for more categories of memory-chunk-types
 * Added support for paramter in listleaked()
 *
 * Revision 1.6  1991/05/25  22:43:26  anders
 * Put #ifdef's around the code to initialize memory from mymalloc()
 * and the memory deallocated by myfree(), cpp-var: PATTERN_MEMORY
 * Added lots of documentation
 * Defined the array fptr containing routines to mark memory, instead
 * of calling these directly (hardcoded) in the code.
 * Put the code in markall() and cleareverything() into listleaked()
 * General cleaning up
 *
 * Revision 1.5  91/04/05  23:09:01  anders
 * Defined a new extern variable to count the memory that has been deallocated
 * The counter for allocated memory is not decremented when deallocating
 * memory, instead the counter deallocated is incremented
 * When new memory is allocated, it is initated to NOT_USED, and when
 * it is deallocated it is set to BEEN_USED
 * A new function is introdused to report statistics about memory usage
 * 
 * Revision 1.4  91/03/27  18:55:47  anders
 * Splitted FLAG() into FLAG() and SEQV(), to allow for a sequence numbering
 * of all chunks of allocated memory.
 * Added code to write out sequence number in listleaked()
 * Added code to listleaked to print out a heading first.
 * 
 * Revision 1.3  90/08/11  00:26:32  anders
 * Added support for marking stack in dynamic memory garbage collection
 * 
 * Revision 1.2  90/08/09  03:51:59  anders
 * Made tracing dynamic memory #ifdef'able (TRACEMEM)
 * Made macros for accessing trace information (COUNT,FLAG...)
 * Put '#include <malloc.h>' into memory
 * Restructuring and commenting code
 * Removed magic numbers, referring to int and pointer lengths
 * 
 * Revision 1.1  90/08/08  02:24:15  anders
 * Initial revision
 * 
 */

/*
 * This interface to malloc() has an optional extention for debugging
 * purposes, which enables it to trace allocated memory. Whenever memory 
 * is allocated, mymalloc() allocates 16 bytes more than needed. These 
 * bytes are used like this:
 *
 *   0       4       8      12       16 bytes
 *   | count |f|m|seq| prev  | next  | start of allocated memory
 *
 * The 'count' is the number of bytes allocated, 'f' (flag) is used in 
 * garbage collection, and 'prev' and 'next' are pointers used in a 
 * double linked list. seqv is a sequence number which is iterated 
 * for each memoryallocation.
 * 
 * count is int, prev and next are char*, f is char and seqv is a 
 * 16 bit integer.
 *
 * You can #ifdef this out by not setting the TRACEMEM cpp variable.
 *
 * An additional option to TRACEMEM is filling allocated and deallocated
 * memory with bitpatterns. If the PATTERN_MEMORY cpp-variable is set,
 * all allocated memory is initated to NOT_USED, and deallocated memory 
 * is set BEEN_USED before deallocation.
 * 
 * Garbage-collection is not implemented, but listleaked will list out
 * every chunk of allocated memory that are not currently in use. The
 * array markptrs contains a list of functions for marking memory.
 *
 * NOTE that #define'ing TRACEMEM requires that your machine follows
 *      this:  sizeof(int) = sizeof(char*) = 32 bits
 */


#include "rexx.h"
#include <stdlib.h>
#include <stdio.h>

#ifdef TRACEMEM
# ifdef PATTERN_MEMORY
/*
 * The two byte values NOT_USED and BEEN_USED are patterns which newly 
 *    allocated dynamic memory will be set to, and memory to be freed 
 *    will be set to, respectively.
 */
#  define NOT_USED 0x42
#  define BEEN_USED 0x69

#  define MAGIC_COOKIE 0xd4

typedef struct memhead
{ 
   int count ;
   struct memhead *prev, *next ;
   unsigned short seqv ;
   unsigned char flag, magic ;
} memheader;

# endif /* PATTERN_MEMORY */


/*
 * Strings used to mark chunks of memory when listing dynamically 
 * allocated memory in listleaked(). Length is max 8 chars.
 * 
 * NOTE: There is a close correspondace between these and the cpp
 *       variables TRC_* in defs.h
 */
char *allocs[] = {
   "leaked",  "hashtab", "procbox", "source",  "srcbox",  "treenode",
   "var_val", "var_nam", "var_box", "stc_box", "stc_line", "sys_info",
   "file_ptr", "proc_arg", "label" } ;


/* 
 * Array of functions to call for marking all active chunks of dynamic
 * allocated memory.
 */
static void (*fptr[])() = {
   markvariables,   /* REXX variables and values */
   marksource,      /* strings containing the sourcecode */
   marktree,        /* the parse-three */
   mark_stack,      /* the lines on the stack */
   mark_systeminfo, /* the system information box */
   mark_filetable,  /* the file descriptor table */
   NULL } ;


/*
 * Counter for dynamically memory allocated, in bytes, and ditto for the
 *    deallocated memory, dynamic memory currently in use are the
 *    difference between these.
 */
static int allocated=0 ;
static int deallocated=0 ;


/* 
 * sequence number for newly allocated memory, incremented for each new 
 *    allocation of dynamic memory
 */
static int sequence=0 ;


/*
 * Pointer to last (most newly) allocated memorychunk in double linked 
 *    list of all allocated dynamic memory.
 */
struct memhead *header=NULL ;
#endif /* TRACEMEM */


/*
 * mymalloc should return a pointer to a chunk of memory of size
 *    'bytes', it is a wrapper for the normal malloc, but performs
 *    some houskeeping tasks
 */
char *mymalloc( int bytes )
{
   char *ptr ;
#ifdef TRACEMEM
   int i,j ;
   extern int allocated ;
   extern struct memhead *header ;
   struct memhead *memptr ;

   allocated += (bytes += sizeof(struct memhead) ) ;
#endif
   if (!(ptr=malloc(bytes))) 
      exiterror( ERR_STORAGE_EXHAUSTED ) ;

#ifdef TRACEMEM
#ifdef PATTERN_MEMORY
   for (j=i=0;i<bytes;i++) 
      ptr[i] = NOT_USED ;
#endif /* PATTERN_MEMORY */

   memptr = (struct memhead*) ptr ;
   memptr->count = bytes ;
   memptr->flag = 0 ;
   memptr->magic = MAGIC_COOKIE ;
   memptr->seqv = ++sequence ;
   memptr->prev = NULL ;
   memptr->next = header ;
   if (header)
      header->prev = memptr ;

   header = memptr ;
   ptr += sizeof(struct memhead) ;
#endif /* TRACEMEM */

   return ptr ;
}


/* myfree takes a pointer to memory to be deallocated, it is a wrapper 
 *    for free(3), and does some housekeeping tasks
 */
void myfree( char *ptr )
{
#ifdef TRACEMEM
   extern int deallocated ;
   extern struct memhead *header ;
   struct memhead *memptr ;
   int i,j,bytes ;

   memptr = (struct memhead*)(ptr-=sizeof(struct memhead)) ;

   if (memptr->magic != MAGIC_COOKIE)
      exiterror( ERR_INTERPRETER_FAILURE ) ;

   deallocated -= ( bytes = memptr->count ) ;
   if (memptr->next)
      memptr->next->prev = memptr->prev ;
   if (memptr->prev)
      memptr->prev->next = memptr->next ;
   else
      header = memptr->next ;

#ifdef PATTERN_MEMORY
   for (j=i=0;i<bytes;i++) 
      ptr[i] = BEEN_USED ;
#endif

#endif /* TRACEMEM */

/*    free(ptr) ; */
}

#ifdef TRACEMEM


/* have_allocated returns the amount of dynamic memory that has been
 *    allocated, in bytes.
 */
int have_allocated( int flag )
{
   extern int allocated ;
   extern int deallocated ;
   int result ;

   switch ( flag ) 
   {
      case ( MEM_CURRENT ) :
         result = allocated - deallocated ;
         break ;
  
      case ( MEM_ALLOC ) :
         result = allocated - deallocated - listleaked( MEMTRC_NONE ) ;
         break ;
   
      case ( MEM_LEAKED ) :
         result = listleaked( MEMTRC_NONE ) ;
         break ;

      default :
         exiterror( ERR_INCORRECT_CALL ) ;
   }

   return result ;
}


/* listleaked will mark all *active* chunks of dynamic memory,
 *    according to what the memory is allocated to. Memory which are
 *    not in use, are listed on standard error. It uses an array of
 *    functions to call subroutines that marks a particular part of
 *    allocated memorychunks.
 */
int listleaked( int pflag ) 
{
   extern char *allocs[] ;
   extern struct memhead *header ;
   struct memhead *memptr ;
/*   extern void (*fptr[])() ; */
   int i, flag, tmp, j, bytes ;
   char *string ;

   for (memptr=header; memptr; memptr=memptr->next) 
      memptr->flag = TRC_LEAKED ;

   mark_listleaked_params() ;
   for (i=0;fptr[i];i++)
      (*(fptr[i]))() ;

   if (! pflag==MEMTRC_NONE)
      fprintf(stderr," Len  Flg Tag      Seqv Contents\n") ;

   for (i=0,memptr=header; memptr; memptr=memptr->next)
      if (((flag=memptr->flag)==TRC_LEAKED)||(pflag==MEMTRC_ALL))
      {
         i += (tmp = memptr->count) ;
         if (!(pflag==MEMTRC_NONE))
	 {
            fprintf(stderr, "%5d %3d %-8s %4d \"", tmp-sizeof(struct memhead), 
                    flag, allocs[flag], memptr->seqv ) ;
            string = (char*)(memptr+1) ; 
            bytes = memptr->count - sizeof(struct memhead) ;
            for (j=0; j<bytes; j++ )
            {
               if (j==20)
               {
                  fprintf(stderr, " ..." ) ;
                  break ;
               }
               if ((string[j]>=' ')&&(string[j]<0x7f))
                  putc( string[j], stderr ) ;
               else
                  putc( '?', stderr ) ;
            }
            
            fprintf( stderr, "\"\n" ) ;
         }
      }

   return i ;
}


/*
 * Marks a chunk of memory pointed to by ptr to be of the kind
 *    referenced in 'i'. Might be defined as a macro.
 */
void markmemory( char *ptr, int flag )
{
  struct memhead *memptr ;
  
  if (ptr)
  {
     memptr = (struct memhead*)(ptr-sizeof(struct memhead)) ;
     memptr->flag = flag ;
  }
  else
     exiterror( ERR_INTERPRETER_FAILURE ) ;
}


/*
 * memory_stats prints statistics of dynamic memory usage on standard
 *    error. It should probably have a parameter telling the format
 */
void memory_stats()
{
   extern int allocated ;
   extern int deallocated ;
   extern int sequence ;

   fprintf(stderr,
        "Allocated %d bytes in %d chunks, of which %d is deallocated\n",
        allocated, deallocated, sequence) ;
}

#endif /* TRACEMEM */







