
#ifndef lint
static char *RCSid = "$Id: files.c,v 1.5 1992/04/05 20:21:05 anders Exp anders $";
#endif

/*
 * Copyright (C) 1992 Anders Christensen <anders@solan.unit.no>
 * Read file README for more information on copying
 */

/*
 * $Log: files.c,v $
 * Revision 1.5  1992/04/05  20:21:05  anders
 * Added copyright notice
 * Packed ftruncate() into some cpp macros: HAS_FTRUNCATE
 *
 * Revision 1.4  1992/03/23  05:15:12  anders
 * Fixed bug that opened files for writing in append mode, instead of
 *    in update mode.
 * Fixed bug that segfault'ed on writing empty lines (i.e positioning)
 *
 * Revision 1.3  1992/03/22  01:41:26  anders
 * More or less a total reimplementation to be compatible to
 *    the type of file io that REXX wants.
 *
 * Revision 1.2  1990/08/26  01:54:07  anders
 * Added support to sence the eof-condition on stdin, thru value of eof()
 * Added new function eof_on_input() and global  int got_eof.
 *
 * Revision 1.1  90/08/08  02:09:41  anders
 * Initial revision
 * 
 */

#define FLAG_PERSIST   0x01
#define FLAG_EOF       0x02
#define FLAG_READ      0x04
#define FLAG_WRITE     0x08
#define FLAG_SURVIVOR  0x10

#define MAX_LINES_IN_FILE 99999

#include "rexx.h"
#include <errno.h>
#include <stdio.h>
#include <string.h>

#if defined(HAS_FTRUNCATE) && defined(FIX_PROTOS)
int ftruncate( int fd, int length ) ;
#endif


typedef struct fileboxtype *fileboxptr ;
typedef struct fileboxtype {
   FILE *fileptr ;
   int flag, linenr ;
   fileboxptr prev, next ;
   char *filename ;
} filebox ;


int openfile( char *name, int access ) ;
fileboxptr getfile( char *name, int access ) ;
void mark_filetable() ;
void positionfile( fileboxptr ptr, int lineno ) ;
int writeoneline( char *name, char *string, int line ) ;

static fileboxptr firstfile=NULL ;
int got_eof ;

#ifdef TRACEMEM
/*
 * Marks all entries in the two filetables. Used by the
 *   memorymanagement.
 */
void mark_filetable() 
{
   extern fileboxptr firstfile ;
   fileboxptr ptr ;

   for (ptr=firstfile; ptr; ptr=ptr->next)
   {
      markmemory( (char*)ptr, TRC_FILEPTR ) ;
      markmemory( ptr->filename, TRC_FILEPTR ) ;
   }
}
#endif


fileboxptr getfileptr( char *name, int access, int create )
{
   extern fileboxptr firstfile ;
   fileboxptr ptr ;

   for (ptr=firstfile;ptr;ptr=ptr->next)
      if ((ptr->flag & access)&&(!strcmp(name,ptr->filename)))
         return ptr ;

   if (create)
   {
      openfile( name, access ) ;
      return getfileptr( name, access, 0 ) ;
   }

   return NULL ;
}



void initfiletable() 
{
   extern fileboxptr firstfile ;
   fileboxptr ptr[3] ;
   int i ;
   static filebox stdio[3] = 
   {
      { NULL,  0, 1, NULL, NULL, "<stdin>"  },
      { NULL, 0, 1, NULL, NULL, "<stdout>" },
      { NULL, 0, 1, NULL, NULL, "<stderr>" }
   } ;
    
   /* SGI can't use stdin, stdout and stderr in the initialization above */
   stdio[0].fileptr = stdin ;
   stdio[1].fileptr = stdout ;
   stdio[2].fileptr = stderr ;

   stdio[0].flag = ( FLAG_SURVIVOR + FLAG_READ ) ;
   stdio[1].flag = ( FLAG_WRITE ) ;
   stdio[2].flag = ( FLAG_WRITE ) ;

   for (i=0;i<3;i++)
   { 
      ptr[i] = (fileboxptr)Malloc(sizeof(filebox)) ;
      memcpy( ptr[i], &stdio[i], sizeof(filebox) ) ;
      ptr[i]->filename = cpy( ptr[i]->filename ) ;
   }

   firstfile = ptr[1]->prev = ptr[0] ;
   ptr[0]->prev = ptr[2]->next = NULL ;
   ptr[0]->next = ptr[2]->prev = ptr[1] ;
   ptr[1]->next = ptr[2] ;
}

int openfile( char *name, int access )
{
   extern fileboxptr firstfile ;
   extern int errno ;   
   char *astring ;
   fileboxptr ptr ;
   FILE *fileptr ;

   if (getfileptr( name, access, 0 ))
      return 1 ;
 
   if (access==FLAG_READ)
      astring = "r" ;
   else if (access==FLAG_WRITE)
      astring = "r+" ;
   else
      exiterror( ERR_INTERPRETER_FAILURE ) ;
   
   fileptr = fopen(name,astring) ;
   if (!fileptr)
      exiterror( 100 + errno ) ;

   ptr = (fileboxptr)Malloc(sizeof(filebox)) ;
   ptr->fileptr = fileptr ;
   ptr->filename = cpy(name) ;
   ptr->flag = 0 ;

   /* Here we should use fstat() to se if this is a TRANSIENT file */
   ptr->flag |= FLAG_PERSIST ;

   ptr->next = firstfile ;
   ptr->prev = NULL ;
   firstfile = ptr ;
   if (ptr->next)
      ptr->next->prev = ptr ;

   ptr->flag |= access ;
   if (ptr->flag & FLAG_PERSIST)
   {
      if (access == FLAG_READ )
         ptr->linenr = 1 ;      
      else
      {
         ptr->linenr = -1 ;  /* signifies unknown */
         if (fseek( ptr->fileptr, 0L, SEEK_SET ))
            exiterror( 100 + errno ) ;
      }
   }
   return 0 ;      
}


void closefile( char *name, int access ) 
{
   fileboxptr ptr ;

   if ( ptr = getfileptr( name, access, 0 )) 
   {
      if (ptr->prev)
         ptr->prev->next = ptr->next ;
      else
         firstfile = ptr->next ;

      if (ptr->next)
         ptr->next->prev = ptr->prev ;

      fclose(ptr->fileptr) ;
      Free(ptr->filename) ;
      Free((char *)ptr) ; 
   }
}
 

char *std_lines( paramboxptr parms )
{
   long oldpoint ;
   int ch, left ;
   char *result, *string ;
   fileboxptr ptr ;

   checkparam( parms, 0, 1 ) ;
   string = (parms->value) ? (parms->value) : "<stdin>" ;

   ptr = getfileptr( string, FLAG_READ, 1 ) ;
   if (!(ptr->flag & FLAG_PERSIST)) 
      left = (!(ptr->flag & FLAG_EOF)) ;
   else
   {
      oldpoint = ftell( ptr->fileptr ) ;
      for(left=0;((ch=getc(ptr->fileptr))!=EOF);)
         if (ch==0x0a)
            left++ ;
      fseek(ptr->fileptr, oldpoint, SEEK_SET) ;
   }
   
   sprintf(result=Malloc(SMALLSTR), "%d", left ) ;
   return result ;
}


char *std_chars( paramboxptr parms )
{
   char *string, *result ;
   fileboxptr ptr ;
   extern int errno ;
   long oldpoint, newpoint ;
   int left ;

   checkparam( parms, 0, 1 ) ;
   string = (parms->value) ? parms->value : "<stdin>" ;

   ptr = getfileptr( string, FLAG_READ, 1 ) ;
   if (!(ptr->flag & FLAG_PERSIST))
      left = ( !(ptr->flag & FLAG_EOF)) ;
   else
   {
      oldpoint = ftell( ptr->fileptr ) ;
      if (fseek(ptr->fileptr,0L,SEEK_END))
          exiterror( 100+errno ) ;

      newpoint = ftell( ptr->fileptr ) ;
      if (fseek(ptr->fileptr,oldpoint,SEEK_SET))
         exiterror( 100+errno ) ;

      left = newpoint - oldpoint ;
   }

   sprintf(result=Malloc(SMALLSTR), "%d", left ) ;
   return result ;
}


char *readoneline( char *file, int line, int count ) 
{
   int i, j ;
   char *ret ;
   static char string[MAXSTREAMSIZE+1] ;
   fileboxptr ptr ;

   if (!(ptr=getfileptr(file,FLAG_READ,1))) 
      return nullstringptr() ;

   if (line)
      positionfile( ptr, line ) ;

   if (!count)
      return nullstringptr() ;

   for (i=0;(i<MAXSTREAMSIZE)&&((j=getc(ptr->fileptr))!=0x0a); i++) 
   {
      if (j==EOF)
      {
         ptr->flag |= FLAG_EOF ;
         if (!((ptr->flag) & (FLAG_PERSIST | FLAG_SURVIVOR )))
            closefile( ptr->filename, FLAG_READ ) ;
         break ;
      }
      string[i]=j ;
   }
   
   string[i] = 0x00 ;
   ret = Malloc(i+1) ;
   strcpy( ret, string ) ;

   if ((j==0x0a) && (ptr->linenr > 0)) 
      ptr->linenr++ ;  /* only if we actually saw the "\n" !!! */

   return ret ;
}


/*
 * There are (at least) two ways to do the backup of the current 
 *    position in the file. First to backup to the start of the file
 *    and then to seek forward, or to seek backwards from the current
 *    position of the file.
 *
 * Perhaps the first is best for the standard case, and the second 
 *    should be activated when the line-paramater is negative ... ?
 */

void positionfile( fileboxptr ptr, int lineno ) 
{
   int ch=0x00 ;

   if (!((ptr->flag) & FLAG_PERSIST ))
      exiterror( ERR_IMPROPER_SEEK ) ;

   ptr->flag &= ( 0xffff - FLAG_EOF ) ;
   if (ptr->linenr <= 0)
   {
      fseek( ptr->fileptr, 0L, SEEK_SET ) ;
      ptr->linenr = 1 ;
   }

   if (lineno>ptr->linenr) 
   {
      while ((lineno>ptr->linenr)) 
      {
         for (;((ch=getc(ptr->fileptr))!=EOF)&&(ch!=0x0a);) ;
         if (ch==0x0a) 
            ptr->linenr++ ;
         else
            break ;
      } 
      if (ch == EOF) 
         ptr->flag |= FLAG_EOF ;
   }
   else /* the second way */
   {
      for (;;) 
      {
         if (fseek(ptr->fileptr,-1L,1))
         {
            if (ftell(ptr->fileptr))
               exiterror( ERR_IMPROPER_SEEK ) ;
            ptr->linenr = 1 ;
            break ;
         }
         ch = getc(ptr->fileptr) ;
         if (ch==0x0a) 
         {
            if (lineno==ptr->linenr)
               break ;
            if (fseek(ptr->fileptr,-2L, 1))  
               exiterror( ERR_IMPROPER_SEEK ) ;
            ptr->linenr-- ; 
         }
         else
            if (fseek(ptr->fileptr, -1L, 1))
               exiterror( ERR_IMPROPER_SEEK ) ;
      } 
   }
}
      



void positioncharfile( char *name, int charno, int access )
{
   fileboxptr fileptr ;

   if (!(fileptr=getfileptr(name,access,1)))
      return ;

   fseek(fileptr->fileptr,(long)(charno-1),SEEK_SET) ;
   if ((charno-1) != ftell(fileptr->fileptr))
      fileptr->flag |= FLAG_EOF ;

   fileptr->linenr = -1 ;
}





char *readbytes( char *file, int length ) 
{
   int i, j ;
   char *retvalue ;
   fileboxptr ptr ;

   if ((ptr=getfileptr(file,FLAG_READ,1))==NULL)
      return nullstringptr() ;

   retvalue = Malloc(length+1) ;
   for (i=0;(i<length)&&((j=getc(ptr->fileptr))!=EOF); i++ )
   {
      retvalue[i] = j ;
      if ((j==0x0a)&&(ptr->linenr>0))
         ptr->linenr++ ;
   }

   retvalue[i] = '\000' ;
   if (j==EOF)
      ptr->flag |= FLAG_EOF ;

   return retvalue ;
}



char *std_charin( paramboxptr parms )
{
   char *filename, *result ;
   int length=1, start=0 ;

   checkparam( parms, 0, 3 ) ;

   filename = (parms->value) ? (parms->value) : "<stdio>" ;
   if ((parms->next)&&(parms->next->value))
   {
      start = atopos(parms->next->value) ;      
      positioncharfile( filename, start, FLAG_READ ) ;
   }

   if ((parms->next)&&(parms->next->next)&&(parms->next->next->value))
      length = atozpos( parms->next->next->value ) ;

   if (length>0)
      result = readbytes( filename, length ) ;
   else
      result = nullstringptr() ;

   return result ;
}




int writebytes( char *name, char *string, int start ) 
{
   fileboxptr ptr ;
   char *i ;

   if (!(ptr=getfileptr(name,FLAG_WRITE,1))) 
      return 0 ;

   if (start) 
   {
      if (!(ptr->flag & FLAG_PERSIST))
         exiterror( ERR_CANT_REWIND ) ;
      
      ptr->linenr = -1 ;
      if (fseek(ptr->fileptr,start,SEEK_SET))
         exiterror( 100+errno ) ;
   }

   for (i=string;(i)&&(*i);i++) 
   {
      fprintf(ptr->fileptr,"%c", *i ) ;
      if ((*i==0x0a)&&(ptr->linenr>0))
         ptr->linenr++ ;
   }
   fflush(ptr->fileptr) ;
   return (i-string) ;
}


char *std_charout( paramboxptr parms )
{
   char *ptr, *filename, *string ;
   int length, pos=0 ;

   checkparam( parms, 0, 3 ) ;

   filename = (parms->value) ? (parms->value) : "<stdout>" ;

   if ((parms->next)&&(parms->next->value))
      string = parms->next->value ;
   else
      string = NULL ;

   if ((parms->next)&&(parms->next->next)&&(parms->next->next->value)) 
      pos = atopos(parms->next->next->value) ;
   else 
      pos = 0 ;
   
   length = writebytes( filename, string, pos ) ;

   sprintf(ptr=Malloc(SMALLSTR),"%d",(string)?(strlen(string)-length):(0)) ;
   return ptr ;
}

char *std_lineout( paramboxptr parms )
{
   char *ptr, *string, *file ;
   int lineno, length, result=0 ;

   checkparam( parms, 0, 3 ) ;
   
   if ((parms->next)&&(parms->next->value))
   {
      string = parms->next->value ;
      length=(strlen(string)+1) ;
   }
   else
      string = NULL ;

   if (parms->value)
      file = parms->value ;
   else 
      file = "<stdout>" ;

   if ((parms->next)&&(parms->next->next)&&(parms->next->next->value))
      lineno = atopos( parms->next->next->value ) ;
   else
      lineno = 0 ;

   result = writeoneline(file, string, lineno) ; 

   sprintf(ptr=Malloc(SMALLSTR),"%d",result) ;
   return ptr ;
}


int eof_on_input()
{
   return 0 ;
}


int writeoneline( char *name, char *string, int line ) 
{
   fileboxptr ptr ;
   char *i ;

   if (!(ptr=getfileptr( name, FLAG_WRITE, 1 )))
      return 1 ;

   if (line)
      positionfile( ptr, line ) ;

   if (string)
   {
#ifdef HAS_FTRUNCATE
      if (ftruncate( fileno(ptr->fileptr), (int)ftell( ptr->fileptr )))
         exiterror( 100+errno ) ;
#endif

      fseek( ptr->fileptr, 0L, SEEK_END ) ;
      for (i=string; *i; i++)
         putc( *i, ptr->fileptr) ;

      putc( (char)0x0a, ptr->fileptr ) ;
      if (fflush( ptr->fileptr ))
         exiterror( 100+errno ) ;
   }
   return 0 ;
}


char *std_linein( paramboxptr parms )
{
   char *filename ;
   paramboxptr ptr ;
   int count=1, line=0 ;

   checkparam( parms, 0, 3 ) ;

   filename = ( parms->value ) ? parms->value : "<stdin>" ;
   if ( ptr = parms->next ) 
   {
      line = ( ptr->value ) ? atopos( ptr->value ) : 0 ;

      if ( ptr = ptr->next )
      {
         count = ( ptr->value ) ? atozpos( ptr->value ) : 1 ;
         if (count>1)
            exiterror( ERR_INCORRECT_CALL ) ;
      }
   }
         
   return readoneline( filename, line, count ) ;
}



/*
 * This routine will traverse the list of open files, and dump relevant
 *   information about each of them. Really a debugging routine. 
 */
void dumpfileptr( void )
{
   extern fileboxptr firstfile ;
   fileboxptr ptr ;
   int i ;
   char string[6] ;

   fprintf(stderr,
         "File Filename                       Flags Linenr\n") ;
   
   for (ptr=firstfile;ptr;ptr=ptr->next)
   {
      fprintf( stderr, "%4d %-30s", fileno(ptr->fileptr), ptr->filename ) ;
      i = 0 ;

      string[0] = ( ptr->flag & FLAG_READ    ) ? 'r' : ' ' ;
      string[1] = ( ptr->flag & FLAG_WRITE   ) ? 'w' : ' ' ;
      string[2] = ( ptr->flag & FLAG_PERSIST ) ? 'p' : 't' ;
      string[3] = ( ptr->flag & FLAG_EOF     ) ? 'e' : ' ' ;
      string[4] = 0x00 ;

      fprintf( stderr, " %5s %6d\n", string, ptr->linenr ) ;
   }
   fprintf( stderr, "r=read, w=write, p=persistent, t=transient, e=eof\n");
}


char *dbg_dumpfiles( paramboxptr parms )
{
   dumpfileptr() ;
   return nullstringptr() ;
}




/* This should really go out .... */
char *readkbdline( void )
{
   extern int got_eof ;
   char *source, ch ;
   int i=0 ;

   source = Malloc(BUFFERSIZE) ;
   if (got_eof) 
      (void)fseek(stdin,SEEK_SET,0) ;
   do {
      if (i<BUFFERSIZE) 
         source[i++] = ch = getc(stdin) ;
   } while((ch!='\012')&&(ch!=EOF)) ;
   got_eof = (ch==EOF) ;
   source[i-1] = '\000' ;

   return source ;
}



char *unx_open( paramboxptr parms )
{
   char *access, *result ;
   int resvalue, iaccess ;

   checkparam( parms, 1, 2 ) ;
   access = ((parms->next)&&(parms->next->value)) ? parms->next->value : "r" ;
   if ( access[0]=='r' )
      iaccess = FLAG_READ ;
   else if ( access[0]=='w' )
      iaccess = FLAG_WRITE ;
   else
      exiterror( ERR_INCORRECT_CALL ) ;
   
   resvalue = openfile( parms->value, iaccess ) ;
   result = Malloc(SMALLSTR) ;
   sprintf(result,"%d",resvalue) ;
   return result ;
}

char *unx_close( paramboxptr parms )
{
   char *result ;
   int resvalue ;

   checkparam( parms, 1, 1 ) ;
   closefile( parms->value, FLAG_READ ) ;
   closefile( parms->value, FLAG_WRITE ) ;
   result = Malloc(SMALLSTR) ;
   sprintf(result,"%d",resvalue=0) ;
   return result ;
}

