#ifndef lint
static char *RCSid = "$Id: builtin.c,v 1.9 1992/04/05 20:35:27 anders Exp anders $";
#endif

/*
 * Copyright (C) 1992 Anders Christensen <anders@solan.unit.no>
 * Read file README for more information on copying
 */

/*
 * $Log: builtin.c,v $
 * Revision 1.9  1992/04/05  20:35:27  anders
 * Added copyright notice
 * Added datatype, trace and delword
 * Fixed bug in date, too little space was allocated
 * Removed funtion getoption
 *
 * Revision 1.8  1992/03/23  05:09:07  anders
 * Added several new functions.
 *
 * Revision 1.7  1992/03/22  01:44:17  anders
 * Added numerous functions
 * Fixed bugs in numberous functions
 *
 * Revision 1.6  1992/03/01  19:02:26  anders
 * Fixed sprintf, which can not be trusted to return a pointer to
 *    the string it has been writing to.
 * Set time_t instead of long as type for storing result from time()
 *
 * Revision 1.5  1991/05/25  02:40:55  anders
 * Fixed bug that appended one character to the end of the result from
 *    the REXX function word()
 *
 * Revision 1.4  91/04/06  00:24:36  anders
 * Fixed bug in std_word()
 * Added std_strip()
 * 
 * Revision 1.3  90/12/10  03:16:45  anders
 * Removed bug that triggered a bus error at  space(''). The problem
 *    was an uninitialized variable.
 * 
 * Revision 1.2  90/08/11  00:41:57  anders
 * Changed definition of std_queued() to call lines_in_stack() instead
 *    of trying to figure it out by itself
 * 
 * Revision 1.1  90/08/08  02:05:46  anders
 * Initial revision
 * 
 */

#include "rexx.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <time.h>
#include <stdio.h>

double myatof() ;

#define UPPERLETTER(a) ((((a)&0xdf)>='A')&&(((a)&0xdf)<='Z'))
#define NUMERIC(a) (((a)>='0')&&((a)<='9'))

int contained_in( char *first, char *second )
{
   for (; (*first)&&(isspace(*first)); first++) ;
   for (; (*second)&&(isspace(*second)); second++) ;
   
   for (; (*first); ) 
   {
      for (; (*first)&&(!isspace(*first)); first++, second++)
         if ((*first)!=(*second))
            return 0 ;

      if ((*second)&&(!isspace(*second)))
         return 0 ;

      if (*first==0x00)
         return 1 ;

      for (; (*first)&&(isspace(*first)); first++) ;
      for (; (*second)&&(isspace(*second)); second++) ;
   }

   return 1 ;
}

char *std_wordpos( paramboxptr parms )
{
   char *seek, *target, *sptr, *tptr ;
   int start=1, res ;
   char *result ;

   checkparam( parms, 2, 3 ) ;
   seek = parms->value ;
   target = parms->next->value ;
   if ((parms->next->next)&&(parms->next->next->value))
      start = atopos(parms->next->next->value) ;

   /* Then lets position right in the target */
   for (tptr=target; isspace(*tptr); tptr++) ;
   for (res=1; (res<start); res++) 
   {   
      for (; (*tptr)&&(!isspace(*tptr)); tptr++ ) ;
      for (; isspace(*tptr); tptr++ ) ;
   }

   for (sptr=seek; isspace(*sptr); sptr++) ;
   if (*sptr)
      for ( ; (*sptr)&&(*tptr); )
      {
         if (contained_in( sptr, tptr ))
            break ;
   
         for (; (*tptr)&&(!isspace(*tptr)); tptr++) ;
         for (; (*tptr)&&(isspace(*tptr)); tptr++) ;
         res ++ ;
      }

   if ((*sptr==0x00)||((*sptr)&&(!(*tptr))))
      res = 0 ;

   sprintf( result = Malloc(SMALLSTR), "%d", res ) ;
   return result ;
}
   

char *std_wordlength( paramboxptr parms )
{
   int i, number ;
   char *string, *ptr, *result ;
   checkparam( parms, 2, 2 ) ;
   string = parms->value ;
   number = atopos(parms->next->value) ;

   for (ptr=string; isspace(*ptr); ptr++) ;
   for (i=0; i<number-1; i++)
   {
      for (; (*ptr)&&(!isspace(*ptr)); ptr++) ;
      for (; (*ptr)&&(isspace(*ptr)); ptr++ ) ;
   }

   for (i=0; (*(ptr+i)&&(!isspace(*(ptr+i)))); i++) ;
   sprintf(result=Malloc(SMALLSTR), "%d", i) ;
   return result ;
}



char *std_wordindex( paramboxptr parms ) 
{
   int i, number ;
   char *string, *ptr, *result ;

   checkparam( parms, 2, 2 ) ;
   string = parms->value ;
   number = atopos( parms->next->value ) ;

   for (ptr=string; isspace(*ptr); ptr++) ;
   for (i=0; i<number-1; i++) {
      for (; (*ptr)&&(!isspace(*ptr)); ptr++) ;
      for (; (*ptr)&&(isspace(*ptr)); ptr++) ;
   }

   sprintf(result=Malloc(SMALLSTR), "%d", (*ptr) ? (ptr - string + 1) : (0)) ;
   return result ;
}


char *std_delword( paramboxptr parms ) 
{
   char *result, *rptr, *cptr, *string ;
   int length=-1, start, i ; 

   checkparam( parms, 2, 3 ) ;
   string = parms->value ;
   start = atopos( parms->next->value ) ;
   if ((parms->next->next)&&(parms->next->next->value))
      length = atozpos(parms->next->next->value) ;
   
   result = rptr = Malloc( strlen( string )+1 ) ;
   for (cptr=string; isspace(*cptr); *(rptr++) = *(cptr++) ) ;
   for (i=0; i<(start-1); i++)
   {
      for (; (*cptr)&&(!isspace(*cptr)); *(rptr++) = *(cptr++)) ;
      for (; isspace(*cptr); *(rptr++) = *(cptr++)) ;
   }
   
   for (i=0; (i<(length))||((length==(-1))&&(*cptr)); i++)
   {
      for (; (*cptr)&&(!isspace(*cptr)); cptr++ ) ;
      for (; isspace(*cptr); cptr++ ) ;
   }
      
   for (; *cptr;)
   {
      for (; (*cptr)&&(!isspace(*cptr)); *(rptr++) = *(cptr++)) ;
      for (; isspace(*cptr); *(rptr++) = *(cptr++)) ;
   }

   *rptr = 0x00 ;
   return result ; 
}


char *std_xrange( paramboxptr parms )
{
   int start=1, stop=0xff, i, j=0, length ;
   char *result ;

   checkparam( parms, 0, 2 ) ;
   if (parms->value)
      start = (unsigned char) getonechar( parms->value ) ;
   
   if ((parms->next)&&(parms->next->value))
      stop = (unsigned char) getonechar( parms->next->value ) ;

   length = stop - start + 1 ;
   if (length<1)
      length = 255 - ( length - 2 ) ;

   result = Malloc( length+1 ) ;
   for (i=start; i!=stop; i++) 	
   {
      if (i==256)
        i = 1 ;
      result[j++] = (char) i ;
   }
   result[j++] = (char) stop ;
   result[j] = 0x00 ;

   return result ;
}


char *std_lastpos( paramboxptr parms )
{
   int res=0, start, i, j, nomore ;
   char *needle, *heystack, *result ;

   checkparam( parms, 2, 3 ) ;
   needle = parms->value ;
   heystack = parms->next->value ;
   if ((parms->next->next)&&(parms->next->next->value))
      start = atopos( parms->next->next->value ) ;
   else
      start = strlen( heystack ) ;

   nomore = strlen( needle ) ;
   if (start>strlen(heystack))
      start = strlen( heystack ) ;

   if (nomore>start) 
      res = 0 ;
   else if (nomore==0)
      res = start ;
   else 
      for (i=start-nomore ; i>=0; i-- )
      { 
         for (j=0; (j<=nomore)&&(needle[j]==heystack[i+j]); j++) ;
         if (j>=nomore)
         {
            res = i + 1 ;
            break ;
         }
      }

   sprintf( result=Malloc(SMALLSTR), "%d", res ) ;
   return result ;
}


char *std_pos( paramboxptr parms )
{
   int start = 1, res ;
   char *result, *needle, *heystack, *found ;
   checkparam( parms, 2, 3 ) ;

   needle = parms->value ;
   heystack = parms->next->value ;
   if ((parms->next->next)&&(parms->next->next->value))
      start = atopos( parms->next->next->value ) ;

   if ((!*needle)&&(!*heystack))
      res = 0 ;
   else
   {
      found = strstr(&heystack[start-1], needle) ;
      res = (found) ? (found - heystack + 1) : 0 ;
   }

   sprintf( result=Malloc(SMALLSTR), "%d", res ) ;
   return result ;
}


char *std_subword( paramboxptr parms )
{
   int i, length, start ;
   char *cptr, *eptr, *string, *result ;

   checkparam( parms, 2, 3 ) ;
   string = parms->value ;
   start = atopos( parms->next->value ) ;
   if ((parms->next->next)&&(parms->next->next->value)) 
      length = atopos( parms->next->next->value ) ;
   else
      length = 0 ;

   cptr = string ;
   for (i=1; i<start; i++) 
   {
      for ( ; (*cptr)&&(*cptr==' '); cptr++) ;
      for ( ; (*cptr)&&(*cptr!=' '); cptr++) ;
   }
   for ( ; (*cptr)&&(*cptr==' '); cptr++) ;

   eptr = cptr ;
   if (length)
   {
      for( i=0; (i<length); i++ )
      {
         for ( ; (*eptr)&&(*eptr==' '); eptr++) ; /* will not hit 1st time */
         for ( ; (*eptr)&&(*eptr!=' '); eptr++) ;
      }
   }
   else
      for( ; *eptr ; eptr++ ) ;

   for( ; ((*eptr==0x00)||(*eptr==' '))&&(eptr>=cptr) ; eptr-- ) ;

   strncpy( result=Malloc(++eptr-cptr+1), cptr, (eptr-cptr) ) ;
   result[eptr-cptr] = 0x00 ;
   return result ;
}



char *std_symbol( paramboxptr parms )
{
   char *start, *str, *tptr ;

   checkparam( parms,1,1 ) ;
   for(start=str=param(parms,1);*str;str++) {
      if (UPPERLETTER(*str)||NUMERIC(*str))
         continue ;

      if (*str=='.')
         break ;

      if ((*str=='@')||(*str=='#')||(*str=='$')||(*str=='!')||
                       (*str=='?')||(*str=='_'))
         continue ;
      return strcpy(Malloc(4),"BAD") ; }

   if ((NUMERIC(*start)||(*start=='.'))&&(*str!='.'))
      return strcpy(Malloc(4),"LIT") ; 

   if (tptr=isvariable(start))
   {
      Free( tptr ) ;    
      return strcpy(Malloc(4),"VAR") ;
   }

   return strcpy(Malloc(4),"LIT") ;
}


char *std_value( paramboxptr parms )
{
   char *ptr, *string ;

   checkparam( parms,1,1 ) ;
   strcpy( string=Malloc(strlen(parms->value)+1), parms->value ) ;
   for (ptr=string; *ptr; ptr++)
      if ((*ptr<='z')&&(*ptr>='a'))
         *ptr -= 0x20 ; 

   ptr = isvariable(string ) ;
   Free( string ) ;   

   if (ptr)
      return ptr ;
   else
      exiterror( ERR_SYMBOL_EXPECTED ) ;

   return nullstringptr() ;  /* this should never happen */
}


char *std_abs( paramboxptr parms )
{
   char *ptr ;
   checkparam( parms,1,1 ) ;
   sprintf(ptr=Malloc(SMALLSTR),"%G",fabs(myatof(parms->value))) ;
   return( ptr ) ;
}

char *std_format( paramboxptr parms ) 
{
   char fmt[30], *retvalue ;
   double number ;
   int before=0, after=0, i ;

   checkparam(parms,1,3) ;
   number = myatof(parms->value) ;

   if ((parms->next)&&(parms->next->value))
      before = atozpos(parms->next->value) ;
   if ((parms->next)&&(parms->next->next)&&(parms->next->next))
      after = atozpos(parms->next->next->value) ;

   retvalue = Malloc(4*SMALLSTR) ;
   if ((before==0)&&(after==0)) 
      sprintf(retvalue,"%d",(int)number) ;
   else if (before==0) {
      sprintf(fmt,"%%.%df",after) ;
      sprintf(retvalue,fmt,number) ; }
   else if (after==0) {
      sprintf(fmt,"%%%d.12f",before+13) ;
      sprintf(retvalue,fmt,number) ; 
      for (i=strlen(retvalue)-1;(retvalue[i]=='0');retvalue[i--]='\000') ;
      if (retvalue[i]=='.')  
         retvalue[i] = '\000' ; }
   else {
      sprintf(fmt,"%%%d.%df",before+after+1,after) ;
      sprintf(retvalue,fmt,number) ; }

   return retvalue ;
}



char *std_overlay( paramboxptr parms )
{
   char *newstr, *oldstr, padch=' ', *retval ;
   int length, spot=0, oldlen, i, j, k ;
   paramboxptr tmpptr ;

   checkparam(parms,2,5) ;
   newstr = parms->value ;
   oldstr = parms->next->value ;
   length = strlen(newstr) ;
   oldlen = strlen(oldstr) ;
   if (parms->next->next) {
      tmpptr = parms->next->next ;
      if (parms->next->next->value) 
         spot = atopos(tmpptr->value) ;

      if (tmpptr->next) {
         tmpptr = tmpptr->next ;
         if (tmpptr->value)
            length = atozpos(tmpptr->value) ;
         if ((tmpptr->next)&&(tmpptr->next->value))
            padch = getonechar(tmpptr->next->value) ; } ; }
   
   retval = Malloc(1+((spot+length-1>oldlen)?spot+length-1:oldlen)) ;
   for (j=i=0;(i<spot-1)&&(oldstr[i]);retval[j++]=oldstr[i++]) ;
   for (;j<spot-1;retval[j++]=padch) ;
   for (k=0;(k<length)&&(newstr[k]);retval[j++]=newstr[k++]) 
      if (oldstr[i]) i++ ;

   for (;k++<length;retval[j++]=padch) if (oldstr[i]) i++ ;
   for (;oldstr[i];retval[j++]=oldstr[i++]) ;
   retval[j] = '\000' ;

   return retval ;
}

char *std_insert( paramboxptr parms )
{
   char *newstr, *oldstr, padch=' ', *retval ;
   int length, spot=0, oldlen, i, j, k ;
   paramboxptr tmpptr ;

   checkparam(parms,2,5) ;
   newstr = parms->value ;
   oldstr = parms->next->value ;
   length = strlen(newstr) ;
   oldlen = strlen(oldstr) ;
   if (parms->next->next) {
      tmpptr = parms->next->next ;
      if (parms->next->next->value) 
         spot = atozpos(tmpptr->value) ;

      if (tmpptr->next) {
         tmpptr = tmpptr->next ;
         if (tmpptr->value)
            length = atozpos(tmpptr->value) ;
         if ((tmpptr->next)&&(tmpptr->next->value))
            padch = getonechar(tmpptr->next->value) ; } ; }
   
   retval = Malloc(length+1+((spot>oldlen)?spot:oldlen)) ;
   for (j=i=0;(i<spot)&&(oldstr[i]);retval[j++]=oldstr[i++]) ;
   for (;j<spot;retval[j++]=padch) ;
   for (k=0;(newstr[k])&&(k<length);retval[j++]=newstr[k++]) ;
   for (;k++<length;retval[j++]=padch) ;
   for (;oldstr[i];retval[j++]=oldstr[i++]) ;
   retval[j] = '\000' ;
   return retval ;
}

   

char *std_time( paramboxptr parms )
{
   extern nodeptr currentnode ;
   int hour ;
   time_t unow, now, rnow, usec, sec ;
   char *ampm, *retval ;
   char format='N' ;
   extern proclevel currlevel ;
   struct tm *tmdata ;

   checkparam( parms, 0, 1 ) ;
   if ((parms)&&(parms->value))
      format = getoption(parms->value) ;

   if (currentnode->now) {
      now = currentnode->now ;
      unow = currentnode->unow ; }
   else  {
      getsecs(&now, &unow) ;
      currentnode->now = now ;
      currentnode->unow = unow ; }

   rnow = now ;
   if (unow>=(500*1000))
      now ++ ;

   tmdata = localtime(&now) ;
   switch (format) {
      case 'C':
         hour = tmdata->tm_hour ;
         ampm = (hour>11) ? "pm" : "am" ;
         if ((hour=hour%12)==0) 
            hour = 12 ;
         retval = Malloc(8) ;
         sprintf(retval, "%d:%02d%s", hour, tmdata->tm_min, ampm) ;
         break ;

      case 'E':
         sec = (currlevel->sec) ? rnow-currlevel->sec : 0 ;
         usec = (currlevel->sec) ? unow-currlevel->usec : 0 ; 
         if (usec<0)
         {
            usec += 1000000 ;
            sec-- ;
         }

         if (!currlevel->sec) 
         {
            currlevel->sec = rnow ;
            currlevel->usec = unow ;
         }

         retval = Malloc(17) ;
         if (sec)
            sprintf(retval,"%ld.%06ld", sec, usec ) ;
         else
            sprintf(retval,".%06ld", usec ) ;
         break ;
 
      case 'H':
         sprintf(retval=Malloc(3), "%d", tmdata->tm_hour) ;
         break ;

      case 'L':
         retval = Malloc(16) ;
         tmdata = localtime(&rnow) ;         
         sprintf(retval, "%02d:%02d:%02d.%06ld", tmdata->tm_hour,
                  tmdata->tm_min, tmdata->tm_sec, unow ) ;
         break ; 

      case 'M':
         retval = Malloc(5) ;
         sprintf(retval, "%d", tmdata->tm_hour*60 + tmdata->tm_min) ;
         break ;

      case 'N':
         retval = Malloc(9) ;
         sprintf(retval, "%02d:%02d:%02d", tmdata->tm_hour, 
                      tmdata->tm_min, tmdata->tm_sec ) ;
         break ;

      case 'R':
         sec = (currlevel->sec) ? rnow-currlevel->sec : 0 ;
         usec = (currlevel->sec) ? unow-currlevel->usec : 0 ;
         if (usec<0) 
         {
            usec += 1000000 ;
            sec-- ;
         }
 
         currlevel->sec =  rnow ;
         currlevel->usec = unow ; 

         retval = Malloc(17) ;
         if (sec)
            sprintf(retval,"%ld.%06ld", sec, usec) ;
         else
            sprintf(retval, ".%06ld", usec) ;
         break ;
     
      case 'S':
         retval = Malloc(6) ;
         sprintf(retval, "%d", ((tmdata->tm_hour*60)+tmdata->tm_min)
                    *60 + tmdata->tm_sec) ;
         break ;

      default:
         exiterror(ERR_INCORRECT_CALL) ; }
 
   return retval ;
}

char *std_date( paramboxptr parms )
{
   extern nodeptr currentnode ;
   char format = 'N' ;
   int length ;
   char *chptr, *retval ;
   struct tm *tmdata ;
   time_t now, unow, rnow ;
   extern char *months[], *WeekDays[] ;
   static char *fmt = "%02d/%02d/%02d" ;
   static char *iso = "%4d%02d%02d" ;

   checkparam( parms, 0, 1 ) ;
   if ((parms)&&(parms->value))
      format = getoption(parms->value) ;

   if (currentnode->now) {
      now = currentnode->now ;
      unow = currentnode->unow ; }
   else  {
      getsecs(&now, &unow) ;
      currentnode->now = now ;
      currentnode->unow = unow ; }

   rnow = now ;
   if (unow>=(500*1000))
      now ++ ;

   tmdata = localtime(&now) ;
   switch (format) {
      case 'C':
         retval = Malloc(6) ;
         length = tmdata->tm_yday + 1 +
                  (int)(((float)tmdata->tm_year-1)*365.25) + 365 ;
         sprintf(retval,"%d",length) ;
         break ;
      
      case 'D':
         retval = Malloc(4) ;
         sprintf(retval, "%d", tmdata->tm_yday+1) ;
         break ;

      case 'E':
         retval = Malloc(9) ;
         sprintf(retval, fmt, tmdata->tm_mday, tmdata->tm_mon+1, 
                              tmdata->tm_year) ;
         break ;
    
      case 'M':
         chptr = months[tmdata->tm_mon] ;
         retval = Malloc(length=strlen(chptr)+1) ;
         memcpy(retval,chptr,length) ;
         break ;

      case 'N':
         retval = Malloc(12) ;
         chptr = months[tmdata->tm_mon] ;
         sprintf(retval,"%d %c%c%c %4d", tmdata->tm_mday, chptr[0], chptr[1],
                           chptr[2], tmdata->tm_year+1900) ;
         /* Watch for the turn of the century, and in year 9999 ... */
         break ;
 
      case 'O':
         retval = Malloc(9) ;
         sprintf(retval, fmt, tmdata->tm_year, tmdata->tm_mon+1, 
                           tmdata->tm_mday);
         break ;

      case 'S':
         retval = Malloc(9) ;
         sprintf(retval, iso, tmdata->tm_year+1900, tmdata->tm_mon+1, 
                           tmdata->tm_mday) ;
         break ;

      case 'U':
         retval = Malloc(9) ;
         sprintf(retval, fmt, tmdata->tm_mon+1, tmdata->tm_mday, 
                                tmdata->tm_year ) ;
         break ;

      case 'W':
         chptr = WeekDays[tmdata->tm_wday] ;
         retval = Malloc(length=strlen(chptr)+1) ;
         memcpy(retval, chptr, length) ;
         break ;
   
      default:
         exiterror(ERR_INCORRECT_CALL) ; }

   return retval ;
}
  

char *std_words( paramboxptr parms )
{
   int space, i, j ;
   char *string, *result ;

   checkparam( parms, 1, 1 ) ;
   string = parms->value ;

   space = 1 ;
   for (i=j=0;string[i];i++) {
      if ((!space)&&(isspace(string[i]))) j++ ;
      space = (isspace(string[i])) ; }
         
   if ((!space)&&(i>0)) j++ ;
   result = Malloc(SMALLSTR) ;
   sprintf(result,"%d",j) ;
   return result ;
}


char *std_word( paramboxptr parms )
{
   char *string, *result ;
   int i, j, finished, start, stop, number, space ;

   checkparam( parms, 2, 2 ) ;
   string = parms->value ;
   number = atopos(parms->next->value) ;

   start = 0 ;
   finished = 0 ;   
   space = 1 ; 
   for (i=j=0;(string[i])&&(!finished);i++) {
      if ((space)&&(!isspace(string[i]))) 
         start = i ;
      if ((!space)&&(isspace(string[i]))) {
         stop = i ;
         finished = (++j==number) ; }
      space = (isspace(string[i])) ; }

   if ((!finished)&&(number==j+1)) {
      stop = i ;
      finished = 1 ; }

   if (finished) {
      result = Malloc(stop-start+1) ;
      strncpy(result,&string[start],stop-start) ; 
      result[stop-start] = 0x00 ; }
   else 
      result = nullstringptr() ;

   return result ;
}
      

   


char *std_address( paramboxptr parms )
{
   extern proclevel currlevel ;
   int length ;
   char *result ;

   checkparam( parms, 0, 0 ) ;
   length = strlen(currlevel->environment) ;
   result = Malloc(++length) ;
   memcpy(result,currlevel->environment,length) ;
   return result ;
}


char *std_digits( paramboxptr parms )
{
   extern proclevel currlevel ;
   char *result ;

   checkparam( parms, 0, 0 ) ;
   sprintf(result=Malloc(SMALLSTR), "%d", currlevel->numsize) ;
   return result ;
}


char *std_form( paramboxptr parms )
{
   extern proclevel currlevel ;
   char *result ;

   checkparam( parms, 0, 0 ) ;
   strcpy(result=Malloc(strlen(currlevel->numform)+1), currlevel->numform) ;
   return result ;
}


char *std_fuzz( paramboxptr parms )
{
   extern proclevel currlevel ;
   char *result ;

   checkparam( parms, 0, 0 ) ;
   sprintf(result=Malloc(SMALLSTR), "%d", currlevel->numfuzz) ;
   return result ;
}


char *std_abbrev( paramboxptr parms ) 
{
   int length ;
   char *longstr, *shortstr, *result ;

   checkparam( parms, 2, 3 ) ;
   longstr = parms->value ;
   shortstr = parms->next->value ;
   
   if ((parms->next->next)&&(parms->next->next->value))
      length = atozpos(parms->next->next->value) ;
   else
      length = strlen(shortstr) ;

   result = Malloc(BOOL_STR_LENGTH) ;
   result[0] = (strncmp(shortstr,longstr,length)) ? '0' : '1' ;
   result[1] = '\000' ;

   if ((length>strlen(shortstr))||(strlen(shortstr)>strlen(longstr)))
      result[0] = '0' ;

   return result ;
}      


char *std_queued( paramboxptr parms )
{
   char *ptr ;
   checkparam( parms, 0, 0 ) ;
   sprintf(ptr=Malloc(SMALLSTR),"%d",lines_in_stack()) ;
   return( ptr ) ;
}



char *std_strip( paramboxptr parms )
{
   char option='B', padch=' ', *input, *retval ;
   int leading, trailing, start, stop ;

   checkparam( parms, 1, 3 ) ;
   if ((parms->next)&&(parms->next->value))
      option = (getonechar(parms->next->value)) & (0xdf) ;

   if ((parms->next)&&(parms->next->next)&&(parms->next->next->value))
      padch = getonechar(parms->next->next->value) ;

   if ((option!='B')&&(option!='T')&&(option!='L'))
      exiterror( 40 ) ;

   input = parms->value ;
   leading = ((option=='B')||(option=='L')) ;
   trailing = ((option=='B')||(option=='T')) ;

   for (start=0;(input[start]==padch)&&(leading);start++) ;
   for (stop=strlen(input)-1;(input[stop]==padch)&&(trailing);stop--) ;
   if (stop<start)
      stop = start ;

   strncpy(retval=Malloc(stop-start+2),&(input[start]),stop-start+1) ;
   retval[stop-start+1] = 0x00 ;
   return retval ;
}



char *std_space( paramboxptr parms )
{
   char *retval, *string, padch=' ' ;
   int i, j, k, l, space=1, length=1, hole=0 ;

   checkparam( parms, 1, 3 ) ;
   if ((parms->next)&&(parms->next->value))
      length = atozpos(parms->next->value) ;   

   if ((parms->next)&&(parms->next->next)&&(parms->next->next->value))
      padch = getonechar(parms->next->next->value) ;

   string = parms->value ;
   for (i=0;string[i];i++) {
      if ((space)&&(string[i]!=' ')) hole++ ;
      space = (string[i]==' ') ; }

   space = 1 ;
   retval = Malloc(i + hole*length + 1) ;
   for (j=l=i=0;string[i];i++) {
      if (!((space)&&(string[i]==' '))) {
         if (space=(string[i]==' '))
            for (l=j,k=0;k<length;k++)
	       retval[j++] = padch ;
         else
            retval[j++] = string[i] ; } ; }

   retval[(space)?l:j] = '\000' ;
   return retval ;
}


char *std_arg( paramboxptr parms )
{
   int number=0, retval ;
   char flag='\000', *value ;
   paramboxptr ptr ;
   extern proclevel currlevel ;

   checkparam( parms, 0, 2 ) ;
   if ((parms)&&(parms->value)) {
      number = atopos( parms->value ) ;
      if (parms->next) {
         flag = (parms->next->value[0]) & '\337' ;
         if ((flag!='E')&&(flag!='O'))
            exiterror( 40 ) ; } ; }

   ptr = currlevel->args ;
   if (number==0) {
      for (retval=0;ptr=ptr->next;retval++) ;
      if ((retval==1)&&(!currlevel->args->next->value))
         retval = 0 ;
      sprintf(value=Malloc(SMALLSTR),"%d",retval) ; }
   
   else {
      for (retval=0;(retval<number)&&(ptr=ptr->next);retval++) ;
      if (flag) {
         retval = ((ptr)&&(ptr->value)) ;
         sprintf(value=Malloc(SMALLSTR),"%d",(flag=='E')?retval:(!retval)) ; }
      else {
         if ((ptr)&&(ptr->value))
            strcpy(value=Malloc(strlen(ptr->value)+1),ptr->value) ;
         else
            value = nullstringptr() ; } ; }

   return value ;
}


#define LOGIC_AND 0
#define LOGIC_OR  1
#define LOGIC_XOR 2


static char logic( char first, char second, int ltype )
{
   switch (ltype) 
   {
      case ( LOGIC_AND ) : return ( first & second ) ;
      case ( LOGIC_OR  ) : return ( first | second ) ;
      case ( LOGIC_XOR ) : return ( xor( first, second ) ) ;
      default : 
         exiterror( ERR_INTERPRETER_FAILURE ) ;
   }
   /* not reached, next line only to satisfy compiler */
   return 'X' ;
}
         

static char *misc_logic( int ltype, paramboxptr parms )
{
   int length1, length2, i ;
   char padch, *pad, *outstr, *str1, *str2 ;

   checkparam( parms, 1, 3 ) ;
   str1 = parms->value ;

   str2 = (parms->next) ? (parms->next->value) : NULL ;
   if (str2 == NULL)
      str2 = "" ;

   if ((parms->next)&&(parms->next->next)) 
      pad = parms->next->next->value ; 
   else 
      pad = NULL ; 

   if (pad) 
   {
      padch = pad[0] ;
      if (pad[1])
        exiterror( 40 ) ; 
   }

   length1 = strlen(str1) ;
   length2 = strlen(str2) ;
   if (length2 > length1 )
   { 
      char *tmp ;
      tmp = str2 ;
      str2 = str1 ;
      str1 = tmp ;
   }

   outstr = Malloc( strlen(str1) + 1 ) ;

   for (i=0; str2[i]; i++)
      outstr[i] = logic( str1[i], str2[i], ltype ) ;

   if (pad)
      for (; str1[i]; i++)
         outstr[i] = logic( str1[i], padch, ltype ) ;
   else 
      for (; str1[i]; i++)
         outstr[i] = str1[i] ;
   
   outstr[i] = '\000' ;
   return outstr ;
}


char *std_bitand( paramboxptr parms )
{
   return misc_logic( LOGIC_AND, parms ) ;
}

char *std_bitor( paramboxptr parms )
{
   return misc_logic( LOGIC_OR, parms ) ;
}

char *std_bitxor( paramboxptr parms )
{
   return misc_logic( LOGIC_XOR, parms ) ;
}


char *std_center( paramboxptr parms )
{
   int length, i, j, start, stop, chars ;
   char padch, *pad, *str, *ptr ;

   checkparam( parms, 2, 3 ) ;
   length = atozpos( parms->next->value ) ;
   str = parms->value ;
   if (parms->next->next!=NULL)
      pad = parms->next->next->value ;
   else 
      pad = NULL ;

   chars = strlen(str) ;
   if (pad==NULL)
      padch = ' ' ;
   else {
      padch = pad[0] ;
      if (pad[1]!='\000')
        exiterror( 40 ) ; }

   start = (chars>length) ? ((chars-length)/2) : 0 ;
   stop = (chars>length) ? (chars-(chars-length+1)/2) : chars ;

   ptr = (char *)Malloc( length+1 ) ;
   for (j=0;j<((length-chars)/2);ptr[j++]=padch) ;
   for (i=start;i<stop;ptr[j++]=str[i++]) ;
   for (;j<length;ptr[j++]=padch) ;

   ptr[length] = '\000' ;   
   
   return ptr ;
    
}

char *std_sourceline( paramboxptr parms )
{
   extern lineboxptr firstline, lastline ;
   static lineboxptr ptr=NULL ;
   static int lineno=0 ;
   int line ;
   char *answer ;

   checkparam( parms, 0, 1 ) ;
   if (!parms->value) {
      line = lastline->lineno ;
      answer = (char *)Malloc(SMALLSTR) ;
      sprintf( answer, "%d", line ) ; }

   else {
      line = atopos( parms->value ) ;
      if (lineno==0) {
         lineno = 1 ;
         ptr = firstline ; }

      for (;(lineno<line);) {
         if ((ptr=ptr->next)==NULL) exiterror( 40 ) ;
         lineno = ptr->lineno ; }

      for (;(lineno>line);) {
         if ((ptr=ptr->prev)==NULL) exiterror( 40 ) ;
         lineno = ptr->lineno ; }

      answer = Malloc(strlen(ptr->line) + 1) ;
      strcpy(answer,ptr->line) ; }
   
   return answer ;
}


char *std_compare( paramboxptr parms )
{
   char padch, *pad, *str1, *str2, *outstr ;
   int i, j, value ;

   checkparam( parms, 2, 3 ) ;
   str1 = parms->value ;
   str2 = parms->next->value ;
   if (parms->next->next)
      pad = parms->next->next->value ;
   else 
      pad = NULL ;

   if (!pad)
      padch = ' ' ;
   else {
      padch = pad[0] ;
      if (pad[1]!='\000')
        exiterror( 40 ) ; }

   i=j=0 ;
   while ((str1[i])||(str2[j])) {
      if (((str1[i])?(str1[i]):(padch))!=((str2[j])?(str2[j]):(padch))) {
         value = (i>j) ? i : j ;
         break ; }
      if (str1[i]) i++ ;
      if (str2[j]) j++ ; }

   if ((!str1[i])&&(!str2[j])) 
      value = 0 ;
   else
     value++ ;

   sprintf(outstr=Malloc(SMALLSTR),"%d",value) ;
   return outstr ;
}      


char *std_errortext( paramboxptr parms ) 
{
   char *ptr, *ptr2 ;

   checkparam( parms, 1, 1 ) ;

   ptr = errortext(myatol(parms->value)) ;
   ptr2 = (char *) Malloc( strlen( ptr ) + 1 ) ;
   strcpy( ptr2, ptr ) ;

   return ptr2 ;
}

char *std_length( paramboxptr parms )
{
   int length, l ;
   char *ptr ;

   checkparam( parms, 1, 1 ) ;
   length = strlen(parms->value) ;
   if ((l=length)==0) l = 1 ;
   ptr = (char *)Malloc(SMALLSTR) ;
   sprintf(ptr,"%d",length) ;
   return ptr ;
}

char *std_left( paramboxptr parms ) 
{
   int length, i ;
   char padch, *pad, *str, *ptr ;

   checkparam( parms, 2, 3 ) ;
   length = atozpos( parms->next->value ) ;
   str = parms->value ;
   if (parms->next->next!=NULL)
      pad = parms->next->next->value ;
   else 
      pad = NULL ;

   if (pad==NULL)
      padch = ' ' ;
   else {
      padch = pad[0] ;
      if (pad[1]!='\000')
        exiterror( 40 ) ; }

   ptr = (char *)Malloc( length+1 ) ;
   for (i=0;(i<length)&&(str[i]!='\000');i++)
      ptr[i] = str[i] ;

   for (;i<length;ptr[i++]=padch) ;
   ptr[length] = '\000' ;   
   
   return ptr ;
}

char *std_right( paramboxptr parms ) 
{
   int length, i, j ;
   char padch, *pad, *str, *ptr ;

   checkparam( parms, 2, 3 ) ;
   length = atozpos( parms->next->value ) ;
   str = parms->value ;
   if (parms->next->next!=NULL)
      pad = parms->next->next->value ;
   else 
      pad = NULL ;

   if (pad==NULL)
      padch = ' ' ;
   else {
      padch = pad[0] ;
      if (pad[1]!='\000')
        exiterror( 40 ) ; }

   ptr = (char *)Malloc( length+1 ) ;
   for (j=0;str[j]!='\000';j++) ;
   for (i=length-1,j--;(i>=0)&&(j>=0);ptr[i--]=str[j--]) ;

   for (;i>=0;ptr[i--]=padch) ;
   ptr[length] = '\000' ;   
   
   return ptr ;
}


char *std_verify( paramboxptr parms )
{
   char tab[256], *str, *ref, ch, *ptr ;
   int inv=0, start=0, res=0, i ;
   checkparam( parms,2,4 ) ;

   str = parms->value ;
   ref = parms->next->value ;
   if (parms->next->next) {
      if (parms->next->next->value) {
         ch = (*(parms->next->next->value)&0xdf) ;
         if (ch=='M')
            inv = 1 ;
         else if (ch!='N')
            exiterror( 40 ) ; }
      if (parms->next->next->next)
         start = atopos(parms->next->next->next->value)-1 ; }

   for (i=0;i<256;tab[i++]=0) ;
   for (i=0;ref[i];tab[ref[i++]]=1) ;
   for (i=start;(str[i])&&(!res);i++) {
      if (inv==tab[str[i]]) 
         res = i+1 ; }

   sprintf(ptr=Malloc(SMALLSTR),"%d",res) ;
   return( ptr ) ;
}
   
   

char *std_substr( paramboxptr parms ) 
{
   int rlength, length, start, i, j ;
   char padch, *pad=NULL, *str, *ptr ;
   paramboxptr bptr ;

   checkparam( parms, 2, 4 ) ;
   str = parms->value ;
   rlength = strlen( str ) ;
   start = atopos( parms->next->value ) ;
   if ((bptr=parms->next->next)&&(parms->next->next->value)) 
      length = atozpos( parms->next->next->value ) ;
   else
      length = rlength-start+1 ;

   if ((bptr)&&(bptr->next)&&(bptr->next->value))
      pad = parms->next->next->next->value ; 
   

   if (pad==NULL)
      padch = ' ' ;
   else {
      padch = pad[0] ;
      if (pad[1]!='\000')
        exiterror( 40 ) ; }

   ptr = (char *)Malloc( length+1 ) ;
   i = ((rlength>=start)?start-1:rlength) ;
   for (j=0;j<length;ptr[j++]=(str[i])?str[i++]:padch) ;
   
   ptr[j] = 0x00 ;
   return ptr ;
}



char *std_max( paramboxptr parms )
{
   double largest, current ;
   paramboxptr ptr ;
   char *result ;

   if (!(ptr=parms)->value)
      exiterror( ERR_INCORRECT_CALL ) ;

   largest = myatof( ptr->value ) ;

   for(;ptr;ptr=ptr->next)
      if ((ptr->value)&&((current=myatof(ptr->value))>largest))
         largest = current ;

   sprintf(result=Malloc(SMALLSTR), "%G", largest) ;
   return result ;
}
         


char *std_min( paramboxptr parms )
{
   double smallest, current ;
   paramboxptr ptr ;
   char *result ;

   if (!(ptr=parms)->value)
      exiterror( ERR_INCORRECT_CALL ) ;

   smallest = myatof( ptr->value ) ;

   for(;ptr;ptr=ptr->next)
      if ((ptr->value)&&((current=myatof(ptr->value))<smallest))
         smallest = current ;

   sprintf(result=Malloc(SMALLSTR), "%G", smallest) ;
   return result ;
}
         


char *std_reverse( paramboxptr parms ) 
{
   char *ptr ;
   int i, j ;

   checkparam( parms, 1, 1 ) ;

   ptr = (char *)Malloc( (j=strlen(parms->value)) + 1 ) ;
   ptr[j--] = '\000' ;
   for (i=0;j>=0;ptr[i++]=parms->value[j--]) ;

   return ptr ;
}

char *std_random( paramboxptr parms )
{
   int min=0, max=999, result ;
#ifdef HAS_RANDOM
   static int seed, sewed=0 ;
#else
   static unsigned int seed, sewed=0 ;
#endif
   char *ptr ;

   if (sewed==0) {
      sewed = 1 ;
#ifdef HAS_RANDOM
      srandom(seed=(time((time_t *)0)%(3600*24))) ; }
#else
      srand(seed=(time((time_t *)0)%(3600*24))) ; }
#endif

   checkparam( parms, 0, 3 ) ;
   if (parms!=NULL) {
      if (parms->value!=NULL)
         min = atozpos( parms->value ) ;

      if (parms->next!=NULL) {
         if (parms->next->value!=NULL)
            max = atozpos( parms->next->value ) ;

         if (parms->next->next!=NULL) {
            seed = atozpos( parms->next->next->value ) ;
#ifdef HAS_RANDOM
	    srandom( seed ) ; 
#else
            srand( seed ) ;
#endif
         }  
      }  
   }

   if (min>max) 
      exiterror( ERR_INCORRECT_CALL ) ;

#ifdef HAS_RANDOM
   result = (random() % (max-min+1)) + min ;
#else
   result = (rand() % (max-min+1)) + min ;
#endif
   ptr = (char *)Malloc( (int)log10((double)((result==0)?1:result)) + 2) ;
   sprintf(ptr,"%d",result) ;
   
   return ptr ;
}
   

char *std_copies( paramboxptr parms )
{
   char *ptr ;
   int copies, i, length ;

   checkparam( parms, 2, 2 ) ;

   length = strlen(parms->value) ;
   copies = atozpos(parms->next->value) * length ;
   ptr = (char *) Malloc( copies + 1 ) ;
   for (i=0;i<copies;i+=length)
      memcpy(&ptr[i],parms->value,length) ;
    
   ptr[i] = '\000' ;
   return ptr ;
}


char *std_sign( paramboxptr parms )
{
   char *result ;
   double number ;
   
   checkparam( parms, 1, 1 ) ;
   result = Malloc(SMALLSTR) ;

   number = myatof( parms->value ) ;
   if (number)
      sprintf(result, "%d", (number>0) ? 1 : -1) ;
   else
      strcpy( result, "0" ) ;

   return result ;
}


char *std_trunc( paramboxptr parms )
{
   int decimals=0, delta ;
   char pattern[10], *result, *string ;
   double number ;

   checkparam( parms, 1, 2 ) ;
   number = myatof(string=parms->value) ;
   if ((parms->next)&&(parms->next->value))
      decimals = atozpos( parms->next->value ) ;

   delta = (number) ? (number/fabs(number)) : 0 ;
   number = ((int)(number * pow(10,decimals) + 0.5*delta))/pow(10,decimals) ;
   result = Malloc(strlen(string)+1+decimals+SMALLSTR) ;
   sprintf( pattern, "%%.%df", decimals ) ;
   sprintf( result, pattern, number ) ;

   return result ;
}


char *std_translate( paramboxptr parms )
{
   char *iptr=NULL, *optr=NULL, padch=' ', *string, *result ;
   paramboxptr ptr ;
   int olength=0, i, ii ;

   checkparam( parms, 1, 4 ) ;

   string = parms->value ;
   if ((ptr=parms->next)&&(parms->next->value))
      iptr = parms->next->value ;

   if ((ptr)&&(ptr=ptr->next)&&(ptr->value))
   {
      optr = ptr->value ;
      olength = strlen(optr) ;
   }

   if ((ptr)&&(ptr=ptr->next)&&(ptr->value))
      padch = getonechar(ptr->value) ;

   result = Malloc( strlen(string) + 1 ) ;
   for (i=0; string[i]; i++) 
   {
      if ((!iptr)&&(!optr))
         result[i] = toupper(string[i]) ;
      else
      {
         if (iptr)
         {
            for (ii=0; iptr[ii]; ii++)
               if (iptr[ii]==string[i])
                  break ;
 
            if (iptr[ii]==0x00)
            {
               result[i] = string[i] ;
               continue ;
            }
         }
         else
            ii = string[i] ;

         if ((optr)&&(ii<olength))
            result[i] = optr[ii] ;
         else
            result[i] = padch ;
      }     
   }

   result[i] = 0x00 ;
   return result ;
}


char *std_delstr( paramboxptr parms )
{
   int i, j, length, sleng, start ;
   char *string, *result ;

   checkparam( parms, 2, 3 ) ;

   sleng = strlen(string = parms->value) ;
   start = atozpos( parms->next->value ) ;

   if ((parms->next->next)&&(parms->next->next->value))
      length = atozpos( parms->next->next->value ) ;
   else
      length = strlen( string ) - start + 1 ;

   if (length<0)
      length = 0 ;

   result = Malloc( sleng-length + 2 ) ;
 
   for (i=j=0; (string[i])&&(i<start-1); result[i++] = string[j++]) ;
   j += length ;
   for (; (j<=sleng)&&(string[j]); result[i++] = string[j++] ) ;

   result[i] = 0x00 ;
   return result ;
}

   
char *std_datatype( paramboxptr parms )
{
   char *string, *type=NULL, *result, option, ch, *cptr ;
   int res ; 
   double junk ;
   
   checkparam( parms, 1, 2 ) ;

   string = parms->value ;

   
   if ((parms->next)&&(parms->next->value))
   {
      option = toupper(getonechar(parms->next->value)) ;
      res = 1 ;
      cptr = string ;
      if ((*cptr==0x00)&&(option!='X'))
         res = 0 ;

      switch ( option )
      {
         case 'A':
            for (; *cptr; res = isalnum(*cptr++) && res) ;
            break ;

         case 'B':
            for (; *cptr; cptr++ ) 
               res &= ((*cptr=='0')||(*cptr=='1')) ;
            break ;

         case 'L':
            for (; *cptr; res = islower(*cptr++) && res ) ;
            break ;

         case 'M':
            for (; *cptr; res = isalpha(*cptr++) && res ) ;
            break ;
 
         case 'N':
            res = myisnumber(cptr,&junk) ;
            break ;

         case 'S':
            /* "... if string only contains characters that are valid
             *    in REXX symbols ...", so it really does not say that 
             *    string should be a valid symbol. Actually, according
             *    to this statement, '1234E+2' is a valid symbol, although
             *    is returns false from datatype('1234E+2','S')
             */
            for (; ch=*cptr; cptr++)
               res &= ( ((ch<='z')&&(ch>='a')) || ((ch<='Z')&&(ch>='A'))
                          || ((ch<='9')&&(ch>='0')) || (ch=='.')
                          || (ch=='@') || (ch=='#') || (ch=='$') 
                          || (ch=='?') || (ch=='_') || (ch=='!')) ;
            break ;
 
         case 'U':
            for (; *cptr; res = isupper(*cptr++) && res ) ;
            break ;
       
         case 'W':
            /* bug: does not recognize the value of NUMERIC DIGITS */
            for (; *cptr; cptr++ )
               res = myisnumber(cptr,&junk) && isdigit(*cptr) && res ;
            break ; 

         case 'X':
            for (; *cptr; cptr++ )
               res = res && (isxdigit(*cptr)||(isspace(*cptr))) ;
            break ;

         default:
            exiterror( ERR_INCORRECT_CALL ) ;
      }
      sprintf(result = Malloc(SMALLSTR), "%d", res ) ;
   }
   else
   {
      cptr = ((*string)&&(myisnumber(string,&junk))) ? "NUM" : "CHAR" ;
      strcpy( result=Malloc(5), cptr ) ;
   }

   return result ;
}


char *std_trace( paramboxptr parms )
{
   extern proclevel currlevel ;
   extern sysinfo systeminfo ;
   char *result, *string ;
   int ptr=0 ;

   checkparam( parms, 0, 1 ) ;

   result = Malloc( 3 ) ;
   if (systeminfo->interactive)
      result[ptr++] = '?' ;

   result[ptr++] = currlevel->tracestat ;
   result[ptr] = 0x00 ;

   ptr = 0 ;
   if (string=parms->value)
   {
      if (string[ptr]=='?')
      {
         ptr++ ;
         systeminfo->interactive = 1 ;
      }
 
      currlevel->tracestat = toupper(string[ptr]) ;
   }

   return result ;
}
     

