/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1995  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.13 of 17-July-95
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	sunsite.unc.edu:/pub/languages/forth/pfe-VERSION.tar.gz
 * or	ftp.cygnus.com:/pub/forth/pfe-VERSION.tar.gz
 *
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * core.c ---         Core words and Core extension words
 * (duz 09Jul93)
 */

#include "forth.h"
#include "support.h"
#include "compiler.h"
#include "dblsub.h"
#include "term.h"

#include <string.h>
#include <float.h>
#include <limits.h>

#include "missing.h"

/************************************************************************/
/* Core Words                                                           */
/************************************************************************/

Code (store)
{
  *(Cell *) sp[0] = sp[1];
  sp += 2;
}

Code (number_sign)
{
  hold (num2dig (u_d_div ((udCell *) sp, BASE)));
}

code (number_sign_greater)
{
  sp[1] = (Cell) HLD;
  sp[0] = (Cell) (PAD - HLD);
}

code (number_sign_s)
{
  do
    number_sign_ ();
  while (sp[0] || sp[1]);
}

Code (tick)
{
  tick ((Xt *) --sp);
}

Code (paren)
{
  char *p;
  uCell n;

  switch (SOURCE_ID)
    {
    case -1:
    case 0:
      parse (')', &p, &n);
      break;
    default:
      while (!parse (')', &p, &n) && refill ());
      break;
    }
}

Code (star)
{
  sp[1] = sp[0] * sp[1];
  sp++;
}

Code (star_slash)
{
  fdiv_t res = fmdiv (mmul (sp[2], sp[1]), sp[0]);

  sp += 2;
  sp[0] = res.quot;
}

Code (star_slash_mod)
{
  *(fdiv_t *) &sp[1] = fmdiv (mmul (sp[2], sp[1]), sp[0]);
  sp++;
}

Code (plus)
{
  sp[1] += sp[0];
  sp++;
}

Code (plus_store)
{
  *(Cell *) sp[0] += sp[1];
  sp += 2;
}

code (plus_loop_execution)
{
  Cell i = *sp++;

  if (i < 0
      ? (*RP += i) >= 0
      : (*RP += i) < 0)
    ip = rp[2];
  else
    rp += 3;
}

Code (plus_loop)
{
  question_pairs (LOOP_MAGIC);
  compile1 ();
  forward_resolve_ ();
}

COMPILES (plus_loop, plus_loop_execution,
	  SKIPS_NOTHING, LOOP_STYLE);

Code (comma)
{
  COMMA (*sp++);
}

Code (minus)
{
  sp[1] -= sp[0];
  sp++;
}

code (dot)
{
  s_to_d_ ();
  d_dot_ ();
}

code (dot_quote_execution)
{
  char *p = (char *) ip;

  type (p + 1, *p);
  SKIP_STRING;
}

Code (dot_quote)
{
  compile1 ();
  alloc_parsed_string ('"');
}

COMPILES (dot_quote, dot_quote_execution,
	  SKIPS_STRING, DEFAULT_STYLE);

Code (slash)
{
  fdiv_t res = fdiv (sp[1], sp[0]);

  *++sp = res.quot;
}

Code (slash_mod)
{
  *(fdiv_t *) sp = fdiv (sp[1], sp[0]);
}

Code (zero_less)
{
  *sp = FLAG (*sp < 0);
}

Code (zero_equal)
{
  *sp = FLAG (*sp == 0);
}

Code (one_plus)
{
  ++*sp;
}

Code (one_minus)
{
  --*sp;
}

Code (two_store)
{
  *(dCell *) *sp = *(dCell *) &sp[1];
  sp += 3;
}

Code (two_star)
{
  *sp <<= 1;
}

Code (two_slash)
{
  *sp >>= 1;
}

Code (two_fetch)
{
  dCell *p = (dCell *) *sp--;

  *(dCell *) sp = *p;
}

Code (two_drop)
{
  sp += 2;
}

Code (two_dup)
{
  sp -= 2;
  sp[0] = sp[2];
  sp[1] = sp[3];
}

Code (two_over)
{
  sp -= 2;
  sp[0] = sp[4];
  sp[1] = sp[5];
}

Code (two_swap)
{
  Cell h;

  h = sp[0];
  sp[0] = sp[2];
  sp[2] = h;
  h = sp[1];
  sp[1] = sp[3];
  sp[3] = h;
}

void
colon_runtime (void)
{
  *--rp = ip;
  ip = (Xt *) PFA;
}

Code (colon)
{
  question_exec_ ();
  header (colon_runtime, SMUDGED);
  store_csp_ ();
  STATE = TRUE;
  sys.locals = NULL;
}

code (semicolon_execution)
{
  ip = *rp++;
}

Code (semicolon)
{
  if (sys.locals)
    {
      compile2 ();
      sys.locals = NULL;
    }
  else
    compile1 ();
  question_csp_ ();
  STATE = FALSE;
  unsmudge_ ();
}

COMPILES2 (semicolon, semicolon_execution, locals_exit_execution,
	   SKIPS_NOTHING, SEMICOLON_STYLE);

Code (less_than)
{
  sp[1] = FLAG (sp[1] < sp[0]);
  sp++;
}

code (less_number_sign)
{
  HLD = PAD;
}

Code (equals)
{
  sp[1] = FLAG (sp[1] == sp[0]);
  sp++;
}

Code (greater_than)
{
  sp[1] = FLAG (sp[1] > sp[0]);
  sp++;
}

Code (to_body)
{
  *sp = (Cell) TO_BODY (*sp);
}

Code (to_number)
{
  sp[1] = (Cell)
    to_number ((char *) sp[1],
	       (uCell *) &sp[0],
	       (udCell *) &sp[2],
	       BASE);
}

Code (to_r)
{
  RPUSH (*sp++);
}

Code (question_dupe)
{
  if (*sp)
    --sp, sp[0] = sp[1];
}

code (fetch)
{
  *sp = *(Cell *) *sp;
}

Code (abort)
{
  tHrow (THROW_ABORT);
}

code (abort_quote_execution)
{
  char *p = (char *) ip;

  SKIP_STRING;
  if (*sp++ == 0)
    return;
  tHrow (THROW_ABORT_QUOTE, p + 1, *(Byte *) p);
}

Code (abort_quote)
{
  compile1 ();
  alloc_parsed_string ('"');
}

COMPILES (abort_quote, abort_quote_execution,
	  SKIPS_STRING, DEFAULT_STYLE);

Code (abs)
{
  if (*sp < 0)
    *sp = -*sp;
}

Code (accept)
{
  sp[1] = aCcept ((char *) sp[1], sp[0]);
  sp += 1;
}

code (align)
{
  while (!ALIGNED (DP))
    *DP++ = 0;
}

code (aligned)
{
  *sp = aligned (*sp);
}

Code (allot)
{
  DP += *sp++;
}

Code (and)
{
  sp[1] &= sp[0];
  sp++;
}

Code (begin)
{
  compile1 ();
  backward_mark_ ();
  *--sp = DEST_MAGIC;
}

COMPILES (begin, noop, SKIPS_NOTHING, BEGIN_STYLE);

Code (c_store)
{
  *(char *) sp[0] = sp[1];
  sp += 2;
}

Code (c_comma)
{
  *DP++ = (Byte) *sp++;
}

Code (c_fetch)
{
  *sp = *(Byte *) *sp;
}

Code (cell_plus)
{
  *sp += sizeof (Cell);
}

Code (cells)
{
  *sp *= sizeof (Cell);
}

Code (char)
{
  char *p;
  uCell n;

  skip_delimiter (' ');
  parse (' ', &p, &n);
  if (n == 0)
    tHrow (THROW_INVALID_NAME);
  *--sp = *(Byte *) p;
}

Code (char_plus)
{
  *sp += sizeof (char);
}

Code (chars)
{
  *sp *= sizeof (char);
}

void
constant_runtime (void)
{
  *--sp = PFA[0];
}

Code (constant)
{
  header (constant_runtime, 0);
  COMMA (*sp++);
}

Code (count)
{
  Byte *p;

  p = (Byte *) (*sp)++;
  *--sp = *p;
}

code (cr)
{
  outc ('\n');
  OUT = 0;
  sys.lines++;
}

void
create_runtime (void)
{
  *--sp = (Cell) PFA;
}

Code (create)
{
  header (create_runtime, 0);
}

code (decimal)
{
  BASE = 10;
}

Code (depth)
{
  size_t n;

  n = memtop.stack - sp;
  *--sp = n;
}

code (do_execution)
{
  rp -= 3;			/* push onto return-stack: */
  rp[2] = ++ip;			/*  ip to jump back to just after DO */
  rp[1] = (Xt *) sp[1];		/*  upper limit */
  rp[0] = (Xt *) (sp[0] -	/*  lower minus */
		  sp[1]);	/*  upper limit */
  sp += 2;
}
/* *INDENT-OFF* */
Code (do)
{
  compile1 ();
  forward_mark_ ();
  *--sp = LOOP_MAGIC;
}
COMPILES (do, do_execution,
	  SKIPS_OFFSET, DO_STYLE);
/* *INDENT-ON* */

void
does_defined_runtime (void)
{
  *--sp = (Cell) PFA;
  *--rp = ip;
  ip = (Xt *) CFA[-1];
}

code (does_execution)
{
  Xt p = runtime ();

  *p = does_defined_runtime;
  --p;
  *(Xt **) p = ip;
  if (lp != RP)
    ip = *rp++;
  else
    locals_exit_execution_ ();
}

Code (does)
{
  question_csp_ ();
  compile1 ();
  sys.locals = NULL;
}

COMPILES (does, does_execution,
	  SKIPS_NOTHING, DOES_STYLE);

code (drop)
{
  sp++;
}

Code (dupe)
{
  --sp;
  sp[0] = sp[1];
}

code (else_execution)
{
  BRANCH;
}
/* *INDENT-OFF* */
Code (else)
{
  question_pairs (ORIG_MAGIC);
  compile1 ();
  ahead_ ();
  rot_ ();
  forward_resolve_ ();
}
COMPILES (else, else_execution,
	  SKIPS_OFFSET, ELSE_STYLE);
/* *INDENT-ON* */

Code (emit)
{
  execute (sys.emit);
}

Code (environment_query)
{
  /* *INDENT-OFF* */
  static struct
    {
      char	*str;
      uCell	val;
    }
  env [] =
    {
      { "/COUNTED-STRING",	UCHAR_MAX },
      { "/HOLD",		MIN_HOLD },
      { "/PAD",			MIN_PAD },
      { "ADDRESS-UNIT-BITS",	CHAR_BIT },
      { "CORE",			(uCell)TRUE },
      { "CORE-EXT",		(uCell)TRUE },
      { "FLOORED",		(uCell)TRUE },
      { "MAX-CHAR",		UCHAR_MAX },
      { "MAX-N",		CELL_MAX },
      { "MAX-U",		UCELL_MAX },
      { "MAX-D",		0 },
      { "MAX-UD",		0 },
      { "STACK-CELLS",		0 },
      { "RETURN-STACK-CELLS",	0 },
      { "BLOCK",		(uCell)TRUE },
      { "BLOCK-EXT",		(uCell)TRUE },
      { "DOUBLE",		(uCell)TRUE },
      { "DOUBLE-EXT",		(uCell)TRUE },
      { "EXCEPTION",		(uCell)TRUE },
      { "EXCEPTION-EXT",	(uCell)TRUE },
      { "FACILITY",		(uCell)TRUE },
      { "FACILITY-EXT",		(uCell)TRUE },
      { "FILE",			(uCell)TRUE },
      { "FILE-EXT",		(uCell)TRUE },
      { "FLOATING",		(uCell)TRUE },
      { "FLOATING-EXT",		(uCell)TRUE },
      { "FLOATING-STACK",	0 },
      { "MAX-FLOAT",		0 },
      { "#LOCALS",		MAX_LOCALS },
      { "LOCALS",		(uCell)TRUE },
      { "LOCALS-EXT",		(uCell)TRUE },
      { "MEMORY-ALLOC",		(uCell)TRUE },
      { "MEMORY-ALLOC-EXT",	(uCell)TRUE },
      { "TOOLS",		(uCell)TRUE },
      { "TOOLS-EXT",		FALSE },
      { "SEARCH-ORDER",		(uCell)TRUE },
      { "SEARCH-ORDER-EXT",	(uCell)TRUE },
      { "WORDLISTS",		ORDER_LEN },
      { "STRING",		(uCell)TRUE },
      { "STRING-EXT",		(uCell)TRUE },
      /* pfe extensions: */
      { "HOST-SYSTEM",		(uCell)host_system },
    };
  /* *INDENT-ON* */

  char buf[32];
  int i;

  store_c_string ((char *) sp[1], sp[0], buf, sizeof buf);
  if (LOWER_CASE)
    upper (buf, sp[0]);
  for (i = 0; i < DIM (env); i++)
    if (strcmp (buf, env[i].str) == 0)
      break;
  ++sp;
  switch (i)
    {
    case DIM (env):
      *sp = FALSE;
      return;
    default:
      *sp = env[i].val;
      break;
    case 10:
      *sp = UINT_MAX;
      *--sp = INT_MAX;
      break;
    case 11:
      *sp = UINT_MAX;
      *--sp = UINT_MAX;
      break;
    case 12:
      *sp = option.stack_size;
      break;
    case 13:
      *sp = option.ret_stack_size;
      break;
    case 26:
      *sp = option.flt_stack_size;
      break;
    case 27:
      ++sp;
      *--fp = DBL_MAX;
      break;
    case 40:
      *sp = env[i].val;
      *--sp = strlen ((char *) env[i].val);
      break;
    }
  *--sp = TRUE;
}

Code (evaluate)
{
  char *p = (char *) sp[1];
  int n = sp[0];

  sp += 2;
  evaluate (p, n);
}

Code (execute)
{
  execute ((Xt) *sp++);
}

code (exit)
{
  if (sys.locals)
    compile2 ();
  else
    compile1 ();
}

COMPILES2 (exit, semicolon_execution, locals_exit_execution,
	   SKIPS_NOTHING, DEFAULT_STYLE);

Code (fill)
{
  memset ((void *) sp[2], sp[0], sp[1]);
  sp += 3;
}

Code (find)
{
  char *p = (char *) *sp;

  p = find (p + 1, *p);
  if (p)
    {
      *sp = (Cell) name_from (p);
      *--sp = *p & IMMEDIATE ? 1 : -1;
    }
  else
    *--sp = 0;
}

Code (f_m_slash_mod)
{
  Cell denom = *sp++;

  *(fdiv_t *) sp = fmdiv (*(dCell *) sp, denom);
}

Code (here)
{
  *--sp = (Cell) DP;
}

Code (hold)
{
  hold ((char) *sp++);
}

Code (i)
{
  *--sp = RP[0] + RP[1];
}

code (if_execution)
{
  if (!*sp++)
    BRANCH;
  else
    ip++;
}
/* *INDENT-OFF* */
Code (if)
{
  compile1 ();
  ahead_ ();
}
COMPILES (if, if_execution,
	  SKIPS_OFFSET, IF_STYLE);
/* *INDENT-ON* */

Code (immediate)
{
  if (LAST)
    *LAST |= IMMEDIATE;
  else
    tHrow (THROW_ARG_TYPE);
}

code (invert)
{
  *sp = ~*sp;
}

Code (j)
{
  *--sp = RP[3] + RP[4];
}

Code (key)
{
  execute (sys.key);
}

Code (leave)
{
  ip = rp[2] - 1;
  rp += 3;
  BRANCH;
}

code (literal_execution)
{
  POP (Cell, ip, *--sp);
}

Code (literal)
{
  if (STATE)
    {
      compile1 ();
      COMMA (*sp++);
    }
}

COMPILES (literal, literal_execution,
	  SKIPS_CELL, DEFAULT_STYLE);

code (loop_execution)
{
  if (++*RP)			/* increment top of return stack */
    ip = rp[2];			/* if nonzero: loop back */
  else
    rp += 3;			/* if zero: terminate loop */
}

Code (loop)
{
  question_pairs (LOOP_MAGIC);
  compile1 ();
  forward_resolve_ ();
}

COMPILES (loop, loop_execution,
	  SKIPS_NOTHING, LOOP_STYLE);

Code (l_shift)
{
  sp[1] <<= sp[0];
  sp++;
}

Code (m_star)
{
  *(dCell *) sp = mmul (sp[0], sp[1]);
}

Code (max)
{
  if (sp[0] > sp[1])
    sp[1] = sp[0];
  sp++;
}

Code (min)
{
  if (sp[0] < sp[1])
    sp[1] = sp[0];
  sp++;
}

Code (mod)
{
  fdiv_t res = fdiv (sp[1], sp[0]);

  *++sp = res.rem;
}

Code (move)
{
  memmove ((void *) sp[1], (void *) sp[2], (size_t) sp[0]);
  sp += 3;
}

Code (negate)
{
  *sp = -*sp;
}

Code (or)
{
  sp[1] |= sp[0];
  sp++;
}

Code (over)
{
  --sp;
  sp[0] = sp[2];
}

code (postpone_execution)
{
  Cell xt;
  POP (Cell, ip, xt);

  COMMA (xt);
}

Code (postpone)
{
  Xt xt;

  question_comp_ ();
  if (!(*tick (&xt) & IMMEDIATE))
    compile1 ();
  COMMA (xt);
}

COMPILES (postpone, postpone_execution,
	  SKIPS_NOTHING, DEFAULT_STYLE);

Code (quit)
{
  tHrow (THROW_QUIT);
}

Code (r_from)
{
  RPOP (*--sp);
}

Code (r_fetch)
{
  *--sp = *RP;
}

Code (recurse)
{
  question_comp_ ();
  if (LAST)
    COMMA (name_from (LAST));
  else
    tHrow (THROW_ARG_TYPE);
}

Code (repeat)
{
  question_pairs (DEST_MAGIC);
  compile1 ();
  backward_resolve_ ();
  question_pairs (ORIG_MAGIC);
  forward_resolve_ ();
}

COMPILES (repeat, else_execution,
	  SKIPS_OFFSET, REPEAT_STYLE);

code (rot)
{
  Cell h = sp[2];

  sp[2] = sp[1];
  sp[1] = sp[0];
  sp[0] = h;
}

Code (r_shift)
{
  *(uCell *) &sp[1] >>= sp[0];
  sp++;
}

code (s_quote_execution)
{
  Byte *p = (Byte *) ip;

  sp -= 2;
  sp[0] = *p;
  sp[1] = (Cell) (p + 1);
  SKIP_STRING;
}

Code (s_quote)
{
  if (STATE)
    {
      compile1 ();
      alloc_parsed_string ('"');
    }
  else
    {
      char *p, *q;
      uCell n;

      p = pocket ();
      parse ('"', &q, &n);
      if (n > 255)
	n = 255;
      *p++ = n;
      memcpy (p, q, n);
      *--sp = (Cell) p;
      *--sp = n;
    }
}

COMPILES (s_quote, s_quote_execution,
	  SKIPS_STRING, DEFAULT_STYLE);

code (s_to_d)
{
  sp--;
  sp[0] = sp[1] < 0 ? -1 : 0;
}

Code (sign)
{
  if (*sp++ < 0)
    hold ('-');
}

Code (s_m_slash_rem)
{
  Cell denom = *sp++;

  *(fdiv_t *) sp = smdiv (*(dCell *) sp, denom);
}

Code (source)
{
  char *p;
  int in;

  source (&p, &in);
  sp -= 2;
  sp[1] = (Cell) p;
  sp[0] = in;
}

code (space)
{
  outc (' ');
}

code (spaces)
{
  spaces (*sp++);
}

code (swap)
{
  Cell h = sp[1];

  sp[1] = sp[0];
  sp[0] = h;
}

Code (then)
{
  compile1 ();
  question_pairs (ORIG_MAGIC);
  forward_resolve_ ();
}

COMPILES (then, noop, SKIPS_NOTHING, THEN_STYLE);

code (type)
{
  execute (sys.type);
}

Code (u_dot)
{
  *--sp = 0;
  d_dot_ ();
}

Code (u_less_than)
{
  sp[1] = FLAG ((uCell) sp[1] < (uCell) sp[0]);
  sp++;
}

Code (u_m_star)
{
  *(udCell *) sp = ummul ((uCell) sp[0], (uCell) sp[1]);
}

Code (u_m_slash_mod)
{
  uCell denom = (uCell) *sp++;

  *(udiv_t *) sp = umdiv (*(udCell *) sp, denom);
}

Code (unloop)
{
  rp += 3;
}

code (until)
{
  question_pairs (DEST_MAGIC);
  compile1 ();
  backward_resolve_ ();
}

COMPILES (until, if_execution,
	  SKIPS_OFFSET, UNTIL_STYLE);

Code (variable)
{
  create_ ();
  COMMA (0);
}
/* *INDENT-OFF* */
Code (while)
{
  question_pairs (DEST_MAGIC);
  *--sp = DEST_MAGIC;
  compile1 ();
  ahead_ ();
  two_swap_ ();
}
COMPILES (while, if_execution,
	  SKIPS_OFFSET, WHILE_STYLE);
/* *INDENT-ON* */

Code (word)
{
  *sp = (Cell) word ((char) *sp);
}

Code (xor)
{
  sp[1] ^= sp[0];
  sp++;
}

Code (left_bracket)
{
  question_comp_ ();
  STATE = FALSE;
}

Code (bracket_tick)
{
  compile1 ();
  tick_ ();
  comma_ ();
}

COMPILES (bracket_tick, literal_execution,
	  SKIPS_NOTHING, DEFAULT_STYLE);

Code (bracket_char)
{
  compile1 ();
  char_ ();
  comma_ ();
}

COMPILES (bracket_char, literal_execution,
	  SKIPS_CELL, DEFAULT_STYLE);

Code (right_bracket)
{
  STATE = TRUE;
}

/************************************************************************/
/* Core Extension Words                                                 */
/************************************************************************/

Code (dot_paren)
{
  char *p;
  uCell n;

  switch (SOURCE_ID)
    {
    case -1:
    case 0:
      parse (')', &p, &n);
      type (p, n);
      break;
    default:
      while (!parse (')', &p, &n))
	{
	  type (p, n);
	  if (!refill ())
	    return;
	  cr_ ();
	}
      type (p, n);
    }
}

code (dot_r)
{
  to_r_ ();
  s_to_d_ ();
  r_from_ ();
  d_dot_r_ ();
}

Code (zero_not_equals)
{
  *sp = FLAG (*sp != 0);
}

Code (zero_greater)
{
  *sp = FLAG (*sp > 0);
}

Code (two_to_r)
{
  RPUSH (sp[1]);
  RPUSH (sp[0]);
  sp += 2;
}

Code (two_r_from)
{
  sp -= 2;
  RPOP (sp[0]);
  RPOP (sp[1]);
}

Code (two_r_fetch)
{
  sp -= 2;
  sp[0] = RP[0];
  sp[1] = RP[1];
}

Code (colon_noname)
{
  question_exec_ ();
  align_ ();
  *--sp = (Cell) DP;
  COMMA (colon_runtime);
  store_csp_ ();
  STATE = TRUE;
}

Code (not_equals)
{
  sp[1] = FLAG (sp[0] != sp[1]);
  sp++;
}

code (question_do_execution)
{
  if (sp[0] == sp[1])		/* if limits are equal */
    sp += 2, BRANCH;		/* drop them and branch */
  else
    do_execution_ ();		/* else like DO */
}

Code (question_do)
{
  compile1 ();
  forward_mark_ ();
  *--sp = LOOP_MAGIC;
}

COMPILES (question_do, question_do_execution,
	  SKIPS_OFFSET, DO_STYLE);

Code (again)
{
  question_pairs (DEST_MAGIC);
  compile1 ();
  backward_resolve_ ();
}

COMPILES (again, else_execution,
	  SKIPS_OFFSET, AGAIN_STYLE);

code (c_quote_execution)
{
  *--sp = (Cell) ip;
  SKIP_STRING;
}

Code (c_quote)
{
  if (STATE)
    {
      compile1 ();
      alloc_parsed_string ('"');
    }
  else
    {
      s_quote_ ();
      drop_ ();
      --*sp;
    }
}

COMPILES (c_quote, c_quote_execution,
	  SKIPS_STRING, DEFAULT_STYLE);
/* *INDENT-OFF* */
Code (case)
{
  compile1 ();
  *--sp = (Cell) CSP;
  CSP = sp;
  *--sp = CASE_MAGIC;
}
/* *INDENT-ON* */

COMPILES (case, noop, SKIPS_NOTHING, CASE_STYLE);

Code (compile_comma)
{
  COMMA (*sp++);
}

Code (convert)
{
  uCell n = UINT_MAX;

  sp[0] = (Cell) to_number ((char *) sp[0] + 1, &n, (udCell *) &sp[1], BASE);
}

Code (endcase)
{
  question_pairs (CASE_MAGIC);
  compile1 ();
  while (sp < CSP)
    forward_resolve_ ();
  CSP = (Cell *) *sp++;
}

COMPILES (endcase, drop,
	  SKIPS_NOTHING, ENDCASE_STYLE);

Code (endof)
{
  question_pairs (OF_MAGIC);
  compile1 ();
  forward_mark_ ();
  swap_ ();
  forward_resolve_ ();
  *--sp = CASE_MAGIC;
}

COMPILES (endof, else_execution,
	  SKIPS_OFFSET, ENDOF_STYLE);

Code (erase)
{
  memset ((void *) sp[1], 0, sp[0]);
  sp += 2;
}

Code (expect)
{
  execute (sys.expect);
}

Code (hex)
{
  BASE = 16;
}

void
marker_runtime (void)
{
  forget (NFA);
}

Code (marker)
{
  header (marker_runtime, 0);
}

Code (nip)
{
  sp[1] = sp[0];
  sp++;
}

code (of_execution)
{
  if (sp[0] != sp[1])		/* tos equals second? */
    sp += 1, BRANCH;		/* no: drop top, branch */
  else
    sp += 2, ip++;		/* yes: drop both, don't branch */
}

Code (of)
{
  question_pairs (CASE_MAGIC);
  compile1 ();
  forward_mark_ ();
  *--sp = OF_MAGIC;
}

COMPILES (of, of_execution,
	  SKIPS_OFFSET, OF_STYLE);

Code (pad)
{
  *--sp = (Cell) PAD;
}

Code (parse)
{
  char delim = *sp;

  --sp;

#if 0
  switch (SOURCE_ID)
    {
    case -1:
    case 0:
      parse (delim, (char **) &sp[1], (uCell *) &sp[0]);
      break;
    default:
      while (!parse (delim, (char **) &sp[1], (uCell *) &sp[0])
	     && refill ());
      break;
    }
#else
  parse (delim, (char **) &sp[1], (uCell *) &sp[0]);
#endif
}

Code (pick)			/* PICK */
{
  *sp = sp[*sp + 1];
}

Code (refill)
{
  *--sp = refill ();
}

Code (restore_input)
{
  if (*sp++ != sizeof (Iframe) / sizeof (Cell))
      tHrow (THROW_ARG_TYPE);

  sp = (Cell *) restore_input (sp);
  *--sp = 0;
}

Code (roll)
{
  Cell i = *sp++;
  Cell h = sp[i];

  for (; i > 0; i--)
    sp[i] = sp[i - 1];
  sp[0] = h;
}

Code (save_input)
{
  sp = (Cell *) save_input (sp);
  *--sp = sizeof (Iframe) / sizeof (Cell);
}

code (to_execution)
{
  *TO_BODY (*ip++) = *sp++;
}

Code (to)
{
  char *p;
  int l, n;
  Xt xt;

  if (STATE)
    {
      p = word (' ');
      l = *(Byte *) p++;
      if (sys.locals && (n = find_local (p, l)) != 0)
	{
	  compile2 ();
	  COMMA (n);
	}
      else
	{
	  if ((p = find (p, l)) == NULL)
	    tHrow (THROW_UNDEFINED);
	  compile1 ();
	  COMMA (name_from (p));
	}
    }
  else
    {
      tick (&xt);
      *TO_BODY (xt) = *sp++;
    }
}

COMPILES2 (to, to_execution, to_local_execution,
	   SKIPS_CELL, DEFAULT_STYLE);

Code (tuck)
{
  --sp;
  sp[0] = sp[1];
  sp[1] = sp[2];
  sp[2] = sp[0];
}

Code (u_dot_r)
{
  *--sp = 0;
  swap_ ();
  d_dot_r_ ();
}

Code (u_greater_than)
{
  sp[1] = FLAG ((uCell) sp[1] > (uCell) sp[0]);
  sp++;
}

Code (unused)
{
  *--sp = memtop.dict - DP;
}

void
value_runtime (void)
{
  *--sp = PFA[0];
}

Code (value)
{
  header (value_runtime, 0);
  COMMA (*sp++);
}

Code (within)
{
  sp[2] = FLAG ((uCell) (sp[2] - sp[1]) <
		(uCell) (sp[0] - sp[1]));
  sp += 2;
}

code (bracket_compile)
{
  question_comp_ ();
  tick_ ();
  comma_ ();
}

Code (backslash)
{
  switch (SOURCE_ID)
    {
    case 0:
      if (BLK)
	{
	  TO_IN += 64 - TO_IN % 64;
	  break;
	}
    case -1:
      TO_IN = NUMBER_TIB;
      break;
    default:
      refill ();
    }
}
/* *INDENT-OFF* */
LISTWORDS (core) =
{
  /* core words */
  CO ("!",		store),
  CO ("#",		number_sign),
  CO ("#>",		number_sign_greater),
  CO ("#S",		number_sign_s),
  CO ("'",		tick),
  CI ("(",		paren),
  CO ("*",		star),
  CO ("*/",		star_slash),
  CO ("*/MOD",		star_slash_mod),
  CO ("+",		plus),
  CO ("+!",		plus_store),
  CS ("+LOOP",		plus_loop),
  CO (",",		comma),
  CO ("-",		minus),
  CO (".",		dot),
  CS (".\"",		dot_quote),
  CO ("/",		slash),
  CO ("/MOD",		slash_mod),
  CO ("0<",		zero_less),
  CO ("0=",		zero_equal),
  CO ("1+",		one_plus),
  CO ("1-",		one_minus),
  CO ("2!",		two_store),
  CO ("2*",		two_star),
  CO ("2/",		two_slash),
  CO ("2@",		two_fetch),
  CO ("2DROP",		two_drop),
  CO ("2DUP",		two_dup),
  CO ("2OVER",		two_over),
  CO ("2SWAP",		two_swap),
  CO (":",		colon),
  CS (";",		semicolon),
  CO ("<",		less_than),
  CO ("<#",		less_number_sign),
  CO ("=",		equals),
  CO (">",		greater_than),
  CO (">BODY",		to_body),
  SV (">IN",		TO_IN),
  CO (">NUMBER",	to_number),
  CO (">R",		to_r),
  CO ("?DUP",		question_dupe),
  CO ("@",		fetch),
  CO ("ABORT",		abort),
  CS ("ABORT\"",	abort_quote),
  CO ("ABS",		abs),
  CO ("ACCEPT",		accept),
  CO ("ALIGN",		align),
  CO ("ALIGNED",	aligned),
  CO ("ALLOT",		allot),
  CO ("AND",		and),
  SV ("BASE",		BASE),
  CS ("BEGIN",		begin),
  OC ("BL",		' '),
  CO ("C!",		c_store),
  CO ("C,",		c_comma),
  CO ("C@",		c_fetch),
  CO ("CELL+",		cell_plus),
  CO ("CELLS",		cells),
  CO ("CHAR",		char),
  CO ("CHAR+",		char_plus),
  CO ("CHARS",		chars),
  CO ("CONSTANT",	constant),
  CO ("COUNT",		count),
  CO ("CR",		cr),
  CO ("CREATE",		create),
  CO ("DECIMAL",	decimal),
  CO ("DEPTH",		depth),
  CS ("DO",		do),
  CS ("DOES>",		does),
  CO ("DROP",		drop),
  CO ("DUP",		dupe),
  CS ("ELSE",		else),
  CO ("EMIT",		emit),
  CO ("ENVIRONMENT?",	environment_query),
  CO ("EVALUATE",	evaluate),
  CO ("EXECUTE",	execute),
  CS ("EXIT",		exit),
  CO ("FILL",		fill),
  CO ("FIND",		find),
  CO ("FM/MOD",		f_m_slash_mod),
  CO ("HERE",		here),
  CO ("HOLD",		hold),
  CO ("I",		i),
  CS ("IF",		if),
  CO ("IMMEDIATE",	immediate),
  CO ("INVERT",		invert),
  CO ("J",		j),
  CO ("KEY",		key),
  CO ("LEAVE",		leave),
  CS ("LITERAL",	literal),
  CS ("LOOP",		loop),
  CO ("LSHIFT",		l_shift),
  CO ("M*",		m_star),
  CO ("MAX",		max),
  CO ("MIN",		min),
  CO ("MOD",		mod),
  CO ("MOVE",		move),
  CO ("NEGATE",		negate),
  CO ("OR",		or),
  CO ("OVER",		over),
  CS ("POSTPONE",	postpone),
  CO ("QUIT",		quit),
  CO ("R>",		r_from),
  CO ("R@",		r_fetch),
  CI ("RECURSE",	recurse),
  CS ("REPEAT",		repeat),
  CO ("ROT",		rot),
  CO ("RSHIFT",		r_shift),
  CS ("S\"",		s_quote),
  CO ("S>D",		s_to_d),
  CO ("SIGN",		sign),
  CO ("SM/REM",		s_m_slash_rem),
  CO ("SOURCE",		source),
  CO ("SPACE",		space),
  CO ("SPACES",		spaces),
  SV ("STATE",		STATE),
  CO ("SWAP",		swap),
  CS ("THEN",		then),
  CO ("TYPE",		type),
  CO ("U.",		u_dot),
  CO ("U<",		u_less_than),
  CO ("UM*",		u_m_star),
  CO ("UM/MOD",		u_m_slash_mod),
  CO ("UNLOOP",		unloop),
  CS ("UNTIL",		until),
  CO ("VARIABLE",	variable),
  CS ("WHILE",		while),
  CO ("WORD",		word),
  CO ("XOR",		xor),
  CI ("[",		left_bracket),
  CS ("[']",		bracket_tick),
  CS ("[CHAR]",		bracket_char),
  CO ("]",		right_bracket),
  /* core extension words */
  SV ("#TIB",		NUMBER_TIB),
  CI (".(",		dot_paren),
  CO (".R",		dot_r),
  CO ("0<>",		zero_not_equals),
  CO ("0>",		zero_greater),
  CO ("2>R",		two_to_r),
  CO ("2R>",		two_r_from),
  CO ("2R@",		two_r_fetch),
  CO (":NONAME",	colon_noname),
  CO ("<>",		not_equals),
  CS ("?DO",		question_do),
  CS ("AGAIN",		again),
  CS ("C\"",		c_quote),
  CS ("CASE",		case),
  CO ("COMPILE,",	compile_comma),
  CO ("CONVERT",	convert),
  CS ("ENDCASE",	endcase),
  CS ("ENDOF",		endof),
  CO ("ERASE",		erase),
  CO ("EXPECT",		expect),
  OC ("FALSE",		FALSE),
  CO ("HEX",		hex),
  CO ("MARKER",		marker),
  CO ("NIP",		nip),
  CS ("OF",		of),
  CO ("PAD",		pad),
  CO ("PARSE",		parse),
  CO ("PICK",		pick),
  CO ("QUERY",		query),
  CO ("REFILL",		refill),
  CO ("RESTORE-INPUT",	restore_input),
  CO ("ROLL",		roll),
  CO ("SAVE-INPUT",	save_input),
  SC ("SOURCE-ID",	SOURCE_ID),
  SV ("SPAN",		SPAN),
  SC ("TIB",		TIB),
  CS ("TO",		to),
  OC ("TRUE",		TRUE),
  CO ("TUCK",		tuck),
  CO ("U.R",		u_dot_r),
  CO ("U>",		u_greater_than),
  CO ("UNUSED",		unused),
  CO ("VALUE",		value),
  CO ("WITHIN",		within),
  CI ("[COMPILE]",	bracket_compile),
  CI ("\\",		backslash)
};
COUNTWORDS (core, "Core words + extensions");
