\ 6809 Assembler, Disassembler and software simulator, written in PFE.
\ Author: L.C. Benschop, Eindhoven, The Netherlands.

only forth also extensions also  forth definitions

marker empty
create 6809mem 65536 chars allot 

\ Words to reference 6809 MEMORY The 6809 is a big-endian machine.
\ These can be adapted so that memory-mapped IO is directed to
\ IO-devices or that those devices are emulated.
: VC@ ( addr --- c)
  65535 and 6809mem + c@ ;
: VC! ( c addr ---)
  65535 and 6809mem + c! ;
: V@ ( addr --- n)
  65535 and dup 6809mem + c@ 8 lshift swap 1+ 65535 and 6809mem + c@ or ;
: V! ( n addr ---)
  65535 and >r dup 8 rshift r@ 6809mem + c! r> 1+ 65535 and 6809mem + c! ;

: VLOAD ( addr --- |name ) \ Load object code in memory.
   BL WORD COUNT R/O OPEN-FILE ABORT" File not Found"
   >R
   6809MEM OVER + 65536 rot - r@ READ-FILE DROP drop 
   r> CLOSE-file DROP ;
: VSAVE ( addr len --- |name ) \ Save object code to disk.
   BL WORD COUNT W/O CREATE-FILE ABORT" No room on disk!" >R
   swap 6809MEM + swap r@ WRITE-FILE ABORT" No room on disk!"
   r> CLOSE-FILE DROP ;

: defer create 0 , does> @ execute ;
: is ' >body ! ;

VOCABULARY 6809ASM
6809ASM ALSO DEFINITIONS

' C, DEFER C, IS C,
' ,  DEFER ,  IS ,
' HERE DEFER HERE IS HERE
' ALLOT DEFER ALLOT IS ALLOT
VARIABLE VDP
: VHERE ( --- addr)
  VDP @ ;
: VALLOT VDP +! ;
: VC, ( c --- )
  VHERE VC! 1 VALLOT ;
: V, ( n ---)
  VHERE V! 2 VALLOT ;
: ORG VDP ! ;

: <MARK ( --- addr )
  HERE ;
: <RESOLVE ( addr ---)
  HERE 1+ - C, ;
: >MARK ( --- addr )
  HERE 0 C, ;
: >RESOLVE ( addr --- )
  HERE OVER 1+ - SWAP VC! ;

VARIABLE ?PREBYTE VARIABLE PREBYTE \ Byte $10 or $11 before opcode
VARIABLE ?OPCODE  VARIABLE OPCODE  \ Opcode byte
VARIABLE ?POSTBYTE VARIABLE POSTBYTE \ Byte after opcode indicating mode.
VARIABLE ?OPERAND  \ Address or data after instruction.
VARIABLE MODE \ True is direct addressing false is other.
VARIABLE DPAGE \ Direct page address.
: SETDP ( n ---) \ Set direct page.
  256 * DPAGE ! ;
0 SETDP

: NOINSTR \ Reset all the instruction flags so there will be no instruction.
  ?PREBYTE OFF ?OPCODE OFF ?POSTBYTE OFF ?OPERAND OFF MODE OFF ;
: A; \ Assemble current instruction and reset instruction flags.
  MODE @  IF \ direct addresiing?
   DUP DPAGE @ - 255 U> IF \ Is address 16 bits?
    2 ?OPERAND ! \ Indicate 16 bits address.
    OPCODE @ $F0 AND 0= \ Change opcode byte.
     IF $70 OPCODE +!
     ELSE $20 OPCODE +!
     THEN
   ELSE 1 ?OPERAND ! \ Indicate 8 bis address.
   THEN
  THEN
  ?PREBYTE @ IF PREBYTE @ C, THEN
  ?OPCODE @ IF OPCODE @ C, THEN
  ?POSTBYTE @ IF POSTBYTE @ C, THEN
  ?OPERAND @ IF
   CASE ?OPERAND @
    1 OF C, ENDOF            \ 8 bits data/address.
    2 OF , ENDOF             \ 16 bits data/address.
    3 OF HERE 1+ - C, ENDOF  \ 8 bits relative address.
    4 OF HERE 2 + - , ENDOF   \ 16 bits realtive address.
   ENDCASE
  THEN NOINSTR ;


: LABEL A; HERE CONSTANT ;

: flag10 \ Indicate that next instruction has prebyte $10
  ?PREBYTE ON $10 PREBYTE ! ;
: flag11 \ Indicate that next instruction has prebyte $11
  ?PREBYTE ON $11 PREBYTE ! ;

: # \ Signal immediate mode.
  MODE OFF $-10 OPCODE +! ;

: USE-POSTBYTE \ Signal that postbyte must be used.
  MODE OFF
  ?POSTBYTE ON
  OPCODE @ $F0 AND 0= IF
   $60 OPCODE +!
  ELSE
   OPCODE @ $80 AND IF
    $10 OPCODE +!
   THEN
  THEN ;

: [] \ Signal indirect mode.
  MODE @ IF \ Indirect addressing with 16-bits addres, no postbyte made yet.
   USE-POSTBYTE
   $9F POSTBYTE !   \ Make postbyte.
   2 ?OPERAND !     \ Indicate 16-bits address.
  ELSE
   POSTBYTE @ $80 AND 0= IF \ 5-bits address format already assembled?
    POSTBYTE @ $1F AND DUP $10 AND 0<> $E0 AND OR
    1 ?OPERAND !            \ Signal operand.
    POSTBYTE @ $60 AND $98 OR POSTBYTE ! \ Change postbyte.
   ELSE
    POSTBYTE @ $10 OR POSTBYTE ! \ Indicate indirect addressing.
   THEN
  THEN ;

: ,R \ Modes with a constant offset from a register.
  CREATE C,
  DOES> USE-POSTBYTE
        C@ POSTBYTE ! \ Make register field in postbyte.
        DUP 0= IF
         $84 POSTBYTE +! DROP \ Zero offset.
         ?OPERAND OFF
        ELSE
         DUP -16 >= OVER 15 <= AND IF \ 5-bit offset.
          $1F AND POSTBYTE +!
          ?OPERAND OFF
         ELSE
          DUP 128 + 256 U< IF \ 8-bit offset.
           $88 POSTBYTE +!
           1 ?OPERAND !
          ELSE
           $89 POSTBYTE +!    \ 16-bit offset.
           2 ?OPERAND !
          THEN
         THEN
        THEN ;
$00 ,R ,X
$20 ,R ,Y
$40 ,R ,U
$60 ,R ,S

: AMODE \ addressing modes with no operands.
  CREATE C,
  DOES> USE-POSTBYTE
        C@ POSTBYTE !
        ?OPERAND OFF ;
$80 AMODE ,X+   $81 AMODE ,X++ $82 AMODE ,-X   $83 AMODE ,--X
$85 AMODE B,X   $86 AMODE A,X  $8B AMODE D,X
$A0 AMODE ,Y+   $A1 AMODE ,Y++ $A2 AMODE ,-Y   $A3 AMODE ,--Y
$A5 AMODE B,Y   $A6 AMODE A,Y  $AB AMODE D,Y
$C0 AMODE ,U+   $C1 AMODE ,U++ $C2 AMODE ,-U   $C3 AMODE ,--U
$C5 AMODE B,U   $C6 AMODE A,U  $CB AMODE D,U
$E0 AMODE ,S+   $E1 AMODE ,S++ $E2 AMODE ,-S   $E3 AMODE ,--S
$E5 AMODE B,S   $E6 AMODE A,S  $EB AMODE D,S

: ,PCR \ Signal program counter relative.
  USE-POSTBYTE
  DUP
  HERE ?PREBYTE @ - 3 + - \ Subtract address after instruction
  128 + 256 U< IF \ 8-bits offset good?
   3 ?OPERAND !
   $8C POSTBYTE !
  ELSE
   4 ?OPERAND !
   $8D POSTBYTE !
  THEN ;

: USE-OPCODE ( c ---)
  ?OPCODE ON
  OPCODE ! ;

: IN1 \ Simple instructions with one byte opcode
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE ;
$12 IN1 NOP    $13 IN1 SYNC
$19 IN1 DAA    $1D IN1 SEX
$39 IN1 RTS    $3A IN1 ABX
$3B IN1 RTI    $3D IN1 MUL
$3F IN1 SWI    : SWI2 SWI flag10 ; : SWI3 SWI flag11 ;
$40 IN1 NEGA   $50 IN1 NEGB
$43 IN1 COMA   $53 IN1 COMB
$44 IN1 LSRA   $54 IN1 LSRB
$46 IN1 RORA   $56 IN1 RORB
$47 IN1 ASRA   $57 IN1 ASRB
$48 IN1 ASLA   $58 IN1 ASLB
$48 IN1 LSLA   $58 IN1 LSLB
$49 IN1 ROLA   $59 IN1 ROLB
$4A IN1 DECA   $5A IN1 DECB
$4C IN1 INCA   $5C IN1 INCB
$4D IN1 TSTA   $5D IN1 TSTB
$4F IN1 CLRA   $5F IN1 CLRB
\ Though not no-operand instructions the LEA instructions
\ are treated correctly as the postbyte is added by the mode words.
$30 IN1 LEAX   $31 IN1 LEAY
$32 IN1 LEAS   $33 IN1 LEAU
: DEX LEAX -1 ,X ; : INX LEAX 1 ,X ;
: DES LEAS -1 ,S ; : INS LEAS 1 ,S ;
: DEY LEAY -1 ,Y ; : INY LEAY 1 ,Y ;

: BR-8 \ relative branches with 8-bit offset
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE 3 ?OPERAND ! ;
  $20 BR-8 BRA   $21 BR-8 BRN
  $22 BR-8 BHI   $23 BR-8 BLS
  $24 BR-8 BCC   $25 BR-8 BCS
  $24 BR-8 BHS   $25 BR-8 BLO
  $26 BR-8 BNE   $27 BR-8 BEQ
  $28 BR-8 BVC   $29 BR-8 BVS
  $2A BR-8 BPL   $2B BR-8 BMI
  $2C BR-8 BGE   $2D BR-8 BLT
  $2E BR-8 BGT   $2F BR-8 BLE
  $8D BR-8 BSR

: LBRA
  A; $16 USE-OPCODE 4 ?OPERAND ! ;
: LBSR
  A; $17 USE-OPCODE 4 ?OPERAND ! ;

: BR16 \ Relative branches with 16-bit offset.
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE flag10 4 ?OPERAND ! ;
                  $21 BR16 LBRN
  $22 BR16 LBHI   $23 BR16 LBLS
  $24 BR16 LBCC   $25 BR16 LBCS
  $24 BR16 LBHS   $25 BR16 LBLO
  $26 BR16 LBNE   $27 BR16 LBEQ
  $28 BR16 LBVC   $29 BR16 LBVS
  $2A BR16 LBPL   $2B BR16 LBMI
  $2C BR16 LBGE   $2D BR16 LBLT
  $2E BR16 LBGT   $2F BR16 LBLE

: IN2 \ Instructions with one immediate data byte.
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE 1 ?OPERAND ! ;
$1A IN2 ORCC  $1C IN2 ANDCC  $3C IN2 CWAI
: CLC ANDCC $FE ; : SEC ORCC $01 ;
: CLF ANDCC $BF ; : SEF ORCC $40 ;
: CLI ANDCC $EF ; : SEI ORCC $10 ;
: CLIF ANDCC $AF ; : SEIF ORCC $50 ;
: CLV ANDCC $FD ; : SEV ORCC $02 ;
: % ( --- n) \ Interpret next word as a binary number.
  BASE @ 2 BASE ! BL WORD NUMBER? drop DROP SWAP BASE ! ;

: REG \ Registers as used in PUSH PULL TFR and EXG instructions.
  CREATE C, C,
  DOES> ?OPERAND @ IF \ Is a PUSH/PULL instruction meant?
         1+ C@ OR
        ELSE
         C@ POSTBYTE +! \ It's a TFR,EXG instruction.
        THEN ;
$06 $00 REG D,  $06 $00 REG D
$10 $10 REG X,  $10 $01 REG X
$20 $20 REG Y,  $20 $02 REG Y
$40 $30 REG U,  $40 $03 REG U
$40 $40 REG S,  $40 $04 REG S
$80 $50 REG PC, $80 $05 REG PC
$02 $80 REG A,  $02 $08 REG A
$04 $90 REG B,  $04 $09 REG B
$01 $A0 REG CC, $01 $0A REG CC
$08 $B0 REG DP, $08 $0B REG DP

: EXG A; $1E USE-OPCODE ?POSTBYTE ON 0 POSTBYTE ! ;
: TFR A; $1F USE-OPCODE ?POSTBYTE ON 0 POSTBYTE ! ;
: STK \ Stack instructions.
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE
        1 ?OPERAND ! 0 ;
$34 STK PSHS  $35 STK PULS
$36 STK PSHU  $37 STK PULU

: OP-8 \ Instructions with 8-bits data.
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE
        MODE ON
        1 ?OPERAND ! ;
$00 OP-8 NEG  $03 OP-8 COM
$04 OP-8 LSR  $06 OP-8 ROR
$07 OP-8 ASR  $08 OP-8 ASL
$08 OP-8 LSL  $09 OP-8 ROL
$0A OP-8 DEC  $0C OP-8 INC
$0D OP-8 TST  $0E OP-8 JMP
$0F OP-8 CLR
$90 OP-8 SUBA $D0 OP-8 SUBB
$91 OP-8 CMPA $D1 OP-8 CMPB
$92 OP-8 SBCA $D2 OP-8 SBCB
$94 OP-8 ANDA $D4 OP-8 ANDB
$95 OP-8 BITA $D5 OP-8 BITB
$96 OP-8 LDA  $D6 OP-8 LDB
$97 OP-8 STA  $D7 OP-8 STB
$98 OP-8 EORA $D8 OP-8 EORB
$99 OP-8 ADCA $D9 OP-8 ADCB
$9A OP-8 ORA  $DA OP-8 ORB
$9B OP-8 ADDA $DB OP-8 ADDB
$9D OP-8 JSR

: OP16 \ Instructions with 16-bits daia.
  CREATE C,
  DOES> >R A; R> C@ USE-OPCODE
        MODE ON
        2 ?OPERAND ! ;
$93 OP16 SUBD  $D3 OP16 ADDD
$9C OP16 CMPX  $DC OP16 LDD  $DD OP16 STD
$9E OP16 LDX   $DE OP16 LDU
$9F OP16 STX   $DF OP16 STU
: CMPD SUBD flag10 ; : CMPY CMPX flag10 ;
: LDY  LDX  flag10 ; : STY  STX  flag10 ;
: LDS  LDU  flag10 ; : STS  STU  flag10 ;
: CMPU SUBD flag11 ; : CMPS CMPX flag11 ;

\ Structured assembler constructs.
: IF >R A; R> C, >MARK ;
: THEN A; >RESOLVE ;
: ELSE A; $20 C, >MARK SWAP >RESOLVE ;
: BEGIN A; <MARK ;
: UNTIL >R A; R> C, <RESOLVE ;
: WHILE >R A; R> C, >MARK ;
: REPEAT A; $20 C, SWAP <RESOLVE >RESOLVE ;
: AGAIN $20 UNTIL ;
$22 CONSTANT U<= $23 CONSTANT U>
$24 CONSTANT U<  $25 CONSTANT U>=
$26 CONSTANT 0=  $27 CONSTANT 0<>
$28 CONSTANT VS  $29 CONSTANT VC
$2A CONSTANT 0<  $2B CONSTANT 0>=
$2C CONSTANT <   $2D CONSTANT >=
$2E CONSTANT <=  $2F CONSTANT >

' VC, IS C,
' V, IS ,
' VHERE IS HERE
' VALLOT IS ALLOT

: ENDASM \ End assembly.
  A; FORTH DEFINITIONS ;
FORTH DEFINITIONS
: ASSEMBLE \ Start assembly.
  6809ASM DEFINITIONS NOINSTR ;
ONLY FORTH ALSO extensions also forth DEFINITIONS

\ 6809 Simulator.

VOCABULARY 6809SIM 6809SIM ALSO DEFINITIONS

\ Processor registers.
VARIABLE AREG VARIABLE BREG
VARIABLE CCREG VARIABLE DPREG VARIABLE PCREG
VARIABLE XREG VARIABLE YREG VARIABLE UREG VARIABLE SREG
VARIABLE IREG \ Instruction register.
: DREG@ ( --- n)
  AREG @ $ff and 8 lshift BREG @ $ff and + ;
: DREG! ( n ---)
  DUP 255 AND BREG ! 8 rshift 255 AND AREG ! ;
: IMM-BYTE ( --- c) \ Get byte at program counter and increment PC.
  PCREG @ VC@ 1 PCREG +! ;
: IMM-WORD ( --- n) \ Get word at program counter and increment PC.
  PCREG @ V@  2 pcreg +! ;
: PSHSBYTE ( c ---) \ Push byte on stack.
  -1 SREG +! SREG @ VC! ;
: PSHSWORD ( n ---) \ Push word on stack.
  -2 SREG +! SREG @ V! ;
: PULSBYTE ( --- c) \ Pull byte from stack.
  SREG @ VC@ 1 SREG +! ;
: PULSWORD ( --- n) \ Pull word from stack.
  SREG @ V@  2 SREG +! ;

: SIGNED ( c --- n) \ Make signed number from signed byte.
  DUP 128 AND IF 256 - THEN ;

CREATE IXREGS XREG , YREG , UREG , SREG ,
VARIABLE INDEX

: ,R+ ( --- addr)
  INDEX @ @  1 INDEX @ +! ;
: ,R++ ( --- addr)
  INDEX @ @  2 INDEX @ +! ;
: ,-R  ( --- addr)
  -1 INDEX @ +!  INDEX @ @ ;
: ,--R ( --- addr)
  -2 INDEX @ +!  INDEX @ @ ;
: ,R ( --- addr)
  INDEX @ @ ;
: A,R ( --- addr)
  INDEX @ @ AREG @ $ff and SIGNED + ;
: B,R ( --- addr)
  INDEX @ @ BREG @ $ff and SIGNED + ;
: N,R ( --- addr)
  INDEX @ @ IMM-BYTE SIGNED + ;
: NN,R ( ---addr)
  INDEX @ @ IMM-WORD + ;
: D,R ( --- addr)
  INDEX @ @ DREG@ + ;
: N,PCR ( --- addr)
  IMM-BYTE SIGNED PCREG @ + ;
: NN,PCR ( --- addr)
  IMM-WORD PCREG @ + ;

CREATE PBTABLE ' ,R+ , ' ,R++ , ' ,-R , ' ,--R ,
               ' ,R ,  ' B,R  , ' A,R ,  ' FALSE ,
               ' N,R , ' NN,R , ' FALSE , ' D,R ,
               ' N,PCR , ' NN,PCR , ' FALSE , ' IMM-WORD ,

: POSTBYTE ( --- addr) \ Postbyte addressing forms.
  IMM-BYTE DUP $60 AND 5 rshift cells IXREGS + @ INDEX !
  DUP $80 AND IF \ Not 5-bit format.
     DUP >R $0F AND cells PBTABLE + @ EXECUTE \ Perform indexing.
     R> $10 AND IF V@ THEN \ Add indirection if necessary.
  ELSE \ 5-bit format.
     $1F AND DUP $10 AND IF $FFF0 OR THEN \ Sign extend 5 bits.
     INDEX @ @ +
  THEN ;

: IMM8 ( --- addr) \ Immediate addressing 8 bits.
  PCREG @ 1 PCREG +! ;
: IMM16 ( --- addr) \ Immediate addressing 16 bits.
  PCREG @ 2 PCREG +! ;
: DIRECT ( --- addr) \ Direct addressing.
  IMM-BYTE DPREG @ $ff and 8 lshift + ;
CREATE E0TABLE ' DIRECT , ' FALSE , ' POSTBYTE , ' IMM-WORD ,
: EADDR0 ( --- addr) \ Get effective address for NEG...CLR instructions.
  IREG @ $30 AND 4 rshift cells E0TABLE + @ EXECUTE ;
CREATE E8TABLE ' IMM8 , ' DIRECT , ' POSTBYTE , ' IMM-WORD ,
: EADDR8 ( --- addr) \ Get effective address for 8-bits instructions.
  IREG @ $30 AND 4 rshift cells E8TABLE + @ EXECUTE ;
CREATE E16TABLE ' IMM16 , ' DIRECT , ' POSTBYTE , ' IMM-WORD ,
: EADDR16 ( --- addr) \ Get effective address for 16-bits instructions.
  IREG @ $30 AND 4 rshift cells E16TABLE + @ EXECUTE ;
: ??? \ Illegal opcode.
  7 EMIT ;


: SEC \ Set carry flag.
  CCREG @ 1 OR CCREG ! ;
: CLC \ Clear carry flag.
  CCREG @ $FE AND CCREG ! ;
: SEZ \ Set zero flag.
  CCREG @ 4 OR CCREG ! ;
: CLZ \ Clear zero flag.
  CCREG @ $FB AND CCREG ! ;
: SEN \ Set sign flag.
  CCREG @ 8 OR CCREG ! ;
: CLN \ Clear sign flag.
  CCREG @ $F7 AND CCREG ! ;
: SEV \ Set overflow flag.
  CCREG @ 2 OR CCREG ! ;
: CLV \ Clear overflow flag.
  CCREG @ $FD AND CCREG ! ;
: SEH \ Set halfcarry flag.
  CCREG @ 32 OR CCREG ! ;
: CLH \ Clear halfcarry flag.
  CCREG @ $DF AND CCREG ! ;

: SETNZ8 \ Set zero and sign flag after 8-bit operation.
  DUP 255 AND IF CLZ ELSE SEZ THEN
  DUP 128 AND IF SEN ELSE CLN THEN ;
: SETNZ16 \ Set zero and sign flags after 16-bit operation.
  DUP $ffff and IF CLZ ELSE SEZ THEN
  DUP $8000 and IF SEN ELSE CLN THEN ;

: SETSTATUS ( n1 n2 n3 --- n3)
\ Set status bits dependent on result of arithmetic function.
  3DUP XOR XOR  $10 AND IF SEH ELSE CLH THEN
  DUP >R DUP 2/ XOR XOR XOR $80 AND IF SEV ELSE CLV THEN
  R> DUP $100 AND IF SEC ELSE CLC THEN
  SETNZ8 ;

: (ADD) ( n1 n2 --- n3) \ Add 8 bits and set status.
  2DUP + SETSTATUS ;
: (ADC) ( n1 n2 --- n3) \ Add with carry 8 bits and set status.
  2DUP + CCREG @ 1 AND + SETSTATUS ;
: (SUB) ( n1 n2 --- n3) \ Subtract 8 bits and set status.
  2DUP - SETSTATUS ;
: (SBC) ( n1 n2 --- n3) \ Subtract with carry 8 bits and set status.
  2DUP - CCREG @ 1 AND - SETSTATUS ;
: ADDA
  AREG @ $ff and EADDR8 VC@ (ADD) AREG ! ;
: ADDB
  BREG @ $ff and EADDR8 VC@ (ADD) BREG ! ;
: ADCA
  AREG @ $ff and EADDR8 VC@ (ADC) AREG ! ;
: ADCB
  BREG @ $ff and EADDR8 VC@ (ADC) BREG ! ;
: SUBA
  AREG @ $ff and EADDR8 VC@ (SUB) AREG ! ;
: SUBB
  BREG @ $ff and EADDR8 VC@ (SUB) BREG ! ;
: SBCA
  AREG @ $ff and EADDR8 VC@ (SBC) AREG ! ;
: SBCB
  BREG @ $ff and EADDR8 VC@ (SBC) BREG ! ;
: CMPA
  AREG @ $ff and EADDR8 VC@ (SUB) DROP ;
: CMPB
  BREG @ $ff and EADDR8 VC@ (SUB) DROP ;

: (AND) ( n1 n2 --- n3) \ AND and set status.
  AND SETNZ8 CLV ;
: (OR) ( n1 n2 --- n3)  \ OR and set status.
  OR SETNZ8 CLV ;
: (EOR) ( n1 n2 --- n3) \ Exclusive OR and set status.
  XOR SETNZ8 CLV ;
: ANDA
  AREG @ $ff and EADDR8 VC@ (AND) AREG ! ;
: ANDB
  BREG @ $ff and EADDR8 VC@ (AND) BREG ! ;
: ORA
  AREG @ $ff and EADDR8 VC@ (OR) AREG ! ;
: ORB
  BREG @ $ff and EADDR8 VC@ (OR) BREG ! ;
: EORA
  AREG @ $ff and EADDR8 VC@ (EOR) AREG ! ;
: EORB
  BREG @ $ff and EADDR8 VC@ (EOR) BREG ! ;
: BITA
  AREG @ $ff and EADDR8 VC@ (AND) DROP ;
: BITB
  BREG @ $ff and EADDR8 VC@ (AND) DROP ;

: LDA
  EADDR8 VC@ SETNZ8 CLV AREG ! ;
: LDB
  EADDR8 VC@ SETNZ8 CLV BREG ! ;
: STA
  AREG @ $ff and SETNZ8 CLV EADDR8 VC! ;
: STB
  BREG @ $ff and SETNZ8 CLV EADDR8 VC! ;

: JSR
  EADDR8  PCREG @ PSHSWORD  PCREG ! ;

: (NEG) ( n --- -n ) \ Negate n and set status register.
  0 SWAP (SUB) ;
: NEGA
  AREG @ $ff and (NEG) AREG ! ;
: NEGB
  BREG @ $ff and (NEG) BREG ! ;
: NEG
  EADDR0 DUP VC@ (NEG) SWAP VC! ;
: (COM) ( n --- nXOR-1) \ Comsplement n and set status register.
  NOT  SETNZ8 SEC CLV ;
: COMA
  AREG @ $ff and (COM) AREG ! ;
: COMB
  BREG @ $ff and (COM) BREG ! ;
: COM
  EADDR0 DUP VC@ (COM) SWAP VC! ;
: (LSR) ( n --- n/2 ) \ Logic shift right and set status.
  DUP 1 AND IF SEC ELSE CLC THEN
  2/ SETNZ8 ;
: LSRA
  AREG @ $ff and (LSR) AREG ! ;
: LSRB
  BREG @ $ff and (LSR) BREG ! ;
: LSR
  EADDR0 DUP VC@ (LSR) SWAP VC! ;
: (ROR) ( n --- n ROT right) \ Rotate right and set status.
  CCREG @ 1 AND >R
  DUP 1 AND IF SEC ELSE CLC THEN
  2/ R> IF $80 OR THEN SETNZ8 ;
: RORA
  AREG @ $ff and (ROR) AREG ! ;
: RORB
  BREG @ $ff and (ROR) BREG ! ;
: ROR
  EADDR0 DUP VC@ (ROR) SWAP VC! ;
: (ASR) ( n --- n/2) \ Arithmetic shift right and set status.
  DUP 1 AND IF SEC ELSE CLC THEN
  2/ DUP $40 AND IF $80 OR THEN
  DUP $10 AND IF SEH ELSE CLH THEN SETNZ8 ;
: ASRA
  AREG @ $ff and (ASR) AREG ! ;
: ASRB
  BREG @ $ff and (ASR) BREG ! ;
: ASR
  EADDR0 DUP VC@ (ASR) SWAP VC! ;
: (ASL) ( n --- n*2) \ Arithmetic (logic) shift left.
  DUP (ADD) ;
: ASLA
  AREG @ $ff and (ASL) AREG ! ;
: ASLB
  BREG @ $ff and (ASL) BREG ! ;
: ASL
  EADDR0 DUP VC@ (ASL) SWAP VC! ;
: (ROL) ( n --- n ROT left) \ Rotate left.
  CCREG @ 1 AND >R
  DUP $80 AND IF SEC ELSE CLC THEN
  2* DUP $80 AND IF SEV ELSE CLV THEN
  R> OR SETNZ8 ;
: ROLA
  AREG @ $ff and (ROL) AREG ! ;
: ROLB
  BREG @ $ff and (ROL) BREG ! ;
: ROL
  EADDR0 DUP VC@ (ROL) SWAP VC! ;
: (DEC) ( n --- n-1) \ Decrement and set status.
  1- DUP $7F = IF SEV ELSE CLV THEN SETNZ8 ;
: DECA
  AREG @ $ff and (DEC) AREG ! ;
: DECB
  BREG @ $ff and (DEC) BREG ! ;
: DEC
  EADDR0 DUP VC@ (DEC) SWAP VC! ;
: (INC) ( n --- n+1) \ Increment and set status.
  1+ DUP $80 = IF SEV ELSE CLV THEN SETNZ8 ;
: INCA
  AREG @ $ff and (INC) AREG ! ;
: INCB
  BREG @ $ff and (INC) BREG ! ;
: INC
  EADDR0 DUP VC@ (INC) SWAP VC! ;
: (TST) ( n --- ) \ Test and set status.
  SETNZ8 CLV DROP ;
: TSTA
  AREG @ $ff and (TST) ;
: TSTB
  BREG @ $ff and (TST) ;
: TST
  EADDR0 VC@ (TST) ;
: JMP
  EADDR0 PCREG ! ;
: (CLR) ( --- 0) \ Set the status flags as n CLR statement.
  SEZ CLN CLV CLC 0 ;
: CLRA
  (CLR) AREG ! ;
: CLRB
  (CLR) BREG ! ;
: CLR
  (CLR) EADDR0 VC! ;
: BSR
  IMM-BYTE
  PCREG @ PSHSWORD
  SIGNED PCREG +! ;

VARIABLE (INSTRTABLE)
VARIABLE FLAG 0 FLAG !
: 0FL
  1 FLAG !
  IMM-BYTE DUP IREG ! cells (INSTRTABLE) @ + @ EXECUTE 0 FLAG ! ;
: 1FL
  2 FLAG !
  IMM-BYTE DUP IREG ! cells (INSTRTABLE) @ + @ EXECUTE 0 FLAG ! ;

: NOP ;
: SYNC ;
: LBRA
  IMM-WORD PCREG +! ;
: LBSR
  IMM-WORD
  PCREG @ PSHSWORD
  PCREG +! ;
: DAA
  AREG @ $ff and AREG !
  CCREG @ $20 AND IF 6 AREG +! THEN
  AREG @ $0F AND 9 > IF 6 AREG +! THEN
  CCREG @ 1 AND IF $60 AREG +! THEN
  AREG @ $F0 AND $90 > IF $60 AREG +! THEN
  AREG @ 255 U> IF SEC THEN ;
: ORCC
  IMM-BYTE CCREG @ OR CCREG ! ;
: ANDCC
  IMM-BYTE CCREG @ AND CCREG ! ;
: MUL
  AREG @ $ff and BREG @ $ff and * DUP DREG!
  DUP $ffff and IF CLZ ELSE SEZ THEN
  $FF00 AND IF SEC ELSE CLC THEN ;
: SEX
  BREG @ $ff and SIGNED SETNZ16 DREG! ;
: ABX
  BREG @ $ff and XREG +! ;
: RTS
  PULSWORD PCREG ! ;
: RTI
  CCREG @ $80 AND
  PULSBYTE CCREG !
  IF
   PULSBYTE AREG !
   PULSBYTE BREG !
   PULSBYTE DPREG !
   PULSWORD XREG !
   PULSWORD YREG !
   PULSWORD UREG !
  THEN
  PULSWORD PCREG ! ;
: PSHALL \ Push all the registers.
  PCREG @ PSHSWORD
  UREG @  PSHSWORD
  YREG @  PSHSWORD
  XREG @  PSHSWORD
  DPREG @ PSHSBYTE
  BREG @ PSHSBYTE
  AREG @ PSHSBYTE
  CCREG @ PSHSBYTE ;

: SWI
  PSHALL
  CCREG @ $80 OR FLAG @ 0= IF $50 OR THEN CCREG !
  CASE FLAG @
  0 OF $FFFA ENDOF
  1 OF $FFF4 ENDOF
  2 OF $FFF2 ENDOF
  ENDCASE V@ PCREG ! ;

: IRQ \ Perform interrupt.
  CCREG @ $10 AND 0= IF
   PSHALL
   CCREG @ $90 OR CCREG !
   $FFF8 V@ PCREG !
  THEN ;
: NMI \ Perform nonmaskable interrupt.
  PSHALL
  CCREG @ $D0 OR CCREG !
  $FFFC V@ PCREG ! ;
: FIRQ \ Perform Fast interrupt.
  CCREG @ $40 AND 0= IF
   PCREG @ PSHSWORD
   CCREG @ PSHSBYTE
   CCREG @ $7F AND $50 OR CCREG !
   $FFF6 V@ PCREG !
  THEN ;
: RESET \ Reset processor.
  CCREG @ $D0 OR CCREG !
  $FFFE V@ PCREG ! ;

: CWAI
  ANDCC
  IRQ ;

VARIABLE ---
CREATE REGS --- , XREG , YREG , UREG , SREG , PCREG , --- , --- ,
            AREG , BREG , CCREG , DPREG , --- , --- , --- , --- ,
: REG@ ( c --- n) \  Get value from register c
  DUP IF cells REGS + @ @ ELSE DROP DREG@ THEN ;
: REG! ( n c ---) \ Store n into register c
  DUP IF cells REGS + @ ! ELSE DROP DREG! THEN ;
: EXG
  IMM-BYTE DUP 4 rshift SWAP $0F AND \ Get register numbers.
  2DUP REG@ >R REG@ SWAP REG! R> SWAP REG! ;
: TFR
  IMM-BYTE DUP $0F AND SWAP 4 rshift REG@ SWAP REG! ;

: (BR) ( f ---) \ Perform a conditional branch.
  FLAG @ IF \ Is it a long branch?
   IF  IMM-WORD PCREG +!
   ELSE       2 PCREG +!
   THEN
  ELSE
   IF  IMM-BYTE SIGNED PCREG +!
   ELSE              1 PCREG +!
   THEN
  THEN ;

: NXORV ( --- f) \ Exclusive or of N and V flag, indicating 'less than'
  CCREG @ DUP $08 AND 0<> SWAP $02 AND 0<> XOR ;

: BRA
  TRUE (BR) ;
: BRN
  FALSE (BR) ;
: BHI  \ branch if carry and zero both 0.
  CCREG @ $05 AND 0= (BR) ;
: BLS
  CCREG @ $05 AND (BR) ;
: BCC
  CCREG @ $01 AND 0= (BR) ;
: BCS
  CCREG @ $01 AND (BR) ;
: BNE
  CCREG @ $04 AND 0= (BR) ;
: BEQ
  CCREG @ $04 AND (BR) ;
: BVC
  CCREG @ $02 AND 0= (BR) ;
: BVS
  CCREG @ $02 AND (BR) ;
: BPL
  CCREG @ $08 AND 0= (BR) ;
: BMI
  CCREG @ $08 AND (BR) ;
: BGE
  NXORV 0= (BR) ;
: BLT
  NXORV (BR) ;
: BGT
  NXORV CCREG @ $04 AND OR 0= (BR) ;
: BLE
  NXORV CCREG @ $04 AND OR (BR) ;

: LEAX
  POSTBYTE $ffff and DUP IF CLZ ELSE SEZ THEN
  XREG ! ;
: LEAY
  POSTBYTE $ffff and DUP IF CLZ ELSE SEZ THEN
  YREG ! ;
: LEAS
  POSTBYTE SREG ! ;
: LEAU
  POSTBYTE UREG ! ;

: SWAPUS \ Swap contents of U and S registers.
  UREG @ SREG @ UREG ! SREG ! ;

: PSHS
  IMM-BYTE
  DUP 128 AND IF PCREG @  PSHSWORD THEN
  DUP  64 AND IF UREG  @  PSHSWORD THEN
  DUP  32 AND IF YREG  @  PSHSWORD THEN
  DUP  16 AND IF XREG  @  PSHSWORD THEN
  DUP   8 AND IF DPREG @  PSHSBYTE THEN
  DUP   4 AND IF BREG  @ PSHSBYTE THEN
  DUP   2 AND IF AREG  @ PSHSBYTE THEN
        1 AND IF CCREG @ PSHSBYTE THEN ;

: PULS
  IMM-BYTE
  DUP   1 AND IF PULSBYTE CCREG ! THEN
  DUP   2 AND IF PULSBYTE AREG  ! THEN
  DUP   4 AND IF PULSBYTE BREG  ! THEN
  DUP   8 AND IF PULSBYTE DPREG ! THEN
  DUP  16 AND IF PULSWORD XREG   ! THEN
  DUP  32 AND IF PULSWORD YREG   ! THEN
  DUP  64 AND IF PULSWORD UREG   ! THEN
      128 AND IF PULSWORD PCREG  ! THEN ;

: PSHU
  SWAPUS PSHS SWAPUS ;
: PULU
  SWAPUS PULS SWAPUS ;

: SETSTATUSD ( n1 n2 n3 cy --- n3 ) \ Set flags according to 16bit operation
  IF SEC $8000 ELSE CLC 0 THEN
  \ Start with carry in bit 15.
  OVER >R \ Preserve result.
  XOR XOR XOR $8000 AND \ Xor carry, orerands and result, giving overflow bit.
  IF SEV ELSE CLV THEN
  R> SETNZ16 ;

: ADDD
  DREG@ EADDR16 V@ 2DUP + dup $10000 and SETSTATUSD DREG! ;
: SUBD
  FLAG @ 2 = IF UREG @ $ffff and ELSE DREG@ THEN
  EADDR16 V@  2DUP - dup $10000 and SETSTATUSD
  FLAG @ IF DROP ELSE DREG! THEN ;
: LDD
  EADDR16 V@ SETNZ16 DREG! ;
: STD
  DREG@ SETNZ16 EADDR16 V! ;
: LDX
  EADDR16 V@ SETNZ16  FLAG @ IF YREG ELSE XREG THEN  ! ;
: STX
  FLAG @ IF YREG ELSE XREG THEN  @  SETNZ16 EADDR16 V! ;
: LDU
  EADDR16 V@ SETNZ16  FLAG @ IF SREG ELSE UREG THEN  ! ;
: STU
  FLAG @ IF SREG ELSE UREG THEN  @  SETNZ16 EADDR16 V! ;
: CMPX
  CASE FLAG @
  0 OF XREG ENDOF
  1 OF YREG ENDOF
  2 OF SREG ENDOF
  ENDCASE @ $ffff and
  EADDR16 V@ 2DUP - dup $10000 and SETSTATUSD DROP ;


CREATE INSTRTABLE INSTRTABLE (INSTRTABLE) !
' NEG , ' ??? , ' ??? , ' COM , ' LSR , ' ??? , ' ROR , ' ASR ,
' ASL , ' ROL , ' DEC , ' ??? , ' INC , ' TST , ' JMP , ' CLR ,
' 0FL , ' 1FL , ' NOP , ' SYNC , ' ??? , ' ??? , ' LBRA , ' LBSR ,
' ??? , ' DAA , ' ORCC , ' ??? , ' ANDCC , ' SEX , ' EXG , ' TFR ,
' BRA , ' BRN , ' BHI , ' BLS , ' BCC , ' BCS , ' BNE , ' BEQ ,
' BVC , ' BVS , ' BPL , ' BMI , ' BGE , ' BLT , ' BGT , ' BLE ,
' LEAX , ' LEAY , ' LEAS , ' LEAU , ' PSHS , ' PULS , ' PSHU , ' PULU ,
' ??? , ' RTS , ' ABX , ' RTI , ' CWAI , ' MUL , ' ??? , ' SWI ,
' NEGA , ' ??? , ' ??? , ' COMA , ' LSRA , ' ??? , ' RORA , ' ASRA ,
' ASLA , ' ROLA , ' DECA , ' ??? , ' INCA , ' TSTA , ' ??? , ' CLRA ,
' NEGB , ' ??? , ' ??? , ' COMB , ' LSRB , ' ??? , ' RORB , ' ASRB ,
' ASLB , ' ROLB , ' DECB , ' ??? , ' INCB , ' TSTB , ' ??? , ' CLRB ,
' NEG , ' ??? , ' ??? , ' COM , ' LSR , ' ??? , ' ROR , ' ASR ,
' ASL , ' ROL , ' DEC , ' ??? , ' INC , ' TST , ' JMP , ' CLR ,
' NEG , ' ??? , ' ??? , ' COM , ' LSR , ' ??? , ' ROR , ' ASR ,
' ASL , ' ROL , ' DEC , ' ??? , ' INC , ' TST , ' JMP , ' CLR ,
' SUBA , ' CMPA , ' SBCA , ' SUBD , ' ANDA , ' BITA , ' LDA , ' STA ,
' EORA , ' ADCA , ' ORA , ' ADDA , ' CMPX , ' BSR , ' LDX , ' STX ,
' SUBA , ' CMPA , ' SBCA , ' SUBD , ' ANDA , ' BITA , ' LDA , ' STA ,
' EORA , ' ADCA , ' ORA , ' ADDA , ' CMPX , ' JSR , ' LDX , ' STX ,
' SUBA , ' CMPA , ' SBCA , ' SUBD , ' ANDA , ' BITA , ' LDA , ' STA ,
' EORA , ' ADCA , ' ORA , ' ADDA , ' CMPX , ' JSR , ' LDX , ' STX ,
' SUBA , ' CMPA , ' SBCA , ' SUBD , ' ANDA , ' BITA , ' LDA , ' STA ,
' EORA , ' ADCA , ' ORA , ' ADDA , ' CMPX , ' JSR , ' LDX , ' STX ,
' SUBB , ' CMPB , ' SBCB , ' ADDD , ' ANDB , ' BITB , ' LDB , ' STB ,
' EORB , ' ADCB , ' ORB , ' ADDB , ' LDD , ' STD , ' LDU , ' STU ,
' SUBB , ' CMPB , ' SBCB , ' ADDD , ' ANDB , ' BITB , ' LDB , ' STB ,
' EORB , ' ADCB , ' ORB , ' ADDB , ' LDD , ' STD , ' LDU , ' STU ,
' SUBB , ' CMPB , ' SBCB , ' ADDD , ' ANDB , ' BITB , ' LDB , ' STB ,
' EORB , ' ADCB , ' ORB , ' ADDB , ' LDD , ' STD , ' LDU , ' STU ,
' SUBB , ' CMPB , ' SBCB , ' ADDD , ' ANDB , ' BITB , ' LDB , ' STB ,
' EORB , ' ADCB , ' ORB , ' ADDB , ' LDD , ' STD , ' LDU , ' STU ,

: SINGLE-STEP  \ Perform one instruction.
  IMM-BYTE    \ Get instruction.
  DUP IREG !   \ Store into instruction register for later use.
  cells INSTRTABLE + @ \ Lookup inbstruction in table.
  EXECUTE ;

VARIABLE BPREG \ Breakpoint address.
: BREAKPOINT ( addr --- ) \ Preform instructions until breakpoint.
  BPREG @
  BEGIN
   SINGLE-STEP
   DUP PCREG @ $ffff and =
  UNTIL DROP ;

: FEMIT dup $60 and 0= if drop [char] . then emit ;

: HDIGIT. ( c ---) \ Print hex digit.
  $0F AND DUP 9 > IF 7 + THEN $30 + FEMIT ;

: B. ( c ---) \ Print byte hexadecimal.
  DUP 4 rshift HDIGIT. HDIGIT. ;
: H. ( n ---) \ Print word hexadecimal.
  DUP 8 rshift B. B. 1 SPACES ;
: BIN. ( c ---) \ Print byte binary.
  BASE @ 2 BASE ! SWAP 0 <# # # # # # # # # #> TYPE SPACE BASE ! ;
VARIABLE CURSOR
: SHOWPAGE ( n ---) \ Show page at addr n.
  0 0 AT-XY
  ."        0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F"
  ."  0123456789ABCDEF"
   256 BOUNDS DO
   CR I H. SPACE
   16 0 DO I J + $ffff and DUP CURSOR @ = IF
                    reVERSE VC@ B. -reVERSE \ Type cursor inverse.
                  ELSE VC@ B.
                  THEN
                  1 SPACES
        LOOP
   16 0 DO I J + VC@ FEMIT
        LOOP
  16 +LOOP CR ;

\ 6809 Disassembler.
VARIABLE IP \ Instruction pointer.
VARIABLE INSTR \ Instruction.

: DIRECTDIS \ Disassemble direct address.
  IP @ VC@  1 IP +!  ." $" B. ;
: EXTENDEDDIS \ Disassemble extended address.
  IP @ V@   2 IP +!  ." $" H. ;
: BINDIS \ Disassemble binary argument.
  IP @ VC@  1 IP +!  ." % " BIN. ;
: RELDIS \ Disassemble realtive branch address.
  IP @ VC@  1 IP +!  SIGNED IP @ + ." $" H. ;
: LONGRELDIS \ Disassemble long relative branch.
  IP @ V@   2 IP +!  IP @ + ." $" H. ;
: IMMDIS \ Disassemble immediate operand.
  INSTR @ $8D =
   IF RELDIS \ Exception for BSR instruction.
   ELSE ." # " INSTR @ $0F AND DUP 3 = SWAP $0C AND $0C = OR
     IF \ 16 bits instruction.
        IP @ V@  2 IP +!  ." $" H.
     ELSE \ 8 bits instruction.
        IP @ VC@ 1 IP +!  ." $" B.
     THEN
   THEN ;

VARIABLE 1STREG \ First register to be printed?
: PPREG. ( regnr ---) \ Type register name for PSH and PUL instructions.
  1STREG @ IF 1STREG OFF ELSE ." , " THEN
  CASE
   0 OF ." PC" ENDOF
   1 OF INSTR @ 2 AND IF ." S" ELSE ." U" THEN ENDOF
   2 OF ." Y" ENDOF
   3 OF ." X" ENDOF
   4 OF ." DP" ENDOF
   5 OF ." B" ENDOF
   6 OF ." A" ENDOF
   7 OF ." CC" ENDOF
  ENDCASE ;

: PSHPULDIS \ Disassemble rigister set after PSH and PUL instructions.
  IP @ VC@  1 IP +!
   1STREG ON
   8 0 DO DUP $80 AND IF I PPREG. THEN 2* LOOP DROP ;

: ETREG. ( regnr ---) \ Type register name for TFR and EXG instructions.
  CASE
   0 OF ." D" ENDOF
   1 OF ." X" ENDOF
   2 OF ." Y" ENDOF
   3 OF ." U" ENDOF
   4 OF ." S" ENDOF
   5 OF ." PC" ENDOF
   8 OF ." A" ENDOF
   9 OF ." B" ENDOF
   10 OF ." CC" ENDOF
   11 OF ." DP" ENDOF
     ." ?"
  ENDCASE ;
: EXGTFRDIS \ Disassemble registers after EXG and TFR instructions.
  IP @ VC@  1 IP +!
  DUP 4 rshift ETREG. ." , " $0F AND ETREG. ;

: INDEXREG. \ Type the index register.
  CASE INDEX @
   $00 OF ." X" ENDOF
   $20 OF ." Y" ENDOF
   $40 OF ." U" ENDOF
   $60 OF ." S" ENDOF
  ENDCASE ;

: 16signed ( n --- n2) 
  dup $8000 and if $ffff0000 or then ;

: PBDIS \ Disassemble instructions with postbyte.
  IP @ VC@  1 IP +!
  DUP $60 AND INDEX !
  DUP $80 < IF \ 5-bit format.
   $1F AND DUP $10 AND IF $FFF0 OR THEN 16signed . ." ," INDEXREG.
  ELSE
   DUP $0F AND
   CASE
    0 OF ." ," INDEXREG. ." +" ENDOF
    1 OF ." ," INDEXREG. ." ++" ENDOF
    2 OF ." ,-" INDEXREG. ENDOF
    3 OF ." ,--" INDEXREG. ENDOF
    4 OF ." 0 ," INDEXREG. ENDOF
    5 OF ." B," INDEXREG. ENDOF
    6 OF ." A," INDEXREG. ENDOF
    8 OF IP @ VC@ 1 IP +! SIGNED . ." ," INDEXREG. ENDOF
    9 OF IP @ V@  2 IP +! 16signed . ." ," INDEXREG. ENDOF
   $B OF ." D," INDEXREG. ENDOF
   $C OF RELDIS ." ,PCR" ENDOF
   $D OF LONGRELDIS ." ,PCR" ENDOF
   $F OF EXTENDEDDIS ENDOF
    ." ???"
   ENDCASE
    $10 AND IF ."  []" THEN
  THEN
  ;
: 1ROW \ Disassemble instructions on row 1.
  CASE INSTR @
  $16 OF LONGRELDIS ENDOF
  $17 OF LONGRELDIS ENDOF
  $1A OF BINDIS ENDOF
  $1C OF BINDIS ENDOF
  $1E OF EXGTFRDIS ENDOF
  $1F OF EXGTFRDIS ENDOF  
  ENDCASE ;
: 3ROW \ Disassemble instructions on row 3.
  INSTR @ $34 < IF PBDIS
  ELSE INSTR @ $38 < IF PSHPULDIS
  ELSE INSTR @ $3C = IF BINDIS THEN
  THEN THEN ;

CREATE DISROWS ' DIRECTDIS , ' 1ROW , ' RELDIS , ' 3ROW ,
               ' NOOP ,      ' NOOP , ' PBDIS  , ' EXTENDEDDIS ,
               ' IMMDIS ,    ' DIRECTDIS , ' PBDIS , ' EXTENDEDDIS ,
               ' IMMDIS ,    ' DIRECTDIS , ' PBDIS , ' EXTENDEDDIS ,

: 10DIS ( n ---) \ Disassemble instruction with prebyte $10
  DUP $F0 AND $20 = IF \ Long branch?
    ." L"  cells INSTRTABLE + @ >NAME count $1f and type space
             \ Print name of instr.
    LONGRELDIS
  ELSE DUP $80 < IF DROP ." SWI2"
                 ELSE CASE DUP $4F AND
                        3 OF ." CMPD " ENDOF
                      $0C OF ." CMPY " ENDOF
                      $0E OF ." LDY " ENDOF
                      $0F OF ." STY " ENDOF
                      $4E OF ." LDS " ENDOF
                      $4F OF ." STS " ENDOF
                            ." ??? "
                      ENDCASE
                      DUP INSTR !
                      $F0 AND 4 rshift cells DISROWS + @ EXECUTE
                 THEN
  THEN ;

: 11DIS ( n ---) \ Disassemble instruction with prebyte $11
  DUP $80 < IF DROP ." SWI3" ELSE
   CASE DUP $4F AND
     3 OF ." CMPU " ENDOF
   $0C OF ." CMPS " ENDOF
      ." ??? "
   ENDCASE
   DUP INSTR !
   $F0 AND 4 rshift cells DISROWS + @ EXECUTE
  THEN ;

: (DIS) \ Disassemble instruction at instruction pointer and advance pointer.
  IP @ VC@ 1 IP +! DUP
  $10 = IF DROP IP @ VC@  1 IP +! 10DIS
        ELSE
        DUP $11 = IF DROP IP @  VC@  1 IP +! 11DIS
                  ELSE
                   DUP INSTR !
                   DUP cells INSTRTABLE + @ >NAME count $1f and type space 
                     \ Print name of instr.
                   4 rshift cells DISROWS + @ EXECUTE \ Treat each row seperately.
                  THEN
        THEN ;

VARIABLE PAGE 0 PAGE !
: SHOWSTATUS
  PAGE @ SHOWPAGE
  ." CC=" CCREG @ BIN. ."  A=$" AREG @ B. ."  B=$" BREG @ B.
  ."  DP=$" DPREG @ B. ."  X=$" XREG @ H. ." Y=$" YREG @ H.
  ." U=$" UREG @ H. ." S=$" SREG @ H. CR ."    EFHINZVC PC=$" PCREG @ H.
  PCREG @ IP ! (DIS) 32 SPACES CR CR 80 SPACES 0 20 AT-XY  ;

VARIABLE COMMAND \ Command key, just typed.
VARIABLE NEWPAGE \ Must entire page be shown next?
: GET# ( ---n) \ Get hexadecimal number from user.
  BASE @ HEX QUERY BL WORD number? 2DROP
   SWAP BASE ! NEWPAGE ON ;

: HEXD \ Process hexadecimal digit from keyboard.
  COMMAND @ [char] 0 - DUP 9 > IF 7 - THEN \ Convert key to hex.
  CURSOR @ VC@ 16 * $F0 AND + CURSOR @ VC!
  CURSOR @ $0F AND 54 +
  CURSOR @ PAGE @ - $ffff and 4 rshift 1+ AT-XY CURSOR @ VC@ FEMIT ;
: GO ." Breakpoint: " GET# BPREG ! BREAKPOINT ;
: STEP \ Set breakpoint after next instruction.
  SHOWSTATUS IP @ BPREG ! BREAKPOINT newpage on ;
: SING SINGLE-STEP NEWPAGE ON ;
: DOIRQ PCREG @ BPREG ! IRQ PCREG @ BPREG @ -
  IF BREAKPOINT THEN NEWPAGE ON ;
: DOFIRQ PCREG @ BPREG ! FIRQ PCREG @ BPREG @ -
  IF BREAKPOINT THEN NEWPAGE ON ;
: DONMI PCREG @ BPREG ! NMI BREAKPOINT NEWPAGE ON ;
: DORESET RESET NEWPAGE ON ;

: upc dup [char] a [char] z 1+ within if 32 - then ;
: REG ." Register: " KEY UPC DUP EMIT ."  Value: " GET#
  SWAP CASE
   [char] D OF DPREG ENDOF
   [char] A OF AREG ENDOF
   [char] B OF BREG ENDOF
   [char] C OF CCREG ENDOF
   [char] P OF PCREG ENDOF
   [char] X OF XREG ENDOF
   [char] Y OF YREG ENDOF
   [char] U OF UREG ENDOF
   [char] S OF SREG ENDOF
    ---
  ENDCASE ! ;
create namebuf 50 allot
s" edit-text " namebuf 1+ swap cmove
: PROG \ Make cursor equal to program counter.
  PCREG @ DUP CURSOR ! $FF00 AND PAGE ! NEWPAGE ON ;
: LOAD namebuf count swap 10 + swap included NEWPAGE ON CLS ;
: EDIT namebuf count 10 + evaluate NEWPAGE ON CLS ;
: NAME ." Filename: " namebuf 11 + 39 accept namebuf c! ;
: INST \ Move cursor to next instruction but do not execute.
  IP @ PCREG ! PROG ;
: LOADM \ Load 6809 memory from disk.
  ." Start address: " GET#
  ." Filename: " QUERY VLOAD NEWPAGE ON ;
: WRITEM \ Write 6809 memory to disk.
  ." Start address: " GET# ." Length: " GET#
  ." Filename: " QUERY VSAVE NEWPAGE ON ;
: CURS \ Make program counter equal to cursor location.
  CURSOR @ PCREG ! NEWPAGE ON ;
: HELP CLS
     ." Cursor keys, Home, End, PgUp, PgDn: Move cursor in memory."
  CR ." ^S ^D ^E ^X: Cursor left/right/up/down."
  CR ." ^A ^F ^R ^C: Home, End, PgUp, PgDn."
  CR ." Space     : Move cursor to next location."
  CR ." 0-9,A-F   : Change memory location at cursor position."
  CR ." ?         : Help."
  CR ." G         : Execute until breakpoint."
  CR ." H         : Reset processor."
  CR ." I         : Perform IRQ interrupt."
  CR ." J         : Perform FIRQ interrupt."
  CR ." K         : Perform NMI interrupt."
  CR ." L         : Load memory from disk."
  CR ." N         : Select Assembler file."
  CR ." P         : Set cursor to program counter."
  CR ." Q         : Quit."
  CR ." R         : Change register."
  CR ." S         : Execute with breakpoint after next instruction."
  CR ." T         : Single step."
  CR ." U         : Set program counter after next instruction."
  CR ." W         : Write memory to disk."
  CR ." X         : Set program counter to cursor location."
  CR ." Y         : Assemble the assembler file."
  CR ." Z         : Edit the assembler file." KEY DROP CLS NEWPAGE ON ;
: HOME CURSOR @ $FFF0 AND CURSOR ! ;
: END CURSOR @  $0F OR CURSOR ! ;
: PGDN $100 CURSOR +! $100 PAGE +! NEWPAGE ON ;
: PGUP $-100 CURSOR +! $-100 PAGE +! NEWPAGE ON ;
: ?PD \ Check if page must go down.
  CURSOR @  PAGE @ 255 + - $ffff and 16signed 0> IF $10 PAGE +! NEWPAGE ON THEN ;
: ?PU \ Check if page must go up.
  CURSOR @  PAGE @ - $ffff and 16signed 0< IF $-10 PAGE +! NEWPAGE ON THEN ;

: DOWN $10 CURSOR +! ?PD ;
: UP   $-10 CURSOR +! ?PU ;
: RIGHT 1 CURSOR +! ?PD ;
: LEFT -1 CURSOR +! ?PU ;
: CURCOORDS ( --- x y ) \ Coordinates of cursor.
  CURSOR @ $0F AND 3 * 6 +
  CURSOR @ PAGE @ - $ffff and 4 rshift 1+ ;
: CUROFF
  CURCOORDS AT-XY CURSOR @ VC@ B. 0 20 AT-XY ;
: CURON
  CURCOORDS AT-XY REVERSE CURSOR @ VC@ B. -REVERSE 0 20 AT-XY ;
CREATE KEYTABLE ' HEXD , ' HEXD , ' HEXD , ' HEXD , ' HEXD ,
                ' HEXD , ' HEXD , ' HEXD , ' HEXD , ' HEXD ,
                ' NOOP , ' NOOP , ' NOOP , ' NOOP , ' NOOP ,
                ' HELP , ' NOOP , ' HEXD , ' HEXD , ' HEXD ,
                ' HEXD , ' HEXD , ' HEXD , ' GO   , ' DORESET ,
                ' DOIRQ , ' DOFIRQ , ' DONMI , ' LOADM , ' NOOP ,
                ' NAME , ' NOOP , ' PROG , ' QUIT , ' REG  ,
                ' STEP , ' SING , ' INST , ' NOOP , ' WRITEM ,
                ' CURS , ' LOAD , ' EDIT ,
create curstable ' left , ' right , ' up , ' down ,
                 ' home , ' end , ' pgdn , ' pgup ,
create ctrltable ' noop  , ' home , ' noop , ' pgdn , ' right , ' up , 
                 ' end , ' noop ,
                 ' noop , ' noop , ' noop , ' noop , ' noop , ' noop , ' noop ,
                 ' noop , ' noop , ' noop , ' pgup , ' left , ' noop , ' noop ,
                 ' noop , ' noop , ' down ,


PREVIOUS FORTH DEFINITIONS

: SIMULATE \ The word that starts the simulator.
  [ 6809SIM ]
  CLS NEWPAGE ON FLAG OFF
  BEGIN
   eKEY? 0= NEWPAGE @ AND IF SHOWSTATUS NEWPAGE OFF THEN
   eKEY upc
   CUROFF DUP COMMAND ! 
   dup 25 < if 
    cells ctrltable + @ execute 
   else
   [char] 0 - DUP 44 U<
   IF cells KEYTABLE + @ EXECUTE
   ELSE 48 + k-left - DUP 8 U<
        IF cells CURSTABLE + @ EXECUTE
        ELSE BL k-left - = IF RIGHT THEN
        THEN
   THEN
   THEN
   cursor @ $ffff and cursor !
   page @ $ffff and page !
   CURON
  0 until ;



: DISAS ( addr1 addr2 ---)
  [ 6809SIM ]
  SWAP IP !
  BEGIN
   CR
   IP @
   (DIS)
   20 ?XY drop - SPACES SPACE
   [char] \ EMIT SPACE DUP H. IP @ SWAP DO I VC@ B. SPACE LOOP
   IP @ OVER U> UNTIL
   DROP ;

FORTH

6809sim definitions \ Add IO capability to 6809 simulator.
  \ Leave out if SWI2,SWi3 and SYNC must retain original functions.
: SWI
  FLAG @ CASE
   0 OF SWI ENDOF
   1 OF BREG @ EMIT ENDOF
   2 OF KEY? IF CLC KEY BREG ! 
                       ELSE SEC THEN
     ENDOF
   ENDCASE
;
' SWI INSTRTABLE $3F cells + ! \ Modify SWI instruction such that SWI2 means

: SYNC 7 emit quit ;

' SYNC INSTRTABLE $13 cells + ! 

                            \ EMIT and SWI3 means KEY.
forth definitions
