/*
 * pthread_dce.c
 *
 * A set of DCE/POSIX draft 4 wrapper routines that call Linux/POSIX .1c
 * threads.
 */
#include <errno.h>
#include "pthread_dce.h"

int
pthd4_attr_create( pthread_attr_t *attr )
{
	int istat;

	if( pthread_attr_init( attr ) != SUCCESS )
		return( FAILURE );

	/*
	 * Draft 4/DCE threads are created as joinable threads.  POSIX threads
	 * are not.
	 */
	istat = pthread_attr_setdetachstate( attr, PTHREAD_CREATE_JOINABLE );
	if( istat != SUCCESS )
		return( FAILURE );

	return( SUCCESS );
}

int 
pthd4_attr_delete ( pthread_attr_t *attr)
{
    if( pthread_attr_destroy( attr ) != SUCCESS )
		return( FAILURE );

    return( SUCCESS );
}

int 
pthd4_attr_setstacksize( pthread_attr_t *attr, long stacksize )
{
    int istat = 0;

    istat = pthread_attr_setstacksize( attr, (size_t)stacksize );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

long
pthd4_attr_getstacksize( pthread_attr_t attr)
{
	size_t stacksize;

    if( pthread_attr_getstacksize( &attr, &stacksize ) != SUCCESS )
		return( FAILURE );

    return( (long) stacksize );
}

int 
pthd4_create( pthread_t *thread,
              pthread_attr_t attr,
              pthread_startroutine_t start_routine,
              pthread_addr_t arg )
{
    int istat = 0, cleanup = 0;

	/*
	 * The default draft 4 behavior is to create joinable threads.  The
	 * new POSIX standard requires that new threads be created detached.
	 * So, if the caller has passed us a default attributes object,
	 * initialize the attributes object as if this were a draft 4 thread.
	 */
    if ( attr == pthread_attr_default )
    {
		if( pthread_attr_init( &attr ) != SUCCESS )
			return( FAILURE );

        istat = pthread_attr_setdetachstate( &attr, PTHREAD_CREATE_JOINABLE );
        if ( istat > 0 )
            return( FAILURE );
		cleanup = TRUE;
    }    

    istat = pthread_create( thread, 
						    &attr, 
						    (thread_proc_t) start_routine, 
						    arg );

	if( istat != SUCCESS )
	{
		if( cleanup )
		{
			pthread_attr_destroy( &attr );
			attr = pthread_attr_default;
		}

		return( FAILURE );
	}

	if( cleanup )
	{
		pthread_attr_destroy( &attr );
		attr = pthread_attr_default;
	}

    return( istat );
}

int 
pthd4_detach( pthread_t *thread )
{
    int istat = 0;

    istat = pthread_detach( *thread );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

void
pthd4_exit( pthread_addr_t status )
{
    pthread_exit( &status );
}

int 
pthd4_join( pthread_t thread, pthread_addr_t *status )
{
    int istat = 0;

    istat = pthread_join( thread, (void **)&status );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

pthread_t 
pthd4_self( void )
{
    return( pthread_self() );
}

int 
pthd4_mutex_init( pthread_mutex_t *mutex, pthread_mutexattr_t attr )
{
    int istat = 0;

    istat = pthread_mutex_init( mutex, &attr );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_mutex_destroy( pthread_mutex_t *mutex )
{
    int istat = 0;

    istat = pthread_mutex_destroy( mutex );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int
pthd4_mutex_lock( pthread_mutex_t *mutex )
{
    int istat = 0;

    istat = pthread_mutex_lock( mutex );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_mutex_trylock( pthread_mutex_t *mutex )
{
    int istat = 0;

    istat = pthread_mutex_trylock( mutex );  
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_mutex_unlock( pthread_mutex_t *mutex )
{
    int istat = 0;

    istat = pthread_mutex_unlock( mutex );  
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cond_init( pthread_cond_t *cond, pthread_condattr_t attr )
{
    int istat = 0;

    istat = pthread_cond_init( cond, &attr );                     
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cond_destroy( pthread_cond_t *cond )
{
    int istat = 0;

    istat = pthread_cond_destroy( cond );  
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cond_broadcast( pthread_cond_t *cond )
{
    int istat = 0;

    istat = pthread_cond_broadcast( cond );  
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cond_signal( pthread_cond_t *cond )
{
    int istat = 0;

    istat = pthread_cond_signal( cond );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cond_wait( pthread_cond_t *cond, pthread_mutex_t *mutex )
{
    int istat = 0;

    istat = pthread_cond_wait( cond, mutex );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cond_timedwait( pthread_cond_t *cond,
                      pthread_mutex_t *mutex,
                      struct timespec *abstime )
{
    int istat = 0;

    istat = pthread_cond_timedwait( cond, mutex, abstime );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_once( pthread_once_t *once_block, void (*init_routine)(void) )
{
    int istat = 0;

    istat = pthread_once( once_block, init_routine );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_keycreate( pthread_key_t *key, pthread_destructor_t destructor )
{
    int istat = 0;

    istat = pthread_key_create( key, destructor );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_setspecific( pthread_key_t key, pthread_addr_t value )
{
    int istat = 0;

    istat = pthread_setspecific( key, value );  
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_getspecific( pthread_key_t key, pthread_addr_t *value )
{
    int istat = 0;

    istat = pthread_getspecific( key, value );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_cancel( pthread_t thread )
{
    int istat = 0;

    istat = pthread_cancel( thread );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

void 
pthd4_testcancel( void )
{
    pthread_testcancel();
}

int 
pthd4_setasynccancel( int state )
{
    int istat = 0;
    int old_type = 0;
    int new_type = 0;

    if ( state == 0 )
        new_type = PTHREAD_CANCEL_DEFERRED;
    else
        new_type = PTHREAD_CANCEL_ASYNCHRONOUS;

    istat = pthread_setcanceltype( new_type, &old_type );
    if ( istat > 0 )
        return( -1 );
    else
        return( old_type );
}

int 
pthd4_setcancel( int state )
{
    int istat = 0;
    int prev_state = 0;
    int new_state = 0;

    if ( state == 0 )
        new_state = PTHREAD_CANCEL_DISABLE;
    else
        new_state = PTHREAD_CANCEL_ENABLE;

    istat = pthread_setcancelstate( new_state, &prev_state );
    if ( istat > 0 )
        return( -1 );
    else
        return( prev_state );
}

int 
pthd4_get_expiration_np( struct timespec *delta, struct timespec *abstime )
{
    int istat = 0;

    istat = pthread_get_expiration_np( delta, abstime );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

int 
pthd4_delay_np( struct timespec *interval )
{
    int istat = 0;

    istat = pthread_delay_np( interval );
    if ( istat > 0 )
        istat = -1;

    return( istat );
}

